/* ************************************************************************
*   File: act.comm.c                                    Part of CircleMUD *
*  Usage: Player-level communication commands                             *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"


#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "handler.h"
#include "db.h"
#include "screen.h"

/* extern variables */
extern struct room_data *world;
extern struct descriptor_data *descriptor_list;
extern struct char_data *character_list;
extern struct clan_stuff clans[];
extern sh_int r_frozen_start_room;
 
/* external functions */
int get_from_q(struct txt_q * queue, char *dest, int *aliased);
int perform_alias(struct descriptor_data * d, char *orig);
int is_ignoring(struct char_data *ch, long idnum);
struct char_data *get_social(struct char_data *ch, char *result, char *argument);
void drunk_speak(char *);

ACMD(do_ignore)
{
 long ig_id;
 struct ignore_type *ignoring;

  one_argument(argument, arg);

  if (!*argument) {
   sprintf(buf, "You are ignoring:");
    if (ch->ignore) {
      for (ignoring = ch->ignore; ignoring; ignoring = ignoring->next)
       sprintf(buf, "%s %s", buf, get_name_by_id(ignoring->idnum));
    } else strcat(buf, "no-one");
   strcat(buf, "\r\n"); 
   send_to_char(buf, ch);
   return;
  }
  /* Assume there is an argument... duh! */
  if ((ig_id = get_id_by_name(arg)) <= 0) {
   send_to_char("No one by that name.\r\n", ch);
   return;
  }
  if (GET_IDNUM(ch) == ig_id) {
   send_to_char("Ignoring yourself... a bit redundant, aren't we?\r\n", ch); 
   return;
  }
  /* Assume we have a valid char to ignore */
  if (is_ignoring(ch, ig_id)) { /* delete ignore */
    for (ignoring = ch->ignore; ignoring; ignoring = ignoring->next) {
      if (ignoring->idnum == ig_id) {
         if (ignoring->prev)
           ignoring->prev->next = ignoring->next;
         if (ignoring->next)
           ignoring->next->prev = ignoring->prev;
         if (ignoring == ch->ignore)
           ch->ignore = NULL;
         free(ignoring);
         sprintf(buf, "You are no longer ignoring %s.\r\n", get_name_by_id(ig_id));
         send_to_char(buf, ch);
         return;
      } /* if */
    } /* for */
  }
  /* Okie, it's ignoring time... */
  CREATE(ignoring, struct ignore_type, 1);
  ignoring->idnum = ig_id;
  ignoring->next = ch->ignore;
  ignoring->prev = NULL;
  if (ch->ignore) 
    ch->ignore->prev = ignoring;
  ch->ignore = ignoring;
  sprintf(buf, "You are now ignoring %s.\r\n", get_name_by_id(ig_id));
  send_to_char(buf, ch);
}

int is_ignoring(struct char_data *ch, long idnum) 
{
  struct ignore_type *ignoring;
  
  if (!ch->ignore) 
    return FALSE;
  for (ignoring = ch->ignore; ignoring; ignoring = ignoring->next)
    if (ignoring->idnum == idnum)
      return TRUE;
  return FALSE;
}

ACMD(do_say)
{
  skip_spaces(&argument);

  if (!*argument)
    send_to_char("Yes, but WHAT do you want to say?\r\n", ch);
  else {
    if (!IS_NPC(ch) && GET_COND(ch, DRUNK) > 0) {
      drunk_speak(argument);
    }
    sprintf(buf, "$n says, '%s&+w'", argument);
    act(buf, FALSE, ch, 0, 0, TO_ROOM);
    if (PRF_FLAGGED(ch, PRF_NOREPEAT))
      send_to_char(OK, ch);
    else {
      sprintf(buf, "You say, '%s&+w'", argument);
      act(buf, FALSE, ch, 0, argument, TO_CHAR);
    }
  }
}


ACMD(do_gossip)
{
  struct descriptor_data *d;
  byte gmote = FALSE;

  ACMD(do_gmote);

  delete_doubledollar(argument);
  half_chop(argument, buf, buf1);
  skip_spaces(&argument);
  if (!ch->desc)
    return;
  if (PRF_FLAGGED(ch, PRF_NOGOSS)) {
    send_to_char("You aren't even on that channel!\r\n", ch);
    return;
  }
  if (PLR_FLAGGED(ch, PLR_NOSHOUT)) {
    send_to_char("You can't do that!\r\n", ch);
    return;
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_SOUNDPROOF)) {
    send_to_char("The walls absorb your words.\r\n", ch);
    return;
  }

  if(subcmd == SCMD_GMOTE || (subcmd == SCMD_GOSSIP && *argument == '@')) {
    subcmd = SCMD_GOSSIP;
    gmote = TRUE;
  }

  if (!*argument)  {
    send_to_char("Yes, but WHAT do you want to gossip?\r\n", ch);
    return; 
  }

  if (gmote) {
    if (*argument == '@')
      do_gmote(ch, argument + 1, 0, 1);
    else
      do_gmote(ch, argument, 0, 1);
    return;
  }

  if (is_abbrev(buf, "-emote") || is_abbrev(buf, "-possessive")) {
    if (!*buf1)  {
      send_to_char("You have to emote *something*..\r\n", ch);
      return; 
    }
    if (is_abbrev(buf, "-emote")) {
      sprintf(buf, "[gossip] %s %s&+y\r\n", GET_NAME(ch), buf1);
      sprintf(buf2, "[gossip] Someone %s&+y\r\n", buf1);
    }
    if (is_abbrev(buf, "-possessive")) {
      sprintf(buf, "[gossip] %s's %s&+y\r\n", GET_NAME(ch), buf1);
      sprintf(buf2, "[gossip] Someone's %s&+y\r\n", buf1);
    }
    for (d=descriptor_list; d; d=d->next) {
      if ((!d->connected) && 
          (!ROOM_FLAGGED(d->character->in_room, ROOM_SOUNDPROOF)) &&
          (!(PLR_FLAGGED(d->character, PLR_WRITING | PLR_MAILING) /*&& 
           PRF_FLAGGED(d->character, PRF_CONCENTRATE)*/)) &&
          (d != ch->desc || (!PRF_FLAGGED(d->character, PRF_NOREPEAT)))) 
        if (!PRF_FLAGGED(d->character, PRF_NOGOSS) && 
            !is_ignoring(d->character, GET_IDNUM(ch))) {
           send_to_char(CCYEL(d->character, C_NRM), d->character);
           if (CAN_SEE(d->character, ch))
             send_to_char(buf, d->character);
           else
             send_to_char(buf2, d->character);
           send_to_char(CCNRM(d->character, C_NRM), d->character);
        }
   }
    return;
  }
  if (GET_COND(ch, DRUNK) > 0) {
     drunk_speak(argument);
  }
  for (d=descriptor_list; d; d=d->next) {
   if(ch==d->character) {
    sprintf(buf, "You gossip, '%s&+y'\r\n", argument);
    send_to_char((CCYEL(ch, C_NRM)), ch);
    send_to_char(buf, ch);
    send_to_char((CCNRM(ch, C_NRM)), ch);
    continue;
  }
  sprintf(buf, "%s gossips, '%s&+y'\r\n", GET_NAME(ch), argument);
  sprintf(buf2, "Someone gossips, '%s&+y'\r\n", argument);
  if ((!d->connected) && 
      (!ROOM_FLAGGED(d->character->in_room, ROOM_SOUNDPROOF)) &&
      (!(PLR_FLAGGED(d->character, PLR_WRITING | PLR_MAILING) /*&&
       PRF_FLAGGED(d->character, PRF_CONCENTRATE)*/)) &&
      (d != ch->desc || (!PRF_FLAGGED(d->character, PRF_NOREPEAT)))) 
      if (!PRF_FLAGGED(d->character, PRF_NOGOSS) && 
      !is_ignoring(d->character, GET_IDNUM(ch))) {
         send_to_char(CCYEL(d->character, C_NRM), d->character);
         if (CAN_SEE(d->character, ch))
           send_to_char(buf, d->character);
         else
           send_to_char(buf2, d->character);
         send_to_char(CCNRM(d->character, C_NRM), d->character);
   }   }
}


ACMD(do_gsay)
{
  struct char_data *k;
  struct follow_type *f;
 
  skip_spaces(&argument);
 
  if (!IS_AFFECTED(ch, AFF_GROUP)) {
    send_to_char("But you are not the member of a group!\r\n", ch);
    return;
  }
  if (!*argument)
    send_to_char("Yes, but WHAT do you want to group-say?\r\n", ch);
  else {
    if (ch->master)
      k = ch->master;
    else
      k = ch;
 
    sprintf(buf, "&+m$n tells the group, '%s&+m'", argument);
 
    if (IS_AFFECTED(k, AFF_GROUP) && (k != ch))
      act(buf, FALSE, ch, 0, k, TO_VICT | TO_SLEEP);
    for (f = k->followers; f; f = f->next)
      if (IS_AFFECTED(f->follower, AFF_GROUP) && (f->follower != ch))
        act(buf, FALSE, ch, 0, f->follower, TO_VICT | TO_SLEEP);
    if (PRF_FLAGGED(ch, PRF_NOREPEAT))
      send_to_char(OK, ch);
    else {
      sprintf(buf, "&+mYou tell the group, '%s&+m'", argument);
      act(buf, FALSE, ch, 0, 0, TO_CHAR | TO_SLEEP);
    }
  }
}


void perform_tell(struct char_data *ch, struct char_data *vict, char *arg)
{
  if (vict == NULL)
    return;

  if (GET_COND(ch, DRUNK) > 0)
    drunk_speak(arg);
  send_to_char(CCRED(vict, C_NRM), vict);
  sprintf(buf, "$n &+rtells you, '%s&+r'", arg);
  act(buf, FALSE, ch, 0, vict, TO_VICT | TO_SLEEP);
  send_to_char(CCNRM(vict, C_NRM), vict);

  if (PRF_FLAGGED(ch, PRF_NOREPEAT))
    send_to_char(OK, ch);
  else {
    send_to_char(CCRED(ch, C_CMP), ch);
    sprintf(buf, "You tell $N, '%s&+r'", arg);
    act(buf, FALSE, ch, 0, vict, TO_CHAR | TO_SLEEP);
    send_to_char(CCNRM(ch, C_CMP), ch);
  }

  GET_LAST_TELL(vict) = GET_IDNUM(ch);

}

/*
 * Yes, do_tell probably could be combined with whisper and ask, but
 * called frequently, and should IMHO be kept as tight as possible.
 */
ACMD(do_tell)
{
  struct char_data *vict;

  half_chop(argument, buf, buf2);

  if (!*buf || !*buf2) {
    send_to_char("Who do you wish to tell what??\r\n", ch);
    return;
  }
  
  if (!(vict = get_player_vis(ch, buf, 0)))
    vict = get_char_vis(ch, buf);
  if (!vict) {
      send_to_char(NOPERSON, ch); 
      return;
    }
  if (ch == vict)
    send_to_char("You try to tell yourself something.\r\n", ch);
  else if (!IS_NPC(ch) && !IS_NPC(vict) && is_ignoring(vict, GET_IDNUM(ch))) 
    send_to_char("Sorry.  That person is ignoring you.",ch);
  else if (is_ignoring(ch, GET_IDNUM(vict)))
    act("You are ignoring $M!", FALSE, ch, 0, vict, TO_CHAR | TO_SLEEP);
  else if (PRF_FLAGGED(ch, PRF_NOTELL))
    send_to_char("You can't tell other people while you have notell on.\r\n", ch);
  else if (ROOM_FLAGGED(ch->in_room, ROOM_SOUNDPROOF) && (ch->in_room != vict->in_room))
    send_to_char("The walls seem to absorb your words.\r\n", ch);
  else if (!IS_NPC(vict) && !vict->desc)	/* linkless */
    act("$E's linkless at the moment.", FALSE, ch, 0, vict, TO_CHAR | TO_SLEEP);
  else if (PLR_FLAGGED(vict, PLR_WRITING)/* && PRF_FLAGGED(vict, 
PRF_CONCENTRATE)*/)
    act("$E's writing a message right now; try again later.",
	FALSE, ch, 0, vict, TO_CHAR | TO_SLEEP);
  else if (PRF_FLAGGED(vict, PRF_NOTELL) || (ROOM_FLAGGED(vict->in_room, 
    ROOM_SOUNDPROOF) && (ch->in_room != vict->in_room)))
    act("$E can't hear you.", FALSE, ch, 0, vict, TO_CHAR | TO_SLEEP);
  else
    perform_tell(ch, vict, buf2);
}


ACMD(do_reply)
{
  struct char_data *tch = character_list;

  skip_spaces(&argument);

  if (GET_LAST_TELL(ch) == NOBODY)
    send_to_char("You have no-one to reply to!\r\n", ch);
  else if (!*argument)
    send_to_char("What is your reply?\r\n", ch);
  else {
    /*
     * Make sure the person you're replying to is still playing by searching
     * for them.  Note, now last tell is stored as player IDnum instead of
     * a pointer, which is much better because it's safer, plus will still
     * work if someone logs out and back in again.
     */
				     
    while (tch != NULL && GET_IDNUM(tch) != GET_LAST_TELL(ch))
      tch = tch->next;

    if (tch == NULL)
      send_to_char("They are no longer playing.\r\n", ch);
    else
     if (PRF_FLAGGED(tch, PRF_NOTELL) || ROOM_FLAGGED(tch->in_room, 
         ROOM_SOUNDPROOF) || ROOM_FLAGGED(ch->in_room, ROOM_SOUNDPROOF))
      act("$E can't hear you.", FALSE, ch, 0, tch, TO_CHAR | TO_SLEEP);
     else
      perform_tell(ch, tch, argument);
  }
}


ACMD(do_spec_comm)
{
  struct char_data *vict;
  char *action_sing, *action_plur, *action_others;

  if (subcmd == SCMD_WHISPER) {
    action_sing = "whisper to";
    action_plur = "whispers to";
    action_others = "$n whispers something to $N.";
  } else {
    action_sing = "ask";
    action_plur = "asks";
    action_others = "$n asks $N a question.";
  }

  half_chop(argument, buf, buf2);

  if (!*buf || !*buf2) {
    sprintf(buf, "Whom do you want to %s.. and what??\r\n", action_sing);
    send_to_char(buf, ch);
  } else if (!(vict = get_char_room_vis(ch, buf)))
    send_to_char(NOPERSON, ch);
  else if (vict == ch)
    send_to_char("You can't get your mouth close enough to your ear...\r\n", ch);
  else {
    if (GET_COND(ch, DRUNK) > 0) {
      drunk_speak(buf2);
    }
    sprintf(buf, "$n %s you, '%s&+w'", action_plur, buf2);
    act(buf, FALSE, ch, 0, vict, TO_VICT);
    if (PRF_FLAGGED(ch, PRF_NOREPEAT))
      send_to_char(OK, ch);
    else {
      sprintf(buf, "You %s %s, '%s&+w'\r\n", action_sing, GET_NAME(vict), buf2);
      act(buf, FALSE, ch, 0, 0, TO_CHAR);
    }
    act(action_others, FALSE, ch, 0, vict, TO_NOTVICT);
  }
}



#define MAX_NOTE_LENGTH 1000	/* arbitrary */

ACMD(do_write)
{
  struct obj_data *paper = 0, *pen = 0;
  char *papername, *penname;

  papername = buf1;
  penname = buf2;

  two_arguments(argument, papername, penname);

  if (!ch->desc)
    return;

  if (!*papername) {		/* nothing was delivered */
    send_to_char("Write?  With what?  ON what?  What are you trying to do?!?\r\n", ch);
    return;
  }
  if (*penname) {		/* there were two arguments */
    if (!(paper = get_obj_in_list_vis(ch, papername, ch->carrying))) {
      sprintf(buf, "You have no %s.\r\n", papername);
      send_to_char(buf, ch);
      return;
    }
    if (!(pen = get_obj_in_list_vis(ch, penname, ch->carrying))) {
      sprintf(buf, "You have no %s.\r\n", penname);
      send_to_char(buf, ch);
      return;
    }
  } else {		/* there was one arg.. let's see what we can find */
    if (!(paper = get_obj_in_list_vis(ch, papername, ch->carrying))) {
      sprintf(buf, "There is no %s in your inventory.\r\n", papername);
      send_to_char(buf, ch);
      return;
    }
    if (GET_OBJ_TYPE(paper) == ITEM_PEN) {	/* oops, a pen.. */
      pen = paper;
      paper = 0;
    } else if (GET_OBJ_TYPE(paper) != ITEM_NOTE) {
      send_to_char("That thing has nothing to do with writing.\r\n", ch);
      return;
    }
    /* One object was found.. now for the other one. */
    if (!GET_EQ(ch, WEAR_HOLD)) {
      sprintf(buf, "You can't write with %s %s alone.\r\n", AN(papername),
	      papername);
      send_to_char(buf, ch);
      return;
    }
    if (!CAN_SEE_OBJ(ch, GET_EQ(ch, WEAR_HOLD))) {
      send_to_char("The stuff in your hand is invisible!  Yeech!!\r\n", ch);
      return;
    }
    if (pen)
      paper = GET_EQ(ch, WEAR_HOLD);
    else
      pen = GET_EQ(ch, WEAR_HOLD);
  }


  /* ok.. now let's see what kind of stuff we've found */
  if (GET_OBJ_TYPE(pen) != ITEM_PEN)
    act("$p is no good for writing with.", FALSE, ch, pen, 0, TO_CHAR);
  else if (GET_OBJ_TYPE(paper) != ITEM_NOTE)
    act("You can't write on $p.", FALSE, ch, paper, 0, TO_CHAR);
  else if (paper->action_description)
    send_to_char("There's something written on it already.\r\n", ch);
  else {
    /* we can write - hooray! */
    SET_BIT(PLR_FLAGS(ch), PLR_WRITING);
    /*impeditmod*/
    ch->desc->backstr = NULL;
    send_to_char("Write your note.  (-s saves -h for help)\r\n", ch);
    if (paper->action_description) {
      ch->desc->backstr = str_dup(paper->action_description);
      send_to_char(paper->action_description, ch);
    }
    act("$n begins to jot down a note.", TRUE, ch, 0, 0, TO_ROOM);
    ch->desc->str = &paper->action_description;
    ch->desc->max_str = MAX_NOTE_LENGTH;
  }
}



ACMD(do_page)
{
  struct descriptor_data *d;
  struct char_data *vict;

  half_chop(argument, arg, buf2);

  if (IS_NPC(ch))
    send_to_char("Monsters can't page.. go away.\r\n", ch);
  else if (!*arg)
    send_to_char("Whom do you wish to page?\r\n", ch);
  else {
    sprintf(buf, "\007\007&+W*&+R%s&+W* &+M%s\r\n", GET_NAME(ch), buf2);
    if (!str_cmp(arg, "all")) {
      if (GET_LEVEL(ch) > LVL_SIMP) {
	for (d = descriptor_list; d; d = d->next)
	  if (!d->connected && d->character)
	    act(buf, FALSE, ch, 0, d->character, TO_VICT);
      } else
	send_to_char("You will never be godly enough to do that!\r\n", ch);
      return;
    }
    if ((vict = get_char_vis(ch, arg)) != NULL) {
      if (PRF_FLAGGED(vict, PRF_NOPAGE) && !IS_IMMORTAL(ch)) {
        send_to_char("Sorry, that person cannot be paged.\r\n", ch);
        return; }
      if (is_ignoring(ch, GET_IDNUM(vict)) || is_ignoring(vict, GET_IDNUM(vict))) {
        send_to_char("Your page cannot reach through the IGNORE barrier.\r\n", ch);
        return;
      }
      if (ROOM_FLAGGED(vict->in_room, ROOM_SOUNDPROOF)) {
        send_to_char("The walls absorb your words.\r\n", ch);
        return;
      }
      if (ROOM_FLAGGED(ch->in_room, ROOM_SOUNDPROOF)) {
        send_to_char("The walls absorb your words.\r\n", ch);
        return;
      }
      
      act(buf, FALSE, ch, 0, vict, TO_VICT);
      if (PRF_FLAGGED(ch, PRF_NOREPEAT))
	send_to_char(OK, ch);
      else
	act(buf, FALSE, ch, 0, vict, TO_CHAR);
      return;
    } else
      send_to_char("There is no such person in the game!\r\n", ch);
  }
}


/**********************************************************************
 * generalized communication func, originally by Fred C. Merkel (Torg) *
  *********************************************************************/

ACMD(do_gen_comm)
{
  extern int level_can_shout;

  struct descriptor_data *i;
  struct char_data *vict = NULL;
  int emote = FALSE;
  char color_on[24], message[MAX_INPUT_LENGTH], socmsg[MAX_INPUT_LENGTH];

  /* Array of flags which must _not_ be set in order for comm to be heard */
  static int channels[] = {
    0,
    PRF_DEAF,
    PRF_NOGOSS,
    PRF_NOAUCT,
    PRF_NOGRATZ,
    0,
    0
  };

  /*
   * com_msgs: [0] Message if you can't perform the action because of noshout
   *           [1] name of the action
   *           [2] message if you're not on the channel
   *           [3] a color string.
   */
  static char *com_msgs[][4] = {
    {"You cannot holler!!\r\n",
      "holler",
      "",
    "&+y"},

    {"You cannot shout!!\r\n",
      "shout",
      "Turn off your noshout flag first!\r\n",
    "&+y"},

    {"You cannot gossip!!\r\n",
      "gossip",
      "You aren't even on the channel!\r\n",
    "&+y"},

    {"You cannot auction!!\r\n",
      "auction",
      "You aren't even on the channel!\r\n",
    "&+m"},

    {"You cannot congratulate!\r\n",
      "congrat",
      "You aren't even on the channel!\r\n",
    "&+g"},
    
    { "", /* This is a blank entry for gemote? */
      "",
      "",
      ""},
      
    {"You cannot Private-Say!!\r\n",
      "private-say",
      "You aren't even on the channel!\r\n",
    "&+R"},
  };

  /* to keep pets, etc from being ordered to shout */
  if (!ch->desc)
    return;

  if (PLR_FLAGGED(ch, PLR_NOSHOUT) || ch->in_room == r_frozen_start_room) {
    send_to_char(com_msgs[subcmd][0], ch);
    return;
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_SOUNDPROOF) && subcmd != SCMD_HOLLER) {
    send_to_char("The walls seem to absorb your words.\r\n", ch);
    return;
  }
  /* level_can_shout defined in config.c */
  if (GET_LEVEL(ch) < level_can_shout) {
    sprintf(buf1, "You must be at least level %d before you can %s.\r\n",
	    level_can_shout, com_msgs[subcmd][1]);
    send_to_char(buf1, ch);
    return;
  }
  /* make sure the char is on the channel */
  if (PRF_FLAGGED(ch, channels[subcmd])) {
    send_to_char(com_msgs[subcmd][2], ch);
    return;
  }
  
  if (!GET_PCHAN(ch) && subcmd == SCMD_PSAY) {
    send_to_char("You are not on a private channel.\r\n", ch);
    return;
  }
  /* skip leading spaces */
  skip_spaces(&argument);

  half_chop(argument, arg, message);
  
  /* make sure that there is something there to say! */
  if (!*argument) {
    sprintf(buf1, "Yes, %s, fine, %s we must, but WHAT???\r\n",
	    com_msgs[subcmd][1], com_msgs[subcmd][1]);
    send_to_char(buf1, ch);
    return;
  }
  if (subcmd == SCMD_HOLLER) {
    if ((GET_MOVE(ch) / 2)<= 1) {
      send_to_char("You're too exhausted to holler.\r\n", ch);
      return;
    } else
      GET_MOVE(ch) /= 2;
      if (GET_LEVEL(ch) >= LVL_IMMORT)
       GET_LEVEL(ch)--;
  }
  /* set up the color on code */
  strcpy(color_on, com_msgs[subcmd][3]);

  /* first, set up strings to be given to the communicator */
  
  if (is_abbrev(arg, "-emote") && *message) {
    emote = TRUE;
    sprintf(buf, "%s[%s] $n %s&+w", color_on, com_msgs[subcmd][1], message);
  } else if (*arg == '@') {
    emote = TRUE;
    vict = get_social(ch, socmsg, argument+1);
    if (socmsg && *socmsg)
      sprintf(buf, "%s[%s] %s&+w", color_on, com_msgs[subcmd][1], socmsg);
    else {
      send_to_char("Error parsing social!\r\n", ch);
      return;
    }
  }             
    
  
  if (PRF_FLAGGED(ch, PRF_NOREPEAT))
    send_to_char(OK, ch);
  else {
    if (!emote)
      sprintf(buf1, "%sYou %s, '%s%s'&+w", color_on, com_msgs[subcmd][1],
	      argument, color_on);
    else
      sprintf(buf1, "%s", buf);
    act(buf1, FALSE, ch, 0, vict, TO_CHAR | TO_SLEEP);
  }

  if (!emote) {
    if (GET_COND(ch, DRUNK) > 0)
      drunk_speak(argument);
    sprintf(buf, "%s$n %ss, '%s%s'&+w", color_on, com_msgs[subcmd][1], argument,
            color_on);
  }

  /* now send all the strings out */
  for (i = descriptor_list; i; i = i->next) {
    if (!i->connected && (i != ch->desc) && i->character &&
	!PRF_FLAGGED(i->character, channels[subcmd]) &&
	!(PLR_FLAGGED(i->character, PLR_WRITING) /*&&
          PRF_FLAGGED(i->character, PRF_CONCENTRATE)*/) &&
	(!ROOM_FLAGGED(i->character->in_room, ROOM_SOUNDPROOF)) &&
        !is_ignoring(i->character, GET_IDNUM(ch))) {

      if (subcmd == SCMD_PSAY && GET_PCHAN(ch) != GET_PCHAN(i->character))
        continue;
      if (!GET_PCHAN(ch) && subcmd == SCMD_PSAY)
        continue;
        
      if (subcmd == SCMD_SHOUT &&
	  ((world[ch->in_room].zone != world[i->character->in_room].zone) ||
	   GET_POS(i->character) < POS_RESTING))
	continue;

      perform_act(buf, ch, 0, vict, i->character);
    }
  }
}


ACMD(do_qcomm)
{
  struct descriptor_data *i;

  if (!PRF_FLAGGED(ch, PRF_QUEST)) {
    send_to_char("You aren't even part of the quest!\r\n", ch);
    return;
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_SOUNDPROOF)) {
     send_to_char("The walls absorb your words.\r\n", ch);
     return;
  }
  skip_spaces(&argument);

  if (!*argument) {
    sprintf(buf, "%s?  Yes, fine, %s we must, but WHAT??\r\n", CMD_NAME,
	    CMD_NAME);
    CAP(buf);
    send_to_char(buf, ch);
  } else {
    if (PRF_FLAGGED(ch, PRF_NOREPEAT))
      send_to_char(OK, ch);
    else {
      if (subcmd == SCMD_QSAY) {
        half_chop(argument, arg, buf1);
        if (is_abbrev(arg, "-emote"))
          sprintf(buf, "[quest-say] %s %s", GET_NAME(ch), buf1);
        else
  	  sprintf(buf, "You quest-say, '%s&+w'", argument);
      }
      else
	strcpy(buf, argument);
      act(buf, FALSE, ch, 0, argument, TO_CHAR);
    }

    if (subcmd == SCMD_QSAY)
      sprintf(buf, "$n quest-says, '%s&+w'", argument);
    else
      strcpy(buf, argument);

    for (i = descriptor_list; i; i = i->next)
      if (!i->connected && i != ch->desc &&
	  PRF_FLAGGED(i->character, PRF_QUEST) &&
          !ROOM_FLAGGED(i->character->in_room, ROOM_SOUNDPROOF))
	act(buf, 0, ch, 0, i->character, TO_VICT | TO_SLEEP);
  }
}
ACMD(do_clan)
{
  struct descriptor_data *d;
  
  skip_spaces(&argument);

  if (!GET_CLAN(ch)) {
    send_to_char("You aren't in a clan.\r\n", ch);
    return;
  }
  if (PRF_FLAGGED(ch, PRF_NOCLAN)) {
    send_to_char("You aren't connected to the clan channel.\r\n", ch);
    return;
  }
  if (!*argument) {
    send_to_char("Clan-say what?!?\r\n", ch);
    return;
  }
   one_argument(argument, arg);
  if (is_abbrev(arg, "-emote")) {
    half_chop(argument, arg, argument);
    sprintf(buf1, "%s [clan] %s %s\r\n", clans[(int)GET_CLAN(ch)].clan_short, GET_NAME(ch), argument);
    sprintf(buf2, "%s [clan] Someone %s\r\n", clans[(int)GET_CLAN(ch)].clan_short, argument);
  } else {
    sprintf(buf1, "%s %s clan-says, '%s&+C'\r\n", clans[(int)GET_CLAN(ch)].clan_short, GET_NAME(ch), argument);
    sprintf(buf2, "%s Someone clan-says, '%s'\r\n", clans[(int)GET_CLAN(ch)].clan_short, argument);
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_SOUNDPROOF)) {
    send_to_char("The walls seem to absorb your words.\r\n", ch);
    return;
  }
    for (d=descriptor_list; d; d=d->next)
      if ((!d->connected) && !ROOM_FLAGGED(d->character->in_room, ROOM_SOUNDPROOF) &&
          (!(PLR_FLAGGED(d->character, PLR_WRITING | PLR_MAILING))) && 
         (d != ch->desc || (!PRF_FLAGGED(d->character, PRF_NOREPEAT)))) 
        if ((GET_CLAN(d->character) == GET_CLAN(ch) ||
            PRF_FLAGGED(d->character, PRF_ALLCLAN)) && 
            !PRF_FLAGGED(d->character, PRF_NOCLAN)) {
           send_to_char(CCBCY(d->character, C_NRM), d->character);
           if (CAN_SEE(d->character, ch))
             send_to_char(buf1, d->character);
           else
	     send_to_char(buf2, d->character);
           send_to_char(CCNRM(d->character, C_NRM), d->character);
        }
}

ACMD(do_gos_emote)
{
  struct descriptor_data *d;
  
  skip_spaces(&argument);

  if (!ch->desc) return;

  if (PLR_FLAGGED(ch, PLR_NOSHOUT)) {
    send_to_char("You can't do that!\r\n", ch);
    return;
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_SOUNDPROOF)) {
    send_to_char("The walls absorb your words.\r\n", ch);
    return;
  }
  if (!*argument) {
    send_to_char("You have to emote SOMETHING.\r\n", ch);
    return;
  }
  if (PRF_FLAGGED(ch, PRF_NOGOSS)) {
    send_to_char("You aren't even on that channel!\r\n", ch);
    return;
  }
    sprintf(buf1, "[gossip] %s %s\r\n", GET_NAME(ch), argument);
    sprintf(buf2, "[gossip] Someone %s\r\n", argument);
  
    for (d=descriptor_list; d; d=d->next)
      if ((!d->connected) &&
          (!(PLR_FLAGGED(d->character, PLR_WRITING | PLR_MAILING)/* &&
           PRF_FLAGGED(d->character, PRF_CONCENTRATE)*/)) &&
          (d != ch->desc || (!PRF_FLAGGED(d->character, PRF_NOREPEAT)))) 
        if (!PRF_FLAGGED(d->character, PRF_NOGOSS) && 
           !is_ignoring(d->character, GET_IDNUM(ch))) {  
           send_to_char(CCYEL(d->character, C_NRM), d->character);
           if (CAN_SEE(d->character, ch))
             send_to_char(buf1, d->character);
           else
	     send_to_char(buf2, d->character);
           send_to_char(CCNRM(d->character, C_NRM), d->character);
        }
    }

/*
ACMD(do_erase)
{
  char *papername;
  struct obj_data *paper = 0;
  struct char_data *writer;

  one_argument(argument, papername);
  
 if (!ch->desc)
  return;

  if (!*papername) {
   send_to_char("What? You want to erase yourself??\r\n", ch);
   return;
  }
  if (!(paper = get_obj_in_list_vis(ch, papername, ch->carrying))) {
    sprintf(buf, "You have no %s.\r\n", papername);
    send_to_char(buf, ch);
    return;
  }
  if (GET_OBJ_TYPE(paper) != ITEM_NOTE) {
   send_to_char("You can't erase that.\r\n", ch);
   return;
  }
  
  if (!(GET_OBJ_VAL(paper, 1)))
   writer = NULL;
  else {
   writer = 

*/

ACMD(do_pchan)
{
  struct char_data *vict;
  struct descriptor_data *i;
  int mode;

  char *pmodes[] = {
    "help",    "open",    "off",
    "close",    "who",    "check",
    "remove",    "add",    "\n" };
    
  half_chop(argument, arg, buf2);
  for (mode = 0; str_cmp(pmodes[mode], "\n"); mode++)
    if (is_abbrev(arg, pmodes[mode]))
      break;
  one_argument(buf2, buf);
  switch(mode) {
    case 0:
      send_to_char("Private Channel (PC) commands\r\n", ch);
      send_to_char("------------------------\r\n", ch);
      send_to_char("open   - opens your own private channel.\r\n", ch);
      send_to_char("add    - adds a player to your PC.\r\n", ch);
      send_to_char("remove - remove a player from your PC.\r\n", ch);
      send_to_char("close  - closes your private channel.\r\n", ch);
      send_to_char("who    - lists all members on the current PC.\r\n", ch);
      send_to_char("off    - exits you from your current PC.\r\n\r\n", ch);
      send_to_char("  NOTE: If you don't want to be added to another\r\n", ch);
      send_to_char("        player's PC open your own with no players.\r\n", ch);
      send_to_char("\r\nTo talk on the channel use -- private, psay or .\r\n",ch);
      break;
    case 1:
      GET_PCHAN(ch) = GET_IDNUM(ch);
      send_to_char("You have just opened your own Private Channel.\r\n", ch);
      break;
    case 2:
    sprintf(buf, "%s has left the Private Channel.\r\n", GET_NAME(ch));
    for (i = descriptor_list; i; i = i->next)
      if ((!i->connected) && 
         (!ROOM_FLAGGED(i->character->in_room, ROOM_SOUNDPROOF)) &&
         (!(PLR_FLAGGED(i->character, PLR_WRITING | PLR_MAILING))) &&
         (i->character != ch) &&
         (GET_PCHAN(ch) == GET_PCHAN(i->character)))
           send_to_char(buf, i->character);
      GET_PCHAN(ch) = 0;
      send_to_char("You have just quit any Private Channels.\r\n", ch);
      break;
    case 3:
      if (GET_PCHAN(ch) != GET_IDNUM(ch)) {
        send_to_char("You are not head of this channel... use pchan off.\r\n",
                     ch);
        return;
      }
      /* now remove all people on the private channel */
      for (i = descriptor_list; i; i = i->next)
        if (!i->connected)
        if ((GET_PCHAN(i->character) == GET_IDNUM(ch)) &&
        (!(PLR_FLAGGED(i->character, PLR_WRITING | PLR_MAILING))) &&
          (ch != i->character)) {
        GET_PCHAN(i->character) = 0;
        sprintf(buf, "%s has just closed their Private Channel.\r\n",
               GET_NAME(ch));
        send_to_char(buf, i->character);
      }
      GET_PCHAN(ch) = 0;
      send_to_char("You have just CLOSED your Private Channel.\r\n", ch);
      break;
    case 4:
      if (GET_PCHAN(ch) == 0)
        send_to_char("You are not on a private channel\r\n",ch);
      else {
        /* show all people on the private channel */
        send_to_char("Private Channel Members\r\n", ch);
        send_to_char("-----------------------\r\n", ch);
        for (i = descriptor_list; i; i = i->next)
          if (!i->connected)
          if (GET_PCHAN(i->character) == GET_PCHAN(ch)) {
            sprintf(buf, "%s\r\n", GET_NAME(i->character));
            send_to_char(buf, ch);
          }
       }
       break;
    case 5:
      /* show all people on the ALL private channels */
      if (GET_LEVEL(ch) <= LVL_IMPL) {
        send_to_char("Not an option.  Type pchan help for help.\r\n", ch);
        break;
      }
      send_to_char("Private Channels\r\n", ch);
      send_to_char("---------------------------------------------\r\n", ch);
      for (i = descriptor_list; i; i = i->next)
        if (!i->connected && GET_PCHAN(i->character)) {
        sprintf(buf, "[%-5ld]  %s\r\n",
                  GET_PCHAN(i->character), GET_NAME(i->character));
          send_to_char(buf, ch);
      }
      break;
    case 6:
      if (!*buf)
        send_to_char("Who do you wish to remove from your private channel?\r\n", ch);
      else if (!(vict = get_char_vis(ch, buf)))
        send_to_char(NOPERSON, ch);
      else if (IS_NPC(vict))
        send_to_char("NPC's cannot be on private channels\r\n", ch);
      else if (GET_PCHAN(vict) != GET_IDNUM(ch)) {
        sprintf(buf,"%s is NOT on your Private Channel!\r\n",
                GET_NAME(vict));
        send_to_char(buf, ch);
      } else {
        GET_PCHAN(vict) = 0;
        sprintf(buf,"You have been REMOVED from %s's Private Channel!\r\n",
                GET_NAME(ch));
        send_to_char(buf, vict);
        sprintf(buf,"%s has been REMOVED from the Private Channel!\r\n",
                GET_NAME(vict));
        for (i = descriptor_list; i; i = i->next) 
          if ((!i->connected) && 
            (!ROOM_FLAGGED(i->character->in_room, ROOM_SOUNDPROOF)) &&
            (!(PLR_FLAGGED(i->character, PLR_WRITING | PLR_MAILING))) &&
            (GET_PCHAN(ch) == GET_PCHAN(i->character)))
              send_to_char(buf, i->character);
      }
      break;
    case 7:
      if (GET_PCHAN(ch) != GET_IDNUM(ch))
        send_to_char("You must open your own private channel first\r\n",ch);
      else if (!*buf)
          send_to_char("Who do you wish to add to you private channel?\r\n", ch);
      else if (!(vict = get_char_vis(ch, buf)))
          send_to_char(NOPERSON, ch);
      else if (ch == vict)
          GET_PCHAN(ch) = GET_IDNUM(ch);
      else if (IS_NPC(vict))
          send_to_char("NPC's cannot be added to private channels\r\n", ch);
      else if (GET_PCHAN(vict) != 0) {
          sprintf(buf,"%s is already on another private channel!\r\n",
                  GET_NAME(vict));
          send_to_char(buf, ch);
      } else if (is_ignoring(vict, GET_IDNUM(ch)))
          act("You can't add $N... $E is ignoring you.", FALSE, ch, 0, vict, 
              TO_CHAR);
      else if (is_ignoring(ch, GET_IDNUM(vict)))
          act("You can't add $N... you are ignoring $M.", FALSE, ch, 0,
              vict, TO_CHAR);
      else {
          GET_PCHAN(vict) = GET_IDNUM(ch);
          sprintf(buf,"You have been ADDED to %s's Private Channel!\r\n",
                  GET_NAME(ch));
          send_to_char(buf, vict);
          sprintf(buf,"%s has been ADDED to the Private Channel!\r\n",
                  GET_NAME(vict));
          for (i = descriptor_list; i; i = i->next)
            if ((!i->connected) && 
              (!ROOM_FLAGGED(i->character->in_room, ROOM_SOUNDPROOF)) &&
              (!(PLR_FLAGGED(i->character, PLR_WRITING | PLR_MAILING))) &&
              (GET_PCHAN(ch) == GET_PCHAN(i->character)) && i->character != vict)
                send_to_char(buf, i->character);
      }
      break;
    default:
      send_to_char("Not an option.  Type pchan help for help.\r\n", ch);
      break;
    }
  }

struct syllable {
  char *org;
  char *new;
};


void drunk_speak(char *speech) {
 
  char retstr[512]; 
  int j, ofs = 0, k;
  int length = strlen(speech);

  *retstr = '\0';
  
  for (ofs = 0; ofs < length; ofs++) {
    if (speech[ofs] == 's') {
      k = number(1,5);
      for (j = 0; j < k; j++) 
        strcat(retstr, "s");
    } else if (speech[ofs] == 'k' && number(1,4) > 2) {
      strcat(retstr, "c");
    } else if (speech[ofs] == 'c' && number(1,4) > 2) {
      strcat(retstr, "k");
    } else if (number(1,10) > 7) {
      if (speech[ofs] > 'a' && speech[ofs] < 'z') { /* capitalize */
        j = speech[ofs] - 32;
      } else if (speech[ofs] > 'A' && speech[ofs] < 'Z')  {
        j = speech[ofs] + 32;
      } else {
        j = speech[ofs];
      }
      sprintf(retstr, "%s%c", retstr, j);
    } else {
      sprintf(retstr, "%s%c", retstr, speech[ofs]);
    }

    switch (number(1,20)) {
      case 3:  strcat(retstr, "*BURP*"); 
               break;
      case 8:  strcat(retstr, "*HIC*"); 
               break;
    }
  }
  if (test_mode) {
    fprintf(stderr, "Drunk Speech: %s -> %s\n", speech, retstr);
  }
  sprintf(speech, "%s", retstr);
}
