/* ************************************************************************
*   File: act.informative.c                             Part of CircleMUD *
*  Usage: Player-level commands of an informative nature                  *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "handler.h"
#include "db.h"
#include "spells.h"
#include "screen.h"

/* extern variables */
extern struct room_data *world;
extern struct descriptor_data *descriptor_list;
extern struct char_data *character_list;
extern struct obj_data *object_list;
extern struct title_type titles[NUM_CLASSES][LVL_SIMP + 1];
extern struct command_info cmd_info[];
extern struct index_data *mob_index;
extern struct index_data *obj_index;

extern char *credits;
extern char *claneq;
extern char *info;
extern char *motd;
extern char *imotd;
extern char *wizlist;
extern char *immlist;
extern char *policies;
extern char *handbook;
extern char *circlemud_license;
extern char *dirs[];
extern char *where[];
extern char *color_liquid[];
extern char *fullness[];
extern char *connected_types[];
extern char *class_abbrevs[];
extern char *room_bits[];
extern char *spells[];
extern char *race_abbrevs[];
extern struct clan_stuff clans[];

long find_class_bitvector(char arg);
long find_race_bitvector(char arg);
long find_clan_bitvector(char *arg);
int parse_class(char arg);
int valid_address(char *str);

void show_obj_to_char(struct obj_data * object, struct char_data * ch,
			int mode)
{
  bool found;

  *buf = '\0';
  if ((mode == 0) && object->description)
    strcpy(buf, object->description);
  else if (object->short_description && ((mode == 1) ||
				 (mode == 2) || (mode == 3) || (mode == 4)))
    strcpy(buf, object->short_description);
  else if (mode == 5) {
    if (GET_OBJ_TYPE(object)==ITEM_GUN || GET_OBJ_TYPE(object)==ITEM_QUIVER) {
      if (GET_OBJ_VAL(object, 1))
        sprintf(buf, "It has %d %s%s left in it.", GET_OBJ_VAL(object, 1),
                GET_OBJ_TYPE(object)==ITEM_GUN ? "bullet" : "arrow",
                GET_OBJ_VAL(object, 1) == 1 ? "" : "s");
      else
        sprintf(buf, "It is empty.");
  } else if (GET_OBJ_TYPE(object) == ITEM_NOTE) {
      if (object->action_description) {
	strcpy(buf, "There is something written upon it:\r\n\r\n");
	if(object->mail) {
	 if (object->mail->date > 0)
	  sprintf(buf, "%s   Date: %s", buf, 
	    (char *) asctime(localtime(&object->mail->date)));
	  sprintf(buf, "%s   From: %s\r\n", buf,
	    get_name_by_id(object->mail->from));
	  sprintf(buf, "%s     To: %s\r\n", buf,
	    get_name_by_id(object->mail->to));
	  sprintf(buf, "%sSubject: %s\r\n\r\n", buf, object->mail->subject ?
	    object->mail->subject : "(none)");
	}
	strcat(buf, object->action_description);
	page_string(ch->desc, buf, 1);
      } else
	act("It's blank.", FALSE, ch, 0, 0, TO_CHAR);
      return;
    } else if (GET_OBJ_TYPE(object) != ITEM_DRINKCON) {
      strcpy(buf, "You see nothing special..");
    } else			/* ITEM_TYPE == ITEM_DRINKCON||FOUNTAIN */
      strcpy(buf, "It looks like a drink container.");
  }
  if (mode != 3) {
    found = FALSE;
    if (IS_OBJ_STAT(object, ITEM_INVISIBLE)) {
      strcat(buf, " &+w(invisible)");
      found = TRUE;
    }
    if (ch->player_specials->targobj == object) {
      strcat(buf, " &+g(TARGETED)&+w");
      found = TRUE;
    }
    if (object->restring && GET_LEVEL(ch) >= LVL_IMMORT) {
      sprintf(buf, "%s &+m(%d)&+w", buf, GET_OBJ_VNUM(object));
      found = TRUE;
    }
    if (IS_OBJ_STAT(object, ITEM_BLESS) && IS_AFFECTED(ch, AFF_DETECT_ALIGN)) {
      strcat(buf, " &+w(&+Bglowing blue&+w)");
      found = TRUE;
    }
    if (IS_OBJ_STAT(object, ITEM_MAGIC) && IS_AFFECTED(ch, AFF_DETECT_MAGIC)) {
      strcat(buf, " &+w(&+Yglowing yellow&+w)");
      found = TRUE;
    }
    if (IS_OBJ_STAT(object, ITEM_GLOW)) {
      strcat(buf, " &+w(&+Wsoftly glowing&+w)");
      found = TRUE;
    }
    if (IS_OBJ_STAT(object, ITEM_HUM)) {
      strcat(buf, " &+w(&+Wf&+wa&+Wi&+wn&+wt&+Wl&+wy &+Wh&+wu&+Wm&+wm&+Wi&+wn&+Wg&+w)");
      found = TRUE;
    }
  }
  strcat(buf, "\r\n");
  page_string(ch->desc, buf, 1);
}


void list_obj_to_char(struct obj_data * list, struct char_data * ch, int mode,
		           bool show)
{
  struct obj_data *i;
  bool found;

  found = FALSE;
  for (i = list; i; i = i->next_content) {
    if (CAN_SEE_OBJ(ch, i)) {
      show_obj_to_char(i, ch, mode);
      found = TRUE;
    }
  }
  if (!found && show)
    send_to_char(" Nothing.\r\n", ch);
}


void diag_char_to_char(struct char_data * i, struct char_data * ch)
{
  int percent;

  if (GET_MAX_HIT(i) > 0)
    percent = (100 * GET_HIT(i)) / GET_MAX_HIT(i);
  else
    percent = -1;		/* How could MAX_HIT be < 1?? */

  strcpy(buf, PERS(i, ch));
  CAP(buf);

  if (percent >= 100)
    strcat(buf, " is in excellent condition.\r\n");
  else if (percent >= 90)
    strcat(buf, " has a few scratches.\r\n");
  else if (percent >= 75)
    strcat(buf, " has some small wounds and bruises.\r\n");
  else if (percent >= 50)
    strcat(buf, " has quite a few wounds.\r\n");
  else if (percent >= 30)
    strcat(buf, " has some big nasty wounds and scratches.\r\n");
  else if (percent >= 15)
    strcat(buf, " looks pretty hurt.\r\n");
  else if (percent >= 0)
    strcat(buf, " is in awful condition.\r\n");
  else
    strcat(buf, " is bleeding awfully from big wounds.\r\n");

  send_to_char(buf, ch);
}


void look_at_char(struct char_data * i, struct char_data * ch, int subcmd)
{
  int j, found;
  struct obj_data *tmp_obj;

  if (ch != i && (GET_CLASS(ch) == CLASS_THIEF || GET_LEVEL(ch) >= LVL_IMMORT)
      && subcmd == SCMD_PEEK) {
    found = FALSE;
    act("\r\nYou attempt to peek at $n's inventory:", FALSE, i, 0, ch, TO_VICT);
    for (tmp_obj = i->carrying; tmp_obj; tmp_obj = tmp_obj->next_content) {
      if (CAN_SEE_OBJ(ch, tmp_obj) && (number(0, 20) < GET_LEVEL(ch))) {
	show_obj_to_char(tmp_obj, ch, 1);
	found = TRUE;
      }
    }

    if (!found)
      send_to_char("You can't see anything.\r\n", ch);
    return;
  }
  if (i->player.description)
    send_to_char(i->player.description, ch);
  else
    act("You see nothing special about $m.", FALSE, i, 0, ch, TO_VICT);

  diag_char_to_char(i, ch);

  found = FALSE;
  for (j = 0; !found && j < NUM_WEARS; j++)
    if (GET_EQ(i, j) && CAN_SEE_OBJ(ch, GET_EQ(i, j)))
      found = TRUE;

  if (found) {
    act("\r\n$n is using:", FALSE, i, 0, ch, TO_VICT);
    for (j = 0; j < NUM_WEARS; j++)
      if (GET_EQ(i, j) && CAN_SEE_OBJ(ch, GET_EQ(i, j))) {
	send_to_char(where[j], ch);
	show_obj_to_char(GET_EQ(i, j), ch, 1);
      }
  }
}


void list_one_char(struct char_data * i, struct char_data * ch)
{
  char *positions[] = {
    " &+yis lying here, dead.",
    " &+yis lying here, mortally wounded.",
    " &+yis lying here, incapacitated.",
    " &+yis lying here, stunned.",
    " &+yis sleeping here.",
    " &+yis resting here.",
    " &+yis sitting here.",
    "!FIGHTING!",
    " &+yis standing here.",
    " &+yis riding around.",
    " &+yis being ridden.",
  };

  if (IS_NPC(i) && i->player.long_descr && GET_POS(i) == GET_DEFAULT_POS(i)) {
    if (IS_AFFECTED(i, AFF_INVISIBLE))
      strcpy(buf, "*");
    else
      *buf = '\0';

    if (IS_AFFECTED(ch, AFF_DETECT_ALIGN)) {
      if (IS_EVIL(i)) {
        if (IS_VEVIL(i)) {
	  strcat(buf, "&+R(Bright Red Aura) ");
	} else {
	  strcat(buf, "&+R(Red Aura) ");
	}
      } else {
        if (IS_GOOD(i)) {
          if (IS_VGOOD(i)) {
            strcat(buf, "&+B(Bright Blue Aura) ");
          } else {
            strcat(buf, "&+B(Blue Aura) ");
          }
	}
      }
    }
    strcat(buf, i->player.long_descr);
    send_to_char(buf, ch);
    send_to_char("&+y", ch);

    if (IS_AFFECTED(i, AFF_SANCTUARY))
      act("&+W...$e glows with a bright light!&+w", FALSE, i, 0, ch, TO_VICT);
    if (IS_SET(AFF2_FLAGS(i), AFF2_DIVINE_PROT))
      act("&+Y...$e has a bright yellow aura!&+w", FALSE, i, 0, ch, TO_VICT);
    if (IS_AFFECTED(i, AFF_BLIND))
      act("&+y...$e is groping around blindly!&+w", FALSE, i, 0, ch, TO_VICT);

    return;
  }
  if (IS_NPC(i)) {
    strcpy(buf, i->player.short_descr);
    CAP(buf);
  } else
    sprintf(buf, "%s %s&+y", i->player.name, GET_TITLE(i));

  if (IS_AFFECTED(i, AFF_INVISIBLE))
    strcat(buf, " (invisible)");
  if (IS_AFFECTED(i, AFF_HIDE))
    strcat(buf, " (hidden)");
  if (!IS_NPC(i) && !i->desc)
    strcat(buf, " (linkless)");
  if (PLR_FLAGGED(i, PLR_WRITING))
    strcat(buf, " (writing)");
  if (GET_INVIS_LEV(i))
    sprintf(buf, "%s (i%d)", buf, GET_INVIS_LEV(i));

  if (GET_POS(i) != POS_FIGHTING)
    strcat(buf, positions[(int) GET_POS(i)]);
  else {
    if (FIGHTING(i)) {
      strcat(buf, " is here, fighting ");
      if (FIGHTING(i) == ch)
	strcat(buf, "YOU!");
      else {
	if (i->in_room == FIGHTING(i)->in_room)
	  strcat(buf, PERS(FIGHTING(i), ch));
	else
	  strcat(buf, "someone who has already left");
	strcat(buf, "!");
      }
    } else			/* NIL fighting pointer */
      strcat(buf, " is here struggling with thin air.");
  }

  if (IS_AFFECTED(ch, AFF_DETECT_ALIGN)) {
    if (IS_EVIL(i)) {
      if (IS_VEVIL(i)) {
        strcat(buf, " &+R(Bright Red Aura)");
      } else {
        strcat(buf, " &+R(Red Aura)");
      }
    } else {
      if (IS_GOOD(i)) {
        if (IS_VGOOD(i)) {
          strcat(buf, " &+B(Bright Blue Aura)");
        } else {
          strcat(buf, " &+B(Blue Aura)");
        }
      }
    }
  }
  strcat(buf, "\r\n");
  send_to_char(buf, ch);
  send_to_char("&+y", ch);

  if (IS_AFFECTED(i, AFF_SANCTUARY))
    act("&+W...$e glows with a bright light!&+w", FALSE, i, 0, ch, TO_VICT);
  if (IS_SET(AFF2_FLAGS(i), AFF2_DIVINE_PROT))
    act("&+Y...$e has a bright yellow aura!&+w", FALSE, i, 0, ch, TO_VICT);
}



void list_char_to_char(struct char_data * list, struct char_data * ch)
{
  struct char_data *i;

  for (i = list; i; i = i->next_in_room)
    if (ch != i) {
      if (CAN_SEE(ch, i))
	list_one_char(i, ch);
      else if (IS_DARK(ch->in_room) && !CAN_SEE_IN_DARK(ch) &&
	       IS_AFFECTED(i, AFF_INFRAVISION))
	send_to_char("You see a pair of glowing red eyes looking your way.\r\n", ch);
    }
}


void do_auto_exits(struct char_data * ch)
{
  int door;
  extern char *dir_abbrev[];

  *buf = '\0';

  for (door = 0; door < NUM_OF_DIRS; door++)
    if (EXIT(ch, door) && EXIT(ch, door)->to_room != NOWHERE)
     if (!IS_SET(EXIT(ch, door)->exit_info, EX_CLOSED))
      sprintf(buf, "%s%s ", buf, dir_abbrev[door]);
     else if (!IS_SET(EXIT(ch, door)->exit_info, EX_SECRET))
      sprintf(buf, "%s(%s) ", buf, dir_abbrev[door]);
      else if (GET_LEVEL(ch) >= LVL_IMMORT ||
       (IS_AFFECTED(ch, AFF_DET_DOOR) && number(1,101) > 70))
       sprintf(buf, "%s*%s* ", buf, dir_abbrev[door]);

  sprintf(buf2, "%s[ Exits: %s]%s\r\n", CCCYN(ch, C_NRM),
	  *buf ? buf : "None! ", CCNRM(ch, C_NRM));

  send_to_char(buf2, ch);
}


ACMD(do_exits)
{
  int door;

  *buf = '\0';

  if (IS_AFFECTED(ch, AFF_BLIND)) {
    send_to_char("You can't see a damned thing, you're blind!\r\n", ch);
    return;
  }
  for (door = 0; door < NUM_OF_DIRS; door++)
    if (EXIT(ch, door) && EXIT(ch, door)->to_room != NOWHERE) {
/*	!IS_SET(EXIT(ch, door)->exit_info, EX_CLOSED)) { */
      if (GET_LEVEL(ch) >= LVL_IMMORT)
        if (IS_SET(EXIT(ch, door)->exit_info, EX_CLOSED))
          sprintf(buf2, "%-5s - [%5d] a closed %s %s\r\n", dirs[door],
                  world[EXIT(ch, door)->to_room].number,
                  EXIT(ch, door)->keyword ? fname(EXIT(ch, door)->keyword) : "something",
                  IS_SET(EXIT(ch, door)->exit_info, EX_SECRET) ? 
                    "(secret)" : "");
        else
   	  sprintf(buf2, "%-5s - [%5d] %s\r\n", dirs[door],
	  	  world[EXIT(ch, door)->to_room].number,
		  world[EXIT(ch, door)->to_room].name);
      else {
	sprintf(buf2, "%-5s - ", dirs[door]);
	if (IS_DARK(EXIT(ch, door)->to_room) && !CAN_SEE_IN_DARK(ch) &&
	    !IS_SET(EXIT(ch, door)->exit_info, EX_CLOSED))
	  strcat(buf2, "Too dark to tell\r\n");
	else 
	  if (IS_SET(EXIT(ch, door)->exit_info, EX_CLOSED)) {
	    if (IS_SET(EXIT(ch, door)->exit_info, EX_SECRET) &&
	        (!IS_AFFECTED(ch, AFF_DET_DOOR) || number(1, 101) > 70)) 
	      *buf2='\0';
	    else
	      sprintf(buf2, "%s a closed %s %s\r\n", buf2, 
	              EXIT(ch, door)->keyword ? fname(EXIT(ch, door)->keyword) : "something",
	              IS_SET(EXIT(ch, door)->exit_info, EX_SECRET) ?
	                "(secret)" : "");
          } else {
	    strcat(buf2, world[EXIT(ch, door)->to_room].name);
	    strcat(buf2, "\r\n");
	}
      }
      strcat(buf, CAP(buf2));
    }
  send_to_char("Obvious exits:\r\n", ch);

  if (*buf)
    send_to_char(buf, ch);
  else
    send_to_char(" None.\r\n", ch);
}



void look_at_room(struct char_data * ch, int ignore_brief)
{
  extern char *sector_types[];
  
  if (IS_DARK(ch->in_room) && !CAN_SEE_IN_DARK(ch)) {
    send_to_char("It is pitch black...\r\n", ch);
    return;
  } else if (IS_AFFECTED(ch, AFF_BLIND)) {
    send_to_char("You see nothing but infinite darkness...\r\n", ch);
    return;
  }
  send_to_char(CCCYN(ch, C_NRM), ch);
  if (PRF_FLAGGED(ch, PRF_ROOMFLAGS)) {
    sprintbit((long) ROOM_FLAGS(ch->in_room), room_bits, buf);
    sprintf(buf2, "[%5d] %s %s&+c [ %s] - %s", world[ch->in_room].number,
            IS_LIGHT(ch->in_room) ? "&+Y(L)&+c" : "&+m(D)&+c",
	    world[ch->in_room].name, buf, sector_types[world[ch->in_room].sector_type]);
    send_to_char(buf2, ch);
  } else
    send_to_char(world[ch->in_room].name, ch);

  send_to_char(CCNRM(ch, C_NRM), ch);
  send_to_char("\r\n", ch);

  if (!PRF_FLAGGED(ch, PRF_BRIEF) || ignore_brief ||
      ROOM_FLAGGED(ch->in_room, ROOM_DEATH))
    send_to_char(world[ch->in_room].description, ch);

  /* autoexits */
  if (PRF_FLAGGED(ch, PRF_AUTOEXIT))
    do_auto_exits(ch);

  /* now list characters & objects */
  send_to_char(CCGRN(ch, C_NRM), ch);
  list_obj_to_char(world[ch->in_room].contents, ch, 0, FALSE);
  send_to_char(CCYEL(ch, C_NRM), ch);
  list_char_to_char(world[ch->in_room].people, ch);
  send_to_char(CCNRM(ch, C_NRM), ch);
}



void look_in_direction(struct char_data * ch, int dir)
{
  if (EXIT(ch, dir)) {
    if (EXIT(ch, dir)->general_description)
      send_to_char(EXIT(ch, dir)->general_description, ch);
    else
      send_to_char("You see nothing special.\r\n", ch);

    if (IS_SET(EXIT(ch, dir)->exit_info, EX_CLOSED) && EXIT(ch, dir)->keyword) {
      if (!(IS_SET(EXIT(ch, dir)->exit_info, EX_SECRET))) { /* normal closed exit */
        sprintf(buf, "The %s is closed.\r\n", fname(EXIT(ch, dir)->keyword));
        send_to_char(buf, ch);
      } else { /* secret closed exit */
        if (IS_IMMORTAL(ch) ||
           (IS_AFFECTED(ch, AFF_DET_DOOR) && number(1,101) > 70)) { /* can see */
          sprintf(buf, "The %s is closed. (secret)\r\n", fname(EXIT(ch, dir)->keyword));
          send_to_char(buf, ch);
        }
      }
    } else if (EXIT(ch, dir)->keyword) {
      if (!(IS_SET(EXIT(ch, dir)->exit_info, EX_SECRET))) { /* nonsecret door */
        sprintf(buf, "The %s is open.\r\n", fname(EXIT(ch, dir)->keyword));
        send_to_char(buf, ch);
      } else { /* secret door */
        if (IS_IMMORTAL(ch) ||
           (IS_AFFECTED(ch, AFF_DET_DOOR) && number(1,101) > 70)) { /* can see */
          sprintf(buf, "The %s is open. (secret)\r\n", fname(EXIT(ch, dir)->keyword));
          send_to_char(buf, ch);
        }
    } }

  } else
    send_to_char("Nothing special there...\r\n", ch);
}



void look_in_obj(struct char_data * ch, char *arg)
{
  struct obj_data *obj = NULL;
  struct char_data *dummy = NULL;
  int amt, bits;

  if (!*arg)
    send_to_char("Look in what?\r\n", ch);
  else if (!(bits = generic_find(arg, FIND_OBJ_INV | FIND_OBJ_ROOM |
				 FIND_OBJ_EQUIP, ch, &dummy, &obj))) {
    sprintf(buf, "There doesn't seem to be %s %s here.\r\n", AN(arg), arg);
    send_to_char(buf, ch);
  } else if ((GET_OBJ_TYPE(obj) != ITEM_DRINKCON) &&
	     (GET_OBJ_TYPE(obj) != ITEM_FOUNTAIN) &&
	     (GET_OBJ_TYPE(obj) != ITEM_CONTAINER))
    send_to_char("There's nothing inside that!\r\n", ch);
  else {
    if (GET_OBJ_TYPE(obj) == ITEM_CONTAINER) {
      if (IS_SET(GET_OBJ_VAL(obj, 1), CONT_CLOSED))
	send_to_char("It is closed.\r\n", ch);
      else {
	send_to_char(fname(obj->name), ch);
	switch (bits) {
	case FIND_OBJ_INV:
	  send_to_char(" (carried): \r\n", ch);
	  break;
	case FIND_OBJ_ROOM:
	  send_to_char(" (here): \r\n", ch);
	  break;
	case FIND_OBJ_EQUIP:
	  send_to_char(" (used): \r\n", ch);
	  break;
	}

	list_obj_to_char(obj->contains, ch, 2, TRUE);
      }
    } else {		/* item must be a fountain or drink container */
      if (GET_OBJ_VAL(obj, 1) <= 0)
	send_to_char("It is empty.\r\n", ch);
      else {
	amt = ((GET_OBJ_VAL(obj, 1) * 3) / GET_OBJ_VAL(obj, 0));
	sprintf(buf, "It's %sfull of a %s liquid.\r\n", fullness[amt],
		color_liquid[GET_OBJ_VAL(obj, 2)]);
	send_to_char(buf, ch);
      }
    }
  }
}



char *find_exdesc(char *word, struct extra_descr_data * list)
{
  struct extra_descr_data *i;

  for (i = list; i; i = i->next)
    if (isname(word, i->keyword))
      return (i->description);

  return NULL;
}


/*
 * Given the argument "look at <target>", figure out what object or char
 * matches the target.  First, see if there is another char in the room
 * with the name.  Then check local objs for exdescs.
 */
void look_at_target(struct char_data * ch, char *arg, int subcmd)
{
  int bits, found = 0, j;
  struct char_data *found_char = NULL;
  struct obj_data *obj = NULL, *found_obj = NULL;
  char *desc;

  if (!*arg) {
    send_to_char("Look at what?\r\n", ch);
    return;
  }
  bits = generic_find(arg, FIND_OBJ_INV | FIND_OBJ_ROOM | FIND_OBJ_EQUIP |
		      FIND_CHAR_ROOM, ch, &found_char, &found_obj);

  /* Is the target a character? */
  if (found_char != NULL) {
    look_at_char(found_char, ch, subcmd);
    if (ch != found_char && subcmd != SCMD_PEEK) {
      if (CAN_SEE(found_char, ch))
	act("$n looks at you.", TRUE, ch, 0, found_char, TO_VICT);
      act("$n looks at $N.", TRUE, ch, 0, found_char, TO_NOTVICT);
    }
    return;
  }
  /* Does the argument match an extra desc in the room? */
  if ((desc = find_exdesc(arg, world[ch->in_room].ex_description)) != NULL) {
    page_string(ch->desc, desc, 0);
    return;
  }
  /* Does the argument match an extra desc in the char's equipment? */
  for (j = 0; j < NUM_WEARS && !found; j++)
    if (GET_EQ(ch, j) && CAN_SEE_OBJ(ch, GET_EQ(ch, j)))
      if ((desc = find_exdesc(arg, GET_EQ(ch, j)->ex_description)) != NULL) {
	send_to_char(desc, ch);
	found = 1;
      }
  /* Does the argument match an extra desc in the char's inventory? */
  for (obj = ch->carrying; obj && !found; obj = obj->next_content) {
    if (GET_OBJ_TYPE(obj) == ITEM_GIFT && GET_OBJ_VAL(obj, 1)) {
      sprintf(buf, "You see a dainty little box wrapped up with a pretty ribbon.\r\n");
      send_to_char(buf, ch);
      if (GET_OBJ_VAL(obj, 3)) {
        sprintf(buf, "The box is tagged with the name '%s'.\r\n", 
                CAP(get_name_by_id(GET_OBJ_VAL(obj, 3))));
        send_to_char(buf, ch);
      }
      found = 1;
    }
    if (CAN_SEE_OBJ(ch, obj))
	if ((desc = find_exdesc(arg, obj->ex_description)) != NULL) {
	send_to_char(desc, ch);
	found = 1;
      }
  }

  /* Does the argument match an extra desc of an object in the room? */
  for (obj = world[ch->in_room].contents; obj && !found; obj = obj->next_content)
    if (CAN_SEE_OBJ(ch, obj))
	if ((desc = find_exdesc(arg, obj->ex_description)) != NULL) {
	send_to_char(desc, ch);
	found = 1;
      }
  if (bits) {			/* If an object was found back in
				 * generic_find */
    if (!found)
      show_obj_to_char(found_obj, ch, 5);	/* Show no-description */
    else
      show_obj_to_char(found_obj, ch, 6);	/* Find hum, glow etc */
  } else if (!found)
    send_to_char("You do not see that here.\r\n", ch);
}


ACMD(do_look)
{
  static char arg2[MAX_INPUT_LENGTH];
  int look_type;

  if (!ch->desc)
    return;

  if (GET_POS(ch) < POS_SLEEPING)
    send_to_char("You can't see anything but stars!\r\n", ch);
  else if (IS_AFFECTED(ch, AFF_BLIND))
    send_to_char("You can't see a damned thing, you're blind!\r\n", ch);
  else if (IS_DARK(ch->in_room) && !CAN_SEE_IN_DARK(ch)) {
    send_to_char("It is pitch black...\r\n", ch);
    list_char_to_char(world[ch->in_room].people, ch);	/* glowing red eyes */
  } else {
    half_chop(argument, arg, arg2);

    if (subcmd == SCMD_READ) {
      if (!*arg)
	send_to_char("Read what?\r\n", ch);
      else
	look_at_target(ch, arg, subcmd);
      return;
    }
    if (subcmd == SCMD_PEEK) {
      if (!*arg)
        send_to_char("Peek at whom?\r\n", ch);
      else
        look_at_target(ch, arg, subcmd);
      return;
    }
    if (!*arg)			/* "look" alone, without an argument at all */
      look_at_room(ch, 1);
    else if (is_abbrev(arg, "in"))
      look_in_obj(ch, arg2);
    /* did the char type 'look <direction>?' */
    else if ((look_type = search_block(arg, dirs, FALSE)) >= 0)
      look_in_direction(ch, look_type);
    else if (is_abbrev(arg, "at"))
      look_at_target(ch, arg2, subcmd);
    else
      look_at_target(ch, arg, subcmd);
  }
}



ACMD(do_examine)
{
  int bits;
  struct char_data *tmp_char;
  struct obj_data *tmp_object;

  one_argument(argument, arg);

  if (!*arg) {
    send_to_char("Examine what?\r\n", ch);
    return;
  }
  look_at_target(ch, arg, subcmd);

  bits = generic_find(arg, FIND_OBJ_INV | FIND_OBJ_ROOM | FIND_CHAR_ROOM |
		      FIND_OBJ_EQUIP, ch, &tmp_char, &tmp_object);

  if (tmp_object) {
    if ((GET_OBJ_TYPE(tmp_object) == ITEM_DRINKCON) ||
	(GET_OBJ_TYPE(tmp_object) == ITEM_FOUNTAIN) ||
	(GET_OBJ_TYPE(tmp_object) == ITEM_CONTAINER)) {
      send_to_char("When you look inside, you see:\r\n", ch);
      look_in_obj(ch, arg);
    }
  }
}



ACMD(do_gold)
{
  if (GET_GOLD(ch) == 0)
    send_to_char("You're broke!\r\n", ch);
  else if (GET_GOLD(ch) == 1)
    send_to_char("You have one miserable little gold coin.\r\n", ch);
  else {
    sprintf(buf, "You have %d gold coins.\r\n", GET_GOLD(ch));
    send_to_char(buf, ch);
  }
}




ACMD(do_inventory)
{
  send_to_char("You are carrying:\r\n", ch);
  list_obj_to_char(ch->carrying, ch, 1, TRUE);
}


ACMD(do_equipment)
{
  int i, found = 0;

  send_to_char("You are using:\r\n", ch);
  for (i = 0; i < NUM_WEARS; i++) {
    if (GET_EQ(ch, i)) {
      if (CAN_SEE_OBJ(ch, GET_EQ(ch, i))) {
	send_to_char(where[i], ch);
	show_obj_to_char(GET_EQ(ch, i), ch, 1);
	found = TRUE;
      } else {
	send_to_char(where[i], ch);
	send_to_char("Something.\r\n", ch);
	found = TRUE;
      }
    }
  }
  if (!found) {
    send_to_char(" Nothing.\r\n", ch);
  }
}


ACMD(do_time)
{
  char *suf;
  int weekday, day;
  extern struct time_info_data time_info;
  extern const char *weekdays[];
  extern const char *month_name[];

  sprintf(buf, "It is %d o'clock %s, on ",
	  ((time_info.hours % 12 == 0) ? 12 : ((time_info.hours) % 12)),
	  ((time_info.hours >= 12) ? "pm" : "am"));

  /* 35 days in a month */
  weekday = ((35 * time_info.month) + time_info.day + 1) % 7;

  strcat(buf, weekdays[weekday]);
  strcat(buf, "\r\n");
  send_to_char(buf, ch);

  day = time_info.day + 1;	/* day in [1..35] */

  if (day == 1)
    suf = "st";
  else if (day == 2)
    suf = "nd";
  else if (day == 3)
    suf = "rd";
  else if (day < 20)
    suf = "th";
  else if ((day % 10) == 1)
    suf = "st";
  else if ((day % 10) == 2)
    suf = "nd";
  else if ((day % 10) == 3)
    suf = "rd";
  else
    suf = "th";

  sprintf(buf, "The %d%s Day of the %s, Year %d.\r\n",
	  day, suf, month_name[(int) time_info.month], time_info.year);

  send_to_char(buf, ch);
}


ACMD(do_weather)
{
  static char *sky_look[] = {
    "cloudless",
    "cloudy",
    "rainy",
  "lit by flashes of lightning"};

  if (OUTSIDE(ch)) {
    sprintf(buf, "The sky is %s and %s.\r\n", sky_look[weather_info.sky],
	    (weather_info.change >= 0 ? "you feel a warm wind from south" :
	     "your foot tells you bad weather is due"));
    send_to_char(buf, ch);
  } else
    send_to_char("You have no feeling about the weather at all.\r\n", ch);
}


ACMD(do_help)
{
  extern int top_of_helpt;
  extern struct help_index_element *help_table;
  extern char *help;

  int chk, bot, top, mid, minlen;
  char buf[MAX_INPUT_LENGTH + 33];

  if (!ch->desc)
    return;

  skip_spaces(&argument);

  if (!*argument) {
    page_string(ch->desc, help, 0);
    return;
  }
  if (!help_table) {
    send_to_char("No help available.\r\n", ch);
    return;
  }
  bot = 0;
  top = top_of_helpt;
  if (!isalpha(*argument)) {
    send_to_char("Don't type that!!\r\n", ch);
    return;
  }

  for (;;) {
    mid = (bot + top) / 2;
    minlen = strlen(argument);

    if (!(chk = strn_cmp(argument, help_table[mid].keyword, minlen))) {

      /* trace backwards to find first matching entry. Thanks Jeff Fink! */
      while ((mid > 0) && help_table[mid-1].keyword &&
	 (!(chk = strn_cmp(argument, help_table[mid - 1].keyword, minlen))))
	mid--;
      page_string(ch->desc, help_table[mid].entry, 0);
      return;
    } else if (bot >= top) {
      send_to_char("There is no help on that word.\r\n", ch);
      sprintf(buf, "HELP: %s attempted to get help on %s", GET_NAME(ch),
	argument);
      mudlog(buf, NRM, LVL_IMMORT, TRUE);
      return;
    } else if (chk > 0)
      bot = ++mid;
    else
      top = --mid;
  }
}


ACMD(do_score)
{
  struct time_info_data playing_time;
  struct time_info_data real_time_passed(time_t t2, time_t t1);
  struct ignore_type *ignoring;
  extern char *pc_class_types[];
  extern struct clan_stuff clans[];
  extern char *race_types[];

if (IS_SET(MOB_FLAGS(ch), MOB_ISNPC)) {
  send_to_char("Sorry, this feature is temporarily broken for mobs.\r\n", ch);
  return;
}
sprintf(buf, "&+g<---------------------------------------------------------------------->&+w\r\n");
sprintf(buf, "%s&+g|      &+BN  o  M  U  D   &+MC  H  A  R  A  C  T  E  R    S  T  A  T  S      &+g|&+w\r\n", buf);
sprintf(buf, "%s&+g<---------------------------------------------------------------------->&+w\r\n",buf);
sprintf(buf, "%s&+cAge: &+G[ &+y%d&+G ]  &+cRace: &+y%s&+c  Class: &+y%s&+c  Level: &+G[ &+y%d&+G ]  &+cRemorts: &+G[ &+y%d&+G ]&+w\r\n", buf, GET_AGE(ch), race_types[(int) GET_RACE(ch)], pc_class_types[(int) GET_CLASS(ch)], GET_LEVEL(ch), GET_REMORT(ch));
sprintf(buf, "%s&+g--------------------------&+mPhysical Attributes&+g---------------------------&+w\r\n",buf);
sprintf(buf, "%s&+c  Strength: &+G[ &+y%d&+G/&+y%d&+G ]      &+cDexterity: [ &+y%d&+G ]      &+cConstitution: &+G[ &+y%d&+G ]&+w\r\n",buf, GET_STR(ch), GET_ADD(ch), GET_DEX(ch), GET_CON(ch));
sprintf(buf, "%s&+c    Armor Class: &+G[ &+y%d&+G ]      ",buf, GET_AC(ch));
sprintf(buf, "%s&+cHitroll: &+G[ &+y%d&+G ]      &+cDamroll: &+G[ &+y%d&+G ]&+w\r\n",buf, GET_HITROLL(ch), GET_DAMROLL(ch));
sprintf(buf, "%s&+c  HP: &+G[ &+y%d&+G/&+y%d&+G ]    &+cMana: &+G[ &+y%d&+G/&+y%d&+G ]    &+cMove: &+G[ &+y%d&+G/&+y%d&+G ]&+w\r\n",buf, GET_HIT(ch), GET_MAX_HIT(ch), GET_MANA(ch), GET_MAX_MANA(ch), GET_MOVE(ch), GET_MAX_MOVE(ch));
sprintf(buf, "%s&+g---------------------------&+mMental Attributes&+g----------------------------&+w\r\n",buf);
sprintf(buf, "%s&+c   Intelligence: &+G[ &+y%d&+G ]       &+cWisdom: &+G[ &+y%d&+G ]       &+cCharisma: &+G[ &+y%d&+G ]&+w\r\n",buf, GET_INT(ch), GET_WIS(ch), GET_CHA(ch));
sprintf(buf, "%s&+c   Alignment: &+G[ &+y%d&+G ]     &+cXP: &+G[ &+y%d&+G ]    ",buf, GET_ALIGNMENT(ch), GET_EXP(ch)); 

if (!IS_NPC(ch)) {
  if (GET_LEVEL(ch) < LVL_IMMORT)
    sprintf(buf, "%s&+cXP Needed: &+G[ &+y%d&+G ]&+w\r\n", buf, (titles[(int) GET_CLASS(ch)][GET_LEVEL(ch) + 1].exp) - GET_EXP(ch));
  else
    sprintf(buf, "%s&+w\r\n", buf);
sprintf(buf, "%s&+g------------------------------------------------------------------------&+w\r\n", buf);

sprintf(buf, "%sYou have %d gold on your person.&+w\r\n", buf, GET_GOLD(ch));
sprintf(buf, "%sYou are known as: %s %s&+w\r\n", buf, GET_NAME(ch), GET_TITLE(ch));
sprintf(buf, "%sYou are a member of: %s&+w\r\n", buf, clans[(int)GET_CLAN(ch)].clan_long);

    playing_time = real_time_passed((time(0) - ch->player.time.logon) +
                                  ch->player.time.played, 0);
    sprintf(buf, "%sYou have been on NoMUD for %d days and %d hours.\r\n",
          buf, playing_time.day, playing_time.hours);

  }

  switch (GET_POS(ch)) {
  case POS_DEAD:
    strcat(buf, "You are DEAD!\r\n");
    break;
  case POS_MORTALLYW:
    strcat(buf, "You are mortally wounded!  You should seek help!\r\n");
    break;
  case POS_INCAP:
    strcat(buf, "You are incapacitated, slowly fading away...\r\n");
    break;
  case POS_STUNNED:
    strcat(buf, "You are stunned!  You can't move!\r\n");
    break;
  case POS_SLEEPING:
    strcat(buf, "You are sleeping.\r\n");
    break;
  case POS_RESTING:
    strcat(buf, "You are resting.\r\n");
    break;
  case POS_SITTING:
    strcat(buf, "You are sitting.\r\n");
    break;
  case POS_FIGHTING:
    if (FIGHTING(ch))
      sprintf(buf, "%sYou are fighting %s.\r\n", buf, PERS(FIGHTING(ch), ch));
    else
      strcat(buf, "You are fighting thin air.\r\n");
    break;
  case POS_STANDING:
    strcat(buf, "You are standing.\r\n");
    break;
  case POS_RIDING:
    sprintf(buf, "%sYou are riding %s.\r\n", buf, PERS(ch->riding, ch));
    break;
  case POS_RIDDEN:
    sprintf(buf, "%sYou are being ridden by %s.\r\n", buf, 
            PERS(ch->ridden_by, ch));
    break;
  default:
    strcat(buf, "You are floating.\r\n");
    break;
  }

  if (GET_COND(ch, DRUNK) > 10)
    strcat(buf, "You are intoxicated.\r\n");

  if (GET_COND(ch, FULL) == 0)
    strcat(buf, "You are hungry.\r\n");

  if (GET_COND(ch, THIRST) == 0)
    strcat(buf, "You are thirsty.\r\n");

  if (IS_AFFECTED(ch, AFF_BLIND))
    strcat(buf, "You have been blinded!\r\n");

  if (IS_AFFECTED(ch, AFF_INVISIBLE))
    strcat(buf, "You are invisible.\r\n");

  if (IS_AFFECTED(ch, AFF_DETECT_INVIS))
    strcat(buf, "You are sensitive to the presence of invisible things.\r\n");

  if (IS_AFFECTED(ch, AFF_SANCTUARY))
    strcat(buf, "You are protected by Sanctuary.\r\n");

  if (IS_AFFECTED(ch, AFF_HASTE))
    strcat(buf, "You are moving very rapidly.\r\n");

  if (IS_AFFECTED(ch, AFF_BLUR))
    strcat(buf, "You are strangely out of focus.\r\n");

  if (IS_AFFECTED(ch, AFF_POISON))
    strcat(buf, "You are poisoned!\r\n");

  if (IS_AFFECTED(ch, AFF_CHARM))
    strcat(buf, "You have been charmed!\r\n");

  if (affected_by_spell(ch, SPELL_ARMOR))
    strcat(buf, "You feel protected.\r\n");

  if (affected_by_spell(ch, SPELL_SHIELD))
    strcat(buf, "You feel shielded.\r\n");

  if (affected_by_spell(ch, SPELL_STONESKIN))
    strcat(buf, "Your skin is as hard as rock.\r\n");

  if (IS_AFFECTED(ch, AFF_INFRAVISION))
    strcat(buf, "Your eyes are glowing red.\r\n");

  if (PRF_FLAGGED(ch, PRF_SUMMONABLE))
    strcat(buf, "You are summonable by other players.\r\n");

  if (IS_AFFECTED(ch, AFF_WATERWALK))   /*   KAKO   */
    strcat(buf, "Your toes feel webby.\r\n");

  if (IS_AFFECTED(ch, AFF_FLYING))
    strcat(buf, "You feel light on your toes.\r\n");

  if (PRF_FLAGGED(ch, PRF_NOHASSLE))
    strcat(buf, "You shrug off aggression from your inferiors.\r\n");

  if (IS_SET(AFF2_FLAGS(ch), AFF2_DIVINE_PROT))
    strcat(buf, "You are under Divine Protection.\r\n");

  if (PLR_FLAGGED(ch, PLR_AFK))
    strcat(buf, "You are Away From Keyboard.\r\n");

  sprintf(buf, "%sYou are ignoring:", buf);
  if (!ch->ignore)
    strcat(buf, "no-one.\r\n");
  else
    for (ignoring = ch->ignore; ignoring; ignoring = ignoring->next) {
      if (ignoring->next)
        sprintf(buf, "%s %s,", buf, get_name_by_id(ignoring->idnum));
      else
        sprintf(buf, "%s %s.\r\n", buf, get_name_by_id(ignoring->idnum));
    }
  if (GET_BINDHB(ch))
    sprintf(buf,"%sHeartBeat binding: %s\r\n",buf,GET_BINDHB(ch));
  if (GET_BINDWIMP(ch))
    sprintf(buf,"%sWimp binding: %s\r\n",buf,GET_BINDWIMP(ch));
  if (GET_BINDTELL(ch))
    sprintf(buf,"%sTell binding: %s\r\n",buf,GET_BINDTELL(ch));
  if (GET_BINDGT(ch))
    sprintf(buf,"%sGroup Tell binding: %s\r\n",buf,GET_BINDGT(ch));

  send_to_char(buf, ch);
}


#define NUM_WHY_MSSG	9

/* just a little existentialism to lighten the mood ;) */
ACMD(do_why)
{
  char *why_mssg[NUM_WHY_MSSG] = {
    "Why not?",
    "Because.",
    "Just cause.",
    "Because I said so!",
    "Ask your father.",
    "Read the friggin manual!",
    "Ask Blackrose, maybe she knows.",
    "Answer unclear, try again later.",
    "That's the way it's always been."
};

  if(GET_POS(ch) < POS_SLEEPING) {
    send_to_char("Guess you were too slow, pardner.\r\n", ch);
    return;
  }

  send_to_char(why_mssg[number(0, NUM_WHY_MSSG - 1)], ch);
  send_to_char("\r\n", ch);
}


#define WHO_FORMAT \
"format: who [minlev[-maxlev]] [-n name] [-c classlist] [-s] [-o] [-q] [-r] [-z]\r\n"

ACMD(do_who)
{
  struct descriptor_data *d;
  struct char_data *tch;
  char name_search[MAX_INPUT_LENGTH];
  char mode;
  int i, low = 0, high = LVL_SIMP, localwho = 0, questwho = 0;
  int showclass = 0, short_list = 0, outlaws = 0, num_can_see = 0;
  int who_room = 0, showclan = 0;
  char wholist[8192];
  char tbuf[256], color[24];

  skip_spaces(&argument);
  strcpy(buf, argument);
  name_search[0] = '\0';

  *wholist = '\0';

  while (*buf) {
    half_chop(buf, arg, buf1);
    if (isdigit(*arg)) {
      sscanf(arg, "%d-%d", &low, &high);
      strcpy(buf, buf1);
    } else if (*arg == '-') {
      mode = *(arg + 1);	/* just in case; we destroy arg in the switch */
      switch (mode) {
      case 'o':
      case 'k':
	outlaws = 1;
	strcpy(buf, buf1);
	break;
      case 'z':
	localwho = 1;
	strcpy(buf, buf1);
	break;
      case 's':
	short_list = 1;
	strcpy(buf, buf1);
	break;
      case 'q':
	questwho = 1;
	strcpy(buf, buf1);
	break;
      case 'n':
	half_chop(buf1, name_search, buf);
	break;
      case 'r':
	who_room = 1;
	strcpy(buf, buf1);
	break;
      case 'c':
	half_chop(buf1, arg, buf);
	for (i = 0; i < strlen(arg); i++)
	  showclass |= find_class_bitvector(arg[i]);
	break;
      case 'l':
        half_chop(buf1, arg, buf);
        for (i = 0; i < strlen(arg); i++)
          showclan |= find_clan_bitvector(&arg[i]);
        break;
      default:
	send_to_char(WHO_FORMAT, ch);
	return;
	break;
      }				/* end of switch */

    } else {			/* endif */
      send_to_char(WHO_FORMAT, ch);
      return;
    }
  }				/* end while (parser) */


  sprintf(wholist, "&+b-======================================================================-\r\n");
  sprintf(wholist, "%s====-                   &+WCurrent Players on NoMUD                   &+b-====\r\n", wholist);
  sprintf(wholist, "%s-======================================================================-\r\n&+c", wholist);
/*  sprintf(wholist,
"        &+WWANTED, DEAD OR ALIVE:  ANY OF THE FOLLOWING INDIVIDUALS.\r\n"
"   &+WALL ARE VERY ADDICTED TO MUDDING AND SHOULD BE CONSIDERED MODERATELY\r\n"
"                             DANGEROUS.\r\n"
"&+W =============================================================================\r\n&+c");
*/
  for (d = descriptor_list; d; d = d->next) {
    if (d->connected)
      continue;

    if (d->original)
      tch = d->original;
    else if (!(tch = d->character))
      continue;

    if (*name_search && str_cmp(GET_NAME(tch), name_search) &&
	!strstr(GET_TITLE(tch), name_search))
      continue;
    if (!CAN_SEE(ch, tch) || GET_LEVEL(tch) < low || GET_LEVEL(tch) > high)
      continue;
    if (outlaws && !PLR_FLAGGED(tch, PLR_KILLER) &&
	!PLR_FLAGGED(tch, PLR_THIEF))
      continue;
    if (questwho && !PRF_FLAGGED(tch, PRF_QUEST))
      continue;
    if (localwho && world[ch->in_room].zone != world[tch->in_room].zone)
      continue;
    if (who_room && (tch->in_room != ch->in_room))
      continue;
    if (showclass && !(showclass & (1 << GET_CLASS(tch))))
      continue;
    if (showclan && !(showclan & (1 << GET_CLAN(tch))))
      continue;
    sprintf(color, "%s", IS_IMMORTAL(tch) ? CCWHT(ch, C_NRM) : CCCYN(ch, C_NRM));

    if (short_list) {
      sprintf(tbuf, "%3d %s&+c-%s&+c ", GET_LEVEL(tch), RACE_ABBR(tch),
	CLASS_ABBR(tch));
      sprintf(wholist, "%s%s[%s] %-15.15s%s%s", wholist, color,
	      GET_LEVEL(tch) >= LVL_IMMORT ? " Immortal " : tbuf, GET_NAME(tch),
              color, ((!(++num_can_see % 3)) ? "\r\n" : ""));
    } else {
      num_can_see++;
      if (IS_IMMORTAL(tch))
         sprintf(tbuf, " &+WI&+wm&+Wm&+wo&+Wr&+wt&+Wa&+wl ");
      else
        sprintf(tbuf, "%3d %s&+c-%s&+c ", GET_LEVEL(tch), RACE_ABBR(tch),
     	        CLASS_ABBR(tch));

      sprintf(wholist, "%s%s[%s] %s ", wholist, color, tbuf, GET_NAME(tch));

      if (PLR_FLAGGED(tch, PLR_TAGGED))
        sprintf(wholist, "%sis IT! (&+M*&+RIT&+M*%s)", wholist, color);
      else
        sprintf(wholist, "%s%s%s", wholist, GET_TITLE(tch), color);

      if (GET_INVIS_LEV(tch))
	sprintf(wholist, "%s (i%d)", wholist, GET_INVIS_LEV(tch));
      else if (IS_AFFECTED(tch, AFF_INVISIBLE))
	strcat(wholist, " &+Y(invis)");

      if (PLR_FLAGGED(tch, PLR_MAILING))
	strcat(wholist, " &+R(mailing)");
      else if (PLR_FLAGGED(tch, PLR_WRITING))
	strcat(wholist, " &+R(writing)");

      if (PLR_FLAGGED(tch, PLR_ELDER))
        strcat(wholist, " &+M(ELDER)");
      if (PRF_FLAGGED(tch, PRF_QUEST))
	sprintf(wholist, "%s %s(quest)%s", wholist, CCGRN(ch, C_NRM), color);
      if (PLR_FLAGGED(tch, PLR_THIEF))
	sprintf(wholist, "%s %s(THIEF)%s", wholist, CCRED(ch, C_NRM), color);

      if (PLR_FLAGGED(tch, PLR_KILLER))
	sprintf(wholist, "%s %s(KILLER)%s", wholist, CCRED(ch, C_NRM), color);
      if (PLR_FLAGGED(tch, PLR_AFK))
        sprintf(wholist, "%s %s(AFK)%s", wholist, CCBRE(ch, C_NRM), color);
      if (tch->char_specials.timer >= tch->player.idleafk && 
          !PLR_FLAGGED(tch, PLR_AFK))
        sprintf(wholist, "%s %s(idle)%s", wholist, CCBRE(ch, C_NRM), color);
      if (PLR_FLAGGED(tch, PLR_COMPETITION))
        sprintf(wholist, "%s %s(COMPETITION)%s", wholist, CCYEL(ch, C_NRM), color);
      if (GET_CLAN(tch) != CLAN_NONE) {
        sprintf(wholist, "%s %s%s%s", wholist, CCBCY(ch, C_NRM), 
                clans[(int)GET_CLAN(tch)].clan_short, color);
      }
      if (GET_LEVEL(ch) >= LVL_IMMORT && PLR_FLAGGED(tch, PLR_PROBATION))
        sprintf(wholist, "%s %s(PrObAtIoN)", wholist, CCBRE(ch, C_NRM));
/*      if (GET_LEVEL(tch) == LVL_DEMI)
        sprintf(wholist, "%s %s(DEMI-GOD)", wholist, CCMAG(ch, C_NRM)); */
      if (PRF_FLAGGED(tch, PRF_DEAF))
        sprintf(wholist, "%s %s(noshout)", wholist, CCBYE(ch, C_NRM));
      if (PRF_FLAGGED(tch, PRF_NOTELL))
        sprintf(wholist, "%s %s(notell)", wholist, CCBYE(ch, C_NRM));
      if (PRF_FLAGGED(tch, PRF_NOAUCT))
        sprintf(wholist, "%s %s(noauct)", wholist, CCBYE(ch, C_NRM));
      if (PRF_FLAGGED(tch, PRF_NOGOSS))
        sprintf(wholist, "%s %s(nogos)", wholist, CCBYE(ch, C_NRM));
      if (PRF_FLAGGED(tch, PRF_NOGRATZ))
        sprintf(wholist, "%s %s(nograt)", wholist, CCBYE(ch, C_NRM));
      if (GET_CLAN(tch) != CLAN_NONE && PRF_FLAGGED(tch, PRF_NOCLAN))
        sprintf(wholist, "%s %s(noclan)", wholist, CCBGR(ch, C_NRM));
      if (IS_IMMORTAL(tch) && PRF_FLAGGED(tch, PRF_NOWIZ))
        sprintf(wholist, "%s %s(nowiz)", wholist, CCBCY(ch, C_NRM));
      if (PLR_FLAGGED(tch, PLR_AUTOBOT))
        sprintf(wholist, "%s %s(*BOT*)", wholist, CCBMA(ch, C_NRM));
      sprintf(wholist, "%s%s\r\n", wholist, CCNRM(ch, C_NRM));
    }				/* endif shortlist */
  }				/* end of for */
  if (short_list && (num_can_see % 4))
    sprintf(wholist, "%s\r\n", wholist);
  if (num_can_see == 0)
    sprintf(wholist, "%s\r\n&+WYou're not here!\r\n", wholist);
  else if (num_can_see == 1)
    sprintf(wholist, "%s\r\n&+WYou're all alone!\r\n", wholist);
  else
    sprintf(wholist, "%s\r\n%s%d %splayers online.\r\n%s",
      wholist, CCCYN(ch, C_NRM), num_can_see, CCBWH(ch, C_NRM), CCNRM(ch, C_NRM));
  page_string(ch->desc, wholist, 1);
}


#define USERS_FORMAT \
"format: users [-l minlevel[-maxlevel]] [-n name] [-h host] [-c classlist] [-o] [-p]\r\n"

ACMD(do_users)
{
  extern char *connected_types[];
  char line[200], line2[220], idletime[10], classname[30];
  char state[30], *timeptr, *format, mode;
  char name_search[MAX_INPUT_LENGTH], host_search[MAX_INPUT_LENGTH];
  struct char_data *tch;
  struct descriptor_data *d;
  int low = 0, high = LVL_SIMP, i, num_can_see = 0, ip = 0;
  int showclass = 0, outlaws = 0, playing = 0, deadweight = 0;
  char pager[8192];

  host_search[0] = name_search[0] = '\0';

  *pager = '\0';

  strcpy(buf, argument);
  while (*buf) {
    half_chop(buf, arg, buf1);
    if (*arg == '-') {
      mode = *(arg + 1);  /* just in case; we destroy arg in the switch */
      switch (mode) {
      case 'o':
      case 'k':
	outlaws = 1;
	playing = 1;
	strcpy(buf, buf1);
	break;
      case 'p':
	playing = 1;
	strcpy(buf, buf1);
	break;
      case 'd':
	deadweight = 1;
	strcpy(buf, buf1);
	break;
      case 'l':
	playing = 1;
	half_chop(buf1, arg, buf);
	sscanf(arg, "%d-%d", &low, &high);
	break;
      case 'n':
	playing = 1;
	half_chop(buf1, name_search, buf);
	break;
      case 'h':
	playing = 1;
	half_chop(buf1, host_search, buf);
	break;
      case 'c':
	playing = 1;
	half_chop(buf1, arg, buf);
	for (i = 0; i < strlen(arg); i++)
	  showclass |= find_class_bitvector(arg[i]);
	break;
      case 'i':
        playing = 1;
        ip = 1;
        strcpy(buf, buf1);
        break;
      default:
	send_to_char(USERS_FORMAT, ch);
	return;
	break;
      }				/* end of switch */

    } else {			/* endif */
      send_to_char(USERS_FORMAT, ch);
      return;
    }
  }				/* end while (parser) */
  strcpy(pager,
	 "&+rNum Class       Name         Status        Idl Login@   Site\r\n");
  strcat(pager,
	 "&+r--- ----------- ------------ ------------- --- -------- --------------------&+w\r\n");

  one_argument(argument, arg);

  for (d = descriptor_list; d; d = d->next) {
    if (d->connected && playing)
      continue;
    if (!d->connected && deadweight)
      continue;
    if (!d->connected) {
      if (d->original)
	tch = d->original;
      else if (!(tch = d->character))
	continue;

      if (*host_search && !strstr(d->host, host_search))
	continue;
      if (*name_search && str_cmp(GET_NAME(tch), name_search))
	continue;
      if (!CAN_SEE(ch, tch) || GET_LEVEL(tch) < low || GET_LEVEL(tch) > high)
	continue;
      if (outlaws && !PLR_FLAGGED(tch, PLR_KILLER) &&
	  !PLR_FLAGGED(tch, PLR_THIEF))
	continue;
      if (showclass && !(showclass & (1 << GET_CLASS(tch))))
	continue;
      if (GET_INVIS_LEV(ch) > GET_LEVEL(ch))
	continue;

      if (d->original)
	sprintf(classname, "[%3d %s%s-%s%s]", GET_LEVEL(d->original),
		RACE_ABBR(d->original), CCNRM(ch, C_NRM), CLASS_ABBR(d->original),
                CCNRM(ch, C_NRM));
      else
	sprintf(classname, "[%3d %s%s-%s%s]", GET_LEVEL(d->character),
		RACE_ABBR(d->character), CCNRM(ch, C_NRM), CLASS_ABBR(d->character),
                CCNRM(ch, C_NRM));
    } else
      strcpy(classname, "   -   -  ");

    timeptr = asctime(localtime(&d->login_time));
    timeptr += 11;
    *(timeptr + 8) = '\0';

    if (!d->connected && d->original)
      strcpy(state, "&+GSwitched&+w     ");
    else
      strcpy(state, connected_types[d->connected]);

    if (d->character && !d->connected /*&& GET_LEVEL(d->character) < LVL_GOD*/)
      sprintf(idletime, "%3d", d->character->char_specials.timer *
	      SECS_PER_MUD_HOUR / SECS_PER_REAL_MIN);
    else
      strcpy(idletime, "");

    format = "%3d %-8s %-12s %-13s %-3s %-8s ";

    if (d->character && d->character->player.name) {
      if (d->original)
	sprintf(line, format, d->desc_num, classname,
		d->original->player.name, state, idletime, timeptr);
      else
	sprintf(line, format, d->desc_num, classname,
		d->character->player.name, state, idletime, timeptr);
    } else
      sprintf(line, format, d->desc_num, "     -     ", "UNDEFINED",
	      state, idletime, timeptr);

    if (!ip) {
      if (d->host && *d->host)
        sprintf(line + strlen(line), "[%s]\r\n", d->host);
      else
        strcat(line, "[Hostname unknown]\r\n");
    } else {
      if (d->ip && *d->ip)
        sprintf(line + strlen(line), "[%s]\r\n", d->ip);
      else
        strcat(line, "[Hostname unknown]\r\n");
    }

    if (d->connected) {
      sprintf(line2, "%s%s%s", CCGRN(ch, C_SPR), line, CCNRM(ch, C_SPR));
      strcpy(line, line2);
    }
    if (d->connected || (!d->connected && CAN_SEE(ch, d->character))) {
      strcat(pager, line);
      num_can_see++;
    }
  }

  sprintf(pager, "%s\r\n%d visible sockets connected.\r\n", 
     pager, num_can_see);
  page_string(ch->desc, pager, 1);
}




/* Generic page_string function for displaying text */
ACMD(do_gen_ps)
{
  extern char circlemud_version[];

  switch (subcmd) {
  case SCMD_CREDITS:
    page_string(ch->desc, credits, 0);
    break;
  case SCMD_CLANEQ:
    page_string(ch->desc, claneq, 0);
    break;
  case SCMD_INFO:
    page_string(ch->desc, info, 0);
    break;
  case SCMD_WIZLIST:
    page_string(ch->desc, wizlist, 0);
    break;
  case SCMD_IMMLIST:
    page_string(ch->desc, immlist, 0);
    break;
  case SCMD_HANDBOOK:
    page_string(ch->desc, handbook, 0);
    break;
  case SCMD_POLICIES:
    page_string(ch->desc, policies, 0);
    break;
  case SCMD_MOTD:
    page_string(ch->desc, motd, 0);
    break;
  case SCMD_IMOTD:
    page_string(ch->desc, imotd, 0);
    break;
  case SCMD_CLEAR:
    send_to_char("\033[H\033[J", ch);
    break;
  case SCMD_VERSION:
    send_to_char(circlemud_version, ch);
    break;
  case SCMD_WHOAMI:
    send_to_char(strcat(strcpy(buf, GET_NAME(ch)), "\r\n"), ch);
    break;
  case SCMD_LICENSE:
    page_string(ch->desc, circlemud_license, 0);
  default:
    return;
    break;
  }
}


void perform_mortal_where(struct char_data * ch, char *arg)
{
  register struct char_data *i;
  register struct descriptor_data *d;

  if (PLR_FLAGGED(ch, PLR_COMPETITION)) {
    send_to_char("You cannot use this command with the COMPETITION flag.\r\n", ch);
    return;
  }
  if (!*arg) {
    send_to_char("Players in your Zone\r\n--------------------\r\n", ch);
    for (d = descriptor_list; d; d = d->next)
      if (!d->connected) {
	i = (d->original ? d->original : d->character);
	if (i && CAN_SEE(ch, i) && (i->in_room != NOWHERE) &&
	    (world[ch->in_room].zone == world[i->in_room].zone)) {
	  sprintf(buf, "%-20s - %s&+w\r\n", GET_NAME(i), world[i->in_room].name);
	  send_to_char(buf, ch);
	}
      }
  } else {			/* print only FIRST char, not all. */
    for (i = character_list; i; i = i->next)
      if (world[i->in_room].zone == world[ch->in_room].zone && CAN_SEE(ch, i) &&
	  (i->in_room != NOWHERE) && isname(arg, i->player.name)) {
	sprintf(buf, "%-25s - %s&+w\r\n", GET_NAME(i), world[i->in_room].name);
	send_to_char(buf, ch);
	return;
      }
    send_to_char("No-one around by that name.\r\n", ch);
  }
}


void print_object_location(int num, struct obj_data * obj, struct char_data * ch,
			        int recur)
{
  if (num > 0)
    sprintf(buf, "O%3d. %-25s - ", num, obj->short_description);
  else
    sprintf(buf, "%33s", " - ");

  if (obj->in_room > NOWHERE) {
    sprintf(buf + strlen(buf), "[%5d] %s&+w\n\r",
	    world[obj->in_room].number, world[obj->in_room].name);
    send_to_char(buf, ch);
  } else if (obj->carried_by) {
    sprintf(buf + strlen(buf), "carried by %s&+w\n\r",
	    PERS(obj->carried_by, ch));
    send_to_char(buf, ch);
  } else if (obj->worn_by) {
    sprintf(buf + strlen(buf), "worn by %s&+w\n\r",
	    PERS(obj->worn_by, ch));
    send_to_char(buf, ch);
  } else if (obj->in_obj) {
    sprintf(buf + strlen(buf), "inside %s%s&+w\n\r",
	    obj->in_obj->short_description, (recur ? ", which is" : " "));
    send_to_char(buf, ch);
    if (recur)
      print_object_location(0, obj->in_obj, ch, recur);
  } else {
    sprintf(buf + strlen(buf), "in an unknown location\n\r");
    send_to_char(buf, ch);
  }
}



void perform_immort_where(struct char_data * ch, char *arg)
{
  register struct char_data *i;
  register struct obj_data *k;
  struct descriptor_data *d;
  int num = 0, found = 0, search = 0;

  if (!*arg) {
    send_to_char("Players\r\n-------\r\n", ch);
    for (d = descriptor_list; d; d = d->next)
      if (!d->connected) {
	i = (d->original ? d->original : d->character);
	if (i && CAN_SEE(ch, i) && (i->in_room != NOWHERE)) {
	  if (d->original)
	    sprintf(buf, "%-20s - [%5d] %s &+w(in %s&+w)\r\n",
		    GET_NAME(i), world[d->character->in_room].number,
		 world[d->character->in_room].name, GET_NAME(d->character));
	  else
	    sprintf(buf, "%-20s - [%5d] %s&+w\r\n", GET_NAME(i),
		    world[i->in_room].number, world[i->in_room].name);
	  send_to_char(buf, ch);
	}
      }
  } else {
    half_chop(arg, buf, buf1);
    if (*buf=='-') {
      switch(buf[1]) {
        case 'c': search = 1; break;
        case 'w': search = 2; break;
        case 'i': search = 3; break;
        case 'r': search = 4; break;
        case 'n': search = 5; break;
         default: search = 0; break;
      }  
    } else
      one_argument(arg, buf1);
    if (!search) {
    for (i = character_list; i; i = i->next)
      if (CAN_SEE(ch, i) && i->in_room != NOWHERE && isname(buf1, i->player.name)) {
	found = 1;
	sprintf(buf, "M%3d. %-25s - [%5d] %s&+w\r\n", ++num, GET_NAME(i),
		world[i->in_room].number, world[i->in_room].name);
	send_to_char(buf, ch);
      }
    }
    for (num = 0, k = object_list; k; k = k->next)
      if (search == 5 && (GET_OBJ_VNUM(k) == atoi(buf1))) {
        found = 1;
        print_object_location(++num, k, ch, TRUE);
      } else
      if (CAN_SEE_OBJ(ch, k) && isname(buf1, k->name)) { 
        switch(search) {
         case 0: found = 1;
      	         print_object_location(++num, k, ch, TRUE);
      	         break;
      	 case 1: if (k->carried_by || k->worn_by || 
     	            (k->in_obj && (k->in_obj->carried_by || k->in_obj->worn_by))) {
     	           found = 1;
     	           print_object_location(++num, k, ch, TRUE);
     	         }
     	         break;
     	 case 2: if (k->worn_by) {
     	           found = 1;
     	           print_object_location(++num, k, ch, TRUE);
     	         }
     	         break;
     	 case 3: if (k->carried_by ||
     	  	    (k->in_obj && (k->in_obj->carried_by || k->in_obj->worn_by))) {
     	  	   found = 1;
     	  	   print_object_location(++num, k, ch, TRUE);
     	  	 }
     	  	 break;
         case 4: if (k->in_room > NOWHERE ||
                     (k->in_obj && k->in_obj->in_room > NOWHERE)) {
     	  	   found = 1;
     	  	   print_object_location(++num, k, ch, TRUE);
     	  	 }
     	  	 break;
       }          
      }
    if (!found)
      send_to_char("Couldn't find any such thing.\r\n", ch);
  }
}



ACMD(do_where)
{
  one_argument(argument, arg);

  if (GET_LEVEL(ch) >= LVL_IMMORT && subcmd != SCMD_MWHERE)
    perform_immort_where(ch, argument);
  else
    perform_mortal_where(ch, arg);
}



ACMD(do_levels)
{
  int i, cnum;

  if (IS_NPC(ch)) {
    send_to_char("You ain't nothin' but a hound-dog.\r\n", ch);
    return;
  }
  *buf = '\0';
  
  if (*argument) {
    cnum = parse_class(*arg);
    if (cnum == CLASS_UNDEFINED)
      cnum = GET_CLASS(ch);
  } else
    cnum = GET_CLASS(ch);

  for (i = 1; i < LVL_IMMORT; i++) {
    sprintf(buf + strlen(buf), "[%2d] %8d-%-8d : ", i,
	    titles[(int) cnum][i].exp, titles[(int) cnum][i + 1].exp);
    switch (GET_SEX(ch)) {
    case SEX_MALE:
    case SEX_NEUTRAL:
      strcat(buf, titles[(int) cnum][i].title_m);
      break;
    case SEX_FEMALE:
      strcat(buf, titles[(int) cnum][i].title_f);
      break;
    default:
      send_to_char("Oh dear.  You seem to be sexless.\r\n", ch);
      break;
    }
    strcat(buf, "\r\n");
  }
  page_string(ch->desc, buf, 1);
}



ACMD(do_consider)
{
  int diff, vlev, klev;
  struct char_data *victim; 
  char onam[50];

  one_argument(argument, buf);

  if (!(victim = get_char_room_vis(ch, buf))) {
    send_to_char("Consider killing who?\r\n", ch);
    return;
  }
  if (victim == ch) {
    send_to_char("Easy!  Very easy indeed!\r\n", ch);
    return;
  }
  if (!IS_NPC(victim)) {
    send_to_char("Would you like to borrow a cross and a shovel?\r\n", ch);
    return;
  }
  klev = (GET_REMORT(ch) * 100) + GET_LEVEL(ch);
  vlev = (GET_REMORT(victim) * 100) + GET_LEVEL(victim);
  diff = (GET_LEVEL(victim) - GET_LEVEL(ch));

  if (diff <= -10)
    send_to_char("Now where did that chicken go?\r\n", ch);
  else if (diff <= -5)
    send_to_char("You could do it with a needle!\r\n", ch);
  else if (diff <= -2)
    send_to_char("Easy.\r\n", ch);
  else if (diff <= -1)
    send_to_char("Fairly easy.\r\n", ch);
  else if (diff == 0)
    send_to_char("The perfect match!\r\n", ch);
  else if (diff <= 1)
    send_to_char("You would need some luck!\r\n", ch);
  else if (diff <= 2)
    send_to_char("You would need a lot of luck!\r\n", ch);
  else if (diff <= 3)
    send_to_char("You would need a lot of luck and great equipment!\r\n", ch);
  else if (diff <= 5)
    send_to_char("Do you feel lucky, punk?\r\n", ch);
  else if (diff <= 10)
    send_to_char("Are you mad!?\r\n", ch);
  else if (diff <= 20)
    send_to_char("You ARE mad!\r\n", ch);
  else if (diff <= 30)
    send_to_char("This is a very bad idea.\r\n", ch);
  else if (diff <= 40)
    send_to_char("Do you have a death wish?\r\n", ch);
  else if (diff <= 50)
    send_to_char("You will die.\r\n", ch);
  else {
    if (GET_EQ(ch, WEAR_WIELD))
      strcpy(onam, GET_EQ(ch, WEAR_WIELD)->short_description);
    else if (GET_EQ(ch, WEAR_BODY))
      strcpy(onam, GET_EQ(ch, WEAR_BODY)->short_description);
    else
      sprintf(onam, "stuff"); 
    sprintf(buf, "When you die, can I have your %s?\r\n", onam);
    send_to_char(buf, ch);
  }
}



ACMD(do_diagnose)
{
  struct char_data *vict;

  one_argument(argument, buf);

  if (*buf) {
    if (!(vict = get_char_room_vis(ch, buf))) {
      send_to_char(NOPERSON, ch);
      return;
    } else
      diag_char_to_char(vict, ch);
  } else {
    if (FIGHTING(ch))
      diag_char_to_char(FIGHTING(ch), ch);
    else
      send_to_char("Diagnose who?\r\n", ch);
  }
}


static char *ctypes[] = {
"off", "sparse", "normal", "complete", "\n"};

ACMD(do_color)
{
  int tp;

  if (IS_NPC(ch))
    return;

  one_argument(argument, arg);

  if (!*arg) {
    sprintf(buf, "Your current color level is %s.\r\n", ctypes[COLOR_LEV(ch)]);
    send_to_char(buf, ch);
    return;
  }
  if (((tp = search_block(arg, ctypes, FALSE)) == -1)) {
    send_to_char("Usage: color { Off | Sparse | Normal | Complete }\r\n", ch);
    return;
  }
  REMOVE_BIT(PRF_FLAGS(ch), PRF_COLOR_1 | PRF_COLOR_2);
  SET_BIT(PRF_FLAGS(ch), (PRF_COLOR_1 * (tp & 1)) | (PRF_COLOR_2 * (tp & 2) >> 1));

  sprintf(buf, "Your %scolor%s is now %s.\r\n", CCRED(ch, C_SPR),
	  CCNRM(ch, C_OFF), ctypes[tp]);
  send_to_char(buf, ch);
}


ACMD(do_toggle)
{
  if (IS_NPC(ch))
    return;
  if (GET_WIMP_LEV(ch) == 0)
    strcpy(buf2, "OFF");
  else
    sprintf(buf2, "%-3d", GET_WIMP_LEV(ch));

  sprintf(buf,
	  "Hit Pnt Display: %-3s    "
	  "     Brief Mode: %-3s    "
	  " Summon Protect: %-3s\r\n"

	  "   Move Display: %-3s    "
	  "   Compact Mode: %-3s    "
	  "       On Quest: %-3s\r\n"

	  "   Mana Display: %-3s    "
	  "         NoTell: %-3s    "
	  "   Repeat Comm.: %-3s\r\n"

	  " Auto Show Exit: %-3s    "
	  "           Deaf: %-3s    "
	  "     Wimp Level: %-3s\r\n"

	  " Gossip Channel: %-3s    "
	  "Auction Channel: %-3s    "
	  "  Grats Channel: %-3s\r\n"

	  "         NoPage: %-3s    "
	  "    Color Level: %-8s"
	  "   Board Update: %-3s",

	  ONOFF(PRF_FLAGGED(ch, PRF_DISPHP)),
	  ONOFF(PRF_FLAGGED(ch, PRF_BRIEF)),
	  ONOFF(!PRF_FLAGGED(ch, PRF_SUMMONABLE)),

	  ONOFF(PRF_FLAGGED(ch, PRF_DISPMOVE)),
	  ONOFF(PRF_FLAGGED(ch, PRF_COMPACT)),
	  YESNO(PRF_FLAGGED(ch, PRF_QUEST)),

	  ONOFF(PRF_FLAGGED(ch, PRF_DISPMANA)),
	  ONOFF(PRF_FLAGGED(ch, PRF_NOTELL)),
	  YESNO(!PRF_FLAGGED(ch, PRF_NOREPEAT)),

	  ONOFF(PRF_FLAGGED(ch, PRF_AUTOEXIT)),
	  YESNO(PRF_FLAGGED(ch, PRF_DEAF)),
	  buf2,

	  ONOFF(!PRF_FLAGGED(ch, PRF_NOGOSS)),
	  ONOFF(!PRF_FLAGGED(ch, PRF_NOAUCT)),
	  ONOFF(!PRF_FLAGGED(ch, PRF_NOGRATZ)),

	  ONOFF(PRF_FLAGGED(ch, PRF_NOPAGE)),
	  ctypes[COLOR_LEV(ch)],
	  ONOFF(PRF2_FLAGGED(ch, PRF2_BD_CHK)));

  send_to_char(buf, ch);
}


struct sort_struct {
  int sort_pos;
  byte is_social;
} *cmd_sort_info = NULL;

int num_of_cmds;


void sort_commands(void)
{
  int a, b, tmp;

  ACMD(do_action);

  num_of_cmds = 0;

  /*
   * first, count commands (num_of_commands is actually one greater than the
   * number of commands; it inclues the '\n'.
   */
  while (*cmd_info[num_of_cmds].command != '\n')
    num_of_cmds++;

  /* create data array */
  CREATE(cmd_sort_info, struct sort_struct, num_of_cmds);

  /* initialize it */
  for (a = 1; a < num_of_cmds; a++) {
    cmd_sort_info[a].sort_pos = a;
    cmd_sort_info[a].is_social = (cmd_info[a].command_pointer == do_action);
  }

  /* the infernal special case */
  cmd_sort_info[find_command("insult")].is_social = TRUE;

  /* Sort.  'a' starts at 1, not 0, to remove 'RESERVED' */
  for (a = 1; a < num_of_cmds - 1; a++)
    for (b = a + 1; b < num_of_cmds; b++)
      if (strcmp(cmd_info[cmd_sort_info[a].sort_pos].command,
		 cmd_info[cmd_sort_info[b].sort_pos].command) > 0) {
	tmp = cmd_sort_info[a].sort_pos;
	cmd_sort_info[a].sort_pos = cmd_sort_info[b].sort_pos;
	cmd_sort_info[b].sort_pos = tmp;
      }
}



ACMD(do_commands)
{
  int no, i, cmd_num;
  int wizhelp = 0, socials = 0;
  struct char_data *vict;

  one_argument(argument, arg);

  if (*arg) {
    if (!(vict = get_char_vis(ch, arg)) || IS_NPC(vict)) {
      send_to_char("Who is that?\r\n", ch);
      return;
    }
    if (GET_LEVEL(ch) < GET_LEVEL(vict)) {
      send_to_char("You can't see the commands of people above your level.\r\n", ch);
      return;
    }
  } else
    vict = ch;

  if (subcmd == SCMD_SOCIALS)
    socials = 1;
  else if (subcmd == SCMD_WIZHELP)
    wizhelp = 1;

  sprintf(buf, "The following %s%s are available to %s:\r\n",
	  wizhelp ? "privileged " : "",
	  socials ? "socials" : "commands",
	  vict == ch ? "you" : GET_NAME(vict));

  /* cmd_num starts at 1, not 0, to remove 'RESERVED' */
  for (no = 1, cmd_num = 1; cmd_num < num_of_cmds; cmd_num++) {
    i = cmd_sort_info[cmd_num].sort_pos;
    if (cmd_info[i].minimum_level >= 0 &&
	GET_LEVEL(vict) >= cmd_info[i].minimum_level &&
	(cmd_info[i].minimum_level >= LVL_IMMORT) == wizhelp &&
	(wizhelp || socials == cmd_sort_info[i].is_social)) {
      sprintf(buf + strlen(buf), "%-11s", cmd_info[i].command);
      if (!(no % 7))
	strcat(buf, "\r\n");
      no++;
    }
  }

  strcat(buf, "\r\n");
  page_string(ch->desc, buf, 1);
}


/*                KAKO               */
ACMD(do_afk)
{
 if PLR_FLAGGED(ch, PLR_AFK) {
   REMOVE_BIT(PLR_FLAGS(ch), PLR_AFK);
   send_to_char("\007You are no longer Away From Keyboard.\r\n", ch);
   act("\007$n has returned to $s keyboard.  Let the games continue.", TRUE,
       ch, 0, 0, TO_ROOM);
 }  else
 if (!PLR_FLAGGED(ch, PLR_AFK)) {
   SET_BIT(PLR_FLAGS(ch), PLR_AFK);
   send_to_char("\007You are now Away from Keyboard.\r\n", ch);
   act("\007$n is Away From Keyboard.  Time out!", TRUE, ch, 0, 0, TO_ROOM);
 }
 }
 
const char *distance[] =
{
  "Nearby ",
  "At some distance ",
  "Far distant ",
  "\n"
};
 
void scan_chars(struct char_data *list, struct char_data *ch, int dir, int dist)
{
  struct char_data *i;
  int vis = FALSE;
 
  *buf = '\0';
  i = list;
  if ( (!i->next_in_room) && (!CAN_SEE(ch, i)) ) return;
  sprintf(buf, "%s%s you see", distance[dist], dirs[dir]);
  for (; i; i = i->next_in_room)
   if (CAN_SEE(ch, i)) {
    sprintf(buf, "%s, %s", buf, GET_NAME(i));
    vis = TRUE; }
  strcat(buf, ".\r\n");
  if (vis) send_to_char(buf, ch);
}
 
ACMD(do_scan)
{
  int door, room1, room2, room3;
 
  send_to_char("You rotate 360 degrees, scanning the perimeter...\r\n", ch);
  for (door = 0; door < NUM_OF_DIRS; door++) {
   if (EXIT(ch, door)) {
    if (!(room1 = world[ch->in_room].dir_option[door]->to_room))
      continue;
    if (room1 < 0)
      continue;
    if (IS_SET(EXIT(ch, door)->exit_info, EX_CLOSED))
       continue;
    if (world[room1].people)
     scan_chars(world[room1].people, ch, door, 0);
    if (world[room1].dir_option[door]) {
     if (!(room2 = world[room1].dir_option[door]->to_room))
       continue;
     if (room2 < 0)
      continue;
     if (IS_SET(world[room1].dir_option[door]->exit_info, EX_CLOSED))
      continue;
     if (world[room2].people)
      scan_chars(world[room2].people, ch, door, 1);
     if (world[room2].dir_option[door]) {
      if (!(room3 = world[room2].dir_option[door]->to_room))
        continue;
      if (room3 < 0)
       continue;
     if (IS_SET(world[room2].dir_option[door]->exit_info, EX_CLOSED))
       continue;
      if (world[room3].people)
       scan_chars(world[room3].people, ch, door, 2);
     }
    }
   }
  }
 }
 

int valid_address(char *str)
{
  char *ptr;

  for(ptr = str; *ptr; ptr++)
    if(!((*ptr >= 'a' && *ptr <= 'z') || (*ptr >= 'A' && *ptr <= 'Z') ||
      isdigit(*ptr) || *ptr == ':' || *ptr == '/' || *ptr == '-' ||
      *ptr == '_' || *ptr == '+' || *ptr == '@' || *ptr == '~' ||
      *ptr == '.'))
      return FALSE;

  return TRUE;
}


ACMD(do_email)
{
  char buf[1024];

  if(!*argument) {
    send_to_char("Usage  : email <your email address>\r\n\r\n", ch);
    send_to_char("Example: email somebody@somwhere.com\r\n", ch);
    send_to_char("This command will be phased out in lieu of the config command\r\n", ch);
    return;
  }

  skip_spaces(&argument);

  if (is_abbrev(argument, "-del")) {
    if (!ch->player.email)
      return;
    free(ch->player.email);
    ch->player.email = NULL;
    send_to_char("Email removed.\r\n",ch);
    return;
  }
  if(!valid_address(argument)) {
    send_to_char("Invalid email address.  If this is really your address,\r\n", ch);
    send_to_char("mudmail it to Samedi so he can fix the address checker.\r\n", ch);
    return;
  }

  if(ch->player.email)
    free(ch->player.email);
  ch->player.email = str_dup(argument);

  sprintf(buf, "Changing your email address to: %s\r\n", argument);
  sprintf(buf, "%sThis command will be phased out in lieu of the \"config\" command.\r\n", buf);
  send_to_char(buf, ch);
}


ACMD(do_homepage)
{
  char buf[1024];

  if(!*argument) {
    send_to_char("Usage  : homepage <your web address>\r\n\r\n", ch);
    send_to_char("Note that you shouldn't precede your address with an 'http://'\r\n\r\n", ch);
    send_to_char("Example: homepage www.somewhere.com/~somebody\r\n", ch);
    send_to_char("This command will be phased out in lieu of the \"config\" command.\r\n", ch);
    return;
  }

  skip_spaces(&argument);

  if (is_abbrev(argument, "-del")) {
    if (!ch->player.homepage)
      return;
    free(ch->player.homepage);
    ch->player.homepage = NULL;
    send_to_char("Homepage removed.\r\n", ch);
    return;
  }
  if(!valid_address(argument)) {
    send_to_char("Invalid homepage address.  If this is really your address,\r\n", ch);
    send_to_char("mudmail it to Samedi so he can fix the address checker.\r\n", ch);
    return;
  }

  if(ch->player.homepage)
    free(ch->player.homepage);
  ch->player.homepage = str_dup(argument);

  sprintf(buf, "Changing your homepage address to: %s\r\n", argument);
  send_to_char(buf, ch);
  send_to_char("This command will be phased out in lieu of the \"config\" command.\r\n", ch);
}


ACMD(do_finger)
{
  char class[64];
  struct char_file_u tmp;
  extern char *pc_class_types[];
  extern char *genders[];
  extern char *race_types[];

  if(!*argument) {
    send_to_char("Usage: finger <player name>\r\n", ch);
    return;
  }

  skip_spaces(&argument);

  if(load_char(argument, &tmp, NULL) > -1) {
    if(IS_SET(tmp.char_specials_saved.act, PLR_NOFINGER)) {
      send_to_char("That player wishes to remain anonymous.\r\n", ch);
      return;
    }
    sprinttype(tmp.class, pc_class_types, class);
    sprintf(buf, "&+cName       &+W: &+B%-15s &+cSex    &+W: &+B%-12s\r\n", 
      tmp.name, genders[(int) tmp.sex]);
    sprintf(buf, "%s&+cClass      &+W: &+B%-15s &+cRace   &+W: &+B%-12s",
      buf, class, race_types[(int)tmp.player_specials_saved.race]);
    sprintf(buf, "%s &+cLevel  &+W: &+B%d\r\n", buf, tmp.level);
    sprintf(buf, "%s&+cClan       &+W: &+B%s\r\n", buf,
      clans[(int)tmp.player_specials_saved.clan].clan_long);
    if(tmp.email)
      sprintf(buf, "%s&+cEmail      &+W: &+B%s\r\n", buf, tmp.email);
    if(tmp.homepage)
      sprintf(buf, "%s&+cHomepage   &+W: &+B%s\r\n", buf, tmp.homepage);
    sprintf(buf, "%s&+cLast Login &+W: &+B%s\r\n", buf, ctime(&tmp.last_logon));
    if(tmp.description && *tmp.description != '\0')
      sprintf(buf, "%s&+cDescription&+W:&+B\r\n%s\r\n", buf, tmp.description);
    send_to_char(buf, ch);
  } else
    send_to_char("There is no such player.\r\n", ch);
}

ACMD(do_affects)
{
  struct affected_type *aff;
  *buf='\0';
  if (ch->affected) {
    for (aff = ch->affected; aff; aff = aff->next) {
      sprintf(buf, "%sSPL: (%3dhr) %s%-21s%s\r\n", buf, aff->duration + 1,
              CCCYN(ch, C_NRM), spells[aff->type], CCNRM(ch, C_NRM));
    }
  } else
    sprintf(buf, "You have no spells affecting you.\r\n");
  page_string(ch->desc, buf, 1);
}

ACMD(do_whatis)
{
  int tmp, length, found = FALSE;

  one_argument(argument, arg);

  if (!*arg)
    return;
  for (length = strlen(arg), tmp = 0; *cmd_info[tmp].command != '\n'; tmp++)
    if (!strncmp(cmd_info[tmp].command, arg, length))
      if (GET_LEVEL(ch) >= cmd_info[tmp].minimum_level) {
        found = TRUE;
	break;
      }

  if (found)
    sprintf(buf, "The command you searched for is: %s\r\n", 
            cmd_info[tmp].command);
  else
    sprintf(buf, "There is no command for that.\r\n");
  send_to_char(buf, ch);
}

