/* ************************************************************************
*   File: act.movement.c                                Part of CircleMUD *
*  Usage: movement commands, door handling, & sleep/rest/etc state        *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"


#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "handler.h"
#include "db.h"
#include "spells.h"
#include "house.h"
#include "screen.h"

/* external vars  */
extern struct room_data *world;
extern struct char_data *character_list;
extern struct descriptor_data *descriptor_list;
extern struct index_data *obj_index;
extern int rev_dir[];
extern char *dirs[];
extern int movement_loss[];
extern sh_int r_mortal_start_room;
extern struct clan_stuff clans[]; /*clanroom*/

/* external functs */
int special(struct char_data *ch, int cmd, char *arg);
void death_cry(struct char_data *ch);
void perform_dismount(struct char_data *ch, struct char_data *mount);
int find_eq_pos(struct char_data * ch, struct obj_data * obj, char *arg);




  

/* do_simple_move assumes
 *    1. That there is no master and no followers.
 *    2. That the direction exists.
 *
 *   Returns :
 *   1 : If succes.
 *   0 : If fail
 */
int do_simple_move(struct char_data *ch, int dir, int need_specials_check)
{
  int was_in, need_movement;
  int has_wings = 0, has_scuba = 0, has_boat = 0, i;
  struct obj_data *obj;
  struct char_data *d;

#define CHR	(ch->riding ? ch->riding : ch)
#define CHRB    (ch->ridden_by ? ch->ridden_by : ch)
#define MOUNTED (ch->riding || ch->ridden_by)
  int special(struct char_data * ch, int cmd, char *arg);

  /*
   * Check for special routines (North is 1 in command list, but 0 here) Note
   * -- only check if following; this avoids 'double spec-proc' bug
   */
  if (need_specials_check && special(ch, dir + 1, ""))
    return 0;
   if (IS_AFFECTED(CHR, AFF_WEB) || IS_AFFECTED(CHRB, AFF_WEB)) {
    send_to_char("You can't move!  You're webbed to this spot!\r\n", CHRB);
    return 0;
  }
  if ( (GET_POS(ch) == POS_RIDING) &&  /* Can this guy ride? */
      (GET_SKILL(ch, SKILL_RIDING) < number(1, 101)) ) {
   act("$N suddenly rears and you fall out of the saddle.", FALSE, ch, 0, 
       ch->riding, TO_CHAR);
   act("$n falls off your back.. ooh $e must be a SKILLED rider!", FALSE, 
       ch, 0, ch->riding, TO_VICT);
act("$n suddenly starts waving $s arms frantically, then falls off $N and \r\nlands on $s kiester. Hehehe",
       FALSE, ch, 0, ch->riding, TO_NOTVICT);
   damage(ch, ch, number(30, 50), TYPE_FALL);
   perform_dismount(ch, ch->riding);
   return 0; } 
  /* charmed? */
  if (IS_AFFECTED(ch, AFF_CHARM) && ch->master && ch->in_room == ch->master->in_room) {
    send_to_char("The thought of leaving your master makes you weep.\r\n", ch);
    act("$n bursts into tears.", FALSE, ch, 0, 0, TO_ROOM);
    return 0;
  }
  for (obj = CHR->carrying; obj; obj = obj->next_content) {
      if (GET_OBJ_TYPE(obj) == ITEM_BOAT)
	has_boat = TRUE;
      if (GET_OBJ_TYPE(obj) == ITEM_FLIGHT)
         has_wings = TRUE;
      if (GET_OBJ_TYPE(obj) == ITEM_SCUBA)
         has_scuba = TRUE;
   }
  for (i = 0; i < NUM_WEARS; i++)
    if ((obj = CHR->equipment[i])) {
      if (GET_OBJ_TYPE(obj) == ITEM_FLIGHT)
         has_wings = TRUE;
      if (GET_OBJ_TYPE(obj) == ITEM_SCUBA)
         has_scuba = TRUE;
      if (GET_OBJ_TYPE(obj) == ITEM_BOAT)
         has_boat = TRUE;
    }

  if (IS_AFFECTED(CHR, AFF_WATERWALK))
     has_boat = TRUE;
  if (IS_AFFECTED(CHR, AFF_FLYING))
     has_wings = TRUE;
  if (IS_AFFECTED(CHR, AFF_WATERBREATH))
     has_scuba = TRUE;
  if ( has_wings )
     has_boat = TRUE;

  /* if this room or the one we're going to needs a boat, check for one */
  if ((world[ch->in_room].sector_type == SECT_WATER_NOSWIM) ||
      (world[EXIT(ch, dir)->to_room].sector_type == SECT_WATER_NOSWIM)) {
    if (!has_boat && GET_LEVEL(CHR) < LVL_IMMORT) {
      send_to_char("You need a boat to go there.\r\n", ch);
      return 0;
    }
  }
  /* KAKO -- Can you fly? */
  if ((world[ch->in_room].sector_type == SECT_FLYING) ||
      (world[EXIT(ch, dir)->to_room].sector_type == SECT_FLYING)) {
    if (!has_wings && GET_LEVEL(CHR) < LVL_IMMORT) {
      send_to_char("You'd drop like a rock.\r\n", ch);
      return 0;
    }
  }
  /* KAKO -- Can you breathe underwater? */
  if (world[ch->in_room].sector_type == SECT_UNDERWATER) {
    if (!has_scuba && GET_LEVEL(ch) < LVL_IMMORT) {
      send_to_char("You can't hold your breath THAT long.\r\n", ch);
      return 0;
    }
  }
  if ( has_wings && 
      (world[EXIT(ch, dir)->to_room].sector_type != SECT_UNDERWATER) ) 
    need_movement = 2; 
  else
    need_movement = (movement_loss[world[ch->in_room].sector_type] +
  		     movement_loss[world[world[ch->in_room].dir_option[dir]->to_room].sector_type]) >> 1;

  if (CHR->ridden_by) need_movement *= 2;

  if (GET_MOVE(CHR) < need_movement && !IS_NPC(CHR)) {
    if (need_specials_check && CHR->master)
      send_to_char("You are too exhausted to follow.\r\n", CHR);
    else
      send_to_char("You are too exhausted.\r\n", CHR);

    return 0;
  }

/*  if (IS_SET(ROOM_FLAGS(EXIT(ch, dir)->to_room), ROOM_IMPL)
        && GET_LEVEL(ch) < LVL_IMPL ) {
    send_to_char("Are you an implementor? I think NOT.\r\n", ch);
    return 0;
  } */

  if (IS_SET(ROOM_FLAGS(EXIT(ch, dir)->to_room), ROOM_SIMP)
      && GET_LEVEL(ch) < LVL_SIMP) {
   send_to_char("Only coders are allowed there.\r\n", ch);
   return 0;
  }

//  if (IS_SET(ROOM_FLAGS(ch->in_room), ROOM_ATRIUM)) {
//    if (!House_can_enter(ch, world[EXIT(ch, dir)->to_room].number)) {
//      send_to_char("That's private property -- no trespassing!\r\n", ch);
//      return 0;
//    }
//  }

  if (IS_SET(ROOM_FLAGS(EXIT(ch, dir)->to_room), ROOM_TUNNEL) &&
      num_pc_in_room(&(world[EXIT(ch, dir)->to_room])) > 1) {
    send_to_char("There isn't enough room there for more than one person!\r\n", ch);
    return 0;
  }

/*clanroom*/
   if (GET_LEVEL(ch) < LVL_SIMP && ROOM_FLAGGED(EXIT(ch, 
     dir)->to_room, ROOM_CLANONLY) && 
     !strstr(world[EXIT(ch, dir)->to_room].name, clans[(int)GET_CLAN(ch)].clan_short)) {
      send_to_char("You're not allowed in here.\r\n", ch);
      return 0;
  }

  if(MOUNTED && ROOM_FLAGGED(EXIT(ch, dir)->to_room, ROOM_CLANONLY)) {
    send_to_char("Mounts won't fit in the clan hall.\r\n", ch);
    return 0;
  }

  if (GET_LEVEL(CHR) < LVL_IMMORT && !IS_NPC(CHR))
    GET_MOVE(CHR) -= need_movement;

  if (MOUNTED) { 
   sprintf(buf, "You ride %s.\r\n", dirs[dir]);
   send_to_char(buf, ch); 
   send_to_char(buf, ch->riding ? ch->riding : ch->ridden_by); }

  if (!IS_AFFECTED(ch, AFF_SNEAK) || MOUNTED) {
    sprintf(buf2, "$n leaves %s.", dirs[dir]);
    sprintf(buf,  "$n rides away %s on $N.", dirs[dir]);
    if (MOUNTED) act(buf, TRUE, CHRB, 0, CHR, TO_NOTVICT);
      else act(buf2, TRUE, ch, 0, 0, TO_ROOM);
  } else
    for (d = world[ch->in_room].people; d; d = d->next_in_room)
      if (( (GET_LEVEL(d) >= LVL_IMMORT) ||
          ( (GET_RACE(ch) == GET_RACE(d)) && !IS_NPC(ch) )) &&
             (CAN_SEE(d, ch)) && (d != ch) ) {
         sprintf(buf2,"$n sneaks off %s.", dirs[dir]);
         act(buf2, TRUE, ch, 0, d, TO_VICT);
      }

  was_in = ch->in_room;
  char_from_room(ch);
/*  if (ch->riding) char_from_room(ch->riding); */
  char_to_room(ch, world[was_in].dir_option[dir]->to_room);
/*  if (ch->riding)  char_to_room(ch->riding, 
                   world[was_in].dir_option[dir]->to_room); */

  if (!IS_AFFECTED(ch, AFF_SNEAK) || MOUNTED) {
    sprintf(buf2, "$n enters.");
    sprintf(buf,  "$n rides in on $N.");
    if (MOUNTED) act(buf, TRUE, CHRB, 0, CHR, TO_NOTVICT);
      else act(buf2, TRUE, ch, 0, 0, TO_ROOM);
  } else
    for (d = world[ch->in_room].people; d; d = d->next_in_room)
      if (( (GET_LEVEL(d) >= LVL_IMMORT) ||
          ( (GET_RACE(ch) == GET_RACE(d)) && !IS_NPC(ch) )) &&
             (CAN_SEE(d, ch)) && (d != ch) ) {
         sprintf(buf2,"$n sneaks in.");
         act(buf2, TRUE, ch, 0, d, TO_VICT);
      }

/*
  if (!IS_AFFECTED(ch, AFF_SNEAK) || MOUNTED) 
   if (MOUNTED) act("$n rides in on $N.", TRUE, CHRB, 0, CHR, TO_NOTVICT);
    else act("$n has arrived.",TRUE, ch, 0, 0, TO_ROOM);
  else
    for (d = world[ch->in_room].people; d; d = d->next_in_room)
      if (( (GET_LEVEL(d) >= LVL_IMMORT) ||
          ( (GET_RACE(ch) == GET_RACE(d)) && !IS_NPC(ch) )) &&
             (CAN_SEE(d, ch)) && (ch != d) ) 
        act("$n sneaks in.", TRUE, ch, 0, d, TO_VICT);
*/

  if (ch->riding) look_at_room(CHR, 0);
  if (ch->ridden_by) look_at_room(CHRB, 0);
  look_at_room(ch, 0);

  if (world[ch->in_room].damage.dam) {
    if (!IS_IMMORTAL(ch)) {
      if (world[ch->in_room].damage.mesg)
        send_to_char(world[ch->in_room].damage.mesg, ch);
      damage(ch, ch, world[ch->in_room].damage.dam, TYPE_DEATHTRAP);
      return 0;
    }
  }
  if (IS_SET(ROOM_FLAGS(ch->in_room), ROOM_DEATH) && GET_LEVEL(ch) < LVL_IMMORT) {
      if(IS_NPC(ch)) {
        death_cry(ch);
        extract_char(ch);
        return 0;
      } else {
    sprintf(buf,"%s                      You have fallen into Death Trap!!!%s", CCRED(ch, C_NRM), CCNRM(ch, C_NRM));
    send_to_char(buf, ch);
    log_death_trap(ch);
    death_cry(ch);
    GET_LOADROOM(ch) = world[r_mortal_start_room].number;
    GET_HIT(ch) = 1;
    extract_char(ch);
    return 0;
  } }
  return 1;
}



int perform_move(struct char_data *ch, int dir, int need_specials_check)
{
  int was_in;
  struct follow_type *k, *next;

  if (ch == NULL || dir < 0 || dir >= NUM_OF_DIRS)
    return 0;
  else if (!EXIT(ch, dir) || EXIT(ch, dir)->to_room == NOWHERE)
    send_to_char("Alas, you cannot go that way...\r\n", ch);
  else if (IS_SET(EXIT(ch, dir)->exit_info, EX_CLOSED)) {
    if (IS_SET(EXIT(ch, dir)->exit_info, EX_SECRET) &&
        !IS_IMMORTAL(ch) && !(IS_AFFECTED(ch, AFF_DET_DOOR) && 
        number(1,101) > 70))
      send_to_char("Alas, you cannot go that way...\r\n", ch);
    else {
      if (EXIT(ch, dir)->keyword) {
        sprintf(buf2, "The %s seems to be closed.\r\n", fname(EXIT(ch, dir)->keyword));
        send_to_char(buf2, ch);
      } else
        send_to_char("It seems to be closed.\r\n", ch);
    }
  } else {
    if (!ch->followers)
      return (do_simple_move(ch, dir, need_specials_check));

    was_in = ch->in_room;
    if (!do_simple_move(ch, dir, need_specials_check))
      return 0;

    for (k = ch->followers; k; k = next) {
      next = k->next;
      if ((k->follower->in_room == was_in) &&
	  (GET_POS(k->follower) >= POS_STANDING)) {
	act("You follow $N.\r\n", FALSE, k->follower, 0, ch, TO_CHAR);
	perform_move(k->follower, dir, 1);
      }
    }
    return 1;
  }
  return 0;
}


ACMD(do_doorbash)
{
  return;
}


ACMD(do_move)
{
  /*
   * This is basically a mapping of cmd numbers to perform_move indices.
   * It cannot be done in perform_move because perform_move is called
   * by other functions which do not require the remapping.
   */
  perform_move(ch, cmd - 1, 0);
}


int find_door(struct char_data *ch, char *type, char *dir, char *cmdname)
{
  int door;

  if (*dir) {			/* a direction was specified */
    if ((door = search_block(dir, dirs, FALSE)) == -1) {	/* Partial Match */
      send_to_char("That's not a direction.\r\n", ch);
      return -1;
    }
    if (EXIT(ch, door))
      if (EXIT(ch, door)->keyword)
	if (isname(type, EXIT(ch, door)->keyword))
	  return door;
	else {
	  sprintf(buf2, "I see no %s there.\r\n", type);
	  send_to_char(buf2, ch);
	  return -1;
      } else
	return door;
    else {
      send_to_char("I really don't see how you can close anything there.\r\n", ch);
      return -1;
    }
  } else {			/* try to locate the keyword */
    if (!*type) {
      sprintf(buf2, "What is it you want to %s?\r\n", cmdname);
      send_to_char(buf2, ch);
      return -1;
    }
    for (door = 0; door < NUM_OF_DIRS; door++)
      if (EXIT(ch, door))
	if (EXIT(ch, door)->keyword)
	  if (isname(type, EXIT(ch, door)->keyword))
	    return door;

    sprintf(buf2, "There doesn't seem to be %s %s here.\r\n", AN(type), type);
    send_to_char(buf2, ch);
    return -1;
  }
}


int has_key(struct char_data *ch, int key)
{
  struct obj_data *o;

  for (o = ch->carrying; o; o = o->next_content)
    if (GET_OBJ_VNUM(o) == key)
      return 1;

  if (GET_EQ(ch, WEAR_HOLD))
    if (GET_OBJ_VNUM(GET_EQ(ch, WEAR_HOLD)) == key)
      return 1;

  return 0;
}



#define NEED_OPEN	1
#define NEED_CLOSED	2
#define NEED_UNLOCKED	4
#define NEED_LOCKED	8

char *cmd_door[] =
{
  "open",
  "close",
  "unlock",
  "lock",
  "pick"
};

const int flags_door[] =
{
  NEED_CLOSED | NEED_UNLOCKED,
  NEED_OPEN,
  NEED_CLOSED | NEED_LOCKED,
  NEED_CLOSED | NEED_UNLOCKED,
  NEED_CLOSED | NEED_LOCKED
};


#define EXITN(room, door)		(world[room].dir_option[door])
#define OPEN_DOOR(room, obj, door)	((obj) ?\
		(TOGGLE_BIT(GET_OBJ_VAL(obj, 1), CONT_CLOSED)) :\
		(TOGGLE_BIT(EXITN(room, door)->exit_info, EX_CLOSED)))
#define LOCK_DOOR(room, obj, door)	((obj) ?\
		(TOGGLE_BIT(GET_OBJ_VAL(obj, 1), CONT_LOCKED)) :\
		(TOGGLE_BIT(EXITN(room, door)->exit_info, EX_LOCKED)))

void do_doorcmd(struct char_data *ch, struct obj_data *obj, int door, int scmd)
{
  int other_room = 0;
  struct room_direction_data *back = 0;
  struct obj_data *gift;

  sprintf(buf, "$n %ss ", cmd_door[scmd]);
  if (!obj && ((other_room = EXIT(ch, door)->to_room) != NOWHERE))
    if ((back = world[other_room].dir_option[rev_dir[door]]))
      if (back->to_room != ch->in_room)
	back = 0;

  switch (scmd) {
  case SCMD_OPEN:
     if (obj) {
       if (GET_OBJ_TYPE(obj)==ITEM_GIFT) {
        if (GET_OBJ_VAL(obj, 2) != GET_IDNUM(ch) && 
            GET_OBJ_VAL(obj, 3) != GET_IDNUM(ch)) {
         send_to_char("Tsk tsk... open your own presents!!\r\n", ch); return;}
        gift = read_object(real_object(GET_OBJ_VAL(obj, 0)), REAL);
        obj_to_char(gift, ch);
        act("You open the gift... *excitement*", TRUE, ch, 0, 0, TO_CHAR);
        act("$n rips off the wrapping and opens the box...", TRUE, ch, 0, 0, TO_ROOM);
        act("It's a $p!", TRUE, ch, gift, 0, TO_ROOM);
        act("It's a $p!", TRUE, ch, gift, 0, TO_CHAR);
        extract_obj(obj);
       return;
       } 
     }
  case SCMD_CLOSE:
    OPEN_DOOR(ch->in_room, obj, door);
    if (back)
      OPEN_DOOR(other_room, obj, rev_dir[door]);
    send_to_char(OK, ch);
    break;
  case SCMD_UNLOCK:
  case SCMD_LOCK:
    LOCK_DOOR(ch->in_room, obj, door);
    if (back)
      LOCK_DOOR(other_room, obj, rev_dir[door]);
    send_to_char("*Click*\r\n", ch);
    break;
  case SCMD_PICK:
    LOCK_DOOR(ch->in_room, obj, door);
    if (back)
      LOCK_DOOR(other_room, obj, rev_dir[door]);
    send_to_char("The lock quickly yields to your skills.\r\n", ch);
    strcpy(buf, "$n skillfully picks the lock on ");
    break;
  }

  /* Notify the room */
  sprintf(buf + strlen(buf), "%s%s.", ((obj) ? "" : "the "), (obj) ? "$p" :
	  (EXIT(ch, door)->keyword ? "$F" : "door"));
  if (!(obj) || (obj->in_room != NOWHERE))
    act(buf, FALSE, ch, obj, obj ? 0 : EXIT(ch, door)->keyword, TO_ROOM);

  /* Notify the other room */
  if ((scmd == SCMD_OPEN || scmd == SCMD_CLOSE) && back) {
    sprintf(buf, "The %s is %s%s from the other side.\r\n",
	 (back->keyword ? fname(back->keyword) : "door"), cmd_door[scmd],
	    (scmd == SCMD_CLOSE) ? "d" : "ed");
    if (world[EXIT(ch, door)->to_room].people) {
      act(buf, FALSE, world[EXIT(ch, door)->to_room].people, 0, 0, TO_ROOM);
      act(buf, FALSE, world[EXIT(ch, door)->to_room].people, 0, 0, TO_CHAR);
    }
  }
}


int ok_pick(struct char_data *ch, int keynum, int pickproof, int scmd)
{
  int percent;
  int probability;
  struct obj_data *pick;

  percent = number(1, 101);
  if(GET_RACE(ch) == RACE_HOBBIT || GET_RACE(ch) == RACE_DWARF)
   percent -= 10;  /* add racial pick lock adjustments - Treb 9/96 */

  probability = GET_SKILL(ch, SKILL_PICK_LOCK);

  if (scmd == SCMD_PICK) {
  if (!(pick = ch->equipment[WEAR_HOLD])) {
    send_to_char("You can't pick a lock without a lockpick!\r\n", ch);
    return(0);
  }
  if (GET_OBJ_TYPE(pick)!=ITEM_LOCKPICK) {
    send_to_char("You can't pick a lock without a lockpick!\r\n", ch);
    return(0);
  }
  if (GET_OBJ_VAL(pick, 1)) {
    send_to_char("Um, these picks are broken...\r\n", ch);
    return(0);
   }
   probability = GET_SKILL(ch, SKILL_PICK_LOCK) + GET_OBJ_VAL(pick, 0);

    if (keynum < 0)
      send_to_char("Odd - you can't seem to find a keyhole.\r\n", ch);
    else if (pickproof)
      send_to_char("It resists your attempts at picking it.\r\n", ch);
    else if (percent > probability) {
      send_to_char("You failed to pick the lock.\r\n", ch);
      if ( (percent - probability) > 30 ) {
       act("*snap* DOH!  You broke your $p!", TRUE, ch, pick, 0, TO_CHAR);
       act("With an audible *SNAP*, $n breaks $s $p", TRUE, ch, pick, 0, TO_ROOM);
       GET_OBJ_VAL(pick, 1)++;
      }
    } else
      return (1);
    return (0);
  }
  return (1);
}


#define DOOR_IS_OPENABLE(ch, obj, door)	((obj) ? \
			((GET_OBJ_TYPE(obj) == ITEM_CONTAINER || \
                          GET_OBJ_TYPE(obj) == ITEM_GIFT) && \
			(IS_SET(GET_OBJ_VAL(obj, 1), CONT_CLOSEABLE))) :\
			(IS_SET(EXIT(ch, door)->exit_info, EX_ISDOOR)))
#define DOOR_IS_OPEN(ch, obj, door)	((obj) ? \
			(!IS_SET(GET_OBJ_VAL(obj, 1), CONT_CLOSED)) :\
			(!IS_SET(EXIT(ch, door)->exit_info, EX_CLOSED)))
#define DOOR_IS_UNLOCKED(ch, obj, door)	((obj) ? \
			(!IS_SET(GET_OBJ_VAL(obj, 1), CONT_LOCKED)) :\
			(!IS_SET(EXIT(ch, door)->exit_info, EX_LOCKED)))
#define DOOR_IS_PICKPROOF(ch, obj, door) ((obj) ? \
			(IS_SET(GET_OBJ_VAL(obj, 1), CONT_PICKPROOF)) : \
			(IS_SET(EXIT(ch, door)->exit_info, EX_PICKPROOF)))

#define DOOR_IS_CLOSED(ch, obj, door)	(!(DOOR_IS_OPEN(ch, obj, door)))
#define DOOR_IS_LOCKED(ch, obj, door)	(!(DOOR_IS_UNLOCKED(ch, obj, door)))
#define DOOR_KEY(ch, obj, door)		((obj) ? (GET_OBJ_VAL(obj, 2)) : \
					(EXIT(ch, door)->key))
#define DOOR_LOCK(ch, obj, door)	((obj) ? (GET_OBJ_VAL(obj, 1)) : \
					(EXIT(ch, door)->exit_info))

ACMD(do_gen_door)
{
  int door = -1, keynum;
  char type[MAX_INPUT_LENGTH], dir[MAX_INPUT_LENGTH];
  struct obj_data *obj = NULL;
  struct char_data *victim = NULL;

  skip_spaces(&argument);
  if (!*argument) {
    sprintf(buf, "%s what?\r\n", cmd_door[subcmd]);
    send_to_char(CAP(buf), ch);
    return;
  }
  two_arguments(argument, type, dir);
  if (!generic_find(type, FIND_OBJ_INV | FIND_OBJ_ROOM, ch, &victim, &obj))
    door = find_door(ch, type, dir, cmd_door[subcmd]);

  if ((obj) || (door >= 0)) {
    keynum = DOOR_KEY(ch, obj, door);
    if (!(DOOR_IS_OPENABLE(ch, obj, door)))
      act("You can't $F that!", FALSE, ch, 0, cmd_door[subcmd], TO_CHAR);
    else if (!DOOR_IS_OPEN(ch, obj, door) &&
	     IS_SET(flags_door[subcmd], NEED_OPEN))
      send_to_char("But it's already closed!\r\n", ch);
    else if (!DOOR_IS_CLOSED(ch, obj, door) &&
	     IS_SET(flags_door[subcmd], NEED_CLOSED))
      send_to_char("But it's currently open!\r\n", ch);
    else if (!(DOOR_IS_LOCKED(ch, obj, door)) &&
	     IS_SET(flags_door[subcmd], NEED_LOCKED))
      send_to_char("Oh.. it wasn't locked, after all..\r\n", ch);
    else if (!(DOOR_IS_UNLOCKED(ch, obj, door)) &&
	     IS_SET(flags_door[subcmd], NEED_UNLOCKED))
      send_to_char("It seems to be locked.\r\n", ch);
    else if (!has_key(ch, keynum) && (GET_LEVEL(ch) < LVL_GOD) &&
	     ((subcmd == SCMD_LOCK) || (subcmd == SCMD_UNLOCK)))
      send_to_char("You don't seem to have the proper key.\r\n", ch);
    else if (ok_pick(ch, keynum, DOOR_IS_PICKPROOF(ch, obj, door), subcmd))
      do_doorcmd(ch, obj, door, subcmd);
  }
  return;
}



ACMD(do_enter)
{
  int door;

  one_argument(argument, buf);

  if (*buf) {			/* an argument was supplied, search for door
				 * keyword */
    for (door = 0; door < NUM_OF_DIRS; door++)
      if (EXIT(ch, door))
	if (EXIT(ch, door)->keyword)
	  if (!str_cmp(EXIT(ch, door)->keyword, buf)) {
	    perform_move(ch, door, 1);
	    return;
	  }
    sprintf(buf2, "There is no %s here.\r\n", buf);
    send_to_char(buf2, ch);
  } else if (IS_SET(ROOM_FLAGS(ch->in_room), ROOM_INDOORS))
    send_to_char("You are already indoors.\r\n", ch);
  else {
    /* try to locate an entrance */
    for (door = 0; door < NUM_OF_DIRS; door++)
      if (EXIT(ch, door))
	if (EXIT(ch, door)->to_room != NOWHERE)
	  if (!IS_SET(EXIT(ch, door)->exit_info, EX_CLOSED) &&
	      IS_SET(ROOM_FLAGS(EXIT(ch, door)->to_room), ROOM_INDOORS)) {
	    perform_move(ch, door, 1);
	    return;
	  }
    send_to_char("You can't seem to find anything to enter.\r\n", ch);
  }
}


ACMD(do_leave)
{
  int door;

  if (!IS_SET(ROOM_FLAGS(ch->in_room), ROOM_INDOORS))
    send_to_char("You are outside.. where do you want to go?\r\n", ch);
  else {
    for (door = 0; door < NUM_OF_DIRS; door++)
      if (EXIT(ch, door))
	if (EXIT(ch, door)->to_room != NOWHERE)
	  if (!IS_SET(EXIT(ch, door)->exit_info, EX_CLOSED) &&
	    !IS_SET(ROOM_FLAGS(EXIT(ch, door)->to_room), ROOM_INDOORS)) {
	    perform_move(ch, door, 1);
	    return;
	  }
    send_to_char("I see no obvious exits to the outside.\r\n", ch);
  }
}


ACMD(do_stand)
{
  switch (GET_POS(ch)) {
  case POS_STANDING:
    act("You are already standing.", FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_SITTING:
    act("You stand up.", FALSE, ch, 0, 0, TO_CHAR);
    act("$n clambers to $s feet.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_STANDING;
    break;
  case POS_RESTING:
    act("You stop resting, and stand up.", FALSE, ch, 0, 0, TO_CHAR);
    act("$n stops resting, and clambers on $s feet.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_STANDING;
    break;
  case POS_SLEEPING:
    act("You have to wake up first!", FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_FIGHTING:
    act("Do you not consider fighting as standing?", FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_RIDING:
    act("Let's try dismounting first...", FALSE, ch, 0, 0, TO_CHAR);
    break;
  default:
    act("You stop floating around, and put your feet on the ground.",
	FALSE, ch, 0, 0, TO_CHAR);
    act("$n stops floating around, and puts $s feet on the ground.",
	TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_STANDING;
    break;
  }
}


ACMD(do_sit)
{
  switch (GET_POS(ch)) {
  case POS_STANDING:
    act("You sit down.", FALSE, ch, 0, 0, TO_CHAR);
    act("$n sits down.", FALSE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_SITTING;
    break;
  case POS_SITTING:
    send_to_char("You're sitting already.\r\n", ch);
    break;
  case POS_RESTING:
    act("You stop resting, and sit up.", FALSE, ch, 0, 0, TO_CHAR);
    act("$n stops resting.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_SITTING;
    break;
  case POS_SLEEPING:
    act("You have to wake up first.", FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_FIGHTING:
    act("Sit down while fighting? are you MAD?", FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_RIDING:
    act("You're sittin' tall in saddle already, pard'ner!", 
        FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_RIDDEN:
    act("You'd love to sit down, but that big behemoth on your back won't let you!",
        FALSE, ch, 0, 0, TO_CHAR);
    break;
  default:
    act("You stop floating around, and sit down.", FALSE, ch, 0, 0, TO_CHAR);
    act("$n stops floating around, and sits down.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_SITTING;
    break;
  }
}


ACMD(do_rest)
{
  switch (GET_POS(ch)) {
  case POS_STANDING:
    act("You sit down and rest your tired bones.", FALSE, ch, 0, 0, TO_CHAR);
    act("$n sits down and rests.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_RESTING;
    break;
  case POS_SITTING:
    act("You rest your tired bones.", FALSE, ch, 0, 0, TO_CHAR);
    act("$n rests.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_RESTING;
    break;
  case POS_RESTING:
    act("You are already resting.", FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_SLEEPING:
    act("You have to wake up first.", FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_FIGHTING:
    act("Rest while fighting?  Are you MAD?", FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_RIDING:
    act("Your conscience won't let you rest until you get off this poor creature.", 
        FALSE, ch, 0, 0, TO_CHAR);
    break;
  case POS_RIDDEN:
    act("You just can't rest until you get that guy off your back.",
        FALSE, ch, 0, 0, TO_CHAR);
    break;
  default:
    act("You stop floating around, and stop to rest your tired bones.",
	FALSE, ch, 0, 0, TO_CHAR);
    act("$n stops floating around, and rests.", FALSE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_SITTING;
    break;
  }
}


ACMD(do_sleep)
{
  switch (GET_POS(ch)) {
  case POS_STANDING:
  case POS_SITTING:
  case POS_RESTING:
    send_to_char("You go to sleep.\r\n", ch);
    act("$n lies down and falls asleep.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_SLEEPING;
    break;
  case POS_SLEEPING:
    send_to_char("You are already sound asleep.\r\n", ch);
    break;
  case POS_FIGHTING:
    send_to_char("Sleep while fighting?  Are you MAD?\r\n", ch);
    break;
  case POS_RIDING:
    send_to_char("Don't sleep.. it's a long fall.\r\n", ch);
    break;
  case POS_RIDDEN:
    send_to_char("You can't sleep with this guy on top of you.\r\n", ch);
  default:
    act("You stop floating around, and lie down to sleep.",
	FALSE, ch, 0, 0, TO_CHAR);
    act("$n stops floating around, and lie down to sleep.",
	TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_SLEEPING;
    break;
  }
}


ACMD(do_wake)
{
  struct char_data *vict;
  int self = 0;

  one_argument(argument, arg);
  if (*arg) {
    if (GET_POS(ch) == POS_SLEEPING)
      send_to_char("Maybe you should wake yourself up first.\r\n", ch);
    else if ((vict = get_char_room_vis(ch, arg)) == NULL)
      send_to_char(NOPERSON, ch);
    else if (vict == ch)
      self = 1;
    else if (GET_POS(vict) > POS_SLEEPING)
      act("$E is already awake.", FALSE, ch, 0, vict, TO_CHAR);
    else if (IS_AFFECTED(vict, AFF_SLEEP))
      act("You can't wake $M up!", FALSE, ch, 0, vict, TO_CHAR);
    else if (GET_POS(vict) < POS_SLEEPING)
      act("$E's in pretty bad shape!", FALSE, ch, 0, vict, TO_CHAR);
    else {
      act("You wake $M up.", FALSE, ch, 0, vict, TO_CHAR);
      act("You are awakened by $n.", FALSE, ch, 0, vict, TO_VICT | TO_SLEEP);
      GET_POS(vict) = POS_STANDING;
    }
    if (!self)
      return;
  }
  if (IS_AFFECTED(ch, AFF_SLEEP))
    send_to_char("You can't wake up!\r\n", ch);
  else if (GET_POS(ch) > POS_SLEEPING)
    send_to_char("You are already awake...\r\n", ch);
  else {
    send_to_char("You awaken, and stand up.\r\n", ch);
    act("$n awakens.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_STANDING;
  }
}


ACMD(do_follow)
{
  struct char_data *leader;

  void stop_follower(struct char_data *ch);
  void add_follower(struct char_data *ch, struct char_data *leader);

  one_argument(argument, buf);

  if (*buf) {
    if (!(leader = get_char_room_vis(ch, buf))) {
      send_to_char(NOPERSON, ch);
      return;
    }
  } else {
    send_to_char("Whom do you wish to follow?\r\n", ch);
    return;
  }

  if (ch->master == leader) {
    act("You are already following $M.", FALSE, ch, 0, leader, TO_CHAR);
    return;
  }
  if (MOB_FLAGGED(ch, MOB_FAMILIAR)) {
   act("No, your familiar only wants to serve you!", TRUE, ch->master, 0, 0,
       TO_CHAR);
   return;
  }
  if (IS_AFFECTED(ch, AFF_CHARM) && (ch->master)) {
    act("But you only feel like following $N!", FALSE, ch, 0, ch->master, TO_CHAR);
  } else {			/* Not Charmed follow person */
    if (leader == ch) {
      if (!ch->master) {
	send_to_char("You are already following yourself.\r\n", ch);
	return;
      }
      stop_follower(ch);
    } else {
      if (circle_follow(ch, leader)) {
	act("Sorry, but following in loops is not allowed.", FALSE, ch, 0, 0, TO_CHAR);
	return;
      }
      if (ch->master)
	stop_follower(ch);
      REMOVE_BIT(AFF_FLAGS(ch), AFF_GROUP);
      add_follower(ch, leader);
    }
  }
}

ACMD(do_jump)      
{
  int src_room, mid_room = 0, dest_room = 0;
  int percent, prob;
  int dir, can_jump = 0;
  if (GET_POS(ch)==POS_RIDDEN) {
    if (number(1,100)<=50) {
      act("$n gathers up $s strength and heaves $N off $s back.",
          TRUE, ch, 0, ch->ridden_by, TO_NOTVICT);
      act("You are suddenly airborne as $n heaves you off $s back.", TRUE,
          ch, 0, ch->ridden_by, TO_VICT);
      act("With a grunt you give $N the heave-ho!", TRUE, ch, 0, 
          ch->ridden_by, TO_CHAR);
      GET_POS(ch)=POS_STANDING;GET_POS(ch->ridden_by)=POS_STANDING;
      ch->ridden_by->riding = NULL; ch->ridden_by = NULL; 
      return; 
    } else { 
      act("You try in vain to heave $N off your back.", 
          TRUE, ch, 0, ch->ridden_by, TO_CHAR);
      return; 
    }
  }
  if (GET_POS(ch) != POS_STANDING) {
   send_to_char("You gotta be standing!\r\n", ch);
   return; }
  if (!*argument) {
    act("$n jumps up and down hurriedly... maybe $e has to potty.", 
        TRUE, ch, 0, 0, TO_ROOM);
    send_to_char("Having fun?\r\n", ch);
    return;
  }
  one_argument(argument, arg);
  if ((dir = search_block(arg, dirs, FALSE)) < 0) {
    send_to_char("You call THAT a direction?\r\n", ch);
    return;
  }
  if ((GET_MOVE(ch) - 20) < 0) {
    send_to_char("*YAWN* You don't really feel up to jumping right now.\r\n", ch);
    return;
  }
  percent = number(1, 101);
  prob = GET_SKILL(ch, SKILL_JUMP);
  if (percent > prob) {
    send_to_char("You get ready to leap, then you stumble and fall.\r\n", ch);
    return;
  }

  src_room = ch->in_room;
  if (world[src_room].dir_option[dir]) {
    mid_room = world[src_room].dir_option[dir]->to_room;
    if (mid_room != NOWHERE && world[mid_room].dir_option[dir]) {
      dest_room = world[mid_room].dir_option[dir]->to_room;
        if (dest_room != NOWHERE)
          can_jump = TRUE;
    }
  }
  if (!can_jump) {
    send_to_char("You try to leap, then notice that you'll go *SPLAT* on a wall.\r\n", ch);
    return;
  }
  if ( (world[dest_room].sector_type == SECT_UNDERWATER) ||
       (world[dest_room].sector_type == SECT_FLYING) ||
       (world[dest_room].sector_type == SECT_WATER_NOSWIM) ) {
	send_to_char("Hmm, you can't quite go there. Sorry...\r\n", ch);
	return;
  }

  if ( (IS_SET(world[src_room].dir_option[dir]->exit_info, EX_CLOSED)) ||
       (IS_SET(world[mid_room].dir_option[dir]->exit_info, EX_CLOSED)) ) {
     send_to_char("You can't jump through a door :P\r\n", ch);
     return;
  }

  if ( (ROOM_FLAGGED(mid_room, ROOM_PRIVATE) && 
       world[mid_room].people && world[mid_room].people->next_in_room) ||
       (ROOM_FLAGGED(mid_room, ROOM_HOUSE) &&
       !House_can_enter(ch, world[mid_room].number)) ||
	ROOM_FLAGGED(mid_room, ROOM_CLANONLY) /*clanroom*/) {
       send_to_char("You cannot jump through that room.\r\n", ch);
       return;
  }

  if ( (ROOM_FLAGGED(dest_room, ROOM_PRIVATE) && 
       world[dest_room].people && world[dest_room].people->next_in_room) ||
       (ROOM_FLAGGED(dest_room, ROOM_HOUSE) &&
       !House_can_enter(ch, world[dest_room].number)) ||
	ROOM_FLAGGED(dest_room, ROOM_CLANONLY) /*clanroom*/) {
       send_to_char("You cannot jump into that room.\r\n", ch);
       return;
  }    

  send_to_char("You get a running start and LEAP!\r\n", ch);
  act("$n backs up a few feet, runs, and LEAPS out of the room.",
      FALSE, ch, 0, 0, TO_ROOM);

  char_from_room(ch);
  GET_MOVE(ch) -= 20;
  char_to_room(ch, dest_room);

  act("$n suddenly flies by you and crashes to the ground.",
      FALSE, ch, 0, 0, TO_ROOM);

  look_at_room(ch, 0);

  if (IS_SET(ROOM_FLAGS(ch->in_room), ROOM_DEATH) && GET_LEVEL(ch) < LVL_IMMORT) {
    log_death_trap(ch);
    death_cry(ch);
    GET_LOADROOM(ch) = world[r_mortal_start_room].number;
    extract_char(ch);
  }
} 
  

ACMD(do_come)
{
  struct char_data *vict = NULL;
  struct follow_type *k;
  int found = FALSE;
  
  if (ch->followers)
   for (k = ch->followers; k; k = k->next)
    if (MOB_FLAGGED(k->follower, MOB_FAMILIAR)) {
     found = TRUE;
     vict = k->follower;
    }
 
  if (!found) {
   send_to_char("Just who do you expect to come?\r\n", ch);
   return;
  }
  act("$n disappears suddenly, gone back to its master.", 
      TRUE, vict, 0, 0, TO_ROOM);
  char_from_room(vict);
  char_to_room(vict, ch->in_room);
  act("$N suddenly appears in the room and alights on $n's shoulder.",
      TRUE, ch, 0, vict, TO_ROOM);
  act("$N suddenly appears and alights on your shoulder.",
      TRUE, ch, 0, vict, TO_CHAR);
}

ACMD(do_mount)
{
  struct char_data *mount;
  int percent;

  one_argument(argument, arg);

 if (ch->riding) {
  send_to_char("Go get yourself a stagecoach!\r\n", ch);
  return;
 }
 if (ch->ridden_by) {
  send_to_char("If you wanna make a pyramid, go be a cheerleader!", ch);
  return;
 }
 if (!(mount = get_char_room_vis(ch, arg))) {
  send_to_char("Hmmm, riding an imaginary steed, eh?  Why don't you just imagine mounting it too?\r\n", ch);
  return; }
 if (mount == ch) {
  send_to_char("Alright, you mount yourself on the wall.  Now everyone can see what a dunce you are.\r\n", ch);
  return; }
 if (!MOB_FLAGGED(mount, MOB_MOUNTABLE) && IS_NPC(mount)) {
  send_to_char("Um, you can't ride that...\r\n", ch);
  return;  }
 if (PLR_FLAGGED(mount, PLR_AUTOBOT)) {
  sprintf(buf, "%s attempted to mount BOT %s in %s (%d)", GET_NAME(ch),
          GET_NAME(mount), world[ch->in_room].name, world[ch->in_room].number);
  mudlog(buf, BRF, LVL_IMMORT, TRUE);
  send_to_char("Want your own private healer?  Think again :P~\r\n", ch);
  return; }
 if (PRF2_FLAGGED(mount, PRF2_NOMOUNT)) {
  act("You can't mount $m.  $n is flagged NOMOUNT.", TRUE, mount, 0, ch, TO_VICT);
  return; }
 if (GET_POS(mount) != POS_STANDING) {
  act("You can't mount $m.  Not standing.", TRUE, mount, 0, ch, TO_VICT);
  return; }
 /* MOUNT */
 percent = number(1, 101);
 if (percent > GET_SKILL(ch, SKILL_RIDING)) {
  act("You make a leap for $N's back, but $E moves out of the way! Hmph!",
      FALSE, ch, 0 , mount, TO_CHAR);
  act("Watch out for that $n character... I think $e wants a ride.", 
      FALSE, ch, 0, mount, TO_VICT);
  act("$n makes a running leap at $N but ends up eating dirt instead.", 
      FALSE, ch, 0, mount, TO_NOTVICT);
  return; }
 ch->riding = mount;
 mount->ridden_by = ch;
 act("You climb up on $N's back.", FALSE, ch, 0, mount, TO_CHAR);
 act("$n climbs on your back. *oomph!*", FALSE, ch, 0, mount, TO_VICT);
 act("$n climbs up on $N's back.", FALSE, ch, 0, mount, TO_NOTVICT);
 GET_POS(ch) = POS_RIDING;
 GET_POS(mount) = POS_RIDDEN;
}

void perform_dismount(struct char_data *ch, struct char_data *mount)
{
 if (!mount || !ch) {
  nmlog("SYSERR: dismount by non-rider!");
  return;
 }
 if (mount->ridden_by != ch || ch->riding != mount) 
  nmlog("SYSERR: dismount by non-rider!");
 if (ch) {
     ch->riding = NULL; ch->ridden_by = NULL;
     GET_POS(ch) = POS_STANDING;
 }
 if (mount) {
     mount->riding = NULL; mount->ridden_by = NULL;
     GET_POS(mount) = POS_STANDING;
 }
}

ACMD(do_dismount)
{
 struct char_data *mount;

 if (!(mount = ch->riding)) {
  send_to_char("You want to dismount... the earth?\r\n", ch);
  return;
 }
 
 act("$n climbs off $N's back.  $N looks quite relieved.\r\n", FALSE, ch, 0,
     mount, TO_NOTVICT);
 act("You gracefully and expertly dismount $N.", FALSE, ch, 0, mount, 
     TO_CHAR);
 act("$n climbs down off your back.  That's a load off your back! *phew*", 
FALSE, ch, 0, mount, TO_VICT);
perform_dismount(ch, mount);
}

