/* ************************************************************************
*   File: act.offensive.c                               Part of CircleMUD *
*  Usage: player-level commands of an offensive nature                    *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"


#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "handler.h"
#include "db.h"
#include "spells.h"
#include "screen.h"

/* extern variables */
extern struct room_data *world;
extern struct descriptor_data *descriptor_list;
extern struct room_data *world;
extern struct time_info_data time_info;
extern int pk_allowed;
extern int rev_dir[];

/* extern functions */
void raw_kill(struct char_data * ch);
int ok_damage_shopkeeper(struct char_data * ch, struct char_data * victim);
int mag_savingthrow(struct char_data * ch, int type, int bonus);


ACMD(do_assist)
{
  struct char_data *helpee, *opponent;

  if (FIGHTING(ch)) {
    send_to_char("You're already fighting!  How can you assist someone else?\r\n", ch);
    return;
  }
  one_argument(argument, arg);

  if (!*arg)
    send_to_char("Whom do you wish to assist?\r\n", ch);
  else if (!(helpee = get_char_room_vis(ch, arg)))
    send_to_char(NOPERSON, ch);
  else if (helpee == ch)
    send_to_char("You can't help yourself any more than this!\r\n", ch);
  else {
    for (opponent = world[ch->in_room].people;
	 opponent && (FIGHTING(opponent) != helpee);
	 opponent = opponent->next_in_room)
		;

    if (!opponent)
      act("But nobody is fighting $M!", FALSE, ch, 0, helpee, TO_CHAR);
    else if (!CAN_SEE(ch, opponent))
      act("You can't see who is fighting $M!", FALSE, ch, 0, helpee, TO_CHAR);
    else if (!pk_allowed && !IS_NPC(opponent) && 
             !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK))	/* prevent accidental pkill */
      act("Use 'murder' if you really want to attack $N.", FALSE,
	  ch, 0, opponent, TO_CHAR);
    else {
      send_to_char("You join the fight!\r\n", ch);
      act("$N assists you!", 0, helpee, 0, ch, TO_CHAR);
      act("$n assists $N.", FALSE, ch, 0, helpee, TO_NOTVICT);
      hit(ch, opponent, TYPE_UNDEFINED);
    }
  }
}


ACMD(do_hit)
{
  struct char_data *vict;

  one_argument(argument, arg);

  if (!*arg)
    send_to_char("Hit who?\r\n", ch);
  else if (!(vict = get_char_room_vis(ch, arg)))
    send_to_char("They don't seem to be here.\r\n", ch);
  else if (vict == ch) {
    send_to_char("You hit yourself...OUCH!.\r\n", ch);
    act("$n hits $mself, and says OUCH!", FALSE, ch, 0, vict, TO_ROOM);
  } else if (IS_AFFECTED(ch, AFF_CHARM) && (ch->master == vict))
    act("$N is just such a good friend, you simply can't hit $M.", FALSE, ch, 0, vict, TO_CHAR);
  else if (MOB_FLAGGED(vict, MOB_FAMILIAR) && (vict->master == ch))
    send_to_char("You don't want to kill your own familiar...\r\n", ch);
  else if (MOB_FLAGGED(ch, MOB_FAMILIAR) && (ch->master == vict))
    send_to_char("Why would your familiar want to attack you?\r\n", vict);
  else if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) 
    send_to_char("It is too tranquil here to fight.\r\n", ch);
  else if ((vict->master) && (vict->master != ch) && (!IS_NPC(ch)))
    send_to_char("You can't kill that!\r\n", ch);
  else {
    if (!pk_allowed && !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
      if (!IS_NPC(vict) && !IS_NPC(ch) && (subcmd != SCMD_MURDER)) {
	send_to_char("Use 'murder' to hit another player.\r\n", ch);
	return;
      }
      if (IS_AFFECTED(ch, AFF_CHARM) && !IS_NPC(ch->master) && !IS_NPC(vict))
	return;			/* you can't order a charmed pet to attack a
				 * player */
    }
    if ((GET_POS(ch) == POS_STANDING || GET_POS(ch) == POS_RIDING) && 
       (vict != FIGHTING(ch))) {
      hit(ch, vict, TYPE_UNDEFINED);
      WAIT_STATE(ch, PULSE_VIOLENCE + 2);
    } else
      send_to_char("You do the best you can!\r\n", ch);
  }
}



ACMD(do_kill)
{
  struct char_data *vict;

if (subcmd != SCMD_BUTCHER) {
   do_hit(ch, argument, cmd, subcmd);
   return;
  }
  one_argument(argument, arg);

  if (!*arg) {
    send_to_char("Kill who?\r\n", ch);
  } else {
    if (!(vict = get_char_room_vis(ch, arg)))
      send_to_char("They aren't here.\r\n", ch);
    else if (ch == vict)
      send_to_char("Your mother would be so sad.. :(\r\n", ch);
    else {
      act("You chop $M to pieces!  Ah!  The blood!", FALSE, ch, 0, vict, TO_CHAR);
      act("$N chops you to pieces!", FALSE, vict, 0, ch, TO_CHAR);
      act("$n brutally slays $N!", FALSE, ch, 0, vict, TO_NOTVICT);
      if (!IS_NPC(vict)) {
        sprintf(buf, "%s butchered by %s at %s (#%d)", GET_NAME(vict), GET_NAME(ch),
                world[vict->in_room].name, world[vict->in_room].number);
        mudlog(buf, NRM, LVL_IMMORT, TRUE);           
      }
      raw_kill(vict);
      
    }
  }
}



ACMD(do_backstab)
{
  struct char_data *vict;
  int percent, prob;
  int double_bs = FALSE; 

  one_argument(argument, buf);

  if ( !IS_THIEF(ch) && !IS_APALADIN(ch) && !IS_IMMORTAL(ch) && !IS_NPC(ch) ) {
    send_to_char("You'd better leave the sneaky arts to rogues.\r\n", ch);
    return;
  }
  if (!(vict = get_char_room_vis(ch, buf))) {
    send_to_char("Backstab who?\r\n", ch);
    return;
  }
  if (vict == ch) {
    send_to_char("How can you sneak up on yourself?\r\n", ch);
    return;
  }
  if (!GET_EQ(ch, WEAR_WIELD)) {
    send_to_char("You need to wield a weapon to make it a success.\r\n", ch);
    return;
  }
  if (GET_OBJ_VAL(GET_EQ(ch, WEAR_WIELD), 3) != TYPE_PIERCE - TYPE_HIT) {
    send_to_char("Only piercing weapons can be used for backstabbing.\r\n", ch);
    return;
  }
  if (!IS_NPC(vict) && (vict != FIGHTING(ch)) && !pk_allowed && 
      !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
    send_to_char("Backstab is not a social!\r\n", ch);
    return;
   }
 
/*double */
  if ((ch->equipment[WEAR_HOLD]) &&
      (GET_OBJ_TYPE(ch->equipment[WEAR_HOLD]) == ITEM_WEAPON) &&
      (GET_OBJ_VAL(ch->equipment[WEAR_HOLD], 3) == TYPE_PIERCE - TYPE_HIT))
    double_bs = TRUE;
    
  if (FIGHTING(vict)) {
    send_to_char("You can't backstab a fighting person -- they're too alert!\r\n", ch);
    return;
  }

  if (MOB_FLAGGED(vict, MOB_AWARE)) {
    act("You notice $N lunging at you!", FALSE, vict, 0, ch, TO_CHAR);
    act("$e notices you lunging at $m!", FALSE, vict, 0, ch, TO_VICT);
    act("$n notices $N lunging at $m!", FALSE, vict, 0, ch, TO_NOTVICT);
    hit(vict, ch, TYPE_UNDEFINED);
    return;
  }

  percent = number(1, 101);	/* 101% is a complete failure */
  if (GET_RACE(ch) == RACE_HOBBIT) 
    percent -= 10;  /* add racial modifier for backstab - Treb 9/96 */
/*double */
  if(double_bs) {
    prob = GET_SKILL(ch, SKILL_DOUBLE_BS);
    if(AWAKE(vict) && (percent > prob)) {
      double_bs = FALSE;
      prob = GET_SKILL(ch, SKILL_BACKSTAB);
    }
  } else
    prob = GET_SKILL(ch, SKILL_BACKSTAB);


  if (AWAKE(vict) && (percent > prob))
    damage(ch, vict, 0, SKILL_BACKSTAB);
  else 
    if (double_bs)
      hit(ch, vict, SKILL_DOUBLE_BS);
    else
      hit(ch, vict, SKILL_BACKSTAB);
  
}

ACMD(do_circle)
{
  struct char_data *vict;
  int percent, prob;

  one_argument(argument, buf);

  if ( !IS_THIEF(ch) && !IS_IMMORTAL(ch) && !IS_NPC(ch) ) {
    send_to_char("You'd better leave the sneaky arts to the rogues.\r\n", ch);
    return;
  }
  if (!FIGHTING(ch)) {
    send_to_char("Use this skill only during combat.\r\n", ch);
    return;
  }
  if (!(vict = get_char_room_vis(ch, buf))) {
    if (!(vict = FIGHTING(ch))) {
      send_to_char("Circle around whom?\r\n", ch);
      return;
    }
  }
  if (!FIGHTING(vict)) {
    send_to_char("This skill is not used to start fights.\r\n", ch);
    return;
  }
  if (FIGHTING(vict) == ch) {
    act("$E is paying too much attention to you to circle.", 
        TRUE, ch, 0, vict, TO_CHAR);
    return;
  }
  if (!GET_EQ(ch, WEAR_WIELD)) {
    send_to_char("You need to wield a weapon to make it a success.\r\n", ch);
    return;
  }
  if (GET_OBJ_VAL(GET_EQ(ch, WEAR_WIELD), 3) != TYPE_PIERCE - TYPE_HIT) {
    send_to_char("Only piercing weapons can be used for circle backstab.\r\n", ch);
    return;
  }
  if (!IS_NPC(vict) && (vict != FIGHTING(ch)) && !pk_allowed &&
      !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
    send_to_char("Circle is not a social!\r\n", ch);
    return;
   }
 
  if (MOB_FLAGGED(vict, MOB_AWARE)) {
    act("You notice $N lunging at you!", FALSE, vict, 0, ch, TO_CHAR);
    act("$e notices you lunging at $m!", FALSE, vict, 0, ch, TO_VICT);
    act("$n notices $N lunging at $m!", FALSE, vict, 0, ch, TO_NOTVICT);
    hit(vict, ch, TYPE_UNDEFINED);
    return;
  }

  percent = number(1, 101);	/* 101% is a complete failure */
  if (GET_RACE(ch) == RACE_HOBBIT) 
    percent -= 10;  /* add racial modifier for backstab - Treb 9/96 */
    prob = GET_SKILL(ch, SKILL_CIRCLE);

  act("You attempt to circle around $N for a backstab.", TRUE, ch, 0, vict, TO_CHAR);
  act("$n attempts to circle around $N for a backstab.", TRUE, ch, 0, vict, TO_ROOM);
  act("$n attempt to circle around you for a backstab.", TRUE, ch, 0, vict, TO_VICT);

  if (AWAKE(vict) && (percent > prob)) {
    damage(ch, vict, 0, SKILL_CIRCLE);
    WAIT_STATE(ch, PULSE_VIOLENCE * 2);
  } else {
    hit(ch, vict, SKILL_CIRCLE); 
    WAIT_STATE(ch, PULSE_VIOLENCE * 5);
  }
}


ACMD(do_order)
{
  char name[100], message[256];
  char buf[256];
  bool found = FALSE;
  int org_room;
  struct char_data *vict;
  struct follow_type *k;

  half_chop(argument, name, message);

  if (!*name || !*message)
    send_to_char("Order who to do what?\r\n", ch);
  if (is_abbrev(name, "familiar") && ch->followers) {
    sprintf(buf, "You order your familiar to '%s'.\r\n", message);
    send_to_char(buf, ch);
    for (k = ch->followers; k; k = k->next)
     if (MOB_FLAGGED(k->follower, MOB_FAMILIAR))
       command_interpreter(k->follower, message);
    }
  else if (!(vict = get_char_room_vis(ch, name)) && !is_abbrev(name, "followers"))
    send_to_char("That person isn't here.\r\n", ch);
  else if (ch == vict)
    send_to_char("You obviously suffer from skitzofrenia.\r\n", ch);

  else {
    if (IS_AFFECTED(ch, AFF_CHARM)) {
      send_to_char("Your superior would not aprove of you giving orders.\r\n", ch);
      return;
    }
    if (vict) {
      sprintf(buf, "$N orders you to '%s'", message);
      act(buf, FALSE, vict, 0, ch, TO_CHAR);
      act("$n gives $N an order.", FALSE, ch, 0, vict, TO_ROOM);

      if ((vict->master != ch) || !IS_AFFECTED(vict, AFF_CHARM))
	act("$n has an indifferent look.", FALSE, vict, 0, 0, TO_ROOM);
      else {
	send_to_char(OK, ch);
	command_interpreter(vict, message);
      }
    } else {			/* This is order "followers" */
      sprintf(buf, "$n issues the order '%s'.", message);
      act(buf, FALSE, ch, 0, vict, TO_ROOM);

      org_room = ch->in_room;

      for (k = ch->followers; k; k = k->next) {
	if (org_room == k->follower->in_room)
	  if (IS_AFFECTED(k->follower, AFF_CHARM)) {
	    found = TRUE;
	    command_interpreter(k->follower, message);
	  }
      }
      if (found)
	send_to_char(OK, ch);
      else
	send_to_char("Nobody here is a loyal subject of yours!\r\n", ch);
    }
  }
}



ACMD(do_flee)
{
  int i, attempt, loss;

  if (IS_AFFECTED(ch, AFF_BERSERK)) {
   send_to_char("You're too frenzied to want to flee.\r\n", ch);
   return;
  }
  if (ch->player.roundsout)
    return;

  if (cmd && PLR_FLAGGED(ch, PLR_COMPETITION)) {
   send_to_char("You cannot flee from this battle.\r\n", ch);
   return;
  }

  for (i = 0; i < 6; i++) {
    attempt = number(0, NUM_OF_DIRS - 1);	/* Select a random direction */
    if (CAN_GO(ch, attempt) &&
	!IS_SET(ROOM_FLAGS(EXIT(ch, attempt)->to_room), ROOM_DEATH)) {
      if(IS_NPC(ch) && IS_SET(ROOM_FLAGS(EXIT(ch, attempt)->to_room), 
         ROOM_NOMOB)) {
      act("$n tries to flee, but can't!", TRUE, ch, 0, 0, TO_ROOM);
      return;
      }
      act("$n panics, and attempts to flee!", TRUE, ch, 0, 0, TO_ROOM);
      if (do_simple_move(ch, attempt, TRUE)) {
	send_to_char("You flee head over heels.\r\n", ch);
	if (FIGHTING(ch)) {
	  if (!IS_NPC(ch)) {
	    loss = GET_MAX_HIT(FIGHTING(ch)) - GET_HIT(FIGHTING(ch));
	    loss *= GET_LEVEL(FIGHTING(ch));
	    gain_exp(ch, -loss);
	  }
	  if (FIGHTING(FIGHTING(ch)) == ch)
	    stop_fighting(FIGHTING(ch));
	  stop_fighting(ch);
	}
      } else {
	act("$n tries to flee, but can't!", TRUE, ch, 0, 0, TO_ROOM);
      }
      return;
    }
  }
  send_to_char("PANIC!  You couldn't escape!\r\n", ch);
}


ACMD(do_bash)
{
  struct char_data *vict;
  int percent, prob;

  one_argument(argument, arg);

  if ( !IS_WARRIOR(ch) && !IS_PALADIN(ch) && !IS_MONK(ch) && !IS_IMMORTAL(ch) 
       && !IS_NPC(ch) ) {
    send_to_char("You'd better leave all the martial arts to fighters.\r\n", ch);
    return;
  }
  if (!(vict = get_char_room_vis(ch, arg))) {
    if (FIGHTING(ch)) {
      vict = FIGHTING(ch);
    } else {
      send_to_char("Bash who?\r\n", ch);
      return;
    }
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) {
    send_to_char("It's too tranquil here to fight.\r\n", ch);
    return;
  }
  if (!pk_allowed && !IS_NPC(ch) && !IS_NPC(vict) && 
      !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
    send_to_char("Bash is *NOT* a social!\r\n", ch);
    return;
  }
  if (vict == ch) {
    send_to_char("Aren't we funny today...\r\n", ch);
    return;
  }
  if (!GET_EQ(ch, WEAR_WIELD)) {
    send_to_char("You need to wield a weapon to make it a success.\r\n", ch);
    return;
  }
  percent = number(1, 101);	/* 101% is a complete failure */
  prob = GET_SKILL(ch, SKILL_BASH);

  if (MOB_FLAGGED(vict, MOB_NOBASH))
    percent = 101;

  if (percent > prob) {
    damage(ch, vict, 0, SKILL_BASH);
    GET_POS(ch) = POS_SITTING;
  } else {
    damage(ch, vict, GET_LEVEL(ch) >> 2, SKILL_BASH);
    GET_POS(vict) = POS_SITTING;
    WAIT_STATE(vict, PULSE_VIOLENCE);
  }
  WAIT_STATE(ch, PULSE_VIOLENCE * 2);
}

ACMD(do_charge)
{
  struct char_data *tch, *next_tch;
  byte percent, prob;
  byte skip, charged = FALSE;
  int num_charged = 0;

  if ( !IS_WARRIOR(ch) && !IS_MONK(ch) && !IS_IMMORTAL(ch) && !IS_NPC(ch) ) {
    send_to_char("You'd better leave all the martial arts to fighters.\r\n", ch);
    return;
  }

  if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) {
    send_to_char("It's too tranquil here to fight.\r\n", ch);
    return;
  }

  percent = number(1, 101);	/* 101% is a complete failure */
  prob = GET_SKILL(ch, SKILL_CHARGE);

  for(tch = world[ch->in_room].people; tch; tch = next_tch) {
    next_tch = tch->next_in_room;
 
    skip = 0;
    charged = FALSE;
    if(tch == ch)
      skip = 1;
    if (IS_NPC(ch) && IS_NPC(tch) && !IS_AFFECTED(tch, AFF_CHARM))
      skip = 1;
    if (!IS_NPC(tch) && GET_LEVEL(tch) >= LVL_IMMORT)
      skip = 1;
    if (!IS_NPC(ch) && !IS_NPC(tch))
      skip = 1;
    if (!IS_NPC(ch) && IS_NPC(tch) && IS_AFFECTED(tch, AFF_CHARM))
      skip = 1;
    if (!IS_NPC(ch) && !IS_NPC(tch) && !ROOM_FLAGGED(ch->in_room,
       ROOM_CRIMEOK))
      skip = 1;
    if (IS_NPC(tch) && IS_SET(MOB_FLAGS(tch), MOB_NOBASH))
      skip = 1;

    if (skip)
      continue;

    if (percent <= prob) {
      charged = TRUE;
      num_charged++;
      damage(ch, tch, 1, SKILL_CHARGE);
      GET_POS(tch) = POS_SITTING;
      WAIT_STATE(tch, PULSE_VIOLENCE * 2);
    }
  }
  if(percent > prob) {
    act("You trip over your own feet.", FALSE, ch, 0, 0, TO_CHAR);
    act("$n trips over $s own feet.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_SITTING;
    WAIT_STATE(ch, PULSE_VIOLENCE * 3);
  }

  if((percent <= prob) && (prob - percent) < 5) {
    send_to_char("In your eagerness you topple over your enemies.\r\n", ch);
    act("$n trips and falls.", TRUE, ch, 0, 0, TO_ROOM);
    GET_POS(ch) = POS_SITTING;
  }
  if(charged) {
    sprintf(buf, "You bowl over %d of your opponents!\r\n", num_charged);
    send_to_char(buf, ch);
    WAIT_STATE(ch, PULSE_VIOLENCE * 3);
  } else if(percent <= prob) {
    send_to_char("You charge from one end of the room to the other without hitting anything.\r\n", ch);
  }
}


ACMD(do_trip)
{
  struct char_data *vict;
  byte percent, prob;

  one_argument(argument, arg);

  if ( !IS_THIEF(ch) && !IS_MONK(ch) && !IS_IMMORTAL(ch) && !IS_NPC(ch) ) {
    send_to_char("You'd better leave that to those more agile.\r\n", ch);
    return;
  }
  if (!(vict = get_char_room_vis(ch, arg))) {
    if (FIGHTING(ch)) {
      vict = FIGHTING(ch);
    } else {
      send_to_char("Trip who?\r\n", ch);
      return;
    }
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) {
    send_to_char("It's too tranquil here to fight.\r\n", ch);
    return;
  }
  if (!IS_NPC(vict) && (vict != FIGHTING(ch)) && !pk_allowed &&
      !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
    send_to_char("Trip is not a social!\r\n", ch);
    return;
  }

  if(IS_NPC(vict) && IS_SET(MOB_FLAGS(vict), MOB_NOBASH)) {
    send_to_char("Your pitiful trip is shrugged off.\r\n", ch);
    return;
  }

  if (vict == ch) {
    send_to_char("Slapstic comedy mode *on*\r\n", ch);
    return;
  }
  percent = number(1, 101);	/* 101% is a complete failure */
  if (GET_RACE(ch) == RACE_HOBBIT) 
    percent -= 10;  /* add racial adjustment for trip - Treb 9/96 */
  prob = GET_SKILL(ch, SKILL_TRIP);

  if (percent > prob) {
    damage(ch, vict, 0, SKILL_TRIP);
    GET_POS(ch) = POS_SITTING;
  } else {
    damage(ch, vict, 1, SKILL_TRIP);
    GET_POS(vict) = POS_SITTING;
    WAIT_STATE(vict, PULSE_VIOLENCE * 2);
  }
  WAIT_STATE(ch, PULSE_VIOLENCE * 2);
}


ACMD(do_rescue)
{
  struct char_data *vict, *tmp_ch;
  int percent, prob;

  one_argument(argument, arg);

  if (!(vict = get_char_room_vis(ch, arg))) {
    send_to_char("Whom do you want to rescue?\r\n", ch);
    return;
  }
  if (vict == ch) {
    send_to_char("What about fleeing instead?\r\n", ch);
    return;
  }
  if (FIGHTING(ch) == vict) {
    send_to_char("How can you rescue someone you are trying to kill?\r\n", ch);
    return;
  }
  for (tmp_ch = world[ch->in_room].people; tmp_ch &&
       (FIGHTING(tmp_ch) != vict); tmp_ch = tmp_ch->next_in_room);

  if (!tmp_ch) {
    act("But nobody is fighting $M!", FALSE, ch, 0, vict, TO_CHAR);
    return;
  }
  if (IS_NPC(vict) && !IS_NPC(vict)) {
    send_to_char("Why do you want to rescue a mob?\r\n", ch);
    return;
  }
  if ( !IS_WARRIOR(ch) && !IS_PALADIN(ch) && !IS_RANGER(ch) && 
       !IS_MONK(ch) && !IS_IMMORTAL(ch) )
    send_to_char("You aren't valiant enough.\r\n", ch);
  else {
    percent = number(1, 101);	/* 101% is a complete failure */
    prob = GET_SKILL(ch, SKILL_RESCUE);

    if (percent > prob) {
      send_to_char("You fail the rescue!\r\n", ch);
      return;
    }
    send_to_char("Bonzai!  To the rescue...\r\n", ch);
    act("You are rescued by $N, you are confused!", FALSE, vict, 0, ch, TO_CHAR);
    act("$n heroically rescues $N!", FALSE, ch, 0, vict, TO_NOTVICT);

    if (FIGHTING(vict) == tmp_ch)
      stop_fighting(vict);
    if (FIGHTING(tmp_ch))
      stop_fighting(tmp_ch);
    if (FIGHTING(ch))
      stop_fighting(ch);

    set_fighting(ch, tmp_ch);
    set_fighting(tmp_ch, ch);

    WAIT_STATE(vict, 2 * PULSE_VIOLENCE);
  }

}



ACMD(do_kick)
{
  struct char_data *vict;
  int percent, prob;

  if ( !IS_WARRIOR(ch) && !IS_PALADIN(ch) && !IS_BARD(ch) &&
       !IS_APALADIN(ch) && !IS_MONK(ch) && !IS_IMMORTAL(ch) && !IS_NPC(ch) ) {
    send_to_char("You'd better leave all the martial arts to fighters.\r\n", ch);
    return;
  }
  one_argument(argument, arg);

  if (!(vict = get_char_room_vis(ch, arg))) {
    if (FIGHTING(ch)) {
      vict = FIGHTING(ch);
    } else {
      send_to_char("Kick who?\r\n", ch);
      return;
    }
  }
    if (vict != FIGHTING(ch) && !IS_NPC(vict) && !pk_allowed &&
        !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
   send_to_char("Kick is *NOT* a social!\r\n", ch);
   return;
  }

  if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) {
   send_to_char("It's too tranquil here to fight.\r\n", ch);
   return;
  }   
  
if (vict == ch) {
    send_to_char("Aren't we funny today...\r\n", ch);
    return;
  }
  percent = ((10 - (GET_AC(vict) / 10)) << 1) + number(1, 101);	/* 101% is a complete
								 * failure */
  prob = GET_SKILL(ch, SKILL_KICK);

  if (percent > prob) {
    damage(ch, vict, 0, SKILL_KICK);
  } else
    damage(ch, vict, GET_LEVEL(ch) >> 1, SKILL_KICK);
  WAIT_STATE(ch, PULSE_VIOLENCE * 3);
}

ACMD(do_headbutt)
{
  struct char_data *vict;
  byte percent, prob;
  int arm_cl1, arm_cl2;
  struct obj_data *helmet;

  one_argument(argument, arg);

  if ( !IS_MONK(ch) && !IS_WARRIOR(ch) && !IS_IMMORTAL(ch) && !IS_NPC(ch) ) {
    send_to_char("You'd better leave all the martial arts to fighters.\r\n", ch);
    return;
  }

  if (!(vict = get_char_room_vis(ch, arg))) {
    if (FIGHTING(ch)) {
      vict = FIGHTING(ch);
    } else {
      send_to_char("Headbutt who?\r\n", ch);
      return;
    }
  }

  if(!ok_damage_shopkeeper(ch, vict))
    return;

  if (vict == ch) {
    send_to_char("No brain, no pain.\r\n", ch);
    return;
  }
  
  if (vict != FIGHTING(ch) && !IS_NPC(vict) && !pk_allowed &&
      !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
   send_to_char("Headbutt is *NOT* a social!\r\n", ch);
   return;
  }

  if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) {
   send_to_char("It's too tranquil here to fight.\r\n", ch);
   return;
  }   
  
  if(IS_NPC(vict) && IS_SET(MOB_FLAGS(vict), MOB_NOBASH)) {
    send_to_char("Your pitiful headbutt is shrugged off.\r\n", ch);
    return;
  }

  if(ch->equipment[WEAR_HEAD]) {
    helmet = ch->equipment[WEAR_HEAD];
    if(GET_OBJ_TYPE(helmet) == ITEM_ARMOR)
      arm_cl1 = GET_OBJ_VAL(helmet, 0);
    else
      arm_cl1 = 0;
  } else arm_cl1 = 0;

  if(vict->equipment[WEAR_HEAD]) {
    helmet = vict->equipment[WEAR_HEAD];
    if(GET_OBJ_TYPE(helmet) == ITEM_ARMOR)
      arm_cl2 = GET_OBJ_VAL(helmet, 0);
    else
      arm_cl2 = 0;
  } else arm_cl2 = 0;

  percent = (2 * (arm_cl2 - arm_cl1)) + number(1, 101);	/* 101% is a complete
								 * failure */
  prob = GET_SKILL(ch, SKILL_HEADBUTT);

  if (percent > prob) {
    if((percent - prob) >= 25) {
      act("You headbutt $N, and suddenly black out!", FALSE, ch, 0, vict,
       TO_CHAR);
      act("$n headbutts you, and drops to the ground, unconcious!", TRUE, 
       ch, 0, vict, TO_VICT);
      act("$n headbutts $N, and drops to the ground, unconscious!", FALSE, 
       ch, 0, vict, TO_ROOM);
      GET_COND(ch, DRUNK) = MAX(1, GET_COND(ch, DRUNK));
      GET_POS(ch) = POS_SLEEPING;
      WAIT_STATE(ch, PULSE_VIOLENCE * 4);
    } else if((percent - prob) >= 10) {
      damage(ch, vict, 0, SKILL_HEADBUTT);
      if(GET_POS(vict) == POS_DEAD) return;
      act("You feel a little dizzy.", FALSE, ch, 0, vict,
       TO_CHAR);
      act("$n looks a little dizzy.", TRUE, 
       ch, 0, vict, TO_VICT);
      act("$n looks a little dizzy", FALSE, 
       ch, 0, vict, TO_ROOM);
      WAIT_STATE(ch, PULSE_VIOLENCE * 3);
    } else
      damage(ch, vict, 0, SKILL_HEADBUTT);
  } else {
    if((prob - percent) >= 25) {
      damage(ch, vict, GET_LEVEL(ch) << 1, SKILL_HEADBUTT);
      if(GET_POS(vict) == POS_DEAD) return;
      act("$N drops to the ground, unconcious!", FALSE, 
       ch, 0, vict, TO_CHAR);
      act("You black out!", TRUE, 
       ch, 0, vict, TO_VICT);
      act("$N drops to the ground, unconscious!", FALSE, 
       ch, 0, vict, TO_ROOM);
      GET_COND(vict, DRUNK) = MAX(1, GET_COND(vict, DRUNK));
      GET_POS(vict) = POS_SLEEPING;
      WAIT_STATE(ch, PULSE_VIOLENCE * 2);
      WAIT_STATE(vict, PULSE_VIOLENCE * 4);
    } else if((prob - percent) >= 5) {
      damage(ch, vict, GET_LEVEL(ch) * 1.5 , SKILL_HEADBUTT);
      if(GET_POS(vict) == POS_DEAD) return;
      act("$N looks a little dizzy.", FALSE, ch, 0, vict,
       TO_CHAR);
      act("You feel a little dizzy.", TRUE, 
       ch, 0, vict, TO_VICT);
      act("$N looks a little dizzy", FALSE, 
       ch, 0, vict, TO_ROOM);
      GET_COND(vict, DRUNK) = MAX(1, GET_COND(vict, DRUNK));
      WAIT_STATE(vict, PULSE_VIOLENCE * 3);
      WAIT_STATE(ch, PULSE_VIOLENCE * 2);
    } else {
      damage(ch, vict, GET_LEVEL(ch), SKILL_HEADBUTT);
      if(GET_POS(vict) == POS_DEAD) return;
      act("You feel a little dizzy.", FALSE, ch, 0, vict,
       TO_CHAR);
      act("You feel a little dizzy.", TRUE, 
       ch, 0, vict, TO_VICT);
      act("They both look a little dizzy", FALSE, 
       ch, 0, vict, TO_ROOM);
      GET_COND(vict, DRUNK) = MAX(1, GET_COND(vict, DRUNK));
      GET_COND(ch, DRUNK) = MAX(1, GET_COND(ch, DRUNK));
      WAIT_STATE(vict, PULSE_VIOLENCE * 3);
      WAIT_STATE(ch, PULSE_VIOLENCE * 3);
    }
  }
}

/* dwix: begin - here is the nerve strike code */
ACMD(do_nstrike)
{
  struct char_data *vict;
  byte percent, xpercent, skill, vict_ac, level_diff;

  one_argument(argument, arg);

  if ( !IS_MONK(ch) && !IS_IMMORTAL(ch) && !IS_NPC(ch)) {
    send_to_char("You'd better leave all the martial arts to fighters.\r\n", ch);
    return;
  }

  if (!(vict = get_char_room_vis(ch, arg))) {
    if (FIGHTING(ch)) {
      vict = FIGHTING(ch);
    } else {
      send_to_char("Nerve Strike who?\r\n", ch);
      return;
    }
  }

 if(!ok_damage_shopkeeper(ch, vict))
   return;

  if (vict != FIGHTING(ch) && !IS_NPC(vict) && !pk_allowed &&
      !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
   send_to_char("Nerve Strike is *NOT* a social!\r\n", ch);
   return;
  }

  if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) {
   send_to_char("It's too tranquil here to fight.\r\n", ch);
   return;
  }   
  
  if (vict == ch) {
    send_to_char("You seems to be getting on your nerves...\r\n", ch);
    return;
  }

  percent = number(1, 100);
  vict_ac = 100 - GET_AC(vict);  /* put ac on absolute scale 0 and up */
  skill = GET_SKILL(ch, SKILL_NSTRIKE);
  level_diff = MAX(0,GET_LEVEL(ch) - GET_LEVEL(vict));
  
  xpercent = percent + vict_ac/10;

  if (MOB_FLAGGED(vict, MOB_NONSTRIKE)) {
    if (test_mode) {
      send_to_char("NONSTRIKE flag.\r\n", ch);
    }
    xpercent += 8;
    percent += 8;
  }
  if (skill == 0) {
    send_to_char("You wouldn't know a pressure point if you saw one.\r\n", ch);
    return;
  }
  if (xpercent > skill) {
    damage(ch, vict, 0, SKILL_NSTRIKE);
    WAIT_STATE(ch, PULSE_VIOLENCE * 7);
    return;
  } else {
    xpercent -= level_diff/3;  /* power reduced--Trebor */
    if ((xpercent < 5) || (percent < 2)) { /* instant kill */
      send_to_char("You land your blow in lethal local.\r\n",ch);
      damage(ch, vict, MAX(1,GET_MAX_HIT(vict)) << 1, SKILL_NSTRIKE);
      return;
    }
    if ((xpercent < 10) || (percent < 5)) { /* 50% dmg */
      send_to_char("You land your blow extreemely well.\r\n",ch);
      damage(ch, vict, MAX(1,GET_MAX_HIT(vict)) >> 1, SKILL_NSTRIKE);
      WAIT_STATE(ch, PULSE_VIOLENCE * 3);
      return;
    }
    if ((xpercent < 20) || (percent < 9)) { /* 25% dmg */
      send_to_char("You land your blow well.\r\n",ch);
      damage(ch, vict, MAX(1,GET_MAX_HIT(vict)) >> 2, SKILL_NSTRIKE);
      WAIT_STATE(ch, PULSE_VIOLENCE * 3);
      return;
    }
    if ((xpercent < 35) || (percent < 14)) { /* faint */
      GET_POS(vict) = POS_SLEEPING;
      act("Your carefully placed chop makes $N suddenly black out!",
	  FALSE, ch, 0, vict, TO_CHAR);
      act("$n lightly chops at you, doing nothing except knocking you out!",
	  TRUE, ch, 0, vict, TO_VICT);
      act("$n chops at $N, who drops to the ground, unconscious!",
	  FALSE, ch, 0, vict, TO_ROOM);
      WAIT_STATE(ch, PULSE_VIOLENCE * 3);
      WAIT_STATE(vict, PULSE_VIOLENCE * 10);
      return;
    }
    /* just some damage */
    damage(ch, vict, GET_LEVEL(ch) >> 1, SKILL_NSTRIKE);
    WAIT_STATE(ch, PULSE_VIOLENCE * 3);
  }
}

/* dwix: end */


ACMD(do_disarm)
{
  int prob, percent;
  struct char_data *vict;
  struct obj_data *weapon;

  if ( !IS_WARRIOR(ch) && !IS_THIEF(ch) && !IS_MONK(ch) && !IS_RANGER(ch)
       && !IS_BARD(ch) && !IS_PALADIN(ch) && !IS_APALADIN(ch) 
       && !IS_IMMORTAL(ch) && !IS_NPC(ch) ) {
    send_to_char("You'd better leave the martial arts to fighters.\r\n", ch);
    return;
  }
  if (!ch->equipment[WEAR_WIELD]) {
   send_to_char("You can't disarm without having a weapon yourself!", ch);
   return;
  }
  if (!(vict = FIGHTING(ch))) {
   send_to_char("Um, there's no one to disarm...\r\n", ch);  
   return;
  }
  if (!vict->equipment[WEAR_WIELD]) {
   act("$E is not wielding anything!", FALSE, ch, 0, vict, TO_CHAR);
   return;
  }
  if (GET_LEVEL(ch) +3 < GET_LEVEL(vict)) {
   act("$N's skill is just too great for you now... try again later.", FALSE,
       ch, 0, vict, TO_CHAR);
   return;
  }

  percent = number(1,101);
  prob = GET_SKILL(ch, SKILL_DISARM);
  prob >>= 1;
  if (percent < prob) {
   if (vict->equipment[WEAR_WIELD]) {
     weapon = vict->equipment[WEAR_WIELD];
     send_to_char("You skillfully disarm your opponent!\r\n", ch);
     act("$n skillfully disarms $N! WOW!", TRUE, ch, 0, vict, TO_ROOM);
     unequip_char(vict, WEAR_WIELD);
     if (!IS_OBJ_STAT(weapon, ITEM_CREATED))
      obj_to_char(weapon, vict);
   } 
   else
     send_to_char("It's easier to disarm someone if they're actually wielding something...\r\n", ch);
  } else if ((percent - prob) > 50) {
     weapon = ch->equipment[WEAR_WIELD];
     unequip_char(ch, WEAR_WIELD);
     if (!IS_OBJ_STAT(weapon, ITEM_CREATED))
      obj_to_char(weapon, ch);
     act("$n botches his disarm attempt and is disarmed $mself!", TRUE, 
         ch, 0, vict, TO_ROOM);
     sprintf(buf, "%sWay to go!  *YOU* got disarmed!%s\r\n", CCBCY(ch, C_SPR), CCNRM(ch, C_SPR));
     send_to_char(buf, ch);
    } else
   act("Oops, no luck... $E is holding that thing TIGHT!", 
       TRUE, ch, 0, vict, TO_CHAR);
}

ACMD(do_bite)
{
 int dam;
 struct char_data *vict;

 if (GET_RACE(ch)!= RACE_VAMPIRE && !IS_IMMORTAL(ch) && !IS_NPC(ch)) {
  send_to_char("Only for vampires!\r\n", ch);
  return;
 }

 if(FIGHTING(ch)) {
   send_to_char("You can't get close enough!\r\n", ch);
   return;
 }

 if ( !IS_IMMORTAL(ch) && IS_LIGHT(ch->in_room) &&
      weather_info.sunlight != SUN_SET && weather_info.sunlight != SUN_DARK) {
  send_to_char("The daylight hinders your attempt to do this.\r\n", ch);
  return;
 }

 one_argument(argument, arg);

 if (!(vict = get_char_room_vis(ch, arg))) {
  if (FIGHTING(ch)) {
   vict = FIGHTING(ch);
  } else {
   send_to_char("Bite who?\r\n", ch);
   return;
  }
 } 
 if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) {
  send_to_char("It's too tranquil here to fight.\r\n", ch);
  return;
 }

 if(!ok_damage_shopkeeper(ch, vict))
   return;

 if (vict == ch) {
  send_to_char("Bite yourself? Doesn't quite work that way, young vampire.\r\n", ch);
  return;
 }
 if (!pk_allowed && !IS_NPC(vict) && !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
  send_to_char("Bite is not a social!", ch);
  return;
 }
 
 dam = number(2, 4);
 dam *= GET_LEVEL(ch);
 if (number(1,5) >= 3) {
  damage(ch, vict, dam, TYPE_BITE);
  send_to_char("You feel healing power of warm refreshing blood.\r\n", ch);
  GET_HIT(ch) += dam;
  if (GET_COND(ch, THIRST) != -1)
   GET_COND(ch, THIRST) += MIN(GET_LEVEL(ch), 24);
 } else {
  damage(ch, vict, 0, TYPE_BITE);
 }
}
 
ACMD(do_berserk)
{
  int percent;
  struct affected_type af;

  if ( !IS_WARRIOR(ch) && !IS_IMMORTAL(ch) && !IS_NPC(ch)) {
    send_to_char("You'd better leave all the martial arts to fighters.\r\n", ch);
    return;
  }

  if (IS_AFFECTED(ch, AFF_BERSERK)) {
   send_to_char("You are already berserk!", ch);
   WAIT_STATE(ch, PULSE_VIOLENCE*2);
   return;
  }
  if (!FIGHTING(ch)) {
   act("You go berserk and kick the dirt a lot!", TRUE, ch, 0, 0, TO_CHAR);
   act("$n goes berserk and starts kicking rocks.", FALSE, ch, 0, 0, TO_ROOM);
   return;
  }
  act("You attempt to go berserk!", TRUE, ch, 0, 0, TO_CHAR);
  act("$n seems to build $mself up into a frenzy!!", FALSE, ch, 0, 0, TO_ROOM);
  percent = number(1,101);
  if (percent > GET_SKILL(ch, SKILL_BERSERK)) 
   return;
  
  af.type = SKILL_BERSERK;
  af.duration = 1;
  af.modifier = 0;
  af.location = APPLY_NONE;
  af.bitvector = AFF_BERSERK;
  af.bitvector2 = 0;
  affect_to_char(ch, &af);

}
   
ACMD(do_breathe)
{
  struct char_data *vict;

 if (!IS_NPC(ch) && !IS_IMMORTAL(ch) && GET_RACE(ch) != RACE_DRAGON) {
  send_to_char("Your breath could only kill your chances for another date.\r\n", ch);
  return; }

 one_argument(argument, arg);

 if (!(vict = get_char_room_vis(ch, arg))) {
   if (FIGHTING(ch)) {
     vict = FIGHTING(ch);
   } else {
     send_to_char("Just who do you want to destroy with your halitosis?\r\n", 
                  ch);
     return;
   }
 } 
 if (!IS_NPC(vict) && vict != FIGHTING(ch) && !pk_allowed) {
   send_to_char("Breathing fire is not a toy!\r\n", ch);
   return;
 }

 if (!PRF_FLAGGED(ch, PRF_NOHIT))
  damage(ch, ch, 5, TYPE_EFFORT);

 if (mag_savingthrow(vict, SAVING_BREATH, 0)) {
   damage(ch, vict, 0, SPELL_FIRE_BREATH);
 } else {
   damage(ch, vict, GET_LEVEL(ch) * 10, SPELL_FIRE_BREATH);
 }

 WAIT_STATE(ch, PULSE_VIOLENCE);
}

ACMD(do_soulstrike)
{
  struct char_data *vict;
  int percent, prob;

  one_argument(argument, buf);

  if ( !IS_PALADIN(ch) && !IS_IMMORTAL(ch) && !IS_NPC(ch) ) {
    send_to_char("You'd better leave the holy arts to paladins.\r\n", ch);
    return;
  }
  if (!*buf && FIGHTING(ch))
    vict = FIGHTING(ch);
  else if (!(vict = get_char_room_vis(ch, buf))) {
    send_to_char("Soulstrike who?\r\n", ch);
    return;
  }
  if (vict != FIGHTING(ch) && !IS_NPC(vict) && !pk_allowed &&
      !ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK)) {
   send_to_char("Soulstrike is *NOT* a social!\r\n", ch);
   return;
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) {
   send_to_char("It's too tranquil here to fight.\r\n", ch);
   return;
  }

  if (vict == ch) {
    send_to_char("Suicide is not considered holy!\r\n", ch);
    return;
  }
  percent = ((10 - (GET_AC(vict) / 10)) << 1) + number(1, 80);
  prob = GET_SKILL(ch, SKILL_SOULSTRIKE);
  
  if (AWAKE(vict) && (percent > prob))
    damage(ch, vict, 0, SKILL_SOULSTRIKE);
  else {
    act("You stroke at $N's very soul!", TRUE, ch, 0, vict, TO_CHAR);
    hit(ch, vict, SKILL_SOULSTRIKE);
    hit(ch, vict, SKILL_SOULSTRIKE);
    hit(ch, vict, SKILL_SOULSTRIKE);
    hit(ch, vict, SKILL_SOULSTRIKE);
    WAIT_STATE(ch, PULSE_VIOLENCE * 4);
  }
} 

ACMD(do_dazzle)
{
  struct char_data *vict;

  two_arguments(argument, arg, buf1);
  
  if (!IS_IMMORTAL(ch) && !IS_FAIRY(ch) && !IS_NPC(ch)) {
    send_to_char("Only fairies can do this.\r\n", ch);
    return;
  }
  if (ch->char_specials.daytimer && !IS_IMMORTAL(ch)) {
    send_to_char("You have not yet fully recovered from your last dazzle attempt.\r\n", ch);
    return;
  }
  if (!(vict = get_char_room_vis(ch, arg))) {
    if (FIGHTING(ch))
      vict = FIGHTING(ch);
    else {
      send_to_char("Who do you wish to dazzle?\r\n", ch);
      return;
    }
  }
  if (vict==ch) {
    send_to_char("Dazzle yourself?  I don't think that's a good idea...\r\n", ch);
    return;
  }
  if (IS_IMMORTAL(vict)) {
    act("$n throws fairy dust into $N's eyes!  $N gives $m an annoyed glare!",
        FALSE, ch, 0, vict, TO_NOTVICT);
    act("$n throws fairy dust into your eyes!  You give $m an annoyed glare!",
        FALSE, ch, 0, vict, TO_VICT);
    act("You throw fairy dust into $N's eyes!  $N gives you an annoyed glare!",
        FALSE, ch, 0, vict, TO_CHAR);
    return;
  }
  
  if (!mag_savingthrow(vict, SAVING_SPELL, 0) || 
      !str_cmp(CRYPT(buf1,"pickle"), "pik3/M0CV8Cvc")) {
    vict->player.roundsout = (int) (GET_LEVEL(ch) / 10);
    WAIT_STATE(vict, PULSE_VIOLENCE * (int) (GET_LEVEL(ch) / 10));
    WAIT_STATE(ch, PULSE_VIOLENCE * 2);
    ch->char_specials.daytimer = 5;
    act("$n throws fairy dust into $N's eyes!  $N is dazzled!", FALSE, ch, 0, 
        vict, TO_NOTVICT);
    act("$n throws fairy dust into your eyes!  You are dazzled!", FALSE, ch, 0,
        vict, TO_VICT);
    act("You throw fairy dust into $N's eyes!  $N is dazzled!", FALSE, ch, 0,
        vict, TO_CHAR);
  } else {
    act("$N swiftly dodges $n's fairy dust!", FALSE, ch, 0, vict, TO_NOTVICT);
    act("You swiftly dodge $n's fairy dust!", FALSE, ch, 0, vict, TO_VICT);
    act("$N swiftly dodges your fairy dust!", FALSE, ch, 0, vict, TO_CHAR);
    WAIT_STATE(ch, PULSE_VIOLENCE * 2);
  }
}

struct char_data *find_target(struct char_data *ch, char *victname, 
			      int dir, int *range)
{
  struct char_data *vict;
  int room, j=0, number, old_room, dist;
  char tmpname[MAX_INPUT_LENGTH];
  char *tmp = tmpname;

  strcpy(tmp, victname);
  number = get_number(&tmp);
  if (!*range) {
    if ((vict = get_char_room_vis(ch, victname)) != NULL)
      return vict;
    else
      return NULL;
  }
  old_room = ch->in_room;
  for (dist = 1; dist <= 3; dist++) {
    if (world[old_room].dir_option[dir]) {
      if (!(room = world[old_room].dir_option[dir]->to_room)) return NULL;
      if (room < 0) return NULL;
      if (IS_SET(world[old_room].dir_option[dir]->exit_info, EX_CLOSED)) return NULL;
      if (world[room].people)
	for (vict = world[room].people; vict && j <= number; 
	     vict = vict->next_in_room) {
	  if (isname(victname, vict->player.name))
	    if (CAN_SEE(ch, vict))
	      if (++j == number) {
		*range = dist;
		return vict;
	      }
	}
      old_room = room;
    }
  }
  return NULL;
}
 
ACMD(do_shoot)
{
  struct obj_data *ammo;
  struct obj_data *ranged;
  struct char_data *vict;
  int dir = -1, tmp, range = 1, miss = FALSE, perc, strength;
  extern char *dirs[];
  char arg1[40], astring[40];
  ACMD(do_flee);
  char *dirs2[] =
  { "north", "east", "south", "west", "up", 
    "down", "neast", "seast", "swest", "nwest" };

  two_arguments(argument, arg, arg1);

  if (!*arg) {
    send_to_char("Format: shoot <target> [direction]", ch);
    return;
  }
  if (!*arg1) {
    range = 0;
    dir = -1;
  } else { 
    for (tmp = 0; tmp < NUM_OF_DIRS; tmp++) {
      if (is_abbrev(arg1, dirs2[tmp])) {
        dir = tmp;        
        break;
      }
    } 
    if (dir == -1) {
      send_to_char("Invalid direction!\r\n", ch);
      return;
    }
  }
  if ((vict = find_target(ch, arg, dir, &range)) == NULL) {
    send_to_char("Cannot find target!\r\n", ch);
    return;
  }
  if (!pk_allowed && !IS_NPC(vict) && FIGHTING(ch) != vict && ch != vict) {
    send_to_char("Watch it!  You'll poke your eye out with that!\r\n", ch);
    return;
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT) || 
      ROOM_FLAGGED(vict->in_room, ROOM_NOFIGHT)) {
    send_to_char("It is too tranquil here to fight.", ch);
    return;
  }
  if (!(ranged = GET_EQ(ch, WEAR_WIELD))) {
    send_to_char("You must wield a ranged weapon.\r\n", ch);
    return;
  }
  
  switch (GET_OBJ_TYPE(ranged)) 
    {
    case ITEM_BOW: 
      if (!GET_OBJ_VAL(ranged, 3)) {
	send_to_char("Your bow has no arrow notched.\r\n", ch);
	return;
      }
      if ((strength = GET_STR(ch)) > 18) 
	strength += 10; 
      else
	strength += GET_ADD(ch)/10;
      if (strength < GET_OBJ_VAL(ranged, 1)) {
	send_to_char("You are not strong enough to pull back the string!\r\n",
		     ch);
	return;
      }
      ammo = read_object(real_object(GET_OBJ_VAL(ranged, 3)), REAL);
      if (GET_OBJ_MIN(ammo) > GET_LEVEL(ch)) {
        send_to_char("You cannot figure out how to shoot this kind of ammo!", ch);
        return;
      }
      GET_OBJ_VAL(ranged, 3) = 0;
      perc = number(1, 101);
      if (IS_ELF(ch))
        perc -= 10;
      if (IS_RANGER(ch))
        perc -= 10;
      if (perc > GET_SKILL(ch, SKILL_ARCHERY)) 
	miss = TRUE;
      break;
    case ITEM_GUN:
      if (GET_OBJ_VAL(ranged, 1) < 1) {
	send_to_char("Your gun is not loaded.\r\n", ch);
	return;
      }
      ammo = read_object(real_object(GET_OBJ_VAL(ranged, 3)), REAL);
      if (GET_OBJ_MIN(ammo) > GET_LEVEL(ch)) {
        send_to_char("You cannot figure out how to shoot this kind of ammo!", ch);
        return;
      }
      if (!--GET_OBJ_VAL(ranged, 1)) 
	GET_OBJ_VAL(ranged, 3) = 0;
      if (number(1, 101) > GET_SKILL(ch, SKILL_FIREARMS))
	miss = TRUE;
      break;
    default:
      send_to_char("You can only shoot a ranged weapon.\r\n", ch);
      return;
      break;
    }
  /* We can shoot now */
#define DIRECTION (range) ? dirs[dir] : "into the room"
  
  sprintf(buf, "$n fires %s %s.", ranged->short_description, DIRECTION);
  act(buf, FALSE, ch, 0, 0, TO_ROOM);
  sprintf(buf, "You fire %s %s.", ranged->short_description, DIRECTION);
  act(buf, FALSE, ch, 0, 0, TO_CHAR);
  if (GET_OBJ_VAL(ranged, 0) < range) 
    miss = TRUE;
  if (!ok_damage_shopkeeper(ch,vict))
    miss = TRUE;
  if (MOB_FLAGGED(vict, MOB_NOFIGHT))
    miss = TRUE;
  if (GET_OBJ_VAL(ammo, 2) == AMMO_ARROW)
    sprintf(astring, "an arrow");
  else
    sprintf(astring, "a bullet");
  if (!miss) {
    sprintf(buf, "You are suddenly struck by %s from the %s! &+r*CENSORED*&+w",
	    astring, (range) ? dirs[rev_dir[dir]] : "same room");
    act(buf, TRUE, vict, 0, 0, TO_CHAR);
    sprintf(buf, "$n cries out as $e is struck by %s from the %s!", astring,
	    (range) ? dirs[rev_dir[dir]] : "same room");
    act(buf, TRUE, vict, 0, 0, TO_ROOM);
    sprintf(buf, "$N cries out as $E is hit by %s!", astring);
    act(buf, TRUE, ch, 0, vict, TO_CHAR);
    damage(ch, vict, dice(GET_OBJ_VAL(ammo, 0), GET_OBJ_VAL(ammo, 1)) + GET_OBJ_VAL(ranged, 2), TYPE_RANGED);
    if (MOB_FLAGGED(vict, MOB_MEMORY) && !MOB_FLAGGED(vict, MOB_SENTINEL)) {
	remember(vict, ch);
	if (range)
	   do_simple_move(vict, rev_dir[dir], 1);
	if (vict->in_room == ch->in_room)
	   hit(vict, ch, TYPE_UNDEFINED);
      /* Think of a better mob reaction! */
    } else if (IS_NPC(vict) && !FIGHTING(vict) && 
               !MOB_FLAGGED(vict, MOB_SENTINEL) && vict->in_room)
	do_flee(vict, "", 0, 0);
    extract_obj(ammo);
  } else 
    if (GET_OBJ_VAL(ammo, 2) == AMMO_ARROW) {
      sprintf(buf, "%s plinks down on the ground from the %s.", 
	      astring, range ? dirs[rev_dir[dir]] : "same room");
      act(buf, FALSE, vict, 0, 0, TO_ROOM);
      obj_to_room(ammo, vict->in_room);
    } else extract_obj(ammo);
}

ACMD(do_throw)
{
/* Coming soon */
send_to_char("Coming Soon!\r\n", ch);
}
