/************************************************************************
 * buffer.c - Advanced Buffer System				v1.2	*
 *									*
 * Designed for CircleMUD 3.0				August	1997	*
 ************************************************************************/

#define _BUFFER_C_

#include "conf.h"
#include "sysdep.h"
#include "structs.h"
#include "utils.h"
#include "interpreter.h"

/* ------------------------------------------------------------------------ */

/*
 * #if 1 = memset()
 * #if 0 = *buffer = '\0' (recommended)
 */
#if 1
byte paranoid_buffer = TRUE;
#else
byte paranoid_buffer = FALSE;
#endif

/*
 * #if 1 = don't check buffer size. (recommended)
 * #if 0 = check buffer size to see how much the function used.
 */
#if 0
byte check_buffer_size = FALSE;
#else
byte check_buffer_size = TRUE;
#endif

/*
 * #if 1 = causes a normal reboot when corrupted. (recommended)
 * #if 0 = jettisons the buffers and creates a new set to run with.
 *	Note that when MAX_OVERFLOW is exceeded, a reboot is forced anyway.
 */
#if 1
byte paranoid_overflow = TRUE;
#else
byte paranoid_overflow = FALSE;
#endif

/*
 * #if 1 = Log only problems and useful messages. (recommended)
 * #if 0 = Log all allocations. WARNING! This creates enormous (>16MB) output.
 */
#if 1
byte verbose_buffer = FALSE;
#else
byte verbose_buffer = TRUE;
#endif

/*
 * Adjust this value to tweak the buffer timeout from the default 5 minutes.
 * This means to free the buffer if it hasn't been _used_ in 5 minutes, not
 * to free the buffer 5 minutes after it was created.  Magic number is
 * any arbitrary hex value.  Max_overflow defines how many times the buffer
 * list can be corrupted before a reboot is forced.
 */
#define BUFFER_LIFE	((600 RL_SEC) / PULSE_BUFFER)
#define MAGIC_NUMBER	(0x3D)
#define MAX_OVERFLOW	3

/*
 * For compilers without __FUNCTION__.  This could cause problems in
 * release_my_buffers() in the future.  Currently the only functions
 * that use it are ACMD() ones and they don't chain into each other to
 * cause an overlap free problem.  The dummy function is important.
 */
void dummy() {
#if !defined(__FUNCTION__)
#define __FUNCTION__	__FILE__
#endif
}

/* ------------------------------------------------------------------------ */

/*
 * Global variables.
 *  buffers - head of the allocation linked list.
 *  last_alloc - pointer to last allocated structure, a cache for freeing.
 *  buffer_overflow - how many times we've corrupted the buffer list.
 */
struct buf_data *buffers;
struct buf_data *last_alloc;
byte buffer_overflow = 0;

/* External functions and variables. */
void send_to_char(char *, struct char_data *);
void send_to_all(char *);
extern int circle_shutdown;
extern int circle_reboot;

/* Private functions. */
truct buf_data *new_buffer(sh_int size, bool persistant);
struct buf_data *malloc_buffer(sh_int size, bool persistant);
struct buf_data *get_buffer_head(void);
void set_buffer_head(struct buf_data *new_head);
void free_buffer(struct buf_data *f);
void clear_buffer(struct buf_data *clearme);
int get_used(struct buf_data *buf);

/* ------------------------------------------------------------------------ */

/*
 * Public: init_buffers()
 * This is called from main() to get everything started.
 */
void init_buffers(void)
{
  set_buffer_head(NULL);
  last_alloc = NULL;

  /*
   * These are persistant to always have some available.
   * We're still using less data memory than the original buffer scheme.
   * Some log messages are 128 byte buffers, some are SMALL_BUFSIZE,
   * we need a standard value like SMALL_LOGBUF (128?).
   */
  new_buffer(LARGE_BUFSIZE + GARBAGE_SPACE, TRUE); /* comm.c:process_output() */
  new_buffer(LARGE_BUFSIZE, TRUE);
  new_buffer(MAX_STRING_LENGTH, TRUE);
  new_buffer(MAX_STRING_LENGTH, TRUE);
  new_buffer(SMALL_BUFSIZE, TRUE);
  new_buffer(SMALL_BUFSIZE, TRUE);
  new_buffer(MAX_RAW_INPUT_LENGTH, TRUE);
  new_buffer(MAX_INPUT_LENGTH, TRUE);
  new_buffer(512, TRUE);
  new_buffer(256, TRUE);
  new_buffer(256, TRUE);
  new_buffer(128, TRUE);
  new_buffer(128, TRUE);

  /*
   * This buffer will be used while booting and as a spare buffer
   * until it times out.  db.c:load_help() uses this if you are
   * wondering why it is so large.
   */
  new_buffer(32384, FALSE);
}

/*
 * Public: as release_my_buffers()
 * This function searches the list of buffers for the name supplied
 * and releases those buffers.  It is preferable to release each
 * buffer by name rather than using this function.  This is useful,
 * however, when there are return statements all over a command.
 * If we ever go to a thread based system, this function will either
 * undergo an overhaul or be removed.  It is not very efficient
 * anyway.
 */
void detach_my_buffers(const char *func, const int line_n)
{
  struct buf_data *clear, *next_clear;

  if (!func) {
    nmlog("SYSERR: BUF: detach_my_buffers: someone gave me a null function name!");
    return;
  }
  for (clear = get_buffer_head(); clear; clear = next_clear) {
    next_clear = clear->next;
    if (clear == clear->next) {
      nmlog("SYSERR: BUF: detach_my_buffers: Infinite loop!");
      clear->next = NULL;
    } else if (clear->used && clear->who && strcmp(clear->who, func) == 0) {
      if (verbose_buffer) {
	char buf[128];
	sprintf(buf, "BUF: %s: %s:%d released %d bytes from %s:%d.",
		__FUNCTION__, func, line_n, clear->size, clear->who, clear->line);
	nmlog(buf);
      }
      clear_buffer(clear);
    }
  }
}

/*
 * Public: release_all_buffers()
 * Forcibly release all buffers currently allocated.  This is useful to
 * reclaim any forgotten buffers.  This will need a little bit of work
 * to be thread safe.  With the current setup, we only release one expired
 * buffer every run through the function which defaults to 10 seconds.
 * See structs.h for PULSE_BUFFER to change the release rate.
 */
void release_all_buffers()
{
  struct buf_data *clear, *next_clear;
  int freed = FALSE;

  for (clear = get_buffer_head(); clear; clear = next_clear) {
    next_clear = clear->next;
    if (clear == clear->next) {
      nmlog("SYSERR: BUF: release_all_buffers: Infinite loop!");
      clear->next = NULL;
    } else if (clear->used) {
      char buf[128];
      sprintf(buf, "BUF: %s:%d forgot to release %d bytes.",
		clear->who, clear->line, clear->size);
      nmlog(buf);
      clear_buffer(clear);
    } else if (clear->life > 0)
      clear->life--;
    else if (clear->life != -1 && !freed) {
      struct buf_data *temp;
      REMOVE_FROM_LIST(clear, buffers, next);
      free_buffer(clear);
      freed = TRUE;
    }
  }
}

/*
 * Private: clear_buffer(buffer to clear)
 * This is used to declare an allocated buffer unused.
 */
void clear_buffer(struct buf_data *clearme)
{
  if (!clearme) {
    nmlog("SYSERR: BUF: NULL buf_data given to clear_buffer.");
    return;
  }

  /*
   * If the magic number we set in acquire_buffer() is not there then
   * we have a suspected buffer overflow.
   */
#if defined(PICKY_BUFFERS)
  if (clearme->data[clearme->req_size] != MAGIC_NUMBER)
#else
  if (clearme->data[clearme->size] != MAGIC_NUMBER)
#endif
  {	/* Down here to prevent problems with programs that count {'s. */
    char buf[128];
    sprintf(buf, "SYSERR: BUF: %s: Overflow in buffer from %s:%d.",
		__FUNCTION__, clearme->who, clearme->line);
    nmlog(buf);
    if (paranoid_overflow || ++buffer_overflow > MAX_OVERFLOW) {
      set_buffer_head(NULL);
      send_to_all("Emergency reboot.. come back in a minute or two.\r\n");
      nmlog("SYSERR: BUF: Emergency reboot, too many buffer corruptions!");
      circle_shutdown = circle_reboot = 1;
    } else {
      init_buffers();
      nmlog("SYSERR: BUF: Buffer list dumped and reinitialized!");
    }
  } else {
    if (check_buffer_size) {
     char buf[128];
     /*
      * If not clearme->data return 0, else if paranoid_buffer, return
      * the result of get_used(), otherwise just strlen() the buffer.
      */
     int used = 0;
     if (clearme->data)
       used = (paranoid_buffer ? get_used(clearme) : strlen(clearme->data));
     sprintf(buf, "BUF: %s:%d used %d/%d bytes.",
		clearme->who, clearme->line, used,
#if defined(PICKY_BUFFERS)
		clearme->req_size
#else
		clearme->size
#endif
      );
      nmlog(buf);
    }
    if (paranoid_buffer)
      memset(clearme->data, '\0', clearme->size);
    else
      *clearme->data = '\0';
    clearme->who = NULL;
    clearme->line = 0;
#if defined(PICKY_BUFFERS)
    clearme->req_size = 0;
#endif
    if (clearme->life != -1)	/* Don't reset persistant buffers! */
      clearme->life = BUFFER_LIFE;
    clearme->used = FALSE;
  }
}

/*
 * Public: as release_buffer(buffer data pointer)
 * Used throughout the code to finish their use of the buffer.
 */
void detach_buffer(char *data, const char *func, const int line_n)
{
  struct buf_data *clear;

  if (!data || !func) {
    nmlog("SYSERR: BUF: Invalid information passed to detach_buffer().");
    return;
  }

  /*
   * We cache the last allocated buffer to speed up cases where
   * we allocate a buffer and then free it shortly afterwards.
   */
  if (last_alloc && data == last_alloc->data) {
    clear = last_alloc;
    last_alloc = NULL;
  } else
    for (clear = get_buffer_head(); clear; clear = clear->next)
      if (clear == clear->next) {
	nmlog("SYSERR: BUF: detach_bufferr: Infinite loop!");
	clear->next = NULL;
      } else if (clear->data == data)
        break;

  if (clear) {
    if (verbose_buffer) {
      char buf[128];
      sprintf(buf, "BUF: %s:%d released %d bytes from %s:%d.",
		func, line_n, clear->size, clear->who, clear->line);
      nmlog(buf);
    }
    clear_buffer(clear);
  } else {
    char buf[128];
    sprintf(buf, "SYSERR: BUF: %s: No such buffer found at %s:%d.",
		__FUNCTION__, func, line_n);
    nmlog(buf);
  }
}
  
/*
 * Private: free_buffer(buffer)
 * Internal function for getting rid of unused buffers after a time period.
 */
void free_buffer(struct buf_data *f)
{
  char buf[128];

  if (!f) {
    nmlog("SYSERR: BUF: free_buffer: NULL pointer given.");
    return;
  } else if (f->life == -1)
    sprintf(buf, "SYSERR: BUF: %s: Freeing %d byte persistant buffer!", __FUNCTION__, f->size);
  else
    sprintf(buf, "BUF: %s: Freeing %d bytes in expired buffer.", __FUNCTION__, f->size);
  nmlog(buf);

  if (f->data)
    free(f->data);
  else
    nmlog("SYSERR: BUF: free_buffer: Hey, no data?");
  free(f);
}

/*
 * Private: get_buffer_head()
 */
struct buf_data *get_buffer_head()
{
  return buffers;
}

/*
 * Private: set_buffer_head(new buffer head)
 */
void set_buffer_head(struct buf_data *new_head)
{
  buffers = new_head;
}

/*
 * Private: new_buffer(size of buffer, persistant flag)
 * Finds where it should place the new buffer it's trying to malloc.
 */
struct buf_data *new_buffer(sh_int size, bool persistant)
{
  struct buf_data *buft = get_buffer_head(), *potential, *prev = NULL;

  if (size == 0) {
    nmlog("SYSERR: BUF: new_buffer: 0 byte buffer requested.");
    return NULL;
  }

  /* There aren't any buffers. */
  if (!buft) {
    set_buffer_head(malloc_buffer(size, persistant));
    return get_buffer_head();
  }

  /* Insert a buffer */
  for (; buft; buft = buft->next) {
    if (buft == buft->next) {
      nmlog("SYSERR: BUF: new_buffer: Infinite loop!");
      buft->next = NULL;
    } else if (size < buft->size) {		/* Found where to insert. */
      potential = malloc_buffer(size, persistant);
      if (get_buffer_head() == buft)
        set_buffer_head(potential);
      potential->next = buft;
      if (prev)
        prev->next = potential;
      return potential;
    }
    prev = buft;
  }

  /* Append. */
  if (prev->next)
    nmlog("SYSERR: BUF: new_buffer: Overwrote a buffer.");
  prev->next = malloc_buffer(size, persistant);
  return prev->next;
}

/*
 * Private: malloc_buffer(size of buffer, persistant flag)
 * Creates a new buffer for use.
 */
struct buf_data *malloc_buffer(sh_int size, bool persistant)
{
  struct buf_data *new_buf;

  /*
   * Technically we could sit in an delayed loop until our malloc()
   * succeeds but if a machine is out of memory, it's probably better
   * that we just die and release the memory.
   */
  if (!(new_buf = (struct buf_data *)malloc(sizeof(struct buf_data)))) {
    nmlog("SYSERR: BUF: malloc_buffer: failed malloc() attempt!");
    perror("malloc_buffer()");
    return NULL;
  }

  new_buf->data = (char *)malloc(size + 1);
  new_buf->used = FALSE;
  new_buf->who = NULL;
  new_buf->line = 0;
  new_buf->life = (persistant ? -1 : BUFFER_LIFE);
  new_buf->next = NULL;
  new_buf->size = size;
#if defined(PICKY_BUFFERS)
  new_buf->req_size = 0;
#endif

  if (verbose_buffer) {
    char buf[128];
    sprintf(buf, "BUF: Allocated %d byte buffer, %d byte overhead.",
		new_buf->size, sizeof(struct buf_data));
    nmlog(buf);
  }
  return new_buf;
}

/*
 * Public: as get_buffer(size of buffer)
 * Requests a buffer from the free pool.  If a buffer of the desired size
 * is not available, one is created.
 */
#if defined(BUFFER_SNPRINTF)
struct buf_data *acquire_buffer(sh_int size, const char *who, ush_int line)
#else
char *acquire_buffer(sh_int size, const char *who, ush_int line)
#endif
{
  struct buf_data *give;

  /*
   * Search the list for an unused buffer that is large enough.
   */
  for (give = get_buffer_head(); give; give = give->next)
    if (give == give->next) {
      nmlog("SYSERR: BUF: acquire_buffer: Infinite loop!");
      give->next = NULL;
    } else if (!give->used && give->size >= size)
      break;

  /*
   * Everything big enough is being used, create a new one.
   */
  if (!give) {
    char buf[128];
    sprintf(buf, "BUF: Didn't find %d byte buffer! Making a new one.", size);
    nmlog(buf);
    give = new_buffer(size, FALSE);
  }

  /*
   * If we don't have a valid pointer by now, we're out of memory so just
   * return NULL and hope things don't crash too soon.
   */
  if (!give) {
    char buf[128];
    sprintf(buf, "SYSERR: BUF: %s: Couldn't get %d byte buffer for %s:%d.",
		__FUNCTION__, size, who, line);
    nmlog(buf);
    return NULL;
  }

  give->used = TRUE;
  give->who = who;
  give->line = line;
#if defined(PICKY_BUFFERS)
  give->req_size = size;
#endif
  if (verbose_buffer) {
    char buf[128];
    sprintf(buf, "BUF: %s:%d requested %d bytes, received %d.",
		who, line, size, give->size);
    nmlog(buf);
  }

  /*
   * Plant a magic number to see if someone overruns the buffer. If the first
   * character of the buffer is not NUL then somehow our memory was
   * overwritten...most likely by someone doing a release_buffer() and
   * keeping a secondary pointer to the buffer.
   */
#if defined(PICKY_BUFFERS)
  give->data[give->req_size] = MAGIC_NUMBER;
#else
  give->data[give->size] = MAGIC_NUMBER;
#endif
  if (*give->data != '\0') {
    nmlog("SYSERR: BUF: Buffer is not NULL as it ought to be!");
    *give->data = '\0';
  }

  last_alloc = give;	/* Cache this entry. */
#if defined(BUFFER_SNPRINTF)
  return give;	/* Hope they don't screw with anything. */
#else
  return give->data;
#endif
}

/*
 * This is really only useful to see who has lingering buffers around
 * or if you are curious.  It can't be called in the middle of a
 * command run by a player so it'll usually show the same thing.
 * You can call this with a NULL parameter to have it logged at any
 * time though.
 */
void show_buffers(struct char_data *ch)
{
  struct buf_data *disp;
  char *buf = get_buffer(MAX_STRING_LENGTH);
  int pos = 0, i = 0;

  for (disp = get_buffer_head(); disp; disp = disp->next)
    if (disp == disp->next) {
      nmlog("SYSERR: BUF: show_buffers: Infinite loop!");
      disp->next = NULL;
    } else
      pos += sprintf(buf + pos, "%3d. %5d bytes, %5d life, %s:%d.\r\n", ++i,
		disp->size, disp->life, disp->used ? disp->who : "unused",
		disp->used ? disp->line : 0);

  if (ch)
    send_to_char(buf, ch);
  else
    nmlog(buf);

  release_buffer(buf);
}

ACMD(do_overflow)
{
  char *buf = get_buffer(130);	/* Will get a 256 byte. */

  strcpy(buf, "01234567890123489798494561613163164984981916513213216546947894"
	"78949491613156194898191698132136484321321467897984132132156416879413"
	"21321654897894651321321564789794446346892656843657843653846578436537");

  release_buffer(buf);

  if (ch)
    send_to_char("Ok!\r\n", ch);
}

char *BUFFER_FORMAT =
"buffer (add | delete) size (persistant | temporary)\r\n";

ACMD(do_buffer)
{
  char *arg1, *arg2, *arg3;
  int size, persistant = FALSE;

  half_chop(argument, (arg1 = get_buffer(MAX_INPUT_LENGTH)), argument);
  half_chop(argument, (arg2 = get_buffer(MAX_INPUT_LENGTH)), argument);
  half_chop(argument, (arg3 = get_buffer(MAX_INPUT_LENGTH)), argument);
  if (!arg1 || !arg2 || !arg3)
    size = -1;
  else if ((size = atoi(arg2)) == 0)
    size = -1;
  else if (is_abbrev(arg3, "persistant"))
    persistant = TRUE;
  else if (is_abbrev(arg3, "temporary"))
    persistant = FALSE;
  else
    persistant = FALSE;

  /* Don't need these now. */
  release_buffer(arg2);
  release_buffer(arg3);

  /* Oops, error. */
  if (size < 0)
    send_to_char(BUFFER_FORMAT, ch);
  else if (is_abbrev(arg1, "delete")) {
    struct buf_data *temp, *toy;
    for (toy = get_buffer_head(); toy; toy = toy->next)
      if (toy == toy->next) {
	nmlog("SYSERR: BUF: do_buffer: Infinite loop!");
	toy->next = NULL;
      } else if (!toy->used && toy->size == size &&
		(persistant ? toy->life == -1 : toy->life != -1)) {
	REMOVE_FROM_LIST(toy, buffers, next);
	free_buffer(toy);
        break;
      }
   if (!toy)
     send_to_char("Not found.\r\n", ch);
  } else if (is_abbrev(arg1, "add"))
    new_buffer(size, persistant); /* So easy. :) */
  else
    send_to_char(BUFFER_FORMAT, ch);

  release_buffer(arg1);
}

int get_used(struct buf_data *buf)
{
#if defined(PICKY_BUFFERS)
  int cnt = buf->req_size;
#else
  int cnt = buf->size;
#endif

  for (; cnt >= 0 && buf->data[cnt] == '\0'; cnt--);

  return cnt;
}

