/* ************************************************************************
*   File: class.c                                       Part of CircleMUD *
*  Usage: Source file for class-specific code                             *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

/*
 * This file attempts to concentrate most of the code which must be changed
 * in order for new classes to be added.  If you're adding a new class,
 * you should go through this entire file from beginning to end and add
 * the appropriate new special cases for your new class.
 */



#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "db.h"
#include "utils.h"
#include "spells.h"
#include "interpreter.h"

#define rint(x)    ((int) ((x)+0.5))   /* dwix */

#define MANA_CAP 32000    /*    KAKO        */

void send_to_char(char *messg, struct char_data *ch);

/* Names first */

const char *class_abbrevs[] = {
  "&+cCi",
  "&+mMu",
  "&+wCl",
  "&+yTh",
  "&+BWa",
  "&+WPa",
  "&+GBa",
  "&+RAp",
  "&+gRa",
  "&+bMo",
  "&+YSa",
  "&+MCm",
  "\n"
};


const char *pc_class_types[] = {
  "&+cCitizen",
  "&+mMagic User",
  "&+wCleric",
  "&+yThief",
  "&+BWarrior",
  "&+WPaladin",
  "&+GBard",
  "&+RAnti-Paladin",
  "&+gRanger",
  "&+bMonk",
  "&+YSamurai",
  "&+MColormage",
  "\n"
};


/* The menu for choosing a class in interpreter.c: */
const char *class_menu []= {
  "\r\n&+c"			/* Human */
  "Select a class:\r\n"
  "  [C]leric\r\n"
  "  [T]hief\r\n"
  "  [W]arrior\r\n"
  "  [M]agic-user\r\n"
  "  [P]aladin\r\n"
  "  [B]ard\r\n"
  "  [A]nti-Paladin\r\n"
  "  [R]anger\r\n",
  "\r\n&+c"			/* Elf */
  "Select a class:\r\n"
  "  [C]leric\r\n"
  "  [T]hief\r\n"
  "  [W]arrior\r\n"
  "  [M]agic-user\r\n"
  "  [B]ard\r\n"
  "  [A]nti-Paladin\r\n"
  "  [R]anger\r\n",
  "\r\n&+c"			/* Dwarf */
  "Select a class:\r\n"
  "  [C]leric\r\n"
  "  [T]hief\r\n"
  "  [W]arrior\r\n"
  "  [B]ard\r\n"
  "  [A]nti-Paladin\n\r"
  "  [R]anger",
  "\r\n&+c"			/* Hobbit */
  "Select a class:\r\n"
  "  [C]leric\r\n"
  "  [T]hief\r\n"
  "  [W]arrior\r\n"
  "  [M]agic-user\r\n"
  "  [B]ard\n\r"
  "  [A]nti-Paladin"
  "  [R]anger",
  "\r\n&+c"			/* Ogre */
  "Select a class:\r\n"
  "  [C]leric\r\n"
  "  [W]arrior\r\n"
  "  [M]agic-user\r\n"
  "  [R]anger\r\n",
  "\r\n&+c"			/* Vampire */
  "Select a class:\r\n"
  "  [T]hief\r\n"
  "  [A]nti-Paladin\r\n",
  "\r\n&+c"			/* Troll */
  "Select a class:\r\n"
  "  [T]hief\r\n"
  "  [W]arrior\r\n"
  "  [M]agic-user\r\n"
  "  [B]ard\r\n"
  "  [A]nti-Paladin\r\n"
  "  [R]anger\r\n",
  "\r\n&+c"			/* Fairy */
  "Select a class:\r\n"
  "  [C]leric\r\n"
  "  [T]hief\r\n"
  "  [W]arrior\r\n"
  "  [M]agic-user\r\n"
  "  [B]ard\r\n"
  "  [R]anger\r\n",
  "\r\n&+c"			/* Dragon */
  "Select a class:\r\n"
  "  [W]arrior\r\n"
  "  [M]agic-user\r\n"
  "  [B]ard\r\n"
  "  [A]nti-Paladin\r\n"
  "  [R]anger\r\n",
  "\r\n&+c"			/* Tiefling */
  "Select a class:\r\n"
  "  [W]arrior\r\n"
  "  [M]agic-user\r\n"
  "  [A]nti-Paladin\r\n"
  "  [C]leric\r\n"
  "  [T]hief\r\n",
  "\r\n&+c"			/* Thri-Kreen */
  "Select a class:\r\n"
  "  [W]arrior\r\n"
  "  [M]agic-user\r\n"
  "  [C]leric\r\n"
  "  [R]anger\r\n",
};



/*
 * The code to interpret a class letter -- used in interpreter.c when a
 * new character is selecting a class and by 'set class' in act.wizard.c.
 */

int parse_class(char arg)
{
  arg = LOWER(arg);

  switch (arg) {
 case 'n':
  return CLASS_NEWBIE;
  break;
 case 'm':
  return CLASS_MAGIC_USER;
  break;
 case 'c':
  return CLASS_CLERIC;
  break;
 case 'w':
  return CLASS_WARRIOR;
  break;
 case 't':
  return CLASS_THIEF;
  break;
 case 'p':
  return CLASS_PALADIN;
  break;
 case 'b':
  return CLASS_BARD;
  break;
 case 'a':
  return CLASS_APALADIN;
  break;
 case 'r':
  return CLASS_RANGER;
  break;
 case 'k':
  return CLASS_MONK;
  break;
 case 'z':
  return CLASS_SAMURAI;
  break;
  default:
    return CLASS_UNDEFINED;
    break;
  }
}

/*
 * bitvectors (i.e., powers of two) for each class, mainly for use in
 * do_who and do_users.  Add new classes at the end so that all classes
 * use sequential powers of two (1 << 0, 1 << 1, 1 << 2, 1 << 3, 1 << 4,
 * 1 << 5, etc.
 */

long find_class_bitvector(char arg)
{
  arg = LOWER(arg);

  switch (arg) {
  case 'n':
    return (1 << CLASS_NEWBIE);
    break;
  case 'm':
    return (1 << CLASS_MAGIC_USER);
    break;
  case 'c':
    return (1 << CLASS_CLERIC);
    break;
  case 't':
    return (1 << CLASS_THIEF);
    break;
  case 'w':
    return (1 << CLASS_WARRIOR);
    break;
  case 'p':
    return (1 << CLASS_PALADIN);
    break;
  case 'b':
    return (1 << CLASS_BARD);
    break;
  case 'a':
    return (1 << CLASS_APALADIN);
    break;
  case 'r':
    return (1 << CLASS_RANGER);
    break;
  case 'k':
    return (1 << CLASS_MONK);
    break;
  case 'z':
    return (1 << CLASS_SAMURAI);
    break;
  case 'l':
    return (1 << CLASS_COLORMAGE);
    break;
  default:
    return 0;
    break;
  }
}


/*
 * These are definitions which control the guildmasters for each class.
 *
 * The first field (top line) controls the highest percentage skill level
 * a character of the class is allowed to attain in any skill.  (After
 * this level, attempts to practice will say "You are already learned in
 * this area."
 *
 * The second line controls the maximum percent gain in learnedness a
 * character is allowed per practice -- in other words, if the random
 * die throw comes out higher than this number, the gain will only be
 * this number instead.
 *
 * The third line controls the minimu percent gain in learnedness a
 * character is allowed per practice -- in other words, if the random
 * die throw comes out below this number, the gain will be set up to
 * this number.
 *
 * The fourth line simply sets whether the character knows 'spells'
 * or 'skills'.  This does not affect anything except the message given
 * to the character when trying to practice (i.e. "You know of the
 * following spells" vs. "You know of the following skills"
 */

#define SPELL	0
#define SKILL	1

/* #define LEARNED_LEVEL	0  % known which is considered "learned" */
/* #define MAX_PER_PRAC		1  max percent gain in skill per practice */
/* #define MIN_PER_PRAC		2  min percent gain in skill per practice */
/* #define PRAC_TYPE		3  should it say 'spell' or 'skill'?	*/

int prac_params[4][NUM_CLASSES] = {
  /* MAG	CLE	THE	WAR */
  /* NEW  MAG    CLE	THE    WAR    PAL    BARD   APAL   RAN	  MON    SAM*/
  {50,    95,	 95,	85,    80,    85,    80,    80,    80,	  90,    100},  /* learne level */
  {50,    100,	 100,	12,    12,    25,    100,   100,   100,	  25,	 100},  /* max per prac */
  {10,    25,	 25,	0,     0,     15,    20,    20,	   20,	  10,	 40},   /* min per pac */
  {SKILL, SPELL, SPELL, SKILL, SKILL, SKILL, SKILL, SKILL, SKILL, SKILL, SKILL} /* prac name */ 
};


/*
 * ...And the appropriate rooms for each guildmaster/guildguard; controls
 * which types of people the various guildguards let through.  i.e., the
 * first line shows that from room 3017, only MAGIC_USERS are allowed
 * to go south.
 */
int guild_info[][3] = {

/* Midgaard */
  {CLASS_MAGIC_USER,	3017,	SCMD_SOUTH},
  {CLASS_CLERIC,	3004,	SCMD_NORTH},
  {CLASS_THIEF,		3027,	SCMD_EAST},
  {CLASS_WARRIOR,	3021,	SCMD_EAST},
  {CLASS_PALADIN,       3067,   SCMD_UP},
  {CLASS_APALADIN,	3073,   SCMD_DOWN},
  {CLASS_BARD,		3070,	SCMD_SOUTH},
  {CLASS_RANGER,	8017,   SCMD_DOWN},
  {CLASS_MONK,		3078,	SCMD_NORTH},


/* Brass Dragon */
  {-999 /* all */ ,	5065,	SCMD_WEST},
  {-999           ,     12590,  SCMD_EAST},

/* New Sparta */
  {CLASS_MAGIC_USER,	21075,	SCMD_NORTH},
  {CLASS_CLERIC,	21019,	SCMD_WEST},
  {CLASS_THIEF,		21014,	SCMD_SOUTH},
  {CLASS_WARRIOR,	21023,	SCMD_SOUTH},

/* this must go last -- add new guards above! */
{-1, -1, -1}};




/* THAC0 for classes and levels.  (To Hit Armor Class 0) */

/* [class], [level] (all) */
const int thaco[NUM_CLASSES][LVL_SIMP + 1] = {

/* NEWBIE */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  10 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  20 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  30 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  40 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  50 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  60 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  70 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  80 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  90 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* -- 100 */
        20, 20, 20, 20, 20,},                   /* -- 105 */

/* MAGE */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  10 */
        19, 19, 19, 19, 19, 19, 19, 19, 19, 19, /* --  20 */
        18, 18, 18, 18, 18, 18, 18, 18, 18, 18, /* --  30 */
        17, 17, 17, 17, 17, 17, 17, 17, 17, 17, /* --  40 */
        16, 16, 16, 16, 16, 16, 16, 16, 16, 16, /* --  50 */
        15, 15, 15, 15, 15, 15, 15, 15, 15, 15, /* --  60 */
        14, 14, 14, 14, 14, 14, 14, 14, 14, 14, /* --  70 */
        13, 13, 13, 13, 13, 13, 13, 13, 13, 13, /* --  80 */
        12, 12, 12, 12, 12, 12, 12, 12, 12, 12, /* --  90 */
        11, 11, 11, 11, 11, 11, 11, 11, 11, 11, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 105 */


/* CLERIC */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  10 */
        18, 18, 18, 18, 18, 18, 18, 18, 18, 18, /* --  20 */
        16, 16, 16, 16, 16, 16, 16, 16, 16, 16, /* --  30 */
        14, 14, 14, 14, 14, 14, 14, 14, 14, 14, /* --  40 */
        12, 12, 12, 12, 12, 12, 12, 12, 12, 12, /* --  50 */
        10, 10, 10, 10, 10, 10, 10, 10, 10, 10, /* --  60 */
         8,  8,  8,  8,  8,  8,  8,  8,  8,  8, /* --  70 */
         6,  6,  6,  6,  6,  6,  6,  6,  6,  6, /* --  80 */
         4,  4,  4,  4,  4,  4,  4,  4,  4,  4, /* --  90 */
         2,  2,  2,  2,  2,  2,  2,  2,  2,  2, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 105 */

/* THIEF */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 20, 20, 20, 20, 19, 19, /* --  10 */
        19, 19, 19, 19, 19, 19, 18, 18, 18, 18, /* --  20 */
        18, 18, 18, 18, 17, 17, 17, 17, 17, 17, /* --  30 */
        17, 17, 16, 16, 16, 16, 16, 16, 16, 16, /* --  40 */
        15, 15, 15, 15, 15, 15, 14, 14, 14, 13, /* --  50 */
        13, 13, 13, 13, 13, 12, 12, 12, 12, 12, /* --  60 */
        12, 11, 11, 11, 11, 11, 11, 10, 10, 10, /* --  70 */
        10, 10, 10,  9,  9,  9,  9,  9,  9,  8, /* --  80 */
         8,  8,  8,  8,  8,  7,  7,  7,  7,  7, /* --  90 */
         7,  6,  6,  6,  6,  6,  6,  5,  5,  5, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 105 */

/* WARRIOR */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 19, 19, 19, 19, 18, 18, /* --  10 */
        18, 18, 17, 17, 17, 17, 16, 16, 16, 16, /* --  20 */
        15, 15, 15, 15, 14, 14, 14, 14, 13, 13, /* --  30 */
        13, 13, 12, 12, 12, 12, 11, 11, 11, 11, /* --  40 */
        10, 10, 10, 10,  9,  9,  9,  9,  8,  8, /* --  50 */
         8,  8,  7,  7,  7,  7,  6,  6,  6,  6, /* --  60 */
         5,  5,  5,  5,  4,  4,  4,  4,  3,  3, /* --  70 */
         3,  3,  2,  2,  2,  2,  1,  1,  1,  1, /* --  80 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* --  90 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 100 */

/* PALADIN */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 19, 19, 19, 19, 18, 18, /* --  10 */
        18, 18, 17, 17, 17, 17, 16, 16, 16, 16, /* --  20 */
        15, 15, 15, 15, 14, 14, 14, 14, 13, 13, /* --  30 */
        13, 13, 12, 12, 12, 12, 11, 11, 11, 11, /* --  40 */
        10, 10, 10, 10,  9,  9,  9,  9,  8,  8, /* --  50 */
         8,  8,  7,  7,  7,  7,  6,  6,  6,  6, /* --  60 */
         5,  5,  5,  5,  4,  4,  4,  4,  3,  3, /* --  70 */
         3,  3,  2,  2,  2,  2,  1,  1,  1,  1, /* --  80 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* --  90 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 105 */

/* BARD */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 19, 19, 19, 19, 18, 18, /* --  10 */
        18, 18, 17, 17, 17, 17, 16, 16, 16, 16, /* --  20 */
        15, 15, 15, 15, 14, 14, 14, 14, 13, 13, /* --  30 */
        13, 13, 12, 12, 12, 12, 11, 11, 11, 11, /* --  40 */
        10, 10, 10, 10,  9,  9,  9,  9,  8,  8, /* --  50 */
         8,  8,  7,  7,  7,  7,  6,  6,  6,  6, /* --  60 */
         5,  5,  5,  5,  4,  4,  4,  4,  3,  3, /* --  70 */
         3,  3,  2,  2,  2,  2,  1,  1,  1,  1, /* --  80 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* --  90 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 105 */

/* ANTI PALADIN */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 19, 19, 19, 19, 18, 18, /* --  10 */
        18, 18, 17, 17, 17, 17, 16, 16, 16, 16, /* --  20 */
        15, 15, 15, 15, 14, 14, 14, 14, 13, 13, /* --  30 */
        13, 13, 12, 12, 12, 12, 11, 11, 11, 11, /* --  40 */
        10, 10, 10, 10,  9,  9,  9,  9,  8,  8, /* --  50 */
         8,  8,  7,  7,  7,  7,  6,  6,  6,  6, /* --  60 */
         5,  5,  5,  5,  4,  4,  4,  4,  3,  3, /* --  70 */
         3,  3,  2,  2,  2,  2,  1,  1,  1,  1, /* --  80 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* --  90 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 105 */

/* RANGER */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 19, 19, 19, 19, 18, 18, /* --  10 */
        18, 18, 17, 17, 17, 17, 16, 16, 16, 16, /* --  20 */
        15, 15, 15, 15, 14, 14, 14, 14, 13, 13, /* --  30 */
        13, 13, 12, 12, 12, 12, 11, 11, 11, 11, /* --  40 */
        10, 10, 10, 10,  9,  9,  9,  9,  8,  8, /* --  50 */
         8,  8,  7,  7,  7,  7,  6,  6,  6,  6, /* --  60 */
         5,  5,  5,  5,  4,  4,  4,  4,  3,  3, /* --  70 */
         3,  3,  2,  2,  2,  2,  1,  1,  1,  1, /* --  80 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* --  90 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 105 */

/* MONK */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  10 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  20 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  30 */
        20, 20, 20, 20, 20, 20, 20, 20, 20, 20, /* --  40 */
        20, 20, 21, 21, 22, 22, 23, 23, 24, 24, /* --  50 */
        25, 25, 26, 26, 27, 27, 28, 28, 29, 29, /* --  60 */
        30, 30, 31, 31, 32, 32, 33, 33, 34, 34, /* --  70 */
        35, 35, 36, 36, 37, 37, 38, 38, 39, 39, /* --  80 */
        40, 40, 41, 41, 42, 42, 43, 43, 44, 44, /* --  90 */
        45, 45, 46, 46, 47, 47, 48, 48, 49, 49, /* -- 100 */
        1,  1,   1,  1,  1,},                   /* -- 105 */

/* SAMURAI */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 19, 19, 19, 19, 18, 18, /* --  10 */
        18, 18, 17, 17, 17, 17, 16, 16, 16, 16, /* --  20 */
        15, 15, 15, 15, 14, 14, 14, 14, 13, 13, /* --  30 */
        13, 13, 12, 12, 12, 12, 11, 11, 11, 11, /* --  40 */
        10, 10, 10, 10,  9,  9,  9,  9,  8,  8, /* --  50 */
         8,  8,  7,  7,  7,  7,  6,  6,  6,  6, /* --  60 */
         5,  5,  5,  5,  4,  4,  4,  4,  3,  3, /* --  70 */
         3,  3,  2,  2,  2,  2,  1,  1,  1,  1, /* --  80 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* --  90 */
         1,  1,  1,  1,  1,  1,  1,  1,  1,  1, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 105 */

/* COLORMAGE */
  /* 0                   5                  10            */
  {100, 20, 20, 20, 20, 20, 19, 19, 19, 18, 18, /* --  10 */
        18, 18, 17, 17, 17, 17, 17, 17, 16, 16, /* --  20 */
        16, 16, 16, 16, 16, 15, 15, 15, 15, 15, /* --  30 */
        15, 15, 15, 14, 14, 14, 14, 14, 14, 14, /* --  40 */
        13, 13, 13, 13, 13, 13, 13, 13, 13, 13, /* --  50 */
        12, 12, 12, 12, 12, 12, 12, 12, 12, 12, /* --  60 */
        11, 11, 11, 11, 11, 11, 11, 11, 11, 11, /* --  70 */
        10, 10, 10, 10, 10, 10, 10, 10, 10, 10, /* --  80 */
         9,  9,  9,  9,  9,  9,  9,  9,  9,  9, /* --  90 */
         8,  8,  8,  8,  8,  8,  8,  8,  8,  8, /* -- 100 */
         1,  1,  1,  1,  1,},                   /* -- 105 */


};

/*
 *
 * This function will apply the racial modifiers to a characters
 * stats.
 *
 */
void apply_race_stat_mods(struct char_data *ch)
{
  switch (GET_RACE(ch)) {
    case RACE_HUMAN:
      break;
    case RACE_ELF:
      ch->real_abils.dex += 1;
      ch->real_abils.con -= 1;
      break;
    case RACE_DWARF:
      ch->real_abils.con += 4;
      ch->real_abils.cha -= 2;
      break;
    case RACE_HOBBIT:
      ch->real_abils.dex += 2;
      ch->real_abils.str -= 4;
      ch->real_abils.str_add = 0;
      break;
    case RACE_OGRE:
      ch->real_abils.str += 2;
      ch->real_abils.dex -= 2;
      ch->real_abils.str_add = 20;
      if(ch->real_abils.str > 18)
	ch->real_abils.str_add = 0;
      break;
    case RACE_FAIRY:
      ch->real_abils.str -= 2;
      ch->real_abils.dex += 2;
      ch->real_abils.str_add = 0;
      break;
    case RACE_VAMPIRE:
      ch->real_abils.str += 1;
      ch->real_abils.dex += 1;
      ch->real_abils.cha += 1;
      break;
    case RACE_DRAGON:
      ch->real_abils.dex -= 1;
      ch->real_abils.wis += 1;
      ch->real_abils.intel += 1;
      break;
    case RACE_TIEFLING:
      if(ch->real_abils.cha > 4)
        ch->real_abils.cha -= 2;
      else ch->real_abils.dex -=1;
      ch->real_abils.con -= 1;
      switch (number(1, 4)) {
        case 1:
          ch->real_abils.str += 2;
	  if(ch->real_abils.str == 18 && number(1, 4) == 4)
	    ch->real_abils.str_add = number(0, 100);
          if(ch->real_abils.str > 18) {
	    ch->real_abils.str = 19;
	    ch->real_abils.str_add = 0;
	    ch->real_abils.con += 1;
	  }
          break;
        case 2:
          ch->real_abils.dex += 2;
          if(ch->real_abils.dex > 18) {
	    ch->real_abils.dex = 19;
	    ch->real_abils.intel += 1;
	  }
          break;
        case 3:
          ch->real_abils.intel += 2;
          if(ch->real_abils.intel > 18) {
	    ch->real_abils.intel = 19;
	    ch->real_abils.wis += 1;
	  }
          break;
        case 4:
          ch->real_abils.wis += 2;
          if(ch->real_abils.wis > 18) {
	    ch->real_abils.wis = 19;
	    ch->real_abils.dex += 1;
	  }
          break;
       }
       break;
    case RACE_THRIKREEN:
      ch->real_abils.str += 2;
      ch->real_abils.cha -= 2;
      ch->real_abils.dex += 1;
      ch->real_abils.intel -= 1;
      if(ch->real_abils.str > 18) ch->real_abils.str_add = 0;
      if(ch->real_abils.str == 18)
        ch->real_abils.str_add = number(0, 100);
      break;
    case RACE_GARGOYLE:
      ch->real_abils.str += 1;
      ch->real_abils.dex += 1;
      ch->real_abils.wis -= 1;
      ch->real_abils.intel -= 1;
      break;
    }
}


/*
 *
 * This function will apply all racial affects which apply to the race.
 *
 */
void apply_race_affects(struct char_data *ch)
{
  int i;

  switch(GET_RACE(ch)) {

  case RACE_HUMAN:
    break;

  case RACE_ELF:
    break;

  case RACE_DWARF:
    SET_BIT(RAFF_FLAGS(ch), AFF_INFRAVISION);
    SET_BIT(AFF_FLAGS(ch), AFF_INFRAVISION);
    for (i=0;i < NUM_SAVING; i++)
     GET_SAVE(ch, i) -= 5*(GET_CON(ch) / 4);
    break;
 /*Dwarf con bonus multiplied by 5 to convert from d20-based AD&D system */
 /*to percentage-based based MUD system - Trebor 8/96 */

  case RACE_HOBBIT:
    break;

  case RACE_OGRE:
    SET_BIT(RAFF_FLAGS(ch), AFF_NOPOISON);
    SET_BIT(AFF_FLAGS(ch), AFF_NOPOISON);
    break;

  case RACE_DRAGON:
    break;

  case RACE_VAMPIRE:
    SET_BIT(RAFF_FLAGS(ch), AFF_INFRAVISION);
    SET_BIT(AFF_FLAGS(ch), AFF_INFRAVISION);
    SET_BIT(RAFF_FLAGS(ch), AFF_SENSE_LIFE);
    SET_BIT(AFF_FLAGS(ch), AFF_SENSE_LIFE);
    SET_BIT(RAFF_FLAGS(ch), AFF_WATERBREATH);
    SET_BIT(AFF_FLAGS(ch), AFF_WATERBREATH);
    break;

  case RACE_TROLL:
    SET_BIT(RAFF_FLAGS(ch), AFF_INFRAVISION);
    SET_BIT(AFF_FLAGS(ch), AFF_INFRAVISION);
    for (i=0;i < NUM_SAVING; i++)
    GET_SAVE(ch, i) += 5*(18 - GET_WIS(ch));
    break;

  case RACE_FAIRY:
    SET_BIT(RAFF_FLAGS(ch), AFF_FLYING);
    SET_BIT(AFF_FLAGS(ch), AFF_FLYING);
    break;

  case RACE_TIEFLING:
    if (number(1, 10) > 8 ) {
    SET_BIT(RAFF_FLAGS(ch), AFF_INFRAVISION);
    SET_BIT(AFF_FLAGS(ch), AFF_INFRAVISION);
    }
    if (number(1, 10) > 8 ) {
    SET_BIT(RAFF_FLAGS(ch), AFF_NOPOISON);
    SET_BIT(AFF_FLAGS(ch), AFF_NOPOISON);
    }
    if (number(1, 10) > 8 ) {
    SET_BIT(RAFF_FLAGS(ch), AFF_SENSE_LIFE);
    SET_BIT(AFF_FLAGS(ch), AFF_SENSE_LIFE);
    }
    if (number(1, 10) > 8 ) {
    SET_BIT(RAFF_FLAGS(ch), AFF_WATERBREATH);
    SET_BIT(AFF_FLAGS(ch), AFF_WATERBREATH);
    }
    if (number(1, 10) > 8 ) {
    SET_BIT(RAFF_FLAGS(ch), AFF_FLYING);
    SET_BIT(AFF_FLAGS(ch), AFF_FLYING);
    }
    if (number(1, 10) > 8 )
      for (i=0;i < NUM_SAVING; i++)
       GET_SAVE(ch, i) += 5*(18 - GET_WIS(ch));
    if (number(1, 10) > 8 )
      for (i=0;i < NUM_SAVING; i++)
       GET_SAVE(ch, i) -= (GET_CON(ch) / 4);
    break;

  case RACE_THRIKREEN:
    break;

  case RACE_GARGOYLE:
    SET_BIT (RAFF_FLAGS(ch), AFF_INFRAVISION);
    SET_BIT (AFF_FLAGS(ch), AFF_INFRAVISION);
    break;
  }
}


/*
 * This function will re-arange the stats of a newbie to the best
 * arangement for the different classes.
 *
 * The newbie arangements has the following priorities:
 * Strength
 * Dexterity
 * Constitution
 * Wisdom
 * Intelegence
 * Charisma
 *
 */
void arrange_stats(struct char_data *ch, int class)
{
struct char_ability_data orig_real;
struct char_ability_data orig_aff;
struct char_ability_data affs;
int intermediate;

orig_real    = ch->real_abils;
orig_aff     = ch->aff_abils;
affs.str     = GET_STR(ch) - ch->real_abils.str;
affs.dex     = GET_DEX(ch) - ch->real_abils.dex;
affs.con     = GET_CON(ch) - ch->real_abils.con;
affs.intel   = GET_INT(ch) - ch->real_abils.intel;
affs.wis     = GET_WIS(ch) - ch->real_abils.wis;
affs.cha     = GET_CHA(ch) - ch->real_abils.cha;
affs.str_add = GET_ADD(ch) - ch->real_abils.str_add;
affs.str    += (int) affs.str_add / 10;

switch(class)
    {
  case CLASS_MAGIC_USER:
/*    intermediate = ch->real_abils.intel;
    ch->real_abils.intel = ch->real_abils.str;
    ch->real_abils.str = ch->real_abils.wis;
    ch->real_abils.wis = ch->real_abils.dex;
    ch->real_abils.dex = ch->real_abils.con;
    ch->real_abils.con = intermediate;
    ch->real_abils.str_add = 0; */
    break;
  case CLASS_CLERIC:
/*    intermediate = ch->real_abils.wis;
    ch->real_abils.wis = ch->real_abils.str;
    ch->real_abils.str = ch->real_abils.con;
    ch->real_abils.con = ch->real_abils.intel;
    ch->real_abils.intel = ch->real_abils.dex;
    ch->real_abils.dex = intermediate;
    ch->real_abils.str_add = 0; */
    break;
  case CLASS_THIEF:
/*    intermediate = ch->real_abils.dex;
    ch->real_abils.dex = ch->real_abils.str;
    ch->real_abils.str = intermediate;
    intermediate = ch->real_abils.intel;
    ch->real_abils.intel = ch->real_abils.wis;
    ch->real_abils.wis = intermediate;
    ch->real_abils.str_add = 0; */
    break;
  case CLASS_WARRIOR:
  case CLASS_APALADIN:
  case CLASS_PALADIN:
  case CLASS_COLORMAGE:
    break;
  case CLASS_BARD:
/*    intermediate = ch->real_abils.dex;
    ch->real_abils.dex = ch->real_abils.str;
    ch->real_abils.str = ch->real_abils.wis;
    ch->real_abils.wis = intermediate;
    intermediate = ch->real_abils.intel;
    ch->real_abils.intel = ch->real_abils.con;
    ch->real_abils.cha = ch->real_abils.con;
    ch->real_abils.con = intermediate;
    if (ch->real_abils.str != 18)
      ch->real_abils.str_add = 0; */
    break;
  case CLASS_RANGER:
/*    intermediate = ch->real_abils.dex;
    ch->real_abils.dex = ch->real_abils.str;
    ch->real_abils.str = intermediate;
    intermediate = ch->real_abils.wis;
    ch->real_abils.wis = ch->real_abils.con;
    ch->real_abils.con = ch->real_abils.intel;
    ch->real_abils.intel = intermediate;
    if (ch->real_abils.str != 18)
      ch->real_abils.str_add = 0; */
    break;
  case CLASS_MONK:
/*    intermediate = ch->real_abils.dex;
    ch->real_abils.dex = ch->real_abils.str;
    ch->real_abils.str = ch->real_abils.con;
    ch->real_abils.con = ch->real_abils.intel;
    ch->real_abils.intel = ch->real_abils.wis;
    ch->real_abils.wis = intermediate;
    if (ch->real_abils.str != 18)
      ch->real_abils.str_add = 0; */
    break;
  case CLASS_SAMURAI:
/*    ch->real_abils.dex = 20;
    ch->real_abils.str = 18;
    ch->real_abils.str_add = 100;
    ch->real_abils.intel = 20;
    ch->real_abils.con = 20;
    ch->real_abils.cha = 20;
    ch->real_abils.wis = 20; */
    break;
    }

apply_race_stat_mods(ch);

GET_DEX(ch) = ch->real_abils.dex     + affs.dex;
GET_CON(ch) = ch->real_abils.con     + affs.con;
GET_INT(ch) = ch->real_abils.intel   + affs.intel;
GET_WIS(ch) = ch->real_abils.wis     + affs.wis;
GET_CHA(ch) = ch->real_abils.cha     + affs.cha;
GET_STR(ch) = ch->real_abils.str;
GET_ADD(ch) = ch->real_abils.str_add;

while ((affs.str > 0) && (GET_STR(ch) < 18)) {
  GET_STR(ch)++;
  affs.str--;
  }
while ((affs.str > 0) && (GET_STR(ch) == 18) && (GET_ADD(ch) < 100)) {
  GET_ADD(ch) = MIN(GET_ADD(ch) + 10, 100);
  affs.str--;
  }
while (affs.str > 0) {
  GET_STR(ch)++;
  affs.str--;
  }

while ((affs.str < 0) && (GET_STR(ch) > 18)) {
  GET_STR(ch)--;
  affs.str++;
  }
while ((affs.str < 0) && (GET_STR(ch) == 18) && (GET_ADD(ch) > 0)) {
  GET_ADD(ch) = MAX(GET_ADD(ch) - 10, 0);
  affs.str++;
  }
while (affs.str < 0) {
  GET_STR(ch)--;
  affs.str++;
  }

}


/*
 * Roll the 6 stats for a character... each stat is made of the sum of
 * the best 3 out of 4 rolls of a 6-sided die.  Each class then decides
 * which priority will be given for the best to worst stats.
 */
void roll_real_abils(struct char_data * ch)
{
  int i, j, k, temp;
  ubyte table[6];
  ubyte rolls[4];

  for (i = 0; i < 6; i++)
    table[i] = 0;

  for (i = 0; i < 6; i++) {

    for (j = 0; j < 4; j++)
      rolls[j] = number(1, 6);

    temp = rolls[0] + rolls[1] + rolls[2] + rolls[3] -
      MIN(rolls[0], MIN(rolls[1], MIN(rolls[2], rolls[3])));

    for (k = 0; k < 6; k++)
      if (table[k] < temp) {
	temp ^= table[k];
	table[k] ^= temp;
	temp ^= table[k];
      }
  }

  ch->real_abils.str_add = 0;

  switch (GET_CLASS(ch)) {
  case CLASS_NEWBIE:
  case CLASS_COLORMAGE:
    ch->real_abils.str = table[0];
    ch->real_abils.dex = table[1];
    ch->real_abils.con = table[2];
    ch->real_abils.wis = table[3];
    ch->real_abils.intel = table[4];
    ch->real_abils.cha = table[5];
    if (ch->real_abils.str == 18)
      ch->real_abils.str_add = number(0, 100);
    break;
  case CLASS_MAGIC_USER:
    ch->real_abils.intel = table[0];
    ch->real_abils.wis = table[1];
    ch->real_abils.dex = table[2];
    ch->real_abils.str = table[3];
    ch->real_abils.con = table[4];
    ch->real_abils.cha = table[5];
    break;
  case CLASS_CLERIC:
    ch->real_abils.wis = table[0];
    ch->real_abils.intel = table[1];
    ch->real_abils.str = table[2];
    ch->real_abils.dex = table[3];
    ch->real_abils.con = table[4];
    ch->real_abils.cha = table[5];
    break;
  case CLASS_THIEF:
    ch->real_abils.dex = table[0];
    ch->real_abils.str = table[1];
    ch->real_abils.con = table[2];
    ch->real_abils.intel = table[3];
    ch->real_abils.wis = table[4];
    ch->real_abils.cha = table[5];
    break;
  case CLASS_PALADIN:
    ch->real_abils.str = table[0];
    ch->real_abils.dex = table[1];
    ch->real_abils.con = table[2];
    ch->real_abils.wis = table[3];
    ch->real_abils.intel = table[4];
    ch->real_abils.cha = table[5];
    if (ch->real_abils.str == 18)
      ch->real_abils.str_add = number(0, 100);
    break;
  case CLASS_BARD:
    ch->real_abils.dex = table[0];
    ch->real_abils.wis = table[1];
    ch->real_abils.intel = table[2];
    ch->real_abils.str = table[3];
    ch->real_abils.cha = table[4];
    ch->real_abils.con = table[5];
    if (ch->real_abils.str == 18)
      ch->real_abils.str_add = number(0, 100);
    break;
  case CLASS_APALADIN:
    ch->real_abils.str = table[0];
    ch->real_abils.dex = table[1];
    ch->real_abils.con = table[2];
    ch->real_abils.wis = table[3];
    ch->real_abils.intel = table[4];
    ch->real_abils.cha = table[5];
    if (ch->real_abils.str == 18)
      ch->real_abils.str_add = number(0, 100);
    break;
  case CLASS_RANGER:
    ch->real_abils.dex = table[0];
    ch->real_abils.str = table[1];
    ch->real_abils.wis = table[2];
    ch->real_abils.intel = table[3];
    ch->real_abils.con = table[4];
    ch->real_abils.cha = table[5];
    if (ch->real_abils.str == 18)
      ch->real_abils.str_add = number(0, 100);
    break;
  case CLASS_MONK:		/* dwix */
    ch->real_abils.dex = table[0];
    ch->real_abils.wis = table[1];
    ch->real_abils.str = table[2];
    ch->real_abils.intel = table[3];
    ch->real_abils.con = table[4];
    ch->real_abils.cha = table[5];
    if (ch->real_abils.str == 18)
      ch->real_abils.str_add = number(0, 100);
    break;
  case CLASS_SAMURAI:
    ch->real_abils.dex = 20;
    ch->real_abils.str = 18;
    ch->real_abils.str_add = 100;
    ch->real_abils.intel = 20;
    ch->real_abils.con = 20;
    ch->real_abils.cha = 20;
    ch->real_abils.wis = 20;
    break;
  }
  if (GET_CLASS(ch) != CLASS_NEWBIE)
    apply_race_stat_mods(ch);

  ch->aff_abils = ch->real_abils;
}


/* Some initializations for characters, including initial skills */
void do_start(struct char_data * ch)
{
  void advance_level(struct char_data * ch);
  int chtitle = TRUE;
  int i;

  GET_LEVEL(ch) = 1;
  GET_EXP(ch) = 1;

  GET_PRACTICES(ch) = 0;

  GET_HITROLL(ch) = 0;
  GET_DAMROLL(ch) = 0;
  GET_AC(ch) = 100;

  for (i=0;i < NUM_SAVING; i++)
  GET_SAVE(ch, i) = 0;

  if (chtitle)
    set_title(ch, NULL);
  ch->points.max_hit = 10;
  ch->points.max_mana = 100;
  ch->points.max_move = 100;

/*
  AFF_FLAGS(ch) = 0;
  AFF2_FLAGS(ch) = 0;
  RAFF_FLAGS(ch) = 0;
  RAFF2_FLAGS(ch) = 0;
*/

  if (GET_CLASS(ch) != CLASS_NEWBIE)
    apply_race_affects(ch);

  switch (GET_CLASS(ch)) {

  case CLASS_NEWBIE:
    break;

  case CLASS_MAGIC_USER:
    break;

  case CLASS_CLERIC:
    break;

  case CLASS_THIEF:
/*    SET_SKILL(ch, SKILL_SNEAK, MAX(10, GET_SKILL(ch, SKILL_SNEAK)));
    SET_SKILL(ch, SKILL_HIDE, MAX(5, GET_SKILL(ch, SKILL_HIDE)));
    SET_SKILL(ch, SKILL_STEAL, MAX(15, GET_SKILL(ch, SKILL_STEAL)));
    SET_SKILL(ch, SKILL_BACKSTAB, MAX(10, GET_SKILL(ch, SKILL_BACKSTAB)));
    SET_SKILL(ch, SKILL_PICK_LOCK, MAX(10, GET_SKILL(ch, SKILL_PICK_LOCK)));
    SET_SKILL(ch, SKILL_TRACK, MAX(10, GET_SKILL(ch, SKILL_TRACK))); */
    break;

  case CLASS_WARRIOR:
    break;

  case CLASS_PALADIN:
    for (i=0;i < NUM_SAVING; i++)
    GET_SAVE(ch, i) -= 2;
    break;

  case CLASS_BARD:
    break;

  case CLASS_APALADIN:
    for (i=0;i < NUM_SAVING; i++)
    GET_SAVE(ch, i) -= 2;
    break;

  case CLASS_RANGER:
    break;

  case CLASS_MONK:
    break;

  case CLASS_SAMURAI:
    break;

  case CLASS_COLORMAGE:
    break;
  }


  advance_level(ch);

  GET_HIT(ch) = GET_MAX_HIT(ch);
  GET_MANA(ch) = GET_MAX_MANA(ch);
  GET_MOVE(ch) = GET_MAX_MOVE(ch);

  GET_COND(ch, THIRST) = 24;
  GET_COND(ch, FULL) = 24;
  GET_COND(ch, DRUNK) = 0;

  if (GET_CLASS(ch) == CLASS_NEWBIE) {
    ch->player.time.played = 0;
    ch->player.time.logon = time(0);
    }
}



/*
 * This function controls the change to maxmove, maxmana, and maxhp for
 * each class every time they gain a level.
 */
void advance_level(struct char_data * ch)
{
  int add_hp = 0, add_mana = 0, add_move = 0, i;

  extern struct wis_app_type wis_app[];
  extern struct con_app_type con_app[];

  add_hp = con_app[GET_CON(ch)].hitp;

  switch (GET_CLASS(ch)) {

  case CLASS_NEWBIE:
    break;

  case CLASS_MAGIC_USER:
    add_hp += number(6, 25);
    add_mana = number(GET_LEVEL(ch), (int) (1.5 * GET_LEVEL(ch)));
    add_mana = MAX(add_mana, 10);
    add_move = number(0, 2);
    break;

  case CLASS_CLERIC:
    add_hp += number(10, 29);
    add_mana = number(GET_LEVEL(ch), (int) (1.5 * GET_LEVEL(ch)));
    add_mana = MAX(add_mana, 10);
    add_move = number(0, 2);
    break;

  case CLASS_THIEF:
    add_hp += number(14, 35);
    add_mana = 0;
    add_move = number(1, 3);
    break;

  case CLASS_WARRIOR:
    add_hp += number(20, 39);
    add_mana = 0;
    add_move = number(1, 3);
    break;

  case CLASS_PALADIN:
    add_hp += number(20, 39);
    add_mana = GET_LEVEL(ch);
    add_move = number(1, 3);
    for (i=0;i < NUM_SAVING; i++)
      GET_SAVE(ch, i) = -2;
    break;

  case CLASS_BARD:
    add_hp += number(14, 35);
    add_mana = GET_LEVEL(ch) >> 1;
    add_move = number(1, 3);
    break;

  case CLASS_APALADIN:
    add_hp += number(20, 39);
    add_mana = GET_LEVEL(ch);
    add_move = number(1, 3);
    for (i=0;i < NUM_SAVING; i++)
      GET_SAVE(ch, i) = -2;
    break;

  case CLASS_RANGER:
    add_hp += number(14, 35);
    add_mana = GET_LEVEL(ch) >> 1;
    add_move = number(1, 4);
    break;

  case CLASS_SAMURAI:
    add_hp += number(20, 39);
    add_mana = 0;
    add_move = number(1, 3);
    break;

  case CLASS_MONK:
    add_hp += number(30, 49);
    add_mana = 0;
    add_move = number(1, 3);
    GET_HITROLL(ch) += rint(GET_LEVEL(ch) / 2) - rint( (GET_LEVEL(ch)-1) / 2);
    GET_DAMROLL(ch) += rint(GET_LEVEL(ch) / 2) - rint( (GET_LEVEL(ch)-1) / 2);
    GET_AC(ch) -= 2;
    break;

  case CLASS_COLORMAGE:
    add_hp += number(2, 23);
    add_move = number(0, 2);
    break;

  }
  switch (GET_RACE(ch)) {

  case RACE_HUMAN:
    break;

  case RACE_DWARF:
    if (GET_LEVEL(ch) >= 50) {
      SET_BIT(RAFF_FLAGS(ch), AFF_DET_DOOR);
      SET_BIT(AFF_FLAGS(ch), AFF_DET_DOOR);
    }
    break;

  case RACE_HOBBIT:
    if (GET_LEVEL(ch) >= 5) {
      SET_BIT(RAFF_FLAGS(ch), AFF_SNEAK);
      SET_BIT(AFF_FLAGS(ch), AFF_SNEAK);
    }
    if (GET_LEVEL(ch) >= 10) {
      SET_BIT(RAFF_FLAGS(ch), AFF_NOTRACK);
      SET_BIT(AFF_FLAGS(ch), AFF_NOTRACK);
    }
    if (GET_LEVEL(ch) >= 25) {
      SET_BIT(RAFF_FLAGS(ch), AFF_DETECT_ALIGN);
      SET_BIT(AFF_FLAGS(ch), AFF_DETECT_ALIGN);
    }
    if (GET_LEVEL(ch) >= 50) {
      SET_BIT(RAFF_FLAGS(ch), AFF_PROTECT_EVIL);
      SET_BIT(AFF_FLAGS(ch), AFF_PROTECT_EVIL);
    }
    break;

  case RACE_DRAGON:
    if (GET_LEVEL(ch) >= 10) {
      SET_BIT(RAFF_FLAGS(ch), AFF_FLYING);
      SET_BIT(AFF_FLAGS(ch), AFF_FLYING);
    }
    if (GET_LEVEL(ch) >= 75) SET_BIT(PRF_FLAGS(ch), PRF_NOHASSLE);
    break;

  case RACE_VAMPIRE:
    break;

  case RACE_TROLL:
    if (GET_LEVEL(ch) >= 50) {
      SET_BIT(RAFF_FLAGS(ch), AFF_PROTECT_GOOD);
      SET_BIT(AFF_FLAGS(ch), AFF_PROTECT_GOOD);
    }
    break;

  case RACE_TIEFLING:
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 50) {
        SET_BIT(RAFF_FLAGS(ch), AFF_DET_DOOR);
        SET_BIT(AFF_FLAGS(ch), AFF_DET_DOOR);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 5) {
        SET_BIT(RAFF_FLAGS(ch), AFF_SNEAK);
        SET_BIT(AFF_FLAGS(ch), AFF_SNEAK);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 10) {
        SET_BIT(RAFF_FLAGS(ch), AFF_NOTRACK);
        SET_BIT(AFF_FLAGS(ch), AFF_NOTRACK);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 25) {
        SET_BIT(RAFF_FLAGS(ch), AFF_DETECT_ALIGN);
        SET_BIT(AFF_FLAGS(ch), AFF_DETECT_ALIGN);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 50) {
        SET_BIT(RAFF_FLAGS(ch), AFF_PROTECT_EVIL);
        SET_BIT(AFF_FLAGS(ch), AFF_PROTECT_EVIL);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 75) SET_BIT(PRF_FLAGS(ch), PRF_NOHASSLE);
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 50) {
        SET_BIT(RAFF_FLAGS(ch), AFF_PROTECT_GOOD);
        SET_BIT(AFF_FLAGS(ch), AFF_PROTECT_GOOD);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 25) {
        SET_BIT(RAFF_FLAGS(ch), AFF_WATERWALK);
        SET_BIT(AFF_FLAGS(ch), AFF_WATERWALK);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 75) {
        SET_BIT(RAFF_FLAGS(ch), AFF_HASTE);
        SET_BIT(AFF_FLAGS(ch), AFF_HASTE);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 30) {
        SET_BIT(RAFF_FLAGS(ch), AFF_DETECT_INVIS);
        SET_BIT(AFF_FLAGS(ch), AFF_DETECT_INVIS);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 70) {
        SET_BIT(RAFF_FLAGS(ch), AFF_SANCTUARY);
        SET_BIT(AFF_FLAGS(ch), AFF_SANCTUARY);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 25) {
        SET_BIT(RAFF_FLAGS(ch), AFF_DETECT_MAGIC);
        SET_BIT(AFF_FLAGS(ch), AFF_DETECT_MAGIC);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 50) {
        SET_BIT(RAFF_FLAGS(ch), AFF_BLUR);
        SET_BIT(AFF_FLAGS(ch), AFF_BLUR);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 25) {
        SET_BIT(RAFF_FLAGS(ch), AFF_NOWEB);
        SET_BIT(AFF_FLAGS(ch), AFF_NOWEB);
      }
    }
    if (number(1, 10) > 8 ) {
      if (GET_LEVEL(ch) == 50) {
        SET_BIT(RAFF2_FLAGS(ch), AFF2_STONESKIN);
        SET_BIT(AFF2_FLAGS(ch), AFF2_STONESKIN);
      }
    }
    break;

  case RACE_THRIKREEN:
    if (GET_LEVEL(ch) >= 25) {
      SET_BIT(RAFF_FLAGS(ch), AFF_WATERWALK);
      SET_BIT(AFF_FLAGS(ch), AFF_WATERWALK);
    }
    if (GET_LEVEL(ch) >= 75) {
      SET_BIT(RAFF_FLAGS(ch), AFF_HASTE);
      SET_BIT(AFF_FLAGS(ch), AFF_HASTE);
    }
    break;

  case RACE_ELF:
    add_move += 3;
    if (GET_LEVEL(ch) >= 30) {
      SET_BIT(RAFF_FLAGS(ch), AFF_DETECT_INVIS);
      SET_BIT(AFF_FLAGS(ch), AFF_DETECT_INVIS);
    }
    if (GET_LEVEL(ch) >= 70) {
      SET_BIT(RAFF_FLAGS(ch), AFF_SANCTUARY);
      SET_BIT(AFF_FLAGS(ch), AFF_SANCTUARY);
    }
    break;

  case RACE_FAIRY:
    if (GET_LEVEL(ch) >= 25) {
      SET_BIT(RAFF_FLAGS(ch), AFF_DETECT_MAGIC);
      SET_BIT(AFF_FLAGS(ch), AFF_DETECT_MAGIC);
    }
    if (GET_LEVEL(ch) >= 50) {
      SET_BIT(RAFF_FLAGS(ch), AFF_BLUR);
      SET_BIT(AFF_FLAGS(ch), AFF_BLUR);
    }
    break;

  case RACE_OGRE:
    if (GET_LEVEL(ch) >= 25) {
      SET_BIT(RAFF_FLAGS(ch), AFF_NOWEB);
      SET_BIT(AFF_FLAGS(ch), AFF_NOWEB);
    }
    break;

  case RACE_GARGOYLE:
    if (GET_LEVEL(ch) >= 25) {
      SET_BIT(RAFF_FLAGS(ch), AFF_FLYING);
      SET_BIT(AFF_FLAGS(ch), AFF_FLYING);
    }
    if (GET_LEVEL(ch) >= 50) {
      SET_BIT(RAFF2_FLAGS(ch), AFF2_STONESKIN);
      SET_BIT(AFF2_FLAGS(ch), AFF2_STONESKIN);
    }
    break;

  }

  ch->points.max_hit += MAX(1, add_hp);
  ch->points.max_mana += add_mana;
  ch->points.max_move += MAX(1, add_move);

/*  ch->points.max_hit += MAX(1, add_hp);
  ch->points.max_mana += add_mana;
  ch->points.max_move += MAX(1, add_move); */

  if (GET_CLASS(ch) == CLASS_MAGIC_USER || GET_CLASS(ch) == CLASS_CLERIC  ||
      GET_CLASS(ch) == CLASS_BARD || GET_CLASS(ch) == CLASS_APALADIN ||
      GET_CLASS(ch) == CLASS_RANGER || GET_CLASS(ch) == CLASS_MONK ||
      GET_CLASS(ch) == CLASS_COLORMAGE)
    GET_PRACTICES(ch) += MAX(2, wis_app[GET_WIS(ch)].bonus);
  else
    GET_PRACTICES(ch) += MIN(2, MAX(1, wis_app[GET_WIS(ch)].bonus));
  if (GET_PRACTICES(ch) < 0) GET_PRACTICES(ch) = 100;

/*  if (GET_LEVEL(ch) >= LVL_DEMI) {
    for (i = 0; i < 3; i++)
      GET_COND(ch, i) = (char) -1;
    GET_MAX_HIT(ch) = 2000;
    GET_MAX_MANA(ch) = 2000;
    GET_MAX_MOVE(ch) = 1000;
    GET_PRACTICES(ch) = 50;
  } */

  if (GET_LEVEL(ch) >= LVL_IMMORT) {
    SET_BIT(PRF_FLAGS(ch), PRF_HOLYLIGHT);
    SET_BIT(PRF_FLAGS(ch), PRF_NOHIT);
    send_to_char("Why don't you read our immortal handbook?\r\n", ch);
  }

  save_char(ch, NOWHERE);

   sprintf(buf, "%s advanced to level %d", GET_NAME(ch), GET_LEVEL(ch));
   nmlog(buf);

  if (ch->player_specials->advance) {
    if (ch->player_specials->advance == GET_LEVEL(ch)) {
      if(PRF_FLAGGED(ch, PRF_NOGRATZ)) {
	sprintf(buf, "%s advanced to level %d (nograt)", GET_NAME(ch), 
	        GET_LEVEL(ch));
        mudlog(buf, BRF, MAX(LVL_IMMORT, GET_INVIS_LEV(ch)), FALSE);
      } else {
        sprintf(buf, "%s advanced to level %d", GET_NAME(ch), GET_LEVEL(ch));
        mudlog(buf, BRF, MAX(LVL_IMMORT, GET_INVIS_LEV(ch)), FALSE);
      }
    } /* else if(PRF_FLAGGED(ch, PRF_NOGRATZ)) {
      sprintf(buf, "%s advanced to level %d (nograt)", GET_NAME(ch), 
              GET_LEVEL(ch));
      mudlog(buf, BRF, MAX(LVL_IMMORT, GET_INVIS_LEV(ch)), FALSE);
      sprintf(buf, "%s advanced to level %d", GET_NAME(ch), GET_LEVEL(ch));
      nmlog(buf);
    } else {
      sprintf(buf, "%s advanced to level %d", GET_NAME(ch), GET_LEVEL(ch));
      mudlog(buf, BRF, MAX(LVL_IMMORT, GET_INVIS_LEV(ch)), TRUE); 
    } */
  }
}


/*
 * This simply calculates the backstab multiplier based on a character's
 * level.  This used to be an array, but was changed to be a function so
 * that it would be easier to add more levels to your MUD.  This doesn't
 * really create a big performance hit because it's not used very often.
 * Extended to 100 levels by Trebor 8/96
 */
int backstab_mult(struct char_data *ch, int level)
{
  if (level <= 0)
    return 1;	  /* level 0 */
  else if (level <= 7)
    return 2;	  /* level 1 - 7 */
  else if (level <= 13)
    return 3;	  /* level 8 - 13 */
  else if (level <= 20)
    return 4;	  /* level 14 - 20 */
  else if (level <= 28)
    return 5;	  /* level 21 - 28 */
  else if (level < LVL_IMMORT && !IS_THIEF(ch))
    return 5;     /* Non-thieves limited to max 5 multiplier-Treb */
  else if (level <= 37)
    return 6;     /* level 29 - 37 */
  else if (level <= 47)
    return 7;     /* level 36 - 47 */
  else if (level <= 58)
    return 8;     /* level 48 - 58 */
  else if (level <= 70)
    return 9;     /* level 59 - 70 */
  else if (level <= 83)
    return 10;    /* level 71 - 83 */
  else if (level < LVL_IMMORT)
    return 11;	  /* level 84-100 */
  else
    return 20;	  /* immortals */
}


/*
 * invalid_class is used by handler.c to determine if a piece of equipment is
 * usable by a particular class, based on the ITEM_ANTI_{class} bitvectors.
 */

int invalid_class(struct char_data *ch, struct obj_data *obj) {
  if (   (IS_OBJ_STAT(obj, ITEM_ANTI_MAGIC_USER) && IS_MAGIC_USER(ch))
      || (IS_OBJ_STAT(obj, ITEM_ANTI_CLERIC) && IS_CLERIC(ch))
      || (IS_OBJ_STAT(obj, ITEM_ANTI_WARRIOR) && IS_WARRIOR(ch))
      || (IS_OBJ_STAT(obj, ITEM_ANTI_THIEF) && IS_THIEF(ch))
      || (IS_OBJ_STAT(obj, ITEM_ANTI_PALADIN) && IS_PALADIN(ch))
      || (IS_OBJ_STAT(obj, ITEM_ANTI_BARD) && IS_BARD(ch))
      || (IS_OBJ_STAT(obj, ITEM_ANTI_APALADIN) && IS_APALADIN(ch))
      || (IS_OBJ_STAT(obj, ITEM_ANTI_RANGER) && IS_RANGER(ch))
      || (IS_OBJ_STAT(obj, ITEM_ANTI_MONK) && IS_MONK(ch))
      /*|| (IS_OBJ_STAT(obj, ITEM_ANTI_COLORMAGE) && IS_COLORMAGE(ch))*/
/* dwix: begin 
      || (IS_OBJ_STAT(obj, ITEM_MAGIC_USER) && !IS_MAGIC_USER(ch))
      || (IS_OBJ_STAT(obj, ITEM_CLERIC) && !IS_CLERIC(ch))
      || (IS_OBJ_STAT(obj, ITEM_WARRIOR) && !IS_WARRIOR(ch))
      || (IS_OBJ_STAT(obj, ITEM_THIEF) && !IS_THIEF(ch))
      || (IS_OBJ_STAT(obj, ITEM_PALADIN) && !IS_PALADIN(ch))
      || (IS_OBJ_STAT(obj, ITEM_BARD) && !IS_BARD(ch))
      || (IS_OBJ_STAT(obj, ITEM_APALADIN) && !IS_APALADIN(ch))
      || (IS_OBJ_STAT(obj, ITEM_RANGER) && !IS_RANGER(ch))
      || (IS_OBJ_STAT(obj, ITEM_MONK) && !IS_MONK(ch))
dwix: end */
     )
	return 1;
  else
	return 0;
}


#define CLASS 3
#define ATTRIB  2
#define RACE  1

struct remort_info {
  char *name;
  int value;
  int type;
  int remort;
};

ACMD(do_remort) /* remortmod */
{
  int stat[7], new_race, new_class, num = 0, loop, cst[4];
  int valid_race(int race, int class);
  
  struct remort_info remort[] = {
    {"human"    , RACE_HUMAN    , RACE, 0},
    {"elf"      , RACE_ELF      , RACE, 0},
    {"dwarf"    , RACE_DWARF    , RACE, 0},
    {"hobbit"   , RACE_HOBBIT   , RACE, 0},
    {"halfling" , RACE_HOBBIT   , RACE, 0},
    {"ogre"     , RACE_OGRE     , RACE, 0},
    {"vampire"  , RACE_VAMPIRE  , RACE, 0},
    {"troll"    , RACE_TROLL    , RACE, 0},
    {"fairy"    , RACE_FAIRY    , RACE, 0},
    {"dragon"   , RACE_DRAGON   , RACE, 0},
    {"tiefling" , RACE_TIEFLING , RACE, 0},
    {"thrikreen", RACE_THRIKREEN, RACE, 0},
    {"gargoyle" , RACE_GARGOYLE , RACE, 0},
    {"str"      , 0             , ATTRIB, 0},
    {"dex"      , 1             , ATTRIB, 0},
    {"con"      , 2             , ATTRIB, 0},
    {"int"      , 3             , ATTRIB, 0},
    {"wis"      , 4             , ATTRIB, 0},
    {"cha"      , 5             , ATTRIB, 0},
    {"null"     , 6             , ATTRIB, 0},
    {"mage"     , CLASS_MAGIC_USER, CLASS, 0},
    {"cleric"   , CLASS_CLERIC  , CLASS, 0},
    {"thief"    , CLASS_THIEF   , CLASS, 0},
    {"warrior"  , CLASS_WARRIOR , CLASS, 0},
    {"paladin"  , CLASS_PALADIN , CLASS, 0},
    {"bard"     , CLASS_BARD    , CLASS, 0},
    {"antipaladin",CLASS_APALADIN, CLASS, 0},
    {"ranger"   , CLASS_RANGER  , CLASS, 0},
    {"monk"     , CLASS_MONK    , CLASS, 0},
  /*{"samurai"  , CLASS_SAMURAI , CLASS, 0},
    {"colormage", CLASS_COLORMAGE, CLASS, 0},*/ /* these two aren't useable */
    {"\n", 0, 0, 0},

  };
  if (subcmd != SCMD_REMORT) {
    send_to_char("Please type the full word 'remort' to remort.\r\n", ch);
    return;
  }
  if (!*argument) {
    cst[0] = 0;
    cst[1] = 0;
    sprintf(buf, "Possible classes/races:\r\n");
    sprintf(buf1, "Possible bonuses:\r\n");
    for (loop = 0; str_cmp(remort[loop].name, "\n"); loop++) {
      if (GET_REMORT(ch) < remort[loop].remort) continue;
      if (remort[loop].type == RACE || remort[loop].type == CLASS) {
        sprintf(buf, "%s%-15s", buf, remort[loop].name);
        if (!(++cst[0] % 3)) strcat(buf, "\r\n");
      } else {
        sprintf(buf1, "%s%-15s", buf1, remort[loop].name);
        if (!(++cst[1] % 3)) strcat(buf1, "\r\n");
      }
    }
    strcat(buf, "\r\n\r\n");
    strcat(buf1, "\r\n\r\n");
    send_to_char(buf, ch);
    send_to_char(buf1, ch);
    return;
  }
  if (GET_LEVEL(ch) != LVL_DEMI) {
    send_to_char("You don't really want to remort, do you?\r\n", ch);
    return;
  }
  /* initialize variables */
  new_race = GET_RACE(ch);
  new_class = GET_CLASS(ch);
  stat[0] = ch->real_abils.str;
  stat[1] = ch->real_abils.dex;
  stat[2] = ch->real_abils.con;
  stat[3] = ch->real_abils.intel;
  stat[4] = ch->real_abils.wis;
  stat[5] = ch->real_abils.cha;
  stat[6] = ch->real_abils.str_add;

  half_chop(argument, arg, buf);
  while (*arg)
    {
      for (loop = 0; str_cmp(remort[loop].name, "\n"); loop++) {
	if (!str_cmp(remort[loop].name, arg)) {
          if (GET_REMORT(ch) < remort[loop].remort) {
            sprintf(buf, "You cannot choose '%s' yet.\r\n", remort[loop].name);
            send_to_char(buf, ch);
            return;
          }
	  switch (remort[loop].type) {
	  case RACE:
	    if (GET_CLASS(ch) != new_class) {
	      send_to_char("You cannot change both race and class.\r\n", ch);
	      return;
	    }
	    new_race = remort[loop].value;
	    if (!valid_race(new_race, new_class)) {
	      send_to_char("Your race cannot be that class.\r\n", ch);
	      return;
	    }
	    cst[0] = loop;
	    break;
	  case CLASS:
	    if (GET_RACE(ch) != new_race) {
	      send_to_char("You cannot change both race and class.\r\n", ch);
	      return;
	    }
	    new_class = remort[loop].value;
	    if (!valid_race(new_race, new_class)) {
	      send_to_char("Your race cannot be that class.\r\n", ch);
	      return;
	    }
	    cst[0] = loop;
	    break;
	  case ATTRIB:
	    if (++num > 3) {
	      send_to_char("You can only increase 3 stats\r\n", ch);
	      return;
	    }
            if (remort[loop].value == 6) { /* special case for NULL */
              cst[num] = loop;
              break;
            }
	    if (!remort[loop].value) /* special case for str! */
	      if (stat[0] == 18 && stat[6] <= 90)
		stat[6] += 10;
	      else {
		stat[0]++;
		stat[6] = 0;
	      }
	    else
	      stat[remort[loop].value]++;
	    if (stat[remort[loop].value] > 20) {
	      send_to_char("You may not increase any stat above 20.\r\n", ch);
	      return;
	    }
	    cst[num] = loop;
	    break;
	  default:
	    send_to_char("Invalid argument!\r\n", ch);
	    break;
	  }
	  break;
	}
      }
      half_chop(buf, arg, buf);
    }
  if ((GET_RACE(ch) == new_race && GET_CLASS(ch) == new_class) || num < 3) {
    send_to_char("You must have a class/race change and 3 stat increases.\r\n",
		 ch);
    return;
  }
  GET_RACE(ch) = new_race;
  GET_CLASS(ch) = new_class;
  ch->real_abils.str = stat[0];
  ch->real_abils.str_add = stat[6];
  ch->real_abils.dex = stat[1];
  ch->real_abils.con = stat[2];
  ch->real_abils.intel = stat[3];
  ch->real_abils.wis = stat[4];
  ch->real_abils.cha = stat[5];
  GET_REMORT(ch)++;
  REMOVE_BIT(PRF_FLAGS(ch), PRF_HOLYLIGHT|PRF_NOHIT);
  do_start(ch);
  if (GET_MAX_MOVE(ch) < 100) {
    sprintf(buf, "REMORT SYSERR : %s had %d movement when remorting.",
      GET_NAME(ch), GET_MAX_MOVE(ch));
    mudlog(buf, NRM, LVL_GOD, TRUE);
    GET_MAX_MOVE(ch) = 110;
    GET_MOVE(ch) = GET_MAX_MOVE(ch);
  }
  sprintf(buf, "REMORT: %s became %s (%s %s %s)", GET_NAME(ch),
	  remort[cst[0]].name, remort[cst[1]].name, remort[cst[2]].name,
	  remort[cst[3]].name);
  mudlog(buf, NRM, LVL_GOD, TRUE);
  sprintf(buf, "Okay, you are now a %s with increased %s, %s, and %s.\r\n",
	  remort[cst[0]].name, remort[cst[1]].name, remort[cst[2]].name,
	  remort[cst[3]].name);
  send_to_char(buf, ch);
}


/*
 * SPELLS AND SKILLS.  This area defines which spells are assigned to
 * which classes, and the minimum level the character must be to use
 * the spell or skill.
 */
void init_spell_levels(void)
{
  /* spell_level(spell name, class, min level, min remorts, max learn); */
  /* NEWBIE */
  spell_level(SKILL_DODGE, CLASS_NEWBIE, 20, 0, -1);

  /* MAGES */
  spell_level(SPELL_MAGIC_MISSILE, CLASS_MAGIC_USER, 1, 0, -1);
  spell_level(SPELL_INFRAVISION, CLASS_MAGIC_USER, 1, 0, -1);
  spell_level(SPELL_INVISIBLE, CLASS_MAGIC_USER, 1, 0, -1);
  spell_level(SPELL_DETECT_INVIS, CLASS_MAGIC_USER, 2, 0, -1);
  spell_level(SPELL_DETECT_MAGIC, CLASS_MAGIC_USER, 2, 0, -1);
  spell_level(SPELL_CHILL_TOUCH, CLASS_MAGIC_USER, 3, 0, -1);
  spell_level(SPELL_BURNING_HANDS, CLASS_MAGIC_USER, 4, 0, -1);
  spell_level(SPELL_ARMOR, CLASS_MAGIC_USER, 5, 0, -1);
  spell_level(SPELL_LOCATE_OBJECT, CLASS_MAGIC_USER, 6, 0, -1);
  spell_level(SPELL_STRENGTH, CLASS_MAGIC_USER, 6, 0, -1);
  spell_level(SPELL_SLEEP, CLASS_MAGIC_USER, 6, 0, -1);
  spell_level(SPELL_SHOCKING_GRASP, CLASS_MAGIC_USER, 7, 0, -1);
  spell_level(SPELL_LIGHTNING_BOLT, CLASS_MAGIC_USER, 9, 0, -1);
  spell_level(SPELL_BLINDNESS, CLASS_MAGIC_USER, 9, 0, -1);
  spell_level(SPELL_COLOR_SPRAY, CLASS_MAGIC_USER, 11, 0, -1);
  spell_level(SPELL_BLUR, CLASS_MAGIC_USER, 12, 0, -1);
  spell_level(SPELL_SCRY, CLASS_MAGIC_USER, 13, 0, -1);
  spell_level(SPELL_PARALYZE, CLASS_MAGIC_USER, 14, 0, -1);
  spell_level(SPELL_FLIGHT, CLASS_MAGIC_USER, 15, 0, -1);
  spell_level(SPELL_FIREBALL, CLASS_MAGIC_USER, 15, 0, -1);
  spell_level(SPELL_ENCHANT_WEAPON, CLASS_MAGIC_USER, 16, 0, -1);
  spell_level(SPELL_CURSE, CLASS_MAGIC_USER, 17, 0, -1);
  spell_level(SPELL_WATERWALK, CLASS_MAGIC_USER, 18, 0, -1);
  spell_level(SPELL_ENERGY_DRAIN, CLASS_MAGIC_USER, 19, 0, -1);
  spell_level(SPELL_SENSE_LIFE, CLASS_MAGIC_USER, 21, 0, -1);
  spell_level(SPELL_POISON, CLASS_MAGIC_USER, 25, 0, -1);
  spell_level(SPELL_ENERGY_BLAST, CLASS_MAGIC_USER, 25, 0, -1);
  spell_level(SPELL_SLOW, CLASS_MAGIC_USER, 28, 0, -1);
  spell_level(SPELL_ASTRAL_WALK, CLASS_MAGIC_USER, 30, 0, -1);
  spell_level(SPELL_REM_INVIS, CLASS_MAGIC_USER, 32, 0, -1);
  spell_level(SPELL_HASTE, CLASS_MAGIC_USER, 35, 0, -1);
  spell_level(SPELL_SHIELD, CLASS_MAGIC_USER, 40, 0, -1);
  spell_level(SPELL_MANA_BOLT, CLASS_MAGIC_USER, 50, 0, -1);
  spell_level(SPELL_REVERSAL, CLASS_MAGIC_USER, 58, 0, -1);
  spell_level(SPELL_STONESKIN, CLASS_MAGIC_USER, 60, 0, -1);
  spell_level(SPELL_DISPEL_MAGIC, CLASS_MAGIC_USER, 65, 0, -1);
  spell_level(SPELL_MANA_WAVE, CLASS_MAGIC_USER, 70, 0, -1);
  spell_level(SPELL_FIRE_STORM, CLASS_MAGIC_USER, 85, 0, -1);
  spell_level(SPELL_DEATH_WIND, CLASS_MAGIC_USER, 95, 0, -1);
  spell_level(SPELL_PORTAL, CLASS_MAGIC_USER, 95, 0, -1);
  spell_level(SPELL_HEROES_FEAST, CLASS_MAGIC_USER, 95, 0, -1);
  spell_level(SPELL_BLINK, CLASS_MAGIC_USER, 95, 0, -1);
  spell_level(SPELL_CREATE_FOOD, CLASS_MAGIC_USER, 95, 0, -1);
  spell_level(SPELL_CREATE_WATER, CLASS_MAGIC_USER, 95, 0, -1);

  /* CLERICS */
  spell_level(SPELL_CURE_LIGHT, CLASS_CLERIC, 1, 0, -1);
  spell_level(SPELL_ARMOR, CLASS_CLERIC, 1, 0, -1);
  spell_level(SPELL_DETECT_POISON, CLASS_CLERIC, 5, 0, -1);
  spell_level(SPELL_DETECT_ALIGN, CLASS_CLERIC, 5, 0, -1);
  spell_level(SPELL_BLESS, CLASS_CLERIC, 10, 0, -1);
  spell_level(SPELL_DETECT_INVIS, CLASS_CLERIC, 6, 0, -1);
  spell_level(SPELL_CURE_BLIND, CLASS_CLERIC, 20, 0, -1);
  spell_level(SPELL_PROT_FROM_EVIL, CLASS_CLERIC, 35, 0, -1);
  spell_level(SPELL_CURE_SERIOUS, CLASS_CLERIC, 25, 0, -1);
  spell_level(SPELL_INFRAVISION, CLASS_CLERIC, 9, 0, -1);
  spell_level(SPELL_WORD_OF_RECALL, CLASS_CLERIC, 30, 0, -1);
  spell_level(SPELL_REMOVE_POISON, CLASS_CLERIC, 15, 0, -1);
  spell_level(SPELL_CURE_CRITIC, CLASS_CLERIC, 50, 0, -1);
  spell_level(SPELL_VIGORIZE, CLASS_CLERIC, 35, 0, -1);
  spell_level(SPELL_DISPEL_EVIL, CLASS_CLERIC, 45, 0, -1);
  spell_level(SPELL_DISPEL_GOOD, CLASS_CLERIC, 45, 0, -1);
  spell_level(SPELL_HARM, CLASS_CLERIC, 35, 0, -1);
  spell_level(SPELL_HEAL, CLASS_CLERIC, 75, 0, -1);
  spell_level(SPELL_REMOVE_CURSE, CLASS_CLERIC, 35, 0, -1);
  spell_level(SPELL_GROUP_ARMOR, CLASS_CLERIC, 55, 0, -1);
  spell_level(SPELL_GROUP_HEAL, CLASS_CLERIC, 75, 0, -1);
  spell_level(SPELL_SENSE_LIFE, CLASS_CLERIC, 30, 0, -1);
  spell_level(SPELL_SUMMON, CLASS_CLERIC, 45, 0, -1);
  spell_level(SPELL_GROUP_RECALL, CLASS_CLERIC, 75, 0, -1);
  spell_level(SPELL_SANCTUARY, CLASS_CLERIC, 50, 0, -1);
  spell_level(SPELL_FEAR, CLASS_CLERIC, 45, 0, -1);
  spell_level(SPELL_PACIFY, CLASS_CLERIC, 45, 0, -1);
  spell_level(SPELL_TURN_UNDEAD, CLASS_CLERIC, 75, 0, -1);
  spell_level(SPELL_PWR_WRD_STUN, CLASS_CLERIC, 80, 0, -1);
  spell_level(SPELL_DIVINE_PROT, CLASS_CLERIC, 90, 0, -1);

  /* THIEVES */
  spell_level(SKILL_SNEAK, CLASS_THIEF, 1, 0, -1);
  spell_level(SKILL_BACKSTAB, CLASS_THIEF, 5, 0, -1);
  spell_level(SKILL_STEAL, CLASS_THIEF, 10, 0, -1);
  spell_level(SKILL_HIDE, CLASS_THIEF, 15, 0, -1);
  spell_level(SKILL_DODGE, CLASS_THIEF, 20, 0, -1);
  spell_level(SKILL_PICK_LOCK, CLASS_THIEF, 25, 0, -1);
  spell_level(SKILL_TRIP, CLASS_THIEF, 30, 0, -1);
  spell_level(SKILL_DOUBLE_BS, CLASS_THIEF, 35, 0, -1);
  spell_level(SKILL_JUMP, CLASS_THIEF, 40, 0, -1);
  spell_level(SKILL_SECOND_ATTACK, CLASS_THIEF, 50, 0, -1);
  spell_level(SKILL_PALM, CLASS_THIEF, 60, 0, -1);
  spell_level(SKILL_CIRCLE, CLASS_THIEF, 75, 0, -1);
  spell_level(SKILL_DISARM, CLASS_THIEF, 65, 0, -1);

  /* WARRIORS */
  spell_level(SKILL_KICK, CLASS_WARRIOR, 1, 0, -1);
  spell_level(SKILL_BASH, CLASS_WARRIOR, 5, 0, -1);
  spell_level(SKILL_PARRY, CLASS_WARRIOR, 10, 0, -1);
  spell_level(SKILL_RESCUE, CLASS_WARRIOR, 20, 0, -1);
  spell_level(SKILL_HEADBUTT, CLASS_WARRIOR, 25, 0, -1);
  spell_level(SKILL_SECOND_ATTACK, CLASS_WARRIOR, 35, 0, -1);
  spell_level(SKILL_CHARGE, CLASS_WARRIOR, 45, 0, -1);
  spell_level(SKILL_DISARM, CLASS_WARRIOR, 55, 0, -1);
  spell_level(SKILL_THIRD_ATTACK, CLASS_WARRIOR, 65, 0, -1);
  spell_level(SKILL_BERSERK, CLASS_WARRIOR, 80, 0, -1);

  /* PALADINS */
  spell_level(SPELL_DETECT_ALIGN, CLASS_PALADIN, 1, 0, -1);
  spell_level(SKILL_KICK, CLASS_PALADIN, 1, 0, -1);
  spell_level(SKILL_BASH, CLASS_PALADIN, 3, 0, -1);
  spell_level(SKILL_PARRY, CLASS_PALADIN, 4, 0, -1);
  spell_level(SPELL_CURE_LIGHT, CLASS_PALADIN, 5, 0, -1);
  spell_level(SKILL_RESCUE, CLASS_PALADIN, 5, 0, -1);
  spell_level(SPELL_CURE_BLIND, CLASS_PALADIN, 10, 0, -1);
  spell_level(SPELL_REMOVE_POISON, CLASS_PALADIN, 10, 0, -1);
  spell_level(SPELL_DISPEL_EVIL, CLASS_PALADIN, 15, 0, -1);
  spell_level(SKILL_SOULSTRIKE, CLASS_PALADIN, 25, 0, -1);
  spell_level(SPELL_GROUP_BLESS, CLASS_PALADIN, 25, 0, -1);
  spell_level(SKILL_SECOND_ATTACK, CLASS_PALADIN, 50, 0, -1);
  spell_level(SPELL_PACIFY, CLASS_PALADIN, 40, 0, -1);
  spell_level(SPELL_TURN_UNDEAD, CLASS_PALADIN, 40, 0, -1);
  spell_level(SPELL_CURE_SERIOUS, CLASS_PALADIN, 50, 0, -1);
  spell_level(SKILL_DISARM, CLASS_PALADIN, 50, 0, -1);
  spell_level(SPELL_BLADE_LEGEND, CLASS_PALADIN, 60, 0, -1);
  spell_level(SPELL_PALADIN_FURY, CLASS_PALADIN, 70, 0, -1);
  spell_level(SPELL_HOLY_AURA, CLASS_PALADIN, 75, 0, -1);

 /* BARDS */
  spell_level(SKILL_KICK, CLASS_BARD, 1, 0, -1);
  spell_level(SPELL_FIREBALL, CLASS_BARD, 1, 0, -1);
  spell_level(SPELL_ARMOR, CLASS_BARD, 5, 0, -1);
  spell_level(SPELL_CREATE_WATER, CLASS_BARD, 10, 0, -1);
  spell_level(SPELL_CHARM, CLASS_BARD, 15, 0, -1);
  spell_level(SPELL_CREATE_FOOD, CLASS_BARD, 15, 0, -1);
  spell_level(SPELL_INFRAVISION, CLASS_BARD, 15, 0, -1);
  spell_level(SPELL_BURNING_HANDS, CLASS_BARD, 20, 0, -1);
  spell_level(SPELL_SLEEP, CLASS_BARD, 20, 0, -1);
  spell_level(SPELL_LOCATE_OBJECT, CLASS_BARD, 25, 0, -1);
  spell_level(SPELL_MAGIC_MISSILE, CLASS_BARD, 27, 0, -1);
  spell_level(SPELL_CALL_LIGHTNING, CLASS_BARD, 30, 0, -1);
  spell_level(SKILL_DODGE, CLASS_BARD, 37, 0, -1);
  spell_level(SKILL_SECOND_ATTACK, CLASS_BARD, 50, 0, -1);
  spell_level(SPELL_ENCHANT_WEAPON, CLASS_BARD, 45, 0, -1);
  spell_level(SKILL_PICK_LOCK, CLASS_BARD, 55, 0, -1);
  spell_level(SKILL_HIDE, CLASS_BARD, 65, 0, -1);
  spell_level(SPELL_DETECT_INVIS, CLASS_BARD, 80, 0, -1);

  /* ANTI-PALADINS */
  spell_level(SPELL_DETECT_ALIGN, CLASS_APALADIN, 1, 0, -1);
  spell_level(SPELL_DISPEL_GOOD, CLASS_APALADIN, 1, 0, -1);
  spell_level(SKILL_PARRY, CLASS_APALADIN, 4, 0, -1);
  spell_level(SPELL_BLINDNESS, CLASS_APALADIN, 8, 0, -1);
  spell_level(SKILL_KICK, CLASS_APALADIN, 15, 0, -1);
  spell_level(SPELL_FLAME_BLADE, CLASS_APALADIN, 25, 0, -1);
  spell_level(SPELL_CHILL_TOUCH, CLASS_APALADIN, 30, 0, -1);
  spell_level(SPELL_CURSE, CLASS_APALADIN, 30, 0, -1);
  spell_level(SPELL_ENERGY_DRAIN, CLASS_APALADIN, 35, 0, -1);
  spell_level(SPELL_VAMPIRIC_TOUCH, CLASS_APALADIN, 35, 0, -1);
  spell_level(SKILL_SECOND_ATTACK, CLASS_APALADIN, 50, 0, -1);
  spell_level(SPELL_FEAR, CLASS_APALADIN, 45, 0, -1);
  spell_level(SPELL_POISON, CLASS_APALADIN, 50, 0, -1);
  spell_level(SKILL_DISARM, CLASS_APALADIN, 50, 0, -1);
  spell_level(SPELL_SHOCKING_GRASP, CLASS_APALADIN, 70, 0, -1);
  spell_level(SPELL_COLOR_SPRAY, CLASS_APALADIN, 70, 0, -1);

  /* RANGERS */
  spell_level(SKILL_HIDE, CLASS_RANGER, 1, 0, -1);
  spell_level(SPELL_SENSE_LIFE, CLASS_RANGER, 1, 0, -1);
  spell_level(SKILL_TRACK, CLASS_RANGER, 1, 0, -1);
  spell_level(SPELL_SLEEP, CLASS_RANGER, 7, 0, -1);
  spell_level(SKILL_RESCUE, CLASS_RANGER, 7, 0, -1);
  spell_level(SPELL_BLUR, CLASS_RANGER, 8, 0, -1);
  spell_level(SPELL_DETECT_POISON, CLASS_RANGER, 9, 0, -1);
  spell_level(SPELL_LIGHTNING_BOLT, CLASS_RANGER, 9, 0, -1);
  spell_level(SKILL_BASH, CLASS_RANGER, 10, 0, -1);
  spell_level(SPELL_WORD_OF_RECALL, CLASS_RANGER, 10, 0, -1);
  spell_level(SPELL_POISON, CLASS_RANGER, 12, 0, -1);
  spell_level(SPELL_WATERWALK, CLASS_RANGER, 17, 0, -1);
  spell_level(SPELL_DETECT_INVIS, CLASS_RANGER, 19, 0, -1);
  spell_level(SPELL_CALL_LIGHTNING, CLASS_RANGER, 30, 0, -1);
  spell_level(SPELL_EARTHQUAKE, CLASS_RANGER, 35, 0, -1);
  spell_level(SPELL_REMOVE_POISON, CLASS_RANGER, 35, 0, -1);
  spell_level(SPELL_CONTROL_WEATHER, CLASS_RANGER, 40, 0, -1);
  spell_level(SKILL_SECOND_ATTACK, CLASS_RANGER, 50, 0, -1);
  spell_level(SKILL_DISARM, CLASS_RANGER, 60, 0, -1);
  spell_level(SPELL_WEB, CLASS_RANGER, 60, 0, -1);
  spell_level(SPELL_DECOMPOSE, CLASS_RANGER, 60, 0, -1);
  spell_level(SPELL_STONESKIN, CLASS_RANGER, 60, 0, -1);
  spell_level(SPELL_SANDSTORM, CLASS_RANGER, 60, 0, -1);
  spell_level(SPELL_FAERIE_FIRE, CLASS_RANGER, 60, 0, -1);
  spell_level(SKILL_RIDING, CLASS_RANGER, 60, 0, -1);
  spell_level(SKILL_ARCHERY, CLASS_RANGER, 60, 0, -1);
  spell_level(SKILL_FIREARMS, CLASS_RANGER, 60, 0, -1);
  spell_level(SKILL_FORAGE, CLASS_RANGER, 60, 0, -1);


  /* MONKS */
  spell_level(SKILL_KICK, CLASS_MONK, 1, 0, -1);
  spell_level(SKILL_BASH, CLASS_MONK, 5, 0, -1);
  spell_level(SKILL_NSTRIKE, CLASS_MONK, 15, 0, -1);
  spell_level(SKILL_DODGE, CLASS_MONK, 20, 0, -1);
  spell_level(SKILL_SECOND_ATTACK, CLASS_MONK, 25, 0, -1);
  spell_level(SKILL_JUMP, CLASS_MONK, 35, 0, -1);
  spell_level(SKILL_THIRD_ATTACK, CLASS_MONK, 50, 0, -1);
  spell_level(SKILL_PARRY, CLASS_MONK, 55, 0, -1);
  spell_level(SKILL_TRIP, CLASS_MONK, 60, 0, -1);
  spell_level(SKILL_DISARM, CLASS_MONK, 55, 0, -1);
  spell_level(SKILL_HEADBUTT, CLASS_MONK, 60, 0, -1);
  spell_level(SKILL_FOURTH_ATTACK, CLASS_MONK, 75, 0, -1);
  spell_level(SKILL_CHARGE, CLASS_MONK, 85, 0, -1);

  /* SAMURAI */
  spell_level(SKILL_2HAND, CLASS_SAMURAI, 25, 0, -1);
}


/* Names of class/levels and exp required for each level */

const struct title_type titles[NUM_CLASSES][LVL_SIMP + 1] = {
{{"the Man", "the Woman", 0},
  {"the Newbie", "the Newbie", 1},
  {"the Newbie", "the Newbie", 1000},
  {"the Newbie", "the Newbie", 2000},
  {"the Newbie", "the Newbie", 3000},
  {"the Newbie", "the Newbie", 4000},
  {"the Newbie", "the Newbie", 5000},
  {"the Newbie", "the Newbie", 6000},
  {"the Newbie", "the Newbie", 7000},
  {"the Newbie", "the Newbie", 8000},
  {"the Newbie", "the Newbie", 9000}, /* 10 */
  {"the Newbie", "the Newbie", 10000},
  {"the Newbie", "the Newbie", 20000},
  {"the Newbie", "the Newbie", 30000},
  {"the Newbie", "the Newbie", 40000},
  {"the Newbie", "the Newbie", 50000},
  {"the Newbie", "the Newbie", 60000},
  {"the Newbie", "the Newbie", 70000},
  {"the Newbie", "the Newbie", 80000},
  {"the Newbie", "the Newbie", 90000},
  {"the Newbie", "the Newbie", 100000}, /* 20 */
  {"the Newbie", "the Newbie", 110000},
  {"the Newbie", "the Newbie", 120000},
  {"the Newbie", "the Newbie", 130000},
  {"the Newbie", "the Newbie", 140000},
  {"the Newbie", "the Newbie", 150000},
  {"the Newbie", "the Newbie", 160000},
  {"the Newbie", "the Newbie", 170000},
  {"the Newbie", "the Newbie", 180000},
  {"the Newbie", "the Newbie", 190000},
  {"the Newbie", "the Newbie", 200000}, /* 30 */
  {"the Newbie", "the Newbie", 210000},
  {"the Newbie", "the Newbie", 220000},
  {"the Newbie", "the Newbie", 230000},
  {"the Newbie", "the Newbie", 240000},
  {"the Newbie", "the Newbie", 250000},
  {"the Newbie", "the Newbie", 260000},
  {"the Newbie", "the Newbie", 270000},
  {"the Newbie", "the Newbie", 280000},
  {"the Newbie", "the Newbie", 290000},
  {"the Newbie", "the Newbie", 300000}, /* 40 */
  {"the Newbie", "the Newbie", 310000},
  {"the Newbie", "the Newbie", 320000},
  {"the Newbie", "the Newbie", 330000},
  {"the Newbie", "the Newbie", 340000},
  {"the Newbie", "the Newbie", 350000},
  {"the Newbie", "the Newbie", 360000},
  {"the Newbie", "the Newbie", 370000},
  {"the Newbie", "the Newbie", 380000},
  {"the Newbie", "the Newbie", 390000},
  {"the Newbie", "the Newbie", 400000}, /* 50 */
  {"the Newbie", "the Newbie", 410000},
  {"the Newbie", "the Newbie", 420000},
  {"the Newbie", "the Newbie", 430000},
  {"the Newbie", "the Newbie", 440000},
  {"the Newbie", "the Newbie", 450000},
  {"the Newbie", "the Newbie", 460000},
  {"the Newbie", "the Newbie", 470000},
  {"the Newbie", "the Newbie", 480000},
  {"the Newbie", "the Newbie", 490000},
  {"the Newbie", "the Newbie", 500000}, /* 60 */
  {"the Newbie", "the Newbie", 510000},
  {"the Newbie", "the Newbie", 520000},
  {"the Newbie", "the Newbie", 530000},
  {"the Newbie", "the Newbie", 540000},
  {"the Newbie", "the Newbie", 550000},
  {"the Newbie", "the Newbie", 560000},
  {"the Newbie", "the Newbie", 570000},
  {"the Newbie", "the Newbie", 580000},
  {"the Newbie", "the Newbie", 590000},
  {"the Newbie", "the Newbie", 600000}, /* 70 */
  {"the Newbie", "the Newbie", 610000},
  {"the Newbie", "the Newbie", 620000},
  {"the Newbie", "the Newbie", 630000},
  {"the Newbie", "the Newbie", 640000},
  {"the Newbie", "the Newbie", 650000},
  {"the Newbie", "the Newbie", 660000},
  {"the Newbie", "the Newbie", 670000},
  {"the Newbie", "the Newbie", 680000},
  {"the Newbie", "the Newbie", 690000},
  {"the Newbie", "the Newbie", 600000}, /* 80 */
  {"the Newbie", "the Newbie", 710000},
  {"the Newbie", "the Newbie", 720000},
  {"the Newbie", "the Newbie", 730000},
  {"the Newbie", "the Newbie", 740000},
  {"the Newbie", "the Newbie", 750000},
  {"the Newbie", "the Newbie", 760000},
  {"the Newbie", "the Newbie", 770000},
  {"the Newbie", "the Newbie", 780000},
  {"the Newbie", "the Newbie", 790000},
  {"the Newbie", "the Newbie", 700000}, /* 90 */
  {"the Newbie", "the Newbie", 810000},
  {"the Newbie", "the Newbie", 820000},
  {"the Newbie", "the Newbie", 830000},
  {"the Newbie", "the Newbie", 840000},
  {"the Newbie", "the Newbie", 850000},
  {"the Newbie", "the Newbie", 860000},
  {"the Newbie", "the Newbie", 870000},
  {"the Newbie", "the Newbie", 880000},
  {"the Newbie", "the Newbie", 890000},
  {"the Newbie", "the Newbie", 1000000000}, /* 100 */
  {"the Immortal Warlock", "the Immortal Enchantress", 110000000},
  {"the Avatar of Magic", "the Empress of Magic", 120000000},
  {"the God of Magic", "the Goddess of Magic", 125000000},
  {"the Implementor", "the Implementress", 130000000},
  {"the Immortal One", "the Immortal One", 150000000}
  },
  {{"the Man", "the Woman", 0}, /* Mage */
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 19802},
  {"the Man", "the Woman", 59406},
  {"the Man", "the Woman", 118812},
  {"the Man", "the Woman", 198020},
  {"the Man", "the Woman", 297030},
  {"the Man", "the Woman", 415842},
  {"the Man", "the Woman", 554456},
  {"the Man", "the Woman", 712872},
  {"the Man", "the Woman", 891090},
  {"the Man", "the Woman", 1089110},
  {"the Man", "the Woman", 1306932},
  {"the Man", "the Woman", 1544556},
  {"the Man", "the Woman", 1801982},
  {"the Man", "the Woman", 2079210},
  {"the Man", "the Woman", 2376240},
  {"the Man", "the Woman", 2693072},
  {"the Man", "the Woman", 3029706},
  {"the Man", "the Woman", 3386142},
  {"the Man", "the Woman", 3762380},
  {"the Man", "the Woman", 4158420},
  {"the Man", "the Woman", 4574262},
  {"the Man", "the Woman", 5009906},
  {"the Man", "the Woman", 5465352},
  {"the Man", "the Woman", 5940600},
  {"the Man", "the Woman", 6435650},
  {"the Man", "the Woman", 6950502},
  {"the Man", "the Woman", 7485156},
  {"the Man", "the Woman", 8039612},
  {"the Man", "the Woman", 8613870},
  {"the Man", "the Woman", 9207930},
  {"the Man", "the Woman", 9821792},
  {"the Man", "the Woman", 10455456},
  {"the Man", "the Woman", 11108922},
  {"the Man", "the Woman", 11782190},
  {"the Man", "the Woman", 12475260},
  {"the Man", "the Woman", 13188132},
  {"the Man", "the Woman", 13920806},
  {"the Man", "the Woman", 14673282},
  {"the Man", "the Woman", 15445560},
  {"the Man", "the Woman", 16237640},
  {"the Man", "the Woman", 17049522},
  {"the Man", "the Woman", 17881206},
  {"the Man", "the Woman", 18732692},
  {"the Man", "the Woman", 19603980},
  {"the Man", "the Woman", 20495070},
  {"the Man", "the Woman", 21405962},
  {"the Man", "the Woman", 22336656},
  {"the Man", "the Woman", 23287152},
  {"the Man", "the Woman", 24257450},
  {"the Man", "the Woman", 25247550},
  {"the Man", "the Woman", 26257452},
  {"the Man", "the Woman", 27287156},
  {"the Man", "the Woman", 28336662},
  {"the Man", "the Woman", 29405970},
  {"the Man", "the Woman", 30495080},
  {"the Man", "the Woman", 31603992},
  {"the Man", "the Woman", 32732706},
  {"the Man", "the Woman", 33881222},
  {"the Man", "the Woman", 35049540},
  {"the Man", "the Woman", 36237660},
  {"the Man", "the Woman", 37445582},
  {"the Man", "the Woman", 38673306},
  {"the Man", "the Woman", 39920832},
  {"the Man", "the Woman", 41188160},
  {"the Man", "the Woman", 42475290},
  {"the Man", "the Woman", 43782222},
  {"the Man", "the Woman", 45108956},
  {"the Man", "the Woman", 46455492},
  {"the Man", "the Woman", 47821830},
  {"the Man", "the Woman", 49207970},
  {"the Man", "the Woman", 50613912},
  {"the Man", "the Woman", 52039656},
  {"the Man", "the Woman", 53485202},
  {"the Man", "the Woman", 54950550},
  {"the Man", "the Woman", 56435700},
  {"the Man", "the Woman", 57940652},
  {"the Man", "the Woman", 59465406},
  {"the Man", "the Woman", 61009962},
  {"the Man", "the Woman", 62574320},
  {"the Man", "the Woman", 64158480},
  {"the Man", "the Woman", 65762442},
  {"the Man", "the Woman", 67386206},
  {"the Man", "the Woman", 69029772},
  {"the Man", "the Woman", 70693140},
  {"the Man", "the Woman", 72376310},
  {"the Man", "the Woman", 74079282},
  {"the Man", "the Woman", 75802056},
  {"the Man", "the Woman", 77544632},
  {"the Man", "the Woman", 79307010},
  {"the Man", "the Woman", 81089190},
  {"the Man", "the Woman", 82891172},
  {"the Man", "the Woman", 84712956},
  {"the Man", "the Woman", 86554542},
  {"the Man", "the Woman", 88415930},
  {"the Man", "the Woman", 90297120},
  {"the Man", "the Woman", 92198112},
  {"the Man", "the Woman", 94118906},
  {"the Man", "the Woman", 96059502},
  {"the Man", "the Woman", 100000000},
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
  {{"the Man", "the Woman", 0}, /* Cleric */
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 3000},
  {"the Man", "the Woman", 6500},
  {"the Man", "the Woman", 10500},
  {"the Man", "the Woman", 15000},
  {"the Man", "the Woman", 20000},
  {"the Man", "the Woman", 25500},
  {"the Man", "the Woman", 31500},
  {"the Man", "the Woman", 38000},
  {"the Man", "the Woman", 45000}, /* 10 */
  {"the Man", "the Woman", 52500},
  {"the Man", "the Woman", 60500},
  {"the Man", "the Woman", 69000},
  {"the Man", "the Woman", 78000},
  {"the Man", "the Woman", 87500},
  {"the Man", "the Woman", 97500},
  {"the Man", "the Woman", 108000},
  {"the Man", "the Woman", 119000},
  {"the Man", "the Woman", 130500},
  {"the Man", "the Woman", 142500}, /* 20 */
  {"the Man", "the Woman", 155000},
  {"the Man", "the Woman", 168000},
  {"the Man", "the Woman", 181500},
  {"the Man", "the Woman", 195500},
  {"the Man", "the Woman", 210000},
  {"the Man", "the Woman", 225000},
  {"the Man", "the Woman", 240500},
  {"the Man", "the Woman", 256500},
  {"the Man", "the Woman", 273000},
  {"the Man", "the Woman", 290000}, /* 30 */
  {"the Man", "the Woman", 307500},
  {"the Man", "the Woman", 325500},
  {"the Man", "the Woman", 344000},
  {"the Man", "the Woman", 363000},
  {"the Man", "the Woman", 382500},
  {"the Man", "the Woman", 402500},
  {"the Man", "the Woman", 423000},
  {"the Man", "the Woman", 444000},
  {"the Man", "the Woman", 465500},
  {"the Man", "the Woman", 487500}, /* 40 */
  {"the Man", "the Woman", 510000},
  {"the Man", "the Woman", 533000},
  {"the Man", "the Woman", 556500},
  {"the Man", "the Woman", 580500},
  {"the Man", "the Woman", 605000},
  {"the Man", "the Woman", 630000},
  {"the Man", "the Woman", 655500},
  {"the Man", "the Woman", 681500},
  {"the Man", "the Woman", 708000},
  {"the Man", "the Woman", 735000}, /* 50 */
  {"the Man", "the Woman", 762500},
  {"the Man", "the Woman", 790500},
  {"the Man", "the Woman", 819000},
  {"the Man", "the Woman", 848000},
  {"the Man", "the Woman", 877500},
  {"the Man", "the Woman", 907500},
  {"the Man", "the Woman", 938000},
  {"the Man", "the Woman", 969000},
  {"the Man", "the Woman", 1000500},
  {"the Man", "the Woman", 1032500}, /* 60 */
  {"the Man", "the Woman", 1065000},
  {"the Man", "the Woman", 1098000},
  {"the Man", "the Woman", 1131500},
  {"the Man", "the Woman", 1165500},
  {"the Man", "the Woman", 1200000},
  {"the Man", "the Woman", 1235000},
  {"the Man", "the Woman", 1270500},
  {"the Man", "the Woman", 1306500},
  {"the Man", "the Woman", 1343000},
  {"the Man", "the Woman", 1380000}, /* 70 */
  {"the Man", "the Woman", 1417500},
  {"the Man", "the Woman", 1455500},
  {"the Man", "the Woman", 1494000},
  {"the Man", "the Woman", 1533000},
  {"the Man", "the Woman", 1572500},
  {"the Man", "the Woman", 1612500},
  {"the Man", "the Woman", 1653000},
  {"the Man", "the Woman", 1694000},
  {"the Man", "the Woman", 1735500},
  {"the Man", "the Woman", 1777500}, /* 80 */
  {"the Man", "the Woman", 1820000},
  {"the Man", "the Woman", 1863000},
  {"the Man", "the Woman", 1906500},
  {"the Man", "the Woman", 1950500},
  {"the Man", "the Woman", 1995000},
  {"the Man", "the Woman", 2040000},
  {"the Man", "the Woman", 2085500},
  {"the Man", "the Woman", 2131500},
  {"the Man", "the Woman", 2178000},
  {"the Man", "the Woman", 2225000}, /* 90 */
  {"the Man", "the Woman", 2272500},
  {"the Man", "the Woman", 2320500},
  {"the Man", "the Woman", 2369000},
  {"the Man", "the Woman", 2418000},
  {"the Man", "the Woman", 2467500},
  {"the Man", "the Woman", 2517500},
  {"the Man", "the Woman", 2568000},
  {"the Man", "the Woman", 2619000},
  {"the Man", "the Woman", 2670500},
  {"the Man", "the Woman", 2722500}, /* 100 */
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
  {{"the Man", "the Woman", 0}, /* Thief */
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 19802},
  {"the Man", "the Woman", 59406},
  {"the Man", "the Woman", 118812},
  {"the Man", "the Woman", 198020},
  {"the Man", "the Woman", 297030},
  {"the Man", "the Woman", 415842},
  {"the Man", "the Woman", 554456},
  {"the Man", "the Woman", 712872},
  {"the Man", "the Woman", 891090},
  {"the Man", "the Woman", 1089110},
  {"the Man", "the Woman", 1306932},
  {"the Man", "the Woman", 1544556},
  {"the Man", "the Woman", 1801982},
  {"the Man", "the Woman", 2079210},
  {"the Man", "the Woman", 2376240},
  {"the Man", "the Woman", 2693072},
  {"the Man", "the Woman", 3029706},
  {"the Man", "the Woman", 3386142},
  {"the Man", "the Woman", 3762380},
  {"the Man", "the Woman", 4158420},
  {"the Man", "the Woman", 4574262},
  {"the Man", "the Woman", 5009906},
  {"the Man", "the Woman", 5465352},
  {"the Man", "the Woman", 5940600},
  {"the Man", "the Woman", 6435650},
  {"the Man", "the Woman", 6950502},
  {"the Man", "the Woman", 7485156},
  {"the Man", "the Woman", 8039612},
  {"the Man", "the Woman", 8613870},
  {"the Man", "the Woman", 9207930},
  {"the Man", "the Woman", 9821792},
  {"the Man", "the Woman", 10455456},
  {"the Man", "the Woman", 11108922},
  {"the Man", "the Woman", 11782190},
  {"the Man", "the Woman", 12475260},
  {"the Man", "the Woman", 13188132},
  {"the Man", "the Woman", 13920806},
  {"the Man", "the Woman", 14673282},
  {"the Man", "the Woman", 15445560},
  {"the Man", "the Woman", 16237640},
  {"the Man", "the Woman", 17049522},
  {"the Man", "the Woman", 17881206},
  {"the Man", "the Woman", 18732692},
  {"the Man", "the Woman", 19603980},
  {"the Man", "the Woman", 20495070},
  {"the Man", "the Woman", 21405962},
  {"the Man", "the Woman", 22336656},
  {"the Man", "the Woman", 23287152},
  {"the Man", "the Woman", 24257450},
  {"the Man", "the Woman", 25247550},
  {"the Man", "the Woman", 26257452},
  {"the Man", "the Woman", 27287156},
  {"the Man", "the Woman", 28336662},
  {"the Man", "the Woman", 29405970},
  {"the Man", "the Woman", 30495080},
  {"the Man", "the Woman", 31603992},
  {"the Man", "the Woman", 32732706},
  {"the Man", "the Woman", 33881222},
  {"the Man", "the Woman", 35049540},
  {"the Man", "the Woman", 36237660},
  {"the Man", "the Woman", 37445582},
  {"the Man", "the Woman", 38673306},
  {"the Man", "the Woman", 39920832},
  {"the Man", "the Woman", 41188160},
  {"the Man", "the Woman", 42475290},
  {"the Man", "the Woman", 43782222},
  {"the Man", "the Woman", 45108956},
  {"the Man", "the Woman", 46455492},
  {"the Man", "the Woman", 47821830},
  {"the Man", "the Woman", 49207970},
  {"the Man", "the Woman", 50613912},
  {"the Man", "the Woman", 52039656},
  {"the Man", "the Woman", 53485202},
  {"the Man", "the Woman", 54950550},
  {"the Man", "the Woman", 56435700},
  {"the Man", "the Woman", 57940652},
  {"the Man", "the Woman", 59465406},
  {"the Man", "the Woman", 61009962},
  {"the Man", "the Woman", 62574320},
  {"the Man", "the Woman", 64158480},
  {"the Man", "the Woman", 65762442},
  {"the Man", "the Woman", 67386206},
  {"the Man", "the Woman", 69029772},
  {"the Man", "the Woman", 70693140},
  {"the Man", "the Woman", 72376310},
  {"the Man", "the Woman", 74079282},
  {"the Man", "the Woman", 75802056},
  {"the Man", "the Woman", 77544632},
  {"the Man", "the Woman", 79307010},
  {"the Man", "the Woman", 81089190},
  {"the Man", "the Woman", 82891172},
  {"the Man", "the Woman", 84712956},
  {"the Man", "the Woman", 86554542},
  {"the Man", "the Woman", 88415930},
  {"the Man", "the Woman", 90297120},
  {"the Man", "the Woman", 92198112},
  {"the Man", "the Woman", 94118906},
  {"the Man", "the Woman", 96059502},
  {"the Man", "the Woman", 100000000},
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
  {{"the Man", "the Woman", 0}, /* Warrior */
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 3000},
  {"the Man", "the Woman", 6500},
  {"the Man", "the Woman", 10500},
  {"the Man", "the Woman", 15000},
  {"the Man", "the Woman", 20000},
  {"the Man", "the Woman", 25500},
  {"the Man", "the Woman", 31500},
  {"the Man", "the Woman", 38000},
  {"the Man", "the Woman", 45000}, /* 10 */
  {"the Man", "the Woman", 52500},
  {"the Man", "the Woman", 60500},
  {"the Man", "the Woman", 69000},
  {"the Man", "the Woman", 78000},
  {"the Man", "the Woman", 87500},
  {"the Man", "the Woman", 97500},
  {"the Man", "the Woman", 108000},
  {"the Man", "the Woman", 119000},
  {"the Man", "the Woman", 130500},
  {"the Man", "the Woman", 142500}, /* 20 */
  {"the Man", "the Woman", 155000},
  {"the Man", "the Woman", 168000},
  {"the Man", "the Woman", 181500},
  {"the Man", "the Woman", 195500},
  {"the Man", "the Woman", 210000},
  {"the Man", "the Woman", 225000},
  {"the Man", "the Woman", 240500},
  {"the Man", "the Woman", 256500},
  {"the Man", "the Woman", 273000},
  {"the Man", "the Woman", 290000}, /* 30 */
  {"the Man", "the Woman", 307500},
  {"the Man", "the Woman", 325500},
  {"the Man", "the Woman", 344000},
  {"the Man", "the Woman", 363000},
  {"the Man", "the Woman", 382500},
  {"the Man", "the Woman", 402500},
  {"the Man", "the Woman", 423000},
  {"the Man", "the Woman", 444000},
  {"the Man", "the Woman", 465500},
  {"the Man", "the Woman", 487500}, /* 40 */
  {"the Man", "the Woman", 510000},
  {"the Man", "the Woman", 533000},
  {"the Man", "the Woman", 556500},
  {"the Man", "the Woman", 580500},
  {"the Man", "the Woman", 605000},
  {"the Man", "the Woman", 630000},
  {"the Man", "the Woman", 655500},
  {"the Man", "the Woman", 681500},
  {"the Man", "the Woman", 708000},
  {"the Man", "the Woman", 735000}, /* 50 */
  {"the Man", "the Woman", 762500},
  {"the Man", "the Woman", 790500},
  {"the Man", "the Woman", 819000},
  {"the Man", "the Woman", 848000},
  {"the Man", "the Woman", 877500},
  {"the Man", "the Woman", 907500},
  {"the Man", "the Woman", 938000},
  {"the Man", "the Woman", 969000},
  {"the Man", "the Woman", 1000500},
  {"the Man", "the Woman", 1032500}, /* 60 */
  {"the Man", "the Woman", 1065000},
  {"the Man", "the Woman", 1098000},
  {"the Man", "the Woman", 1131500},
  {"the Man", "the Woman", 1165500},
  {"the Man", "the Woman", 1200000},
  {"the Man", "the Woman", 1235000},
  {"the Man", "the Woman", 1270500},
  {"the Man", "the Woman", 1306500},
  {"the Man", "the Woman", 1343000},
  {"the Man", "the Woman", 1380000}, /* 70 */
  {"the Man", "the Woman", 1417500},
  {"the Man", "the Woman", 1455500},
  {"the Man", "the Woman", 1494000},
  {"the Man", "the Woman", 1533000},
  {"the Man", "the Woman", 1572500},
  {"the Man", "the Woman", 1612500},
  {"the Man", "the Woman", 1653000},
  {"the Man", "the Woman", 1694000},
  {"the Man", "the Woman", 1735500},
  {"the Man", "the Woman", 1777500}, /* 80 */
  {"the Man", "the Woman", 1820000},
  {"the Man", "the Woman", 1863000},
  {"the Man", "the Woman", 1906500},
  {"the Man", "the Woman", 1950500},
  {"the Man", "the Woman", 1995000},
  {"the Man", "the Woman", 2040000},
  {"the Man", "the Woman", 2085500},
  {"the Man", "the Woman", 2131500},
  {"the Man", "the Woman", 2178000},
  {"the Man", "the Woman", 2225000}, /* 90 */
  {"the Man", "the Woman", 2272500},
  {"the Man", "the Woman", 2320500},
  {"the Man", "the Woman", 2369000},
  {"the Man", "the Woman", 2418000},
  {"the Man", "the Woman", 2467500},
  {"the Man", "the Woman", 2517500},
  {"the Man", "the Woman", 2568000},
  {"the Man", "the Woman", 2619000},
  {"the Man", "the Woman", 2670500},
  {"the Man", "the Woman", 2722500}, /* 100 */
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
  {{"the Man", "the Woman", 0},
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 19802},
  {"the Man", "the Woman", 59406},
  {"the Man", "the Woman", 118812},
  {"the Man", "the Woman", 198020},
  {"the Man", "the Woman", 297030},
  {"the Man", "the Woman", 415842},
  {"the Man", "the Woman", 554456},
  {"the Man", "the Woman", 712872},
  {"the Man", "the Woman", 891090},
  {"the Man", "the Woman", 1089110},
  {"the Man", "the Woman", 1306932},
  {"the Man", "the Woman", 1544556},
  {"the Man", "the Woman", 1801982},
  {"the Man", "the Woman", 2079210},
  {"the Man", "the Woman", 2376240},
  {"the Man", "the Woman", 2693072},
  {"the Man", "the Woman", 3029706},
  {"the Man", "the Woman", 3386142},
  {"the Man", "the Woman", 3762380},
  {"the Man", "the Woman", 4158420},
  {"the Man", "the Woman", 4574262},
  {"the Man", "the Woman", 5009906},
  {"the Man", "the Woman", 5465352},
  {"the Man", "the Woman", 5940600},
  {"the Man", "the Woman", 6435650},
  {"the Man", "the Woman", 6950502},
  {"the Man", "the Woman", 7485156},
  {"the Man", "the Woman", 8039612},
  {"the Man", "the Woman", 8613870},
  {"the Man", "the Woman", 9207930},
  {"the Man", "the Woman", 9821792},
  {"the Man", "the Woman", 10455456},
  {"the Man", "the Woman", 11108922},
  {"the Man", "the Woman", 11782190},
  {"the Man", "the Woman", 12475260},
  {"the Man", "the Woman", 13188132},
  {"the Man", "the Woman", 13920806},
  {"the Man", "the Woman", 14673282},
  {"the Man", "the Woman", 15445560},
  {"the Man", "the Woman", 16237640},
  {"the Man", "the Woman", 17049522},
  {"the Man", "the Woman", 17881206},
  {"the Man", "the Woman", 18732692},
  {"the Man", "the Woman", 19603980},
  {"the Man", "the Woman", 20495070},
  {"the Man", "the Woman", 21405962},
  {"the Man", "the Woman", 22336656},
  {"the Man", "the Woman", 23287152},
  {"the Man", "the Woman", 24257450},
  {"the Man", "the Woman", 25247550},
  {"the Man", "the Woman", 26257452},
  {"the Man", "the Woman", 27287156},
  {"the Man", "the Woman", 28336662},
  {"the Man", "the Woman", 29405970},
  {"the Man", "the Woman", 30495080},
  {"the Man", "the Woman", 31603992},
  {"the Man", "the Woman", 32732706},
  {"the Man", "the Woman", 33881222},
  {"the Man", "the Woman", 35049540},
  {"the Man", "the Woman", 36237660},
  {"the Man", "the Woman", 37445582},
  {"the Man", "the Woman", 38673306},
  {"the Man", "the Woman", 39920832},
  {"the Man", "the Woman", 41188160},
  {"the Man", "the Woman", 42475290},
  {"the Man", "the Woman", 43782222},
  {"the Man", "the Woman", 45108956},
  {"the Man", "the Woman", 46455492},
  {"the Man", "the Woman", 47821830},
  {"the Man", "the Woman", 49207970},
  {"the Man", "the Woman", 50613912},
  {"the Man", "the Woman", 52039656},
  {"the Man", "the Woman", 53485202},
  {"the Man", "the Woman", 54950550},
  {"the Man", "the Woman", 56435700},
  {"the Man", "the Woman", 57940652},
  {"the Man", "the Woman", 59465406},
  {"the Man", "the Woman", 61009962},
  {"the Man", "the Woman", 62574320},
  {"the Man", "the Woman", 64158480},
  {"the Man", "the Woman", 65762442},
  {"the Man", "the Woman", 67386206},
  {"the Man", "the Woman", 69029772},
  {"the Man", "the Woman", 70693140},
  {"the Man", "the Woman", 72376310},
  {"the Man", "the Woman", 74079282},
  {"the Man", "the Woman", 75802056},
  {"the Man", "the Woman", 77544632},
  {"the Man", "the Woman", 79307010},
  {"the Man", "the Woman", 81089190},
  {"the Man", "the Woman", 82891172},
  {"the Man", "the Woman", 84712956},
  {"the Man", "the Woman", 86554542},
  {"the Man", "the Woman", 88415930},
  {"the Man", "the Woman", 90297120},
  {"the Man", "the Woman", 92198112},
  {"the Man", "the Woman", 94118906},
  {"the Man", "the Woman", 96059502},
  {"the Man", "the Woman", 100000000},
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
  {{"the Man", "the Woman", 0},
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 19802},
  {"the Man", "the Woman", 59406},
  {"the Man", "the Woman", 118812},
  {"the Man", "the Woman", 198020},
  {"the Man", "the Woman", 297030},
  {"the Man", "the Woman", 415842},
  {"the Man", "the Woman", 554456},
  {"the Man", "the Woman", 712872},
  {"the Man", "the Woman", 891090},
  {"the Man", "the Woman", 1089110},
  {"the Man", "the Woman", 1306932},
  {"the Man", "the Woman", 1544556},
  {"the Man", "the Woman", 1801982},
  {"the Man", "the Woman", 2079210},
  {"the Man", "the Woman", 2376240},
  {"the Man", "the Woman", 2693072},
  {"the Man", "the Woman", 3029706},
  {"the Man", "the Woman", 3386142},
  {"the Man", "the Woman", 3762380},
  {"the Man", "the Woman", 4158420},
  {"the Man", "the Woman", 4574262},
  {"the Man", "the Woman", 5009906},
  {"the Man", "the Woman", 5465352},
  {"the Man", "the Woman", 5940600},
  {"the Man", "the Woman", 6435650},
  {"the Man", "the Woman", 6950502},
  {"the Man", "the Woman", 7485156},
  {"the Man", "the Woman", 8039612},
  {"the Man", "the Woman", 8613870},
  {"the Man", "the Woman", 9207930},
  {"the Man", "the Woman", 9821792},
  {"the Man", "the Woman", 10455456},
  {"the Man", "the Woman", 11108922},
  {"the Man", "the Woman", 11782190},
  {"the Man", "the Woman", 12475260},
  {"the Man", "the Woman", 13188132},
  {"the Man", "the Woman", 13920806},
  {"the Man", "the Woman", 14673282},
  {"the Man", "the Woman", 15445560},
  {"the Man", "the Woman", 16237640},
  {"the Man", "the Woman", 17049522},
  {"the Man", "the Woman", 17881206},
  {"the Man", "the Woman", 18732692},
  {"the Man", "the Woman", 19603980},
  {"the Man", "the Woman", 20495070},
  {"the Man", "the Woman", 21405962},
  {"the Man", "the Woman", 22336656},
  {"the Man", "the Woman", 23287152},
  {"the Man", "the Woman", 24257450},
  {"the Man", "the Woman", 25247550},
  {"the Man", "the Woman", 26257452},
  {"the Man", "the Woman", 27287156},
  {"the Man", "the Woman", 28336662},
  {"the Man", "the Woman", 29405970},
  {"the Man", "the Woman", 30495080},
  {"the Man", "the Woman", 31603992},
  {"the Man", "the Woman", 32732706},
  {"the Man", "the Woman", 33881222},
  {"the Man", "the Woman", 35049540},
  {"the Man", "the Woman", 36237660},
  {"the Man", "the Woman", 37445582},
  {"the Man", "the Woman", 38673306},
  {"the Man", "the Woman", 39920832},
  {"the Man", "the Woman", 41188160},
  {"the Man", "the Woman", 42475290},
  {"the Man", "the Woman", 43782222},
  {"the Man", "the Woman", 45108956},
  {"the Man", "the Woman", 46455492},
  {"the Man", "the Woman", 47821830},
  {"the Man", "the Woman", 49207970},
  {"the Man", "the Woman", 50613912},
  {"the Man", "the Woman", 52039656},
  {"the Man", "the Woman", 53485202},
  {"the Man", "the Woman", 54950550},
  {"the Man", "the Woman", 56435700},
  {"the Man", "the Woman", 57940652},
  {"the Man", "the Woman", 59465406},
  {"the Man", "the Woman", 61009962},
  {"the Man", "the Woman", 62574320},
  {"the Man", "the Woman", 64158480},
  {"the Man", "the Woman", 65762442},
  {"the Man", "the Woman", 67386206},
  {"the Man", "the Woman", 69029772},
  {"the Man", "the Woman", 70693140},
  {"the Man", "the Woman", 72376310},
  {"the Man", "the Woman", 74079282},
  {"the Man", "the Woman", 75802056},
  {"the Man", "the Woman", 77544632},
  {"the Man", "the Woman", 79307010},
  {"the Man", "the Woman", 81089190},
  {"the Man", "the Woman", 82891172},
  {"the Man", "the Woman", 84712956},
  {"the Man", "the Woman", 86554542},
  {"the Man", "the Woman", 88415930},
  {"the Man", "the Woman", 90297120},
  {"the Man", "the Woman", 92198112},
  {"the Man", "the Woman", 94118906},
  {"the Man", "the Woman", 96059502},
  {"the Man", "the Woman", 100000000},
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
  {{"the Man", "the Woman", 0},
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 19802},
  {"the Man", "the Woman", 59406},
  {"the Man", "the Woman", 118812},
  {"the Man", "the Woman", 198020},
  {"the Man", "the Woman", 297030},
  {"the Man", "the Woman", 415842},
  {"the Man", "the Woman", 554456},
  {"the Man", "the Woman", 712872},
  {"the Man", "the Woman", 891090},
  {"the Man", "the Woman", 1089110},
  {"the Man", "the Woman", 1306932},
  {"the Man", "the Woman", 1544556},
  {"the Man", "the Woman", 1801982},
  {"the Man", "the Woman", 2079210},
  {"the Man", "the Woman", 2376240},
  {"the Man", "the Woman", 2693072},
  {"the Man", "the Woman", 3029706},
  {"the Man", "the Woman", 3386142},
  {"the Man", "the Woman", 3762380},
  {"the Man", "the Woman", 4158420},
  {"the Man", "the Woman", 4574262},
  {"the Man", "the Woman", 5009906},
  {"the Man", "the Woman", 5465352},
  {"the Man", "the Woman", 5940600},
  {"the Man", "the Woman", 6435650},
  {"the Man", "the Woman", 6950502},
  {"the Man", "the Woman", 7485156},
  {"the Man", "the Woman", 8039612},
  {"the Man", "the Woman", 8613870},
  {"the Man", "the Woman", 9207930},
  {"the Man", "the Woman", 9821792},
  {"the Man", "the Woman", 10455456},
  {"the Man", "the Woman", 11108922},
  {"the Man", "the Woman", 11782190},
  {"the Man", "the Woman", 12475260},
  {"the Man", "the Woman", 13188132},
  {"the Man", "the Woman", 13920806},
  {"the Man", "the Woman", 14673282},
  {"the Man", "the Woman", 15445560},
  {"the Man", "the Woman", 16237640},
  {"the Man", "the Woman", 17049522},
  {"the Man", "the Woman", 17881206},
  {"the Man", "the Woman", 18732692},
  {"the Man", "the Woman", 19603980},
  {"the Man", "the Woman", 20495070},
  {"the Man", "the Woman", 21405962},
  {"the Man", "the Woman", 22336656},
  {"the Man", "the Woman", 23287152},
  {"the Man", "the Woman", 24257450},
  {"the Man", "the Woman", 25247550},
  {"the Man", "the Woman", 26257452},
  {"the Man", "the Woman", 27287156},
  {"the Man", "the Woman", 28336662},
  {"the Man", "the Woman", 29405970},
  {"the Man", "the Woman", 30495080},
  {"the Man", "the Woman", 31603992},
  {"the Man", "the Woman", 32732706},
  {"the Man", "the Woman", 33881222},
  {"the Man", "the Woman", 35049540},
  {"the Man", "the Woman", 36237660},
  {"the Man", "the Woman", 37445582},
  {"the Man", "the Woman", 38673306},
  {"the Man", "the Woman", 39920832},
  {"the Man", "the Woman", 41188160},
  {"the Man", "the Woman", 42475290},
  {"the Man", "the Woman", 43782222},
  {"the Man", "the Woman", 45108956},
  {"the Man", "the Woman", 46455492},
  {"the Man", "the Woman", 47821830},
  {"the Man", "the Woman", 49207970},
  {"the Man", "the Woman", 50613912},
  {"the Man", "the Woman", 52039656},
  {"the Man", "the Woman", 53485202},
  {"the Man", "the Woman", 54950550},
  {"the Man", "the Woman", 56435700},
  {"the Man", "the Woman", 57940652},
  {"the Man", "the Woman", 59465406},
  {"the Man", "the Woman", 61009962},
  {"the Man", "the Woman", 62574320},
  {"the Man", "the Woman", 64158480},
  {"the Man", "the Woman", 65762442},
  {"the Man", "the Woman", 67386206},
  {"the Man", "the Woman", 69029772},
  {"the Man", "the Woman", 70693140},
  {"the Man", "the Woman", 72376310},
  {"the Man", "the Woman", 74079282},
  {"the Man", "the Woman", 75802056},
  {"the Man", "the Woman", 77544632},
  {"the Man", "the Woman", 79307010},
  {"the Man", "the Woman", 81089190},
  {"the Man", "the Woman", 82891172},
  {"the Man", "the Woman", 84712956},
  {"the Man", "the Woman", 86554542},
  {"the Man", "the Woman", 88415930},
  {"the Man", "the Woman", 90297120},
  {"the Man", "the Woman", 92198112},
  {"the Man", "the Woman", 94118906},
  {"the Man", "the Woman", 96059502},
  {"the Man", "the Woman", 100000000},
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
  {{"the Man", "the Woman", 0},
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 19802},
  {"the Man", "the Woman", 59406},
  {"the Man", "the Woman", 118812},
  {"the Man", "the Woman", 198020},
  {"the Man", "the Woman", 297030},
  {"the Man", "the Woman", 415842},
  {"the Man", "the Woman", 554456},
  {"the Man", "the Woman", 712872},
  {"the Man", "the Woman", 891090},
  {"the Man", "the Woman", 1089110},
  {"the Man", "the Woman", 1306932},
  {"the Man", "the Woman", 1544556},
  {"the Man", "the Woman", 1801982},
  {"the Man", "the Woman", 2079210},
  {"the Man", "the Woman", 2376240},
  {"the Man", "the Woman", 2693072},
  {"the Man", "the Woman", 3029706},
  {"the Man", "the Woman", 3386142},
  {"the Man", "the Woman", 3762380},
  {"the Man", "the Woman", 4158420},
  {"the Man", "the Woman", 4574262},
  {"the Man", "the Woman", 5009906},
  {"the Man", "the Woman", 5465352},
  {"the Man", "the Woman", 5940600},
  {"the Man", "the Woman", 6435650},
  {"the Man", "the Woman", 6950502},
  {"the Man", "the Woman", 7485156},
  {"the Man", "the Woman", 8039612},
  {"the Man", "the Woman", 8613870},
  {"the Man", "the Woman", 9207930},
  {"the Man", "the Woman", 9821792},
  {"the Man", "the Woman", 10455456},
  {"the Man", "the Woman", 11108922},
  {"the Man", "the Woman", 11782190},
  {"the Man", "the Woman", 12475260},
  {"the Man", "the Woman", 13188132},
  {"the Man", "the Woman", 13920806},
  {"the Man", "the Woman", 14673282},
  {"the Man", "the Woman", 15445560},
  {"the Man", "the Woman", 16237640},
  {"the Man", "the Woman", 17049522},
  {"the Man", "the Woman", 17881206},
  {"the Man", "the Woman", 18732692},
  {"the Man", "the Woman", 19603980},
  {"the Man", "the Woman", 20495070},
  {"the Man", "the Woman", 21405962},
  {"the Man", "the Woman", 22336656},
  {"the Man", "the Woman", 23287152},
  {"the Man", "the Woman", 24257450},
  {"the Man", "the Woman", 25247550},
  {"the Man", "the Woman", 26257452},
  {"the Man", "the Woman", 27287156},
  {"the Man", "the Woman", 28336662},
  {"the Man", "the Woman", 29405970},
  {"the Man", "the Woman", 30495080},
  {"the Man", "the Woman", 31603992},
  {"the Man", "the Woman", 32732706},
  {"the Man", "the Woman", 33881222},
  {"the Man", "the Woman", 35049540},
  {"the Man", "the Woman", 36237660},
  {"the Man", "the Woman", 37445582},
  {"the Man", "the Woman", 38673306},
  {"the Man", "the Woman", 39920832},
  {"the Man", "the Woman", 41188160},
  {"the Man", "the Woman", 42475290},
  {"the Man", "the Woman", 43782222},
  {"the Man", "the Woman", 45108956},
  {"the Man", "the Woman", 46455492},
  {"the Man", "the Woman", 47821830},
  {"the Man", "the Woman", 49207970},
  {"the Man", "the Woman", 50613912},
  {"the Man", "the Woman", 52039656},
  {"the Man", "the Woman", 53485202},
  {"the Man", "the Woman", 54950550},
  {"the Man", "the Woman", 56435700},
  {"the Man", "the Woman", 57940652},
  {"the Man", "the Woman", 59465406},
  {"the Man", "the Woman", 61009962},
  {"the Man", "the Woman", 62574320},
  {"the Man", "the Woman", 64158480},
  {"the Man", "the Woman", 65762442},
  {"the Man", "the Woman", 67386206},
  {"the Man", "the Woman", 69029772},
  {"the Man", "the Woman", 70693140},
  {"the Man", "the Woman", 72376310},
  {"the Man", "the Woman", 74079282},
  {"the Man", "the Woman", 75802056},
  {"the Man", "the Woman", 77544632},
  {"the Man", "the Woman", 79307010},
  {"the Man", "the Woman", 81089190},
  {"the Man", "the Woman", 82891172},
  {"the Man", "the Woman", 84712956},
  {"the Man", "the Woman", 86554542},
  {"the Man", "the Woman", 88415930},
  {"the Man", "the Woman", 90297120},
  {"the Man", "the Woman", 92198112},
  {"the Man", "the Woman", 94118906},
  {"the Man", "the Woman", 96059502},
  {"the Man", "the Woman", 100000000},
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
  {{"the Man", "the Woman", 0},
  {"the Observer", "Observer", 1},
  {"the Wanderer", "the Wanderer", 4000},
  {"the Explorer", "the Explorer", 8000},
  {"the Curious", "the Curious", 16000},
  {"the Pursuer of Nature", "the Pursuer of Nature", 32000},
  {"the Dog", "the Dog", 64000},
  {"the Wolf", "the Wolf", 125000},
  {"the Coyote", "the Coyote", 250000},
  {"the Snake", "the Snake", 500000},
  {"the Feline", "the Feline", 750000},
  {"the Lynx", "the Lynx", 1000000},
  {"the Cheetah", "the Cheetah", 1250000},
  {"the Lion", "the Lion", 1500000},
  {"the Tiger", "the Tiger", 1850000},
  {"the Puma", "the Puma", 2200000},
  {"the Panther", "the Panther", 2550000},
  {"the Sparrow", "the Sparrow", 2900000},
  {"the Crow", "the Crow", 3250000},
  {"the Falcon", "the Falcon", 3600000},
  {"the Eagle", "the Eagle", 3900000}, /* level 20 */
  {"the Shrike", "the Shrike",  4300000},
  {"the Grass", "the Grass",  4700000},
  {"the Sunflower", "the Sunflower",  5100000},
  {"the Rose", "the Rose",  5500000},
  {"the Palm", "the Palm",  6000000},
  {"the Willow", "the Willow",  7000000},
  {"the Oak", "the Oak",  8000000},
  {"the Redwood", "the Redwood",  9000000},
  {"the Pursuer of Self", "the Pursuer of Self", 10000000},
  {"the Foot", "the Foot", 11000000},
  {"the Finger", "the Finger",  12000000},
  {"the Hand", "the Hand", 13000000},
  {"the Leg", "the Leg", 14000000},
  {"the Arm", "the Arm", 15000000},
  {"the Chest", "the Chest", 16000000},
  {"the Abdomen", "the Abdomen", 17000000},
  {"the Body", "the Body", 18000000},
  {"the Persuer of Mind", "the Persuer of Mind", 19000000},
  {"the Essence of Calm", "the Essence of Clam", 20000000},
  {"the Essence of Creativity", "the Essence of Creativity", 21000000},
  {"the Essence of Imagination", "the Essence of Imagination", 22000000},
  {"the Essence of Reason", "the Essence of Reason", 23000000},
  {"the Essence of Login", "the Essence of Login", 24000000},
  {"the Essence of Understanding", "the Essence of Understanding", 25000000},
  {"the Essence of Hope", "the Essence of Hope", 26000000},
  {"the Essence of Caring", "the Essence of Caring", 27000000},
  {"the Steady of Mind", "the Steady of Mind", 28000000},
  {"the Unwavering", "the Unwavering", 29000000},
  {"the Persuer of the Soul", "the Persuer of the Soul", 30000000},
  {"the Pure", "the Pure", 32000000}, /* level 50 */
  {"the Chaste", "the Chaste", 34000000},
  {"the Clean", "the Clean", 36000000},
  {"the One", "the One", 38000000},
  {"the Essence of Nature", "the Essence of Nature", 40000000},
  {"the SouthEast Wind", "the SouthEast Wind", 42000000},
  {"the SouthWest Wind", "the SouthWest Wind", 44000000},
  {"the NorthEast Wind", "the NorthEast Wind", 46000000},
  {"the NorthWest Wind", "the NorthWest Wind", 48000000},
  {"the East Wind", "the East Wind", 50000000},
  {"the West Wind", "the West Wind", 52000000},
  {"the South Wind", "the South Wind", 54000000},
  {"the North Wind", "the North Wind", 56000000},
  {"the Wind", "the Wind", 58000000},
  {"the Rain", "the Rain", 60000000},
  {"the Snow", "the Snow", 62000000},
  {"the Tornado", "the Tornado", 64000000},
  {"the Hurricane", "the Hurricane", 66000000},
  {"the Learned", "the Learned", 68000000},
  {"the Studious", "the Studious", 70000000},
   {"the Scribe", "the Scribe", 74000000}, /* level 70 */
  {"the Master", "the Mistress", 78000000},
  {"the Master of the SouthEast Wind", "the Mistress of the SouthEast Wind", 82000000},
  {"the Master of the SouthWest Wind", "the Mistress of the SouthWest Wind", 86000000},
  {"the Master of the NorthEast Wind", "the Mistress of the NorthEasy Wind", 90000000},
  {"the Master of the NorthWest Wind", "the Mistress of the NorthWest Wind", 94000000},
  {"the Master of the East Wind", "the Mistress of the East Wind", 98000000},
  {"the Master of the West Wind", "the Mistress of the West Wind", 102000000},
  {"the Master of the South Wind", "the Mistress of the South Wind", 106000000},
  {"the Master of the North Wind", "the Mistress of the North Wind", 110000000},
  {"the Master of the Winds", "the Mistress of the Winds", 114000000},
  {"the Master of the Rains", "the Mistress of the Rains", 118000000},
  {"the Master of the Snows", "the Mistress of the Snows", 122000000},
  {"the Master of the Tornados", "the Mistress of the Tornados", 126000000},
  {"the Master of the Hurricanes", "the Mistress of the Hurricanes", 130000000},
  {"the Master Scribe", "the Mistress Scribe", 134000000},
  {"the Master of Nature", "the Mistress of Nature", 138000000},
  {"the Master of Flowers", "the Mistress of Flowers", 142000000},
  {"the Grand Master of Winds", "the Grand Mistress of Winds", 146000000},
  {"the Grand Master of Rains", "the Grand Mistress of Rains", 150000000},
  {"the Grand Master of Storms", "the Grand Mistress of Storms", 154000000},
  {"the Grand Master of Hurricanes", "the Grand Mistress of Hurricanes", 158000000},
  {"the Grand Master of Scribes", "the Grand Mistress of Scribes", 162000000},
  {"the Grand Master of Nature", "the Grand Mistress of Nature", 166000000},
  {"the Grand Master of Flowers", "the Grand Mistress of Flowers", 170000000},
  {"the Master of Body", "the Grand Mistress of Body", 174000000},
  {"the Master of Mind", "the Grand Mistress of Mind", 178000000},
  {"the Master of Soul", "the Grand Mistress of Soul", 182000000},
  {"the Grand Master of Body", "the Grand Mistress of Body", 186000000},
  {"the Grand Master of Mind", "the Grand Mistress of Mind", 190000000},
  {"the Grand Master of Soul", "the Grand Mistress of Soul", 250000000},
  {"the Immortal Lord of Justice", "the Immortal Lady of Justice", 260000000},
  {"the Dispenser of Good", "the Dispenser of Good", 270000000},
  {"the Bringer of Joy", "the Bringer of Joy", 280000000},
  {"the Immortal Lord of Good", "the Immortal Lady of Good", 290000000},
  {"the Immortal King of Joy", "the Immortal Queen of Joy", 300000000}
  },
  {{"the Man", "the Woman", 0},
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 19802},
  {"the Man", "the Woman", 59406},
  {"the Man", "the Woman", 118812},
  {"the Man", "the Woman", 198020},
  {"the Man", "the Woman", 297030},
  {"the Man", "the Woman", 415842},
  {"the Man", "the Woman", 554456},
  {"the Man", "the Woman", 712872},
  {"the Man", "the Woman", 891090},
  {"the Man", "the Woman", 1089110},
  {"the Man", "the Woman", 1306932},
  {"the Man", "the Woman", 1544556},
  {"the Man", "the Woman", 1801982},
  {"the Man", "the Woman", 2079210},
  {"the Man", "the Woman", 2376240},
  {"the Man", "the Woman", 2693072},
  {"the Man", "the Woman", 3029706},
  {"the Man", "the Woman", 3386142},
  {"the Man", "the Woman", 3762380},
  {"the Man", "the Woman", 4158420},
  {"the Man", "the Woman", 4574262},
  {"the Man", "the Woman", 5009906},
  {"the Man", "the Woman", 5465352},
  {"the Man", "the Woman", 5940600},
  {"the Man", "the Woman", 6435650},
  {"the Man", "the Woman", 6950502},
  {"the Man", "the Woman", 7485156},
  {"the Man", "the Woman", 8039612},
  {"the Man", "the Woman", 8613870},
  {"the Man", "the Woman", 9207930},
  {"the Man", "the Woman", 9821792},
  {"the Man", "the Woman", 10455456},
  {"the Man", "the Woman", 11108922},
  {"the Man", "the Woman", 11782190},
  {"the Man", "the Woman", 12475260},
  {"the Man", "the Woman", 13188132},
  {"the Man", "the Woman", 13920806},
  {"the Man", "the Woman", 14673282},
  {"the Man", "the Woman", 15445560},
  {"the Man", "the Woman", 16237640},
  {"the Man", "the Woman", 17049522},
  {"the Man", "the Woman", 17881206},
  {"the Man", "the Woman", 18732692},
  {"the Man", "the Woman", 19603980},
  {"the Man", "the Woman", 20495070},
  {"the Man", "the Woman", 21405962},
  {"the Man", "the Woman", 22336656},
  {"the Man", "the Woman", 23287152},
  {"the Man", "the Woman", 24257450},
  {"the Man", "the Woman", 25247550},
  {"the Man", "the Woman", 26257452},
  {"the Man", "the Woman", 27287156},
  {"the Man", "the Woman", 28336662},
  {"the Man", "the Woman", 29405970},
  {"the Man", "the Woman", 30495080},
  {"the Man", "the Woman", 31603992},
  {"the Man", "the Woman", 32732706},
  {"the Man", "the Woman", 33881222},
  {"the Man", "the Woman", 35049540},
  {"the Man", "the Woman", 36237660},
  {"the Man", "the Woman", 37445582},
  {"the Man", "the Woman", 38673306},
  {"the Man", "the Woman", 39920832},
  {"the Man", "the Woman", 41188160},
  {"the Man", "the Woman", 42475290},
  {"the Man", "the Woman", 43782222},
  {"the Man", "the Woman", 45108956},
  {"the Man", "the Woman", 46455492},
  {"the Man", "the Woman", 47821830},
  {"the Man", "the Woman", 49207970},
  {"the Man", "the Woman", 50613912},
  {"the Man", "the Woman", 52039656},
  {"the Man", "the Woman", 53485202},
  {"the Man", "the Woman", 54950550},
  {"the Man", "the Woman", 56435700},
  {"the Man", "the Woman", 57940652},
  {"the Man", "the Woman", 59465406},
  {"the Man", "the Woman", 61009962},
  {"the Man", "the Woman", 62574320},
  {"the Man", "the Woman", 64158480},
  {"the Man", "the Woman", 65762442},
  {"the Man", "the Woman", 67386206},
  {"the Man", "the Woman", 69029772},
  {"the Man", "the Woman", 70693140},
  {"the Man", "the Woman", 72376310},
  {"the Man", "the Woman", 74079282},
  {"the Man", "the Woman", 75802056},
  {"the Man", "the Woman", 77544632},
  {"the Man", "the Woman", 79307010},
  {"the Man", "the Woman", 81089190},
  {"the Man", "the Woman", 82891172},
  {"the Man", "the Woman", 84712956},
  {"the Man", "the Woman", 86554542},
  {"the Man", "the Woman", 88415930},
  {"the Man", "the Woman", 90297120},
  {"the Man", "the Woman", 92198112},
  {"the Man", "the Woman", 94118906},
  {"the Man", "the Woman", 96059502},
  {"the Man", "the Woman", 100000000},
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
  {{"the Man", "the Woman", 0},
  {"the Man", "the Woman", 1},
  {"the Man", "the Woman", 19802},
  {"the Man", "the Woman", 59406},
  {"the Man", "the Woman", 118812},
  {"the Man", "the Woman", 198020},
  {"the Man", "the Woman", 297030},
  {"the Man", "the Woman", 415842},
  {"the Man", "the Woman", 554456},
  {"the Man", "the Woman", 712872},
  {"the Man", "the Woman", 891090},
  {"the Man", "the Woman", 1089110},
  {"the Man", "the Woman", 1306932},
  {"the Man", "the Woman", 1544556},
  {"the Man", "the Woman", 1801982},
  {"the Man", "the Woman", 2079210},
  {"the Man", "the Woman", 2376240},
  {"the Man", "the Woman", 2693072},
  {"the Man", "the Woman", 3029706},
  {"the Man", "the Woman", 3386142},
  {"the Man", "the Woman", 3762380},
  {"the Man", "the Woman", 4158420},
  {"the Man", "the Woman", 4574262},
  {"the Man", "the Woman", 5009906},
  {"the Man", "the Woman", 5465352},
  {"the Man", "the Woman", 5940600},
  {"the Man", "the Woman", 6435650},
  {"the Man", "the Woman", 6950502},
  {"the Man", "the Woman", 7485156},
  {"the Man", "the Woman", 8039612},
  {"the Man", "the Woman", 8613870},
  {"the Man", "the Woman", 9207930},
  {"the Man", "the Woman", 9821792},
  {"the Man", "the Woman", 10455456},
  {"the Man", "the Woman", 11108922},
  {"the Man", "the Woman", 11782190},
  {"the Man", "the Woman", 12475260},
  {"the Man", "the Woman", 13188132},
  {"the Man", "the Woman", 13920806},
  {"the Man", "the Woman", 14673282},
  {"the Man", "the Woman", 15445560},
  {"the Man", "the Woman", 16237640},
  {"the Man", "the Woman", 17049522},
  {"the Man", "the Woman", 17881206},
  {"the Man", "the Woman", 18732692},
  {"the Man", "the Woman", 19603980},
  {"the Man", "the Woman", 20495070},
  {"the Man", "the Woman", 21405962},
  {"the Man", "the Woman", 22336656},
  {"the Man", "the Woman", 23287152},
  {"the Man", "the Woman", 24257450},
  {"the Man", "the Woman", 25247550},
  {"the Man", "the Woman", 26257452},
  {"the Man", "the Woman", 27287156},
  {"the Man", "the Woman", 28336662},
  {"the Man", "the Woman", 29405970},
  {"the Man", "the Woman", 30495080},
  {"the Man", "the Woman", 31603992},
  {"the Man", "the Woman", 32732706},
  {"the Man", "the Woman", 33881222},
  {"the Man", "the Woman", 35049540},
  {"the Man", "the Woman", 36237660},
  {"the Man", "the Woman", 37445582},
  {"the Man", "the Woman", 38673306},
  {"the Man", "the Woman", 39920832},
  {"the Man", "the Woman", 41188160},
  {"the Man", "the Woman", 42475290},
  {"the Man", "the Woman", 43782222},
  {"the Man", "the Woman", 45108956},
  {"the Man", "the Woman", 46455492},
  {"the Man", "the Woman", 47821830},
  {"the Man", "the Woman", 49207970},
  {"the Man", "the Woman", 50613912},
  {"the Man", "the Woman", 52039656},
  {"the Man", "the Woman", 53485202},
  {"the Man", "the Woman", 54950550},
  {"the Man", "the Woman", 56435700},
  {"the Man", "the Woman", 57940652},
  {"the Man", "the Woman", 59465406},
  {"the Man", "the Woman", 61009962},
  {"the Man", "the Woman", 62574320},
  {"the Man", "the Woman", 64158480},
  {"the Man", "the Woman", 65762442},
  {"the Man", "the Woman", 67386206},
  {"the Man", "the Woman", 69029772},
  {"the Man", "the Woman", 70693140},
  {"the Man", "the Woman", 72376310},
  {"the Man", "the Woman", 74079282},
  {"the Man", "the Woman", 75802056},
  {"the Man", "the Woman", 77544632},
  {"the Man", "the Woman", 79307010},
  {"the Man", "the Woman", 81089190},
  {"the Man", "the Woman", 82891172},
  {"the Man", "the Woman", 84712956},
  {"the Man", "the Woman", 86554542},
  {"the Man", "the Woman", 88415930},
  {"the Man", "the Woman", 90297120},
  {"the Man", "the Woman", 92198112},
  {"the Man", "the Woman", 94118906},
  {"the Man", "the Woman", 96059502},
  {"the Man", "the Woman", 100000000},
  {"the Immortal", "the Immortal", 555555555},
  {"the Immortal", "the Immortal", 666666666},
  {"the Immortal", "the Immortal", 777777777},
  {"the Immortal", "the Immortal", 888888888},
  {"the Immortal", "the Immortal", 999999999}
  },
};
