/* ************************************************************************
*   File: db.c                                          Part of CircleMUD *
*  Usage: Loading/saving chars, booting/resetting world, internal funcs   *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#define __DB_C__

#include "conf.h"
#include "sysdep.h"


#include "structs.h"
#include "utils.h"
#include "db.h"
#include "comm.h"
#include "handler.h"
#include "spells.h"
#include "mail.h"
#include "interpreter.h"
#include "house.h"
#include "saves.h"

char err_buf[MAX_STRING_LENGTH];
#define MAXENTRIES 10000

/**************************************************************************
*  declarations of most of the 'global' variables                         *
************************************************************************ */

struct room_data *world = NULL;	/* array of rooms		 */
int top_of_world = 0;		/* ref to top element of world	 */
struct room_data *room_combat_damage = NULL; /* linked list of damage rooms */
struct room_data *room_tick_damage = NULL;

struct char_data *character_list = NULL;	/* global linked list of
						 * chars	 */
struct index_data *mob_index;	/* index table for mobile file	 */
struct char_data *mob_proto;	/* prototypes for mobs		 */
int top_of_mobt = 0;		/* top of mobile index table	 */

struct obj_data *object_list = NULL;	/* global linked list of objs	 */
struct index_data *obj_index;	/* index table for object file	 */
struct obj_data *obj_proto;	/* prototypes for objs		 */
int top_of_objt = 0;		/* top of object index table	 */
extern int max_new_objects;     /* oeditmod - see config.c       */
extern int max_new_rooms;       /* reditmod - see config.c       */
extern int max_new_mobs;        /* meditmod - see config.c       */
extern int new_objects;         /* .06                           */
extern int new_rooms;           /* .06                           */
extern int new_mobs;            /* .06                           */

/* search buffering */
sh_int mob_sbuf = 0;
sh_int obj_sbuf = 0;
sh_int room_sbuf = 0;

struct zone_data *zone_table;	/* zone table			 */
int top_of_zone_table = 0;	/* top element of zone tab	 */
struct message_list fight_messages[MAX_MESSAGES];	/* fighting messages	 */

struct player_index_element *player_table = NULL;	/* index to plr file	 */
FILE *player_fl = NULL;		/* file desc of player file	 */
int top_of_p_table = 0;		/* ref to top of table		 */
int top_of_p_file = 0;		/* ref of size of p file	 */
long top_idnum = 0;		/* highest idnum in use		 */

int mini_mud = 0;		/* mini-mud mode?		 */
int no_rent_check = 0;		/* skip rent check on boot?	 */
int test_mode = 0;              /* testing BETA code?            */
time_t boot_time = 0;		/* time of mud boot		 */
int mudrestrict = 0;		/* level of game restriction	 */
sh_int r_newbie_start_room;     /* rnum of newbie start room     */
sh_int r_mortal_start_room;	/* rnum of mortal start room	 */
sh_int r_immort_start_room;	/* rnum of immort start room	 */
sh_int r_frozen_start_room;	/* rnum of frozen start room	 */
struct room_s_index_e *room_s_table = NULL;/* index to roomsave files */

char *credits = NULL;		/* game credits			 */
char *claneq = NULL;		/* mud claneq rules		 */
char *motd = NULL;		/* message of the day - mortals */
char *imotd = NULL;		/* message of the day - immorts */
char *help = NULL;		/* help screen			 */
char *info = NULL;		/* info page			 */
char *wizlist = NULL;		/* list of higher gods		 */
char *immlist = NULL;		/* list of peon gods		 */
char *background = NULL;	/* background story		 */
char *handbook = NULL;		/* handbook for new immortals	 */
char *policies = NULL;		/* policies page		 */
char *circlemud_license = NULL; /* the Diku/Circle license       */

struct help_index_element *help_table = 0;	/* the help table	 */
int top_of_helpt = 0;		/* top of help index table	 */

int clan_stats[40] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
   0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
struct dns_entry *dns_cache[DNS_HASH_NUM]; /* for dns caching * dnsmod */
int dns_size = 0;

struct time_info_data time_info;/* the infomation about the time    */
struct weather_data weather_info;	/* the infomation about the weather */
struct player_special_data dummy_mob;	/* dummy spec area for mobs	 */
struct reset_q_type reset_q;	/* queue of zones to be reset	 */
struct wholist_entry *wholist = NULL;
int wholist_total = 0;

/* local functions */
void setup_dir(FILE * fl, int room, int dir);
void index_boot(int mode);
void discrete_load(FILE * fl, int mode);
void parse_room(FILE * fl, int virtual_nr);
void parse_mobile(FILE * mob_f, int nr);
char *parse_object(FILE * obj_f, int nr);
void load_zones(FILE * fl, char *zonename);
void load_help(FILE *fl);
void assign_mobiles(void);
void assign_objects(void);
void assign_rooms(void);
void assign_the_shopkeepers(void);
void build_player_index(void);
void build_room_s_index(void);
void boot_saved_rooms(void);
void char_to_store(struct char_data * ch, struct char_file_u * st);
void store_to_char(struct char_file_u * st, struct char_data * ch);
int is_empty(int zone_nr);
void reset_zone(int zone);
int file_to_string(char *name, char *buf);
int file_to_string_alloc(char *name, char **buf);
void check_start_rooms(void);
void renum_world(void);
void renum_zone_table(void);
void log_zone_error(int zone, int cmd_no, char *message);
void reset_time(void);
void clear_char(struct char_data * ch);
void tag_argument(char *argument, char *tag);
void boot_dns(void); /* dnsmod */
void save_dns_cache(void); /* dnsmod */
int get_host_from_cache(struct dns_entry *dnsd); /* dnsmod */
void add_dns_host(struct dns_entry *dnsd, char *hostname); /* dnsmod */
void remove_player(long id, byte stomp);
void clean_pfiles(void);

/* external functions */
extern struct descriptor_data *descriptor_list;
void load_messages(void);
void weather_and_time(int mode);
void mag_assign_spells(void);
void boot_social_messages(void);
/*void update_obj_file(void);*/	/* In objsave.c */
void sort_commands(void);
void sort_spells(void);
void load_banned(void);
void Read_Invalid_List(void);
void boot_the_shops(FILE * shop_f, char *filename, int rec_count);
int hsort(const void *a, const void *b);
void sprintbits(long vektor,char *outstring);
/*void build_mail_index(void);*/
void dummy(void);
void init_boards(void);
void strip_ems(char *str);
void kill_ems(char *str);

/* external vars */
extern int no_specials;
/* Kako's variables */
int group_limit = 6;
int newbie_zone;
void check_autowiz(struct char_data * ch);

#define READ_SIZE 256

/*************************************************************************
*  routines for booting the system                                       *
*********************************************************************** */

/* this is necessary for the autowiz system */
void reboot_wizlists(void)
{
  file_to_string_alloc(WIZLIST_FILE, &wizlist);
  file_to_string_alloc(IMMLIST_FILE, &immlist);
}


ACMD(do_reboot)
{
  int i;

  one_argument(argument, arg);

  if (!str_cmp(arg, "all") || *arg == '*') {
    file_to_string_alloc(WIZLIST_FILE, &wizlist);
    file_to_string_alloc(IMMLIST_FILE, &immlist);
    file_to_string_alloc(CLANEQ_FILE, &claneq);
    file_to_string_alloc(CREDITS_FILE, &credits);
    file_to_string_alloc(MOTD_FILE, &motd);
    file_to_string_alloc(IMOTD_FILE, &imotd);
    file_to_string_alloc(HELP_PAGE_FILE, &help);
    file_to_string_alloc(INFO_FILE, &info);
    file_to_string_alloc(POLICIES_FILE, &policies);
    file_to_string_alloc(HANDBOOK_FILE, &handbook);
    file_to_string_alloc(BACKGROUND_FILE, &background);
    file_to_string_alloc(LICENSE_FILE, &circlemud_license);
  } else if (!str_cmp(arg, "wizlist"))
    file_to_string_alloc(WIZLIST_FILE, &wizlist);
  else if (!str_cmp(arg, "immlist"))
    file_to_string_alloc(IMMLIST_FILE, &immlist);
  else if (!str_cmp(arg, "claneq"))
    file_to_string_alloc(CLANEQ_FILE, &claneq);
  else if (!str_cmp(arg, "credits"))
    file_to_string_alloc(CREDITS_FILE, &credits);
  else if (!str_cmp(arg, "motd"))
    file_to_string_alloc(MOTD_FILE, &motd);
  else if (!str_cmp(arg, "imotd"))
    file_to_string_alloc(IMOTD_FILE, &imotd);
  else if (!str_cmp(arg, "help"))
    file_to_string_alloc(HELP_PAGE_FILE, &help);
  else if (!str_cmp(arg, "info"))
    file_to_string_alloc(INFO_FILE, &info);
  else if (!str_cmp(arg, "policy"))
    file_to_string_alloc(POLICIES_FILE, &policies);
  else if (!str_cmp(arg, "handbook"))
    file_to_string_alloc(HANDBOOK_FILE, &handbook);
  else if (!str_cmp(arg, "background"))
    file_to_string_alloc(BACKGROUND_FILE, &background);
  else if (!str_cmp(arg, "autowiz"))
    check_autowiz(ch);
  else if (!str_cmp(arg, "xhelp")) {
    if (help_table) {
      for (i = 0; i < top_of_helpt; i++) {
        if (help_table[i].keyword)
	  free(help_table[i].keyword);
        if (help_table[i].entry && !help_table[i].duplicate)
	  free(help_table[i].entry);
      }
      free(help_table);
    }
    top_of_helpt = 0;
    index_boot(DB_BOOT_HLP);
  } else if (!str_cmp(arg, "license"))  
    file_to_string_alloc(LICENSE_FILE, &circlemud_license);
  else {
    send_to_char("Unknown reload option.\r\n", ch);
    return;
  }

  send_to_char(OK, ch);
}


void boot_world(void)
{
  nmlog("Loading zone table.");
  index_boot(DB_BOOT_ZON);

  nmlog("Loading rooms.");
  index_boot(DB_BOOT_WLD);

  nmlog("Renumbering rooms.");
  renum_world();

  nmlog("Checking start rooms.");
  check_start_rooms();

  nmlog("Loading mobs and generating index.");
  index_boot(DB_BOOT_MOB);

  nmlog("Loading objs and generating index.");
  index_boot(DB_BOOT_OBJ);

  nmlog("Renumbering zone table.");
  renum_zone_table();

  if (!no_specials) {
    nmlog("Loading shops.");
    index_boot(DB_BOOT_SHP);
  }
}

  

/* body of the booting system */
void boot_db(void)
{
  int i;

  nmlog("Boot db -- BEGIN.");

  nmlog("Resetting the game time:");
  reset_time();

  nmlog("Reading claneq, credits, help, bground, info & motds.");
  file_to_string_alloc(CLANEQ_FILE, &claneq);
  file_to_string_alloc(CREDITS_FILE, &credits);
  file_to_string_alloc(MOTD_FILE, &motd);
  file_to_string_alloc(IMOTD_FILE, &imotd);
  file_to_string_alloc(HELP_PAGE_FILE, &help);
  file_to_string_alloc(INFO_FILE, &info);
  file_to_string_alloc(WIZLIST_FILE, &wizlist);
  file_to_string_alloc(IMMLIST_FILE, &immlist);
  file_to_string_alloc(POLICIES_FILE, &policies);
  file_to_string_alloc(HANDBOOK_FILE, &handbook);
  file_to_string_alloc(BACKGROUND_FILE, &background);


  boot_world();

  nmlog("Loading help entries.");
  index_boot(DB_BOOT_HLP);

  nmlog("Generating player index.");
  build_player_index();

  nmlog("Cleaning Pfiles. -- Skipped");
  /*clean_pfiles();*/

/*  nmlog("Generating mail index.");  
  build_mail_index();
*/

  nmlog("Generating roomsave index.");  
  build_room_s_index();
  
  nmlog("Initializing boards.");
  init_boards();

  if(room_s_table) {
    nmlog("Reloading saved rooms.");
    boot_saved_rooms();
  }

  nmlog("Booting dns cache."); /* dnsmod */
  boot_dns();

  nmlog("Loading fight messages.");
  load_messages();

  nmlog("Loading social messages.");
  boot_social_messages();

  nmlog("Assigning function pointers:");

  if (!no_specials) {
    nmlog("   Mobiles.");
    assign_mobiles();
    nmlog("   Shopkeepers.");
    assign_the_shopkeepers();
    nmlog("   Objects.");
    assign_objects();
    nmlog("   Rooms.");
    assign_rooms();
  }
  nmlog("   Spells.");
  mag_assign_spells();

  nmlog("Assigning spell and skill levels.");
  init_spell_levels();

  nmlog("Sorting command list and spells.");
  sort_commands();
  sort_spells();

  nmlog("Reading banned site and invalid-name list.");
  load_banned();
  Read_Invalid_List();

  for (i = 0; i <= top_of_zone_table; i++) {
/*    if(zone_table[i].number == 7 || zone_table[i].number == 58)
      continue; */
    sprintf(buf2, "Resetting %s (rooms %d-%d).",
	    zone_table[i].name, (i ? (zone_table[i - 1].top + 1) : 0),
	    zone_table[i].top);
    nmlog(buf2);
      if (zone_table[i].number == 44)
      newbie_zone = i;
  reset_zone(i);
  }

  reset_q.head = reset_q.tail = NULL;

  if (!mini_mud) {
    nmlog("Booting houses.");
    House_boot();
  }
  boot_time = time(0);

  nmlog("Boot db -- DONE.");
}


/* reset the time in the game from file */
void reset_time(void)
{
  long beginning_of_time = 650336715;
  struct time_info_data mud_time_passed(time_t t2, time_t t1);

  time_info = mud_time_passed(time(0), beginning_of_time);

  if (time_info.hours <= 4)
    weather_info.sunlight = SUN_DARK;
  else if (time_info.hours == 5)
    weather_info.sunlight = SUN_RISE;
  else if (time_info.hours <= 20)
    weather_info.sunlight = SUN_LIGHT;
  else if (time_info.hours == 21)
    weather_info.sunlight = SUN_SET;
  else
    weather_info.sunlight = SUN_DARK;

  sprintf(buf, "   Current Gametime: %dH %dD %dM %dY.", time_info.hours,
	  time_info.day, time_info.month, time_info.year);
  nmlog(buf);

  weather_info.humidity = 50; 
  weather_info.change = 0; 

  if ((time_info.month >= 1) && (time_info.month <= 5)) {
    if (weather_info.humidity <= 10)
      weather_info.sky = SKY_SUNNY; 
    else if (weather_info.humidity <= 30)
      weather_info.sky = SKY_CLEAR; 
    else if (weather_info.humidity <= 40)
      weather_info.sky = SKY_PARTCLOUD; 
    else if (weather_info.humidity <= 50)
      weather_info.sky = SKY_CLOUDY; 
    else if (weather_info.humidity <= 75)
      weather_info.sky = SKY_SNOWING; 
    else if (weather_info.humidity <= 90)
      weather_info.sky = SKY_SNOWSTORM; 
    else
      weather_info.sky = SKY_BLIZZARD; 
  } else if ((time_info.month >= 6) && (time_info.month <= 9)) {
    if (weather_info.humidity <= 10)
      weather_info.sky = SKY_SIZZLING; 
    else if (weather_info.humidity <= 30)
      weather_info.sky = SKY_SUNNY; 
    else if (weather_info.humidity <= 40)
      weather_info.sky = SKY_CLEAR; 
    else if (weather_info.humidity <= 50)
      weather_info.sky = SKY_PARTCLOUD; 
    else if (weather_info.humidity <= 55)
      weather_info.sky = SKY_CLOUDY; 
    else if (weather_info.humidity <= 75)
      weather_info.sky = SKY_RAINING; 
    else if (weather_info.humidity <= 90)
      weather_info.sky = SKY_POURING; 
    else
      weather_info.sky = SKY_THUNDERSTORM; 
  } else if ((time_info.month >= 10) && (time_info.month <= 13)) {
    if (weather_info.humidity <= 25)
      weather_info.sky = SKY_SIZZLING; 
    else if (weather_info.humidity <= 60)
      weather_info.sky = SKY_SUNNY; 
    else if (weather_info.humidity <= 70)
      weather_info.sky = SKY_CLEAR; 
    else if (weather_info.humidity <= 75)
      weather_info.sky = SKY_PARTCLOUD; 
    else if (weather_info.humidity <= 85)
      weather_info.sky = SKY_CLOUDY; 
    else if (weather_info.humidity <= 90)
      weather_info.sky = SKY_RAINING; 
    else if (weather_info.humidity <= 95)
      weather_info.sky = SKY_POURING; 
    else
      weather_info.sky = SKY_THUNDERSTORM; 
  } else {
    if (weather_info.humidity <= 15)
      weather_info.sky = SKY_SUNNY; 
    else if (weather_info.humidity <= 40)
      weather_info.sky = SKY_CLEAR; 
    else if (weather_info.humidity <= 55)
      weather_info.sky = SKY_PARTCLOUD; 
    else if (weather_info.humidity <= 75)
      weather_info.sky = SKY_CLOUDY; 
    else if (weather_info.humidity <= 90)
      weather_info.sky = SKY_SNOWING; 
    else if (weather_info.humidity <= 95)
      weather_info.sky = SKY_SNOWSTORM; 
    else
      weather_info.sky = SKY_BLIZZARD; 
    }
  }


void build_player_index(void)
{
  int i, xxx;
  FILE *plr_index;
  char index_name[40], line[256];
  char arg1[64];
  int ia[6];

  sprintf(index_name, "%s", PLR_INDEX_FILE);
  if(!(plr_index = fopen(index_name, "r"))) {
    perror("Error opening player index file pfiles/plr_index");
    system("touch ../.killscript");
    exit(1);
  }

  /* count the number of players in the index */
  while(get_line(plr_index, line))
    top_idnum++;
  rewind(plr_index);

  CREATE(player_table, struct player_index_element, MAXENTRIES);
  xxx = 8000;
  
  for(;!feof(plr_index);) {
    get_line(plr_index, line);
    sscanf(line, "%s %d %d %d %d %d %d", arg1, ia, ia+1, ia+2, ia+3,
      ia+4, ia+5);
    if (ia[0] >= 0 && ia[0] < MAXENTRIES && player_table[ia[0]].id == 0)
      i = ia[0];
    else {
      i = ++xxx;
    }
    CREATE(player_table[i].name, char, strlen(arg1) + 1);
    strcpy(player_table[i].name, arg1);
    player_table[i].id = (long) i;
    player_table[i].level = ia[1];
    player_table[i].race = ia[2];
    player_table[i].class = ia[3];
    player_table[i].clan = ia[4];
    player_table[i].last = ia[5];
    top_of_p_table= MAX(top_of_p_table, player_table[i].id);
    clan_stats[ia[4]]++;
  }
  top_of_p_file = top_of_p_table;
}


void build_room_s_index(void)
{
  int n1, n2;
  FILE *room_s_index;
  struct room_s_index_e *temp_e, *last_e = NULL;
  char index_name[40], line[256];

  sprintf(index_name, "%s", ROOM_S_INDEX_FILE);
  if(!(room_s_index = fopen(index_name, "r"))) {
    nmlog("No rooms to recover");
    room_s_table = NULL;
    return;
  }

  do {
    get_line(room_s_index, line);
    if(*line == '~') {
      fclose(room_s_index);
      return;
    }
    CREATE(temp_e, struct room_s_index_e, 1);
    sscanf(line, "%d %d", &n1, &n2);
    temp_e->vnum = n1;
    temp_e->max_save = n2;
    temp_e->next = NULL;
    if(!last_e) {
      room_s_table = temp_e;
      last_e = room_s_table;
    } else
      last_e->next = temp_e;
  } while(!feof(room_s_index));
}


/* boot the saved room files */
void boot_saved_rooms(void)
{
  FILE *fl;
  char fname[128];
  struct room_s_index_e *room_e = room_s_table;

  while(room_e) {
    sprintf(fname, "%s/%d", ROOM_S_PREFIX, room_e->vnum);
    if(!(fl = fopen(fname, "r"))) {
      sprintf(fname, "SYSERR: can't open %s/%d.", ROOM_S_PREFIX, room_e->vnum);
      nmlog(fname);
      continue;
    }
    world[real_room(room_e->vnum)].contents = Crash_load(NULL, fl);
    fclose(fl);
    room_e = room_e->next;
  }
}


/* function to count how many hash-mark delimited records exist in a file */
int count_hash_records(FILE * fl)
{
  char buf[128];
  int count = 0;

  while (fgets(buf, 128, fl))
    if (*buf == '#')
      count++;

  return count;
}



void index_boot(int mode)
{
  char *index_filename, *prefix;
  FILE *index, *db_file;
  int rec_count = 0;

  switch (mode) {
  case DB_BOOT_WLD:
    prefix = WLD_PREFIX;
    break;
  case DB_BOOT_MOB:
    prefix = MOB_PREFIX;
    break;
  case DB_BOOT_OBJ:
    prefix = OBJ_PREFIX;
    break;
  case DB_BOOT_ZON:
    prefix = ZON_PREFIX;
    break;
  case DB_BOOT_SHP:
    prefix = SHP_PREFIX;
    break;
  case DB_BOOT_HLP:
    prefix = HLP_PREFIX;
    break;
  default:
    nmlog("SYSERR: Unknown subcommand to index_boot!");
         system("touch ../.killscript");
    exit(1);
    break;
  }

  if (mini_mud)
    index_filename = MINDEX_FILE;
  else
    index_filename = INDEX_FILE;

  sprintf(buf2, "%s/%s", prefix, index_filename);

  if (!(index = fopen(buf2, "r"))) {
    sprintf(buf1, "Error opening index file '%s'", buf2);
    perror(buf1);
             system("touch ../.killscript");
    exit(1);
  }

  /* first, count the number of records in the file so we can malloc */
  fscanf(index, "%s\n", buf1);
  while (*buf1 != '$') {
    sprintf(buf2, "%s/%s", prefix, buf1);
    if (!(db_file = fopen(buf2, "r"))) {
      /*perror(buf2);
      nmlog("file listed in index not found");
       system("touch ../.killscript");
	exit(1); sam 9/9/96 if the file is missing log it but let it boot */
    } else {
      if (mode == DB_BOOT_ZON)
	rec_count++;
      else
	rec_count += count_hash_records(db_file);
    }

    fclose(db_file);
    fscanf(index, "%s\n", buf1);
  }

  /* Exit if 0 records, unless this is shops */
  if (!rec_count) {
    if (mode == DB_BOOT_SHP)
      return;
    nmlog("SYSERR: boot error - 0 records counted");
       system("touch ../.killscript");
    exit(1);
  }

  rec_count++;

  switch (mode) {
  case DB_BOOT_WLD:
    CREATE(world, struct room_data, rec_count + max_new_rooms);
                                                /*reditmod ^ */
    break;
  case DB_BOOT_MOB:
    CREATE(mob_proto, struct char_data, rec_count + max_new_mobs);
                                                /*meditmod ^ */
    CREATE(mob_index, struct index_data, rec_count + max_new_mobs);
                                                /*meditmod ^ */
    break;
  case DB_BOOT_OBJ:
    CREATE(obj_proto, struct obj_data, rec_count + max_new_objects);
                                                   /*oeditmod ^ */
    CREATE(obj_index, struct index_data, rec_count + max_new_objects);
                                                   /*oeditmod ^ */
    break;
  case DB_BOOT_ZON:
    CREATE(zone_table, struct zone_data, rec_count);
    break;
  case DB_BOOT_HLP:
    CREATE(help_table, struct help_index_element, rec_count * 2);
    break;
  }

  rewind(index);
  fscanf(index, "%s\n", buf1);
  while (*buf1 != '$') {
    sprintf(buf2, "%s/%s", prefix, buf1);
    if (!(db_file = fopen(buf2, "r"))) {
      perror(buf2);
         system("touch ../.killscript");
    exit(1);
    }
    switch (mode) {
    case DB_BOOT_WLD:
    case DB_BOOT_OBJ:
    case DB_BOOT_MOB:
      discrete_load(db_file, mode);
      break;
    case DB_BOOT_ZON:
      load_zones(db_file, buf2);
      break;
    case DB_BOOT_HLP:
      load_help(db_file);
      break;
    case DB_BOOT_SHP:
      boot_the_shops(db_file, buf2, rec_count);
      break;
    }

    fclose(db_file);
    fscanf(index, "%s\n", buf1);
  }

  /* sort the help index */
  if (mode == DB_BOOT_HLP)
    qsort(help_table, top_of_helpt, sizeof(struct help_index_element), hsort);

}


void discrete_load(FILE * fl, int mode)
{
  int nr = -1, last = 0;
  char line[256];

  char *modes[] = {"world", "mob", "obj"};

  for (;;) {
    /*
     * we have to do special processing with the obj files because they have
     * no end-of-record marker :(
     */
    if (mode != DB_BOOT_OBJ || nr < 0)
      if (!get_line(fl, line)) {
	fprintf(stderr, "Format error after %s #%d\n", modes[mode], nr);
         system("touch ../.killscript");
	exit(1);
      }
/*
    if (*line == '\0') {
      fprintf(stderr, "Format error after %s #%d\n", modes[mode], last);
      */
    if (*line == '$')
      return;

    if (*line == '#') {
      last = nr;
      if (sscanf(line, "#%d", &nr) != 1) {
	fprintf(stderr, "Format error after %s #%d\n", modes[mode], last);
         system("touch ../.killscript");
	exit(1);
      }
      if (nr >= 99999)
	return;
      else
	switch (mode) {
	case DB_BOOT_WLD:
	  parse_room(fl, nr);
	  break;
	case DB_BOOT_MOB:
	  parse_mobile(fl, nr);
	  break;
	case DB_BOOT_OBJ:
	  strcpy(line, parse_object(fl, nr));
	  break;
	}
    } else {
      fprintf(stderr, "Format error in %s file near %s #%d\n",
	      modes[mode], modes[mode], nr);
      fprintf(stderr, "Offending line: '%s'\n", line);
          system("touch ../.killscript");
     exit(1);
    }
  }
}


long asciiflag_conv(char *flag)
{
  long flags = 0;
  int is_number = 1;
  register char *p;

  for (p = flag; *p; p++) {
    if (islower(*p))
      flags |= 1 << (*p - 'a');
    else if (isupper(*p))
      flags |= 1 << (26 + (*p - 'A'));

    if (!isdigit(*p))
      is_number = 0;
  }

  if (is_number)
    flags = atol(flag);

  return flags;
}

char fread_letter(FILE *fp)
{
  char c;
  do {
   c = fgetc(fp);
  } while (isspace(c));
  return c;
}

/* load the rooms */
void parse_room(FILE * fl, int virtual_nr)
{
  static int room_nr = 0, zone = 0;
  int t[10], i;
  char line[256], flags[128], fl2[128];
  struct extra_descr_data *new_descr;
  struct prog_data *new_prog;
  struct prog_cmd_data *new_cmd, *last_cmd = NULL;

  sprintf(buf2, "room #%d", virtual_nr);

  if (virtual_nr <= (zone ? zone_table[zone - 1].top : -1)) {
    fprintf(stderr, "Room #%d is below zone %d.\n", virtual_nr, zone);
          system("touch ../.killscript");
    exit(1);
  }
  while (virtual_nr > zone_table[zone].top)
    if (++zone > top_of_zone_table) {
      fprintf(stderr, "Room %d is outside of any zone.\n", virtual_nr);
            system("touch ../.killscript");
    exit(1);
    }
  world[room_nr].zone = zone;
  world[room_nr].number = virtual_nr;
  world[room_nr].name = fread_string(fl, buf2);
  world[room_nr].description = fread_string(fl, buf2);

  if (!get_line(fl, line) || sscanf(line, " %d %s %d ", t, flags, t + 2) != 3) {
    fprintf(stderr, "Format error in room #%d\n", virtual_nr);
            system("touch ../.killscript");
    exit(1);
  }
  /* t[0] is the zone number; ignored with the zone-file system */
  world[room_nr].room_flags = asciiflag_conv(flags);
  world[room_nr].sector_type = t[2];
 
  world[room_nr].func = NULL;
  world[room_nr].contents = NULL;
  world[room_nr].people = NULL;
  world[room_nr].light = 0;	/* Zero light sources */

  for (i = 0; i < NUM_OF_DIRS; i++)
    world[room_nr].dir_option[i] = NULL;

  world[room_nr].ex_description = NULL;

  sprintf(buf, "Format error in room #%d (expecting D/E/S)", virtual_nr);

  for (;;) {
    if (!get_line(fl, line)) {
      fprintf(stderr, "%s\n", buf);
             system("touch ../.killscript");
     exit(1);
    }
    switch (*line) {
    case 'D':
      setup_dir(fl, room_nr, atoi(line + 1));
      break;
    case 'E':
      CREATE(new_descr, struct extra_descr_data, 1);
      new_descr->keyword = fread_string(fl, buf2);
      new_descr->description = fread_string(fl, buf2);
      new_descr->next = world[room_nr].ex_description;
      world[room_nr].ex_description = new_descr;
      break;
    case 'P':
      CREATE(new_prog, struct prog_data, 1);
      new_prog->keyword = fread_string(fl, buf2);
      do {
        CREATE(new_cmd, struct prog_cmd_data, 1);
        if (!get_line(fl, line)) {
          fprintf(stderr, "%s\n", buf);
          system("touch ../.killscript");
          exit(1);
        }
        if (*line == '~') 
          break;
        new_cmd->command = str_dup(line);
        if (new_prog->cmd)  {
          last_cmd->next = new_cmd;
          last_cmd = new_cmd;
        } else {
          new_prog->cmd = new_cmd;
          last_cmd = new_cmd;
        }
      } while (1); 
        
      new_prog->next = world[room_nr].prog;
      world[room_nr].prog = new_prog;
      break;
    case 'R':
      if (!get_line(fl, line) || sscanf(line, " %d %s %s %d ", t, flags, 
          fl2, t + 1) != 4) {
        fprintf(stderr, "Damage room error in room #%d\n", virtual_nr);
        system("touch ../.killscript");
        exit(1);
      }
      world[room_nr].damage.dam = t[0];
      world[room_nr].damage.neutralizer = asciiflag_conv(flags);
      world[room_nr].damage.neutralizer2 = asciiflag_conv(fl2);
      world[room_nr].damage.interval = t[1];
      world[room_nr].damage.mesg = fread_string(fl, buf2);
      world[room_nr].next_death = NULL;
      if (world[room_nr].damage.interval == 1) {  /* combat round */
        if (room_combat_damage) {
          world[room_nr].next_death = room_combat_damage;
          room_combat_damage = &world[room_nr];
        } else
          room_combat_damage = &world[room_nr];
      } else {
        if (room_tick_damage) {
          world[room_nr].next_death = room_tick_damage;
          room_tick_damage = &world[room_nr];
        } else
          room_tick_damage = &world[room_nr];
      }
      break;
    case 'S':			/* end of room */
      top_of_world = room_nr++;
      
      return;
      break;
    default:
      fprintf(stderr, "%s\n", buf);
             system("touch ../.killscript");
      exit(1);
      break;
    }
  }
}



/* read direction data */
void setup_dir(FILE * fl, int room, int dir)
{
  int t[5];
  char line[256];

  sprintf(buf2, "room #%d, direction D%d", world[room].number, dir);

  CREATE(world[room].dir_option[dir], struct room_direction_data, 1);
  world[room].dir_option[dir]->general_description = fread_string(fl, buf2);
  world[room].dir_option[dir]->keyword = fread_string(fl, buf2);

  if (!get_line(fl, line)) {
    fprintf(stderr, "Format error, %s\n", buf2);
             system("touch ../.killscript");
    exit(1);
  }
  if (sscanf(line, " %d %d %d ", t, t + 1, t + 2) != 3) {
    fprintf(stderr, "Format error, %s\n", buf2);
             system("touch ../.killscript");
    exit(1);
  }
  if (t[0] == 1)
    world[room].dir_option[dir]->exit_info = EX_ISDOOR;
  else if (t[0] == 2)
    world[room].dir_option[dir]->exit_info = EX_ISDOOR | EX_PICKPROOF;
  else if (t[0] == 3)
    world[room].dir_option[dir]->exit_info = EX_ISDOOR | EX_SECRET;
  else if (t[0] == 4)
    world[room].dir_option[dir]->exit_info = EX_ISDOOR | EX_PICKPROOF | EX_SECRET;
  else
    world[room].dir_option[dir]->exit_info = 0;

  world[room].dir_option[dir]->key = t[1];
  world[room].dir_option[dir]->to_room = t[2];
}


/* make sure the start rooms exist & resolve their vnums to rnums */
void check_start_rooms(void)
{
  extern sh_int newbie_start_room;
  extern sh_int mortal_start_room;
  extern sh_int immort_start_room;
  extern sh_int frozen_start_room;

  if ((r_mortal_start_room = real_room(mortal_start_room)) < 0) {
    nmlog("SYSERR:  Mortal start room does not exist.  Change in config.c.");
             system("touch ../.killscript");
    exit(1);
  }
  if ((r_newbie_start_room = real_room(newbie_start_room)) < 0) {
    if (!mini_mud)
      nmlog("SYSERR:  Warning: Newbie start room does not exist.  Change in config.c.");
    r_newbie_start_room = r_mortal_start_room;
  }
  if ((r_immort_start_room = real_room(immort_start_room)) < 0) {
    if (!mini_mud)
      nmlog("SYSERR:  Warning: Immort start room does not exist.  Change in config.c.");
    r_immort_start_room = r_mortal_start_room;
  }
  if ((r_frozen_start_room = real_room(frozen_start_room)) < 0) {
    if (!mini_mud)
      nmlog("SYSERR:  Warning: Frozen start room does not exist.  Change in config.c.");
    r_frozen_start_room = r_mortal_start_room;
  }
}


/* resolve all vnums into rnums in the world */
void renum_world(void)
{
  register int room, door;

  for (room = 0; room <= top_of_world; room++)
    for (door = 0; door < NUM_OF_DIRS; door++)
      if (world[room].dir_option[door])
	if (world[room].dir_option[door]->to_room != NOWHERE)
	  world[room].dir_option[door]->to_room =
	    real_room(world[room].dir_option[door]->to_room);
}


#define ZCMD zone_table[zone].cmd[cmd_no]

/* resulve vnums into rnums in the zone reset tables */
void renum_zone_table(void)
{
  int zone, cmd_no, a, b;

  for (zone = 0; zone <= top_of_zone_table; zone++)
    for (cmd_no = 0; ZCMD.command != 'S'; cmd_no++) {
      a = b = 0;
      switch (ZCMD.command) {
      case 'M':
	a = ZCMD.arg1 = real_mobile(ZCMD.arg1);
	b = ZCMD.arg3 = real_room(ZCMD.arg3);
	break;
      case 'O':
	a = ZCMD.arg1 = real_object(ZCMD.arg1);
	if (ZCMD.arg3 != NOWHERE)
	  b = ZCMD.arg3 = real_room(ZCMD.arg3);
	break;
      case 'G':
	a = ZCMD.arg1 = real_object(ZCMD.arg1);
	break;
      case 'E':
	a = ZCMD.arg1 = real_object(ZCMD.arg1);
	break;
      case 'P':
	a = ZCMD.arg1 = real_object(ZCMD.arg1);
	b = ZCMD.arg3 = real_object(ZCMD.arg3);
	break;
      case 'D':
	a = ZCMD.arg1 = real_room(ZCMD.arg1);
	break;
      case 'R': /* rem obj from room */
        a = ZCMD.arg1 = real_room(ZCMD.arg1);
	b = ZCMD.arg2 = real_object(ZCMD.arg2);
        break;
      }
      if (a < 0) {
	if (!mini_mud)
	  log_zone_error(zone, cmd_no, "Invalid vnum, cmd disabled - a");
	ZCMD.command = '*';
      }
      if (b < 0) {
	if (!mini_mud)
	  log_zone_error(zone, cmd_no, "Invalid vnum, cmd disabled - b");
	ZCMD.command = '*';
      }
    }
}



void parse_simple_mob(FILE *mob_f, int i, int nr)
{
  int j, t[10];
  char line[256];

    mob_proto[i].real_abils.str = 11;
    mob_proto[i].real_abils.intel = 11;
    mob_proto[i].real_abils.wis = 11;
    mob_proto[i].real_abils.dex = 11;
    mob_proto[i].real_abils.con = 11;
    mob_proto[i].real_abils.cha = 11;

    get_line(mob_f, line);
    if (sscanf(line, " %d %d %d %dd%d+%d %dd%d+%d ",
	  t, t + 1, t + 2, t + 3, t + 4, t + 5, t + 6, t + 7, t + 8) != 9) {
      fprintf(stderr, "Format error in mob #%d, first line after S flag\n"
	      "...expecting line of form '# # # #d#+# #d#+#'\n", nr);
              system("touch ../.killscript");
     exit(1);
    }
    GET_LEVEL(mob_proto + i) = t[0];
    mob_proto[i].points.hitroll = 20 - t[1];
    mob_proto[i].points.armor = 10 * t[2];

    /* max hit = 0 is a flag that H, M, V is xdy+z */
    mob_proto[i].points.max_hit = 0;
    mob_proto[i].points.hit = t[3];
    mob_proto[i].points.mana = t[4];
    mob_proto[i].points.move = t[5];

    mob_proto[i].points.max_mana = 10;
    mob_proto[i].points.max_move = 50;

    mob_proto[i].mob_specials.damnodice = t[6];
    mob_proto[i].mob_specials.damsizedice = t[7];
    mob_proto[i].points.damroll = t[8];

    get_line(mob_f, line);
    sscanf(line, " %d %d ", t, t + 1);
    GET_GOLD(mob_proto + i) = t[0];
    GET_EXP(mob_proto + i) = t[1];
 
    get_line(mob_f, line);
    if (sscanf(line, " %d %d %d %d ", t, t + 1, t + 2, t + 3) == 4)
      mob_proto[i].mob_specials.attack_type = t[3];
    else
      mob_proto[i].mob_specials.attack_type = 0;
 
    mob_proto[i].char_specials.position = t[0];
    mob_proto[i].mob_specials.default_pos = t[1];
    mob_proto[i].player.sex = t[2];

    mob_proto[i].player.class = 30;
    mob_proto[i].player.weight = 200;
    mob_proto[i].player.height = 198;

    for (j = 0; j < 3; j++)
      GET_COND(mob_proto + i, j) = -1;

    /*
     * these are now save applies; base save numbers for MOBs are now from
     * the warrior save table.
     */
    for (j = 0; j < 5; j++)
      GET_SAVE(mob_proto + i, j) = 0;
}


/*
 * interpret_espec is the function that takes espec keywords and values
 * and assigns the correct value to the mob as appropriate.  Adding new
 * e-specs is absurdly easy -- just add a new CASE statement to this
 * function!  No other changes need to be made anywhere in the code.
 */

#define CASE(test) if (!matched && !str_cmp(keyword, test) && (matched = 1))
#define RANGE(low, high) (num_arg = MAX((low), MIN((high), (num_arg))))

void interpret_espec(char *keyword, char *value, int i, int nr)
{
  int num_arg, matched = 0;

  num_arg = atoi(value);

  CASE("BareHandAttack") {
    RANGE(0, 99);
    mob_proto[i].mob_specials.attack_type = num_arg;
  }

  CASE("Str") {
    RANGE(3, 25);
    mob_proto[i].real_abils.str = num_arg;
  }

  CASE("StrAdd") {
    RANGE(0, 100);
    mob_proto[i].real_abils.str_add = num_arg;    
  }

  CASE("Int") {
    RANGE(3, 25);
    mob_proto[i].real_abils.intel = num_arg;
  }

  CASE("Wis") {
    RANGE(3, 25);
    mob_proto[i].real_abils.wis = num_arg;
  }

  CASE("Dex") {
    RANGE(3, 25);
    mob_proto[i].real_abils.dex = num_arg;
  }

  CASE("Con") {
    RANGE(3, 25);
    mob_proto[i].real_abils.con = num_arg;
  }

  CASE("Cha") {
    RANGE(3, 25);
    mob_proto[i].real_abils.cha = num_arg;
  }

  CASE("Class") {  /*mobclass*/
    if (num_arg < CLASS_DEFAULT)
      mob_proto[i].player.class = CLASS_DEFAULT;
    else
      mob_proto[i].player.class = num_arg;
  }
  
  CASE("Para") {
    GET_SAVE(mob_proto + i, SAVING_PARA) = num_arg;
  }
  CASE("Rod") {
    GET_SAVE(mob_proto + i, SAVING_ROD) = num_arg;
  }
  CASE("Petri") {
    GET_SAVE(mob_proto + i, SAVING_PETRI) = num_arg;
  }
  CASE("Breath") {
    GET_SAVE(mob_proto + i, SAVING_BREATH) = num_arg;
  }
  CASE("Spell") {
    GET_SAVE(mob_proto + i, SAVING_SPELL) = num_arg;
  }
  if (!matched) {
    fprintf(stderr, "Warning: unrecognized espec keyword %s in mob #%d\n",
	    keyword, nr);
  }    
}

#undef CASE
#undef RANGE

void parse_espec(char *buf, int i, int nr)
{
  char *ptr;

  if ((ptr = strchr(buf, ':')) != NULL) {
    *(ptr++) = '\0';
    while (isspace(*ptr))
      ptr++;
  } else
    ptr = "";

  interpret_espec(buf, ptr, i, nr);
}


void parse_enhanced_mob(FILE *mob_f, int i, int nr)
{
  char line[256];

  parse_simple_mob(mob_f, i, nr);

  while (get_line(mob_f, line)) {
    if (!strcmp(line, "E"))	/* end of the ehanced section */
      return;
    else if (*line == '#') {	/* we've hit the next mob, maybe? */
      fprintf(stderr, "Unterminated E section in mob #%d\n", nr);
              system("touch ../.killscript");
      exit(1);
    } else
      parse_espec(line, i, nr);
  }

  fprintf(stderr, "Unexpected end of file reached after mob #%d\n", nr);
              system("touch ../.killscript");
  exit(1);
}


void parse_mobile(FILE * mob_f, int nr)
{
  static int i = 0;
  int j, t[10];
  char line[256], *tmpptr, letter;
  char f1[128], f2[128], f3[128];
  int retval = 0;
  char tbuf[256];

  mob_index[i].virtualrm = nr;
  mob_index[i].number = 0;
  mob_index[i].func = NULL;
  mob_index[i].deleted = FALSE;

  clear_char(mob_proto + i);

  mob_proto[i].player_specials = &dummy_mob;
  sprintf(buf2, "mob vnum %d", nr);

  /***** String data *** */
  mob_proto[i].player.name = fread_string(mob_f, buf2);

  if(!mob_proto[i].player.name) {
    sprintf(tbuf, "SYSERR: Mob #%d has no name.", mob_index[i].virtualrm);
    nmlog(tbuf);
    mob_proto[i].player.name = str_dup("Nobody!");
  }

  tmpptr = mob_proto[i].player.short_descr = fread_string(mob_f, buf2);
  if (tmpptr && *tmpptr)
    if (!str_cmp(fname(tmpptr), "a") || !str_cmp(fname(tmpptr), "an") ||
	!str_cmp(fname(tmpptr), "the"))
      *tmpptr = LOWER(*tmpptr);
  mob_proto[i].player.long_descr = fread_string(mob_f, buf2);
  mob_proto[i].player.description = fread_string(mob_f, buf2);
  mob_proto[i].player.title = NULL;

  /* *** Numeric data *** */
  get_line(mob_f, line);
  retval = sscanf(line, "%s %s %d %c %i %i %s %i", f1, f2, t + 2, &letter,
     t + 3, t + 4, f3, t + 5);
  MOB_FLAGS(mob_proto + i) = asciiflag_conv(f1);
  SET_BIT(MOB_FLAGS(mob_proto + i), MOB_ISNPC);
  AFF_FLAGS(mob_proto + i) = asciiflag_conv(f2);
  GET_ALIGNMENT(mob_proto + i) = t[2];
  if(retval > 4)
    mob_proto[i].mob_specials.num_attacks = t[3];
  else
    mob_proto[i].mob_specials.num_attacks = 1;
  if(retval >= 6 && t[4] >= 30)
    GET_CLASS(mob_proto + i) = t[4];
  else 
    GET_CLASS(mob_proto + i) = 30;
  if(retval >= 7)
    AFF2_FLAGS(mob_proto + i) = asciiflag_conv(f3);
  if (retval >= 8)
    GET_REMORT(mob_proto + i) = t[5];
  else
    GET_REMORT(mob_proto + i) = 0;


  switch (letter) {
  case 'S':	/* Simple monsters */
    parse_simple_mob(mob_f, i, nr);
    break;
  case 'E':	/* Circle3 Enhanced monsters */
    parse_enhanced_mob(mob_f, i, nr);
    break;
  /* add new mob types here.. */
  default:
    fprintf(stderr, "Unsupported mob type '%c' in mob #%d\n", letter, nr);
              system("touch ../.killscript");
    exit(1);
    break;
  }

  mob_proto[i].aff_abils = mob_proto[i].real_abils;

  for (j = 0; j < NUM_WEARS; j++)
    mob_proto[i].equipment[j] = NULL;

  mob_proto[i].nr = i;
  mob_proto[i].desc = NULL;
  top_of_mobt = i++;
}




/* read all objects from obj file; generate index and prototypes */
char *parse_object(FILE * obj_f, int nr)
{
  static int i = 0, retval;
  static char line[256];
  int t[10], j;
  char *tmpptr, tbuf[512];
  char f1[256], f2[256];
  struct extra_descr_data *new_descr;

  obj_index[i].virtualrm = nr;
  obj_index[i].number = 0;
  obj_index[i].func = NULL;
  obj_index[i].deleted = FALSE;

  clear_object(obj_proto + i);
  obj_proto[i].in_room = NOWHERE;
  obj_proto[i].item_number = i;

  sprintf(buf2, "object #%d", nr);

  /* *** string data *** */
  if ((obj_proto[i].name = fread_string(obj_f, buf2)) == NULL) {
    fprintf(stderr, "Null obj name or format error at or near %s\n", buf2);
              system("touch ../.killscript");
    exit(1);
  }
  tmpptr = obj_proto[i].short_description = fread_string(obj_f, buf2);
  if (*tmpptr)
    if (!str_cmp(fname(tmpptr), "a") || !str_cmp(fname(tmpptr), "an") ||
	!str_cmp(fname(tmpptr), "the"))
      *tmpptr = LOWER(*tmpptr);

  tmpptr = obj_proto[i].description = fread_string(obj_f, buf2);
  if (tmpptr && *tmpptr)
    *tmpptr = UPPER(*tmpptr);
  obj_proto[i].action_description = fread_string(obj_f, buf2);

  /* *** numeric data *** */
  if (!get_line(obj_f, line) ||
      (retval = sscanf(line, " %d %s %s %d %d", t, f1, f2, t + 1, t + 2)) < 3) {
    fprintf(stderr, "Format error in first numeric line (expecting 4 args, got %d), %s\n", retval, buf2);
              system("touch ../.killscript");
    exit(1);
  }
  obj_proto[i].obj_flags.type_flag = t[0];
  obj_proto[i].obj_flags.extra_flags = asciiflag_conv(f1);
  obj_proto[i].obj_flags.wear_flags = asciiflag_conv(f2);
  if(retval >= 4)
    obj_proto[i].obj_flags.min_level = t[1];
  else
    obj_proto[i].obj_flags.min_level = LVL_IMMORT;
  if(retval == 5)
    obj_proto[i].obj_flags.min_remort = t[2];
  else
    obj_proto[i].obj_flags.min_remort = 0;
    

  if (!get_line(obj_f, line) ||
      (retval = sscanf(line, "%d %d %d %d %d", t, t + 1, t + 2, t + 3, t + 4)) < 4) {
    fprintf(stderr, "Format error in second numeric line (expecting 4 args, got %d), %s\n", retval, buf2);
              system("touch ../.killscript");
    exit(1);
  }
  obj_proto[i].obj_flags.value[0] = t[0];
  obj_proto[i].obj_flags.value[1] = t[1];
  obj_proto[i].obj_flags.value[2] = t[2];
  obj_proto[i].obj_flags.value[3] = t[3];
  if (retval == 5)
    obj_proto[i].obj_flags.value[4] = t[4];

  if (!get_line(obj_f, line) ||
      (retval = sscanf(line, "%d %d %d %s %s", t, t + 1, t + 2, f1, f2)) < 3) {
    fprintf(stderr, "Format error in third numeric line (expecting 3 args, got %d), %s\n", retval, buf2);
              system("touch ../.killscript");
    exit(1);
  }
  obj_proto[i].obj_flags.weight = t[0];
  obj_proto[i].obj_flags.cost = t[1];
  obj_proto[i].obj_flags.cost_per_day = t[2];
  if (retval >= 4)
    obj_proto[i].obj_flags.bitvector = asciiflag_conv(f1);
  if (retval == 5)
    obj_proto[i].obj_flags.bitvector2 = asciiflag_conv(f2);

  /* check to make sure that weight of containers exceeds curr. quantity */
/*  if (obj_proto[i].obj_flags.type_flag == ITEM_DRINKCON ||
      obj_proto[i].obj_flags.type_flag == ITEM_FOUNTAIN) {
    if (obj_proto[i].obj_flags.weight < obj_proto[i].obj_flags.value[1])
      obj_proto[i].obj_flags.weight = obj_proto[i].obj_flags.value[1] + 5;
  } removed so the waterball works..sam 01July96 */

  /* *** extra descriptions and affect fields *** */

  for (j = 0; j < MAX_OBJ_AFFECT; j++) {
    obj_proto[i].affected[j].location = APPLY_NONE;
    obj_proto[i].affected[j].modifier = 0;
  }

  strcat(buf2, ", after numeric constants (expecting E/A/#xxx)");
  j = 0;

  for (;;) {
    if (!get_line(obj_f, line)) {
      fprintf(stderr, "Format error in %s\n", buf2);
              system("touch ../.killscript");
      exit(1);
    }
    switch (*line) {
    case 'E':
      CREATE(new_descr, struct extra_descr_data, 1);
      new_descr->keyword = fread_string(obj_f, buf2);
      if(!new_descr->keyword) { /*sam */
	sprintf(tbuf, "SYSERR:  Obj #%d has empty edesc keyword list.",
	  obj_index[i].virtualrm);
	nmlog(tbuf);
	new_descr->keyword = str_dup("empty");
      }
      new_descr->description = fread_string(obj_f, buf2);
      new_descr->next = obj_proto[i].ex_description;
      obj_proto[i].ex_description = new_descr;
      break;
    case 'A':
      if (j >= MAX_OBJ_AFFECT) {
	fprintf(stderr, "Too many A fields (%d max), %s\n", MAX_OBJ_AFFECT, buf2);
system("touch ../.killscript");
	exit(1);
      }
      get_line(obj_f, line);
      sscanf(line, " %d %d ", t, t + 1);
      obj_proto[i].affected[j].location = t[0];
      obj_proto[i].affected[j].modifier = t[1];
      j++;
      break;
    case '$':
    case '#':
      top_of_objt = i++;
      return line;
      break;
    default:
      fprintf(stderr, "Format error in %s\n", buf2);
system("touch ../.killscript");
      exit(1);
      break;
    }
  }
}


#define Z	zone_table[zone]

/* load the zone table and command tables */
void load_zones(FILE * fl, char *zonename)
{
  static int zone = 0;
  int cmd_no = 0, num_of_cmds = 0, line_num = 0, tmp, error;
  char *ptr, buf[256], zname[256];

  strcpy(zname, zonename);

  while (get_line(fl, buf))
    num_of_cmds++;		/* this should be correct within 3 or so */
  rewind(fl);

  if (num_of_cmds == 0) {
    fprintf(stderr, "%s is empty!\n", zname);
system("touch ../.killscript");
    exit(0);
  } else
    CREATE(Z.cmd, struct reset_com, num_of_cmds);

  line_num += get_line(fl, buf);

  if (sscanf(buf, "#%d", &Z.number) != 1) {
    fprintf(stderr, "Format error in %s, line %d\n", zname, line_num);
system("touch ../.killscript");
    exit(0);
  }
  sprintf(buf2, "beginning of zone #%d", Z.number);

  line_num += get_line(fl, buf);
  if ((ptr = strchr(buf, '~')) != NULL)	/* take off the '~' if it's there */
    *ptr = '\0';
  Z.name = str_dup(buf);

  line_num += get_line(fl, buf);
  if (sscanf(buf, " %d %d %d ", &Z.top, &Z.lifespan, &Z.reset_mode) != 3) {
    fprintf(stderr, "Format error in 3-constant line of %s", zname);
system("touch ../.killscript");
    exit(0);
  }
  cmd_no = 0;

  for (;;) {
    if ((tmp = get_line(fl, buf)) == 0) {
      fprintf(stderr, "Format error in %s - premature end of file\n", zname);
system("touch ../.killscript");
      exit(0);
    }
    line_num += tmp;
    ptr = buf;
    skip_spaces(&ptr);

    if ((ZCMD.command = *ptr) == '*')
      continue;

    ptr++;

    if (ZCMD.command == 'S' || ZCMD.command == '$') {
      ZCMD.command = 'S';
      break;
    }
    error = 0;
    if (strchr("MOEPD", ZCMD.command) == NULL) {	/* a 3-arg command */
      if (sscanf(ptr, " %d %d %d ", &tmp, &ZCMD.arg1, &ZCMD.arg2) != 3)
	error = 1;
    } else {
      if (sscanf(ptr, " %d %d %d %d ", &tmp, &ZCMD.arg1, &ZCMD.arg2,
		 &ZCMD.arg3) != 4)
	error = 1;
    }

    ZCMD.if_flag = tmp;

    if (error) {
      fprintf(stderr, "Format error in %s, line %d: '%s'\n", zname, line_num, buf);
system("touch ../.killscript");
      exit(0);
    }
    ZCMD.line = line_num;
    cmd_no++;
  }

  top_of_zone_table = zone++;
}

#undef Z


void get_one_line(FILE *fl, char *buf)
{
  if (fgets(buf, READ_SIZE, fl) == NULL) {
    nmlog("error reading help file: not terminated with $?");
system("touch ../.killscript");
    exit(1);
  }

  buf[strlen(buf) - 1] = '\0'; /* take off the trailing \n */
}


void load_help(FILE *fl)
{
  char key[READ_SIZE+1], next_key[READ_SIZE+1], entry[32384];
  char line[READ_SIZE+1], *scan;
  struct help_index_element el;

  /* get the first keyword line */
  get_one_line(fl, key);
  while (*key != '$') {
    /* read in the corresponding help entry */
    strcpy(entry, strcat(key, "\r\n"));
    get_one_line(fl, line);
    while (*line != '#') {
      strcat(entry, strcat(line, "\r\n"));
      get_one_line(fl, line);
    }

    /* now, add the entry to the index with each keyword on the keyword line */
    el.duplicate = 0;
    el.entry = str_dup(entry);
    scan = one_word(key, next_key);
    while (*next_key) {
      el.keyword = str_dup(next_key);
      help_table[top_of_helpt++] = el;
      el.duplicate++;
      scan = one_word(scan, next_key);
    }

    /* get next keyword line (or $) */
    get_one_line(fl, key);
  }
}


int hsort(const void *a, const void *b)
{
  struct help_index_element *a1, *b1;

  a1 = (struct help_index_element *) a;
  b1 = (struct help_index_element *) b;

  return (str_cmp(a1->keyword, b1->keyword));
}


/*************************************************************************
*  procedures for resetting, both play-time and boot-time	 	 *
*********************************************************************** */

int vnum_room(char *searchname, struct char_data * ch)
{
  int nr, found = 0;
  
  *buf = '\0';
  for (nr = 0; nr <= top_of_world; nr++) {
    if (isname(searchname, world[nr].name)) {
      sprintf(buf, "%s&+w%3d. [%5d] %s&+w\r\n", buf, ++found,
              world[nr].number,
              world[nr].name);
/*      send_to_char(buf, ch); */
    }
  }

  page_string(ch->desc, buf, 1);
  return (found);
}

int vnum_mobile(char *searchname, struct char_data * ch)
{
  int nr, found = 0;

  *buf = '\0';
  for (nr = 0; nr <= top_of_mobt; nr++) {
    if (isname(searchname, mob_proto[nr].player.name)) {
      sprintf(buf, "%s&+w%3d. [%5d] %s&+w\r\n", buf, ++found,
	      mob_index[nr].virtualrm,
	      mob_proto[nr].player.short_descr);
/*      send_to_char(buf, ch); */
    }
  }
  page_string(ch->desc, buf, 1);
  return (found);
}



int vnum_object(char *searchname, struct char_data * ch)
{
  int nr, found = 0;

  *buf = '\0';
  for (nr = 0; nr <= top_of_objt; nr++) {
    if (isname(searchname, obj_proto[nr].name)) {
      sprintf(buf, "%s&+w%3d. [%5d] %s&+w\r\n", buf, ++found,
	      obj_index[nr].virtualrm,
	      obj_proto[nr].short_description);
/*      send_to_char(buf, ch); */
    }
  }
  page_string(ch->desc, buf, 1);
  return (found);
}


/* create a character, and add it to the char list */
struct char_data *create_char(void)
{
  struct char_data *ch;

  CREATE(ch, struct char_data, 1);
  clear_char(ch);
  ch->next = character_list;
  character_list = ch;

  return ch;
}


/* create a new mobile from a prototype */
struct char_data *read_mobile(int nr, int type)
{
  int i;
  struct char_data *mob;

  if (type == VIRTUALRM) {
    if ((i = real_mobile(nr)) < 0) {
      sprintf(buf, "Mobile (V) %d does not exist in database.", nr);
      return (0);
    }
  } else
    i = nr;

  CREATE(mob, struct char_data, 1);
  clear_char(mob);
  *mob = mob_proto[i];
  mob->next = character_list;
  character_list = mob;

  if (!mob->points.max_hit) {
    mob->points.max_hit = dice(mob->points.hit, mob->points.mana) +
      mob->points.move;
  } else
    mob->points.max_hit = number(mob->points.hit, mob->points.mana);

  mob->points.hit = mob->points.max_hit;
  mob->points.mana = mob->points.max_mana;
  mob->points.move = mob->points.max_move;

  mob->player.time.birth = time(0);
  mob->player.time.played = 0;
  mob->player.time.logon = time(0);

  mob_index[i].number++;

  return mob;
}


/* create an object, and add it to the object list */
struct obj_data *create_obj(void)
{
  struct obj_data *obj;

  CREATE(obj, struct obj_data, 1);
  clear_object(obj);
  obj->next = object_list;
  object_list = obj;

  return obj;
}


/* create a new object from a prototype */
struct obj_data *read_object(int nr, int type)
{
  struct obj_data *obj;
  int i;

  if (nr < 0) {
    nmlog("SYSERR: trying to create obj with negative num!");
    return NULL;
  }
  if (type == VIRTUALRM) {
    if ((i = real_object(nr)) < 0) {
      sprintf(buf, "Object (V) %d does not exist in database.", nr);
      return NULL;
    }
  } else
    i = nr;

  CREATE(obj, struct obj_data, 1);
  clear_object(obj);
  *obj = obj_proto[i];
  obj->next = object_list;
  object_list = obj;

  obj_index[i].number++;

  return obj;
}



#define ZO_DEAD  999

/* update zone ages, queue for reset if necessary, and dequeue when possible */
void zone_update(void)
{
  int i;
  struct reset_q_element *update_u, *temp;
  static int timer = 0;
  char buf[128];

  /* jelson 10/22/92 */
  if (((++timer * PULSE_ZONE) / PASSES_PER_SEC) >= 60) {
    /* one minute has passed */
    /*
     * NOT accurate unless PULSE_ZONE is a multiple of PASSES_PER_SEC or a
     * factor of 60
     */

    timer = 0;

    /* since one minute has passed, increment zone ages */
    for (i = 0; i <= top_of_zone_table; i++) {
      if (zone_table[i].age < zone_table[i].lifespan &&
	  zone_table[i].reset_mode)
	(zone_table[i].age)++;

      if (zone_table[i].age >= zone_table[i].lifespan &&
	  zone_table[i].age < ZO_DEAD && zone_table[i].reset_mode) {
	/* enqueue zone */

	CREATE(update_u, struct reset_q_element, 1);

	update_u->zone_to_reset = i;
	update_u->next = 0;

	if (!reset_q.head)
	  reset_q.head = reset_q.tail = update_u;
	else {
	  reset_q.tail->next = update_u;
	  reset_q.tail = update_u;
	}

	zone_table[i].age = ZO_DEAD;
      }
    }
  }	/* end - one minute has passed */


  /* dequeue zones (if possible) and reset */
  /* this code is executed every 10 seconds (i.e. PULSE_ZONE) */
  for (update_u = reset_q.head; update_u; update_u = update_u->next)
    if (zone_table[update_u->zone_to_reset].reset_mode == 2 ||
	is_empty(update_u->zone_to_reset)) {
      reset_zone(update_u->zone_to_reset);
      sprintf(buf, "Auto zone reset: %s",
	      zone_table[update_u->zone_to_reset].name);
      mudlog(buf, CMP, LVL_GOD, FALSE);
      /* dequeue */
      if (update_u == reset_q.head)
	reset_q.head = reset_q.head->next;
      else {
	for (temp = reset_q.head; temp->next != update_u;
	     temp = temp->next);

	if (!update_u->next)
	  reset_q.tail = temp;

	temp->next = update_u->next;
      }

      free(update_u);
      break;
    }
}

void log_zone_error(int zone, int cmd_no, char *message)
{
  char buf[256];

  sprintf(buf, "SYSERR: error in zone file: %s", message);
  mudlog(buf, NRM, LVL_GOD, TRUE);

  sprintf(buf, "SYSERR: ...offending cmd: '%c' cmd in zone #%d, line %d",
	  ZCMD.command, zone_table[zone].number, ZCMD.line);
  mudlog(buf, NRM, LVL_GOD, TRUE);
}

#define ZONE_ERROR(message) \
	{ log_zone_error(zone, cmd_no, message); last_cmd = 0; }

/* execute the reset command table of a given zone */
void reset_zone(int zone)
{
  int cmd_no, last_cmd = 0;
  struct char_data *mob = NULL;
  struct obj_data *obj, *obj_to;

  for (cmd_no = 0; ZCMD.command != 'S'; cmd_no++) {

    if (ZCMD.if_flag && !last_cmd)
      continue;

    switch (ZCMD.command) {
    case '*':			/* ignore command */
      last_cmd = 0;
      break;

    case 'M':			/* read a mobile */
      if (mob_index[ZCMD.arg1].number < ZCMD.arg2) {
	mob = read_mobile(ZCMD.arg1, REAL);
	char_to_room(mob, ZCMD.arg3);
	last_cmd = 1;
      } else
	last_cmd = 0;
      break;

    case 'O':			/* read an object */
      if (obj_index[ZCMD.arg1].number < ZCMD.arg2) {
	if (ZCMD.arg3 >= 0) {
	  obj = read_object(ZCMD.arg1, REAL);
	  obj_to_room(obj, ZCMD.arg3);
	  last_cmd = 1;
	} else {
	  obj = read_object(ZCMD.arg1, REAL);
	  obj->in_room = NOWHERE;
	  last_cmd = 1;
	}
      } else
	last_cmd = 0;
      break;

    case 'P':			/* object to object */
      if (obj_index[ZCMD.arg1].number < ZCMD.arg2) {
	obj = read_object(ZCMD.arg1, REAL);
	if (!(obj_to = get_obj_num(ZCMD.arg3))) {
	  ZONE_ERROR("target obj not found");
	  break;
	}
	obj_to_obj(obj, obj_to);
	last_cmd = 1;
      } else
	last_cmd = 0;
      break;

    case 'G':			/* obj_to_char */
      if (!mob) {
	ZONE_ERROR("attempt to give obj to non-existant mob");
	break;
      }
      if (obj_index[ZCMD.arg1].number < ZCMD.arg2) {
	obj = read_object(ZCMD.arg1, REAL);
	obj_to_char(obj, mob);
	last_cmd = 1;
      } else
	last_cmd = 0;
      break;

    case 'E':			/* object to equipment list */
      if (!mob) {
	ZONE_ERROR("trying to equip non-existant mob");
	break;
      }
      if (obj_index[ZCMD.arg1].number < ZCMD.arg2) {
	if (ZCMD.arg3 < 0 || ZCMD.arg3 >= NUM_WEARS) {
	  ZONE_ERROR("invalid equipment pos number");
	} else {
	  obj = read_object(ZCMD.arg1, REAL);
	  equip_char(mob, obj, ZCMD.arg3);
	  last_cmd = 1;
	}
      } else
	last_cmd = 0;
      break;

    case 'R': /* rem obj from room */
      if ((obj = get_obj_in_list_num(ZCMD.arg2, world[ZCMD.arg1].contents)) != NULL) {
        obj_from_room(obj);
        extract_obj(obj);
      }
      last_cmd = 1;
      break;


    case 'D':			/* set state of door */
      if (ZCMD.arg2 < 0 || ZCMD.arg2 >= NUM_OF_DIRS ||
	  (world[ZCMD.arg1].dir_option[ZCMD.arg2] == NULL)) {
	ZONE_ERROR("door does not exist");
      } else
	switch (ZCMD.arg3) {
	case 0:
	  REMOVE_BIT(world[ZCMD.arg1].dir_option[ZCMD.arg2]->exit_info,
		     EX_LOCKED);
	  REMOVE_BIT(world[ZCMD.arg1].dir_option[ZCMD.arg2]->exit_info,
		     EX_CLOSED);
	  break;
	case 1:
	  SET_BIT(world[ZCMD.arg1].dir_option[ZCMD.arg2]->exit_info,
		  EX_CLOSED);
	  REMOVE_BIT(world[ZCMD.arg1].dir_option[ZCMD.arg2]->exit_info,
		     EX_LOCKED);
	  break;
	case 2:
	  SET_BIT(world[ZCMD.arg1].dir_option[ZCMD.arg2]->exit_info,
		  EX_LOCKED);
	  SET_BIT(world[ZCMD.arg1].dir_option[ZCMD.arg2]->exit_info,
		  EX_CLOSED);
	  break;
	}
      last_cmd = 1;
      break;

    default:
      ZONE_ERROR("unknown cmd in reset table; cmd disabled");
      ZCMD.command = '*';
      break;
    }
  }

  zone_table[zone].age = 0;
}



/* for use in reset_zone; return TRUE if zone 'nr' is free of PC's  */
int is_empty(int zone_nr)
{
  struct descriptor_data *i;

  for (i = descriptor_list; i; i = i->next)
    if (!i->connected)
      if (world[i->character->in_room].zone == zone_nr)
	return 0;

  return 1;
}





/*************************************************************************
*  stuff related to the save/load player system				 *
*********************************************************************** */


long get_id_by_name(char *name)
{
  int i;

  one_argument(name, arg);
  for (i = 1; i < top_of_p_table; i++) {
    if (!player_table[i].name) 
      continue;
    if (!strcmp((player_table + i)->name, arg))
      return ((player_table + i)->id);
  }
  return -1;
}


char *get_name_by_id(long id)
{
  if (id >= MAXENTRIES || id < 0)
    return NULL;
  if ((player_table + (int) id)->id != 0)
    return ((player_table +(int) id)->name);

  return NULL;
}


/* new load_char that reads an ascii pfile */
/* Load a char, TRUE if loaded, FALSE if not */
int load_char(char *name, struct char_file_u * char_element,
	struct char_data *ch)
{
  int id, num = 0, num2 = 0, num3 = 0, num4 = 0, num5 = 0, num6 = 0, i;
  FILE *fl;
  char filename[40];
  char buf[128], line[MAX_INPUT_LENGTH+1], tag[6], *buf2 = NULL;
  struct char_file_u *player = char_element;
  struct char_special_data_saved *csds;
  struct player_special_data_saved *psds;
  struct char_ability_data *cad;
  struct char_point_data *cpd;
  struct affected_type *aff;
  struct ignore_type *new_ignore, *ignore = NULL;

  int find_name(char *name);

  csds = &(player->char_specials_saved);
  psds = &(player->player_specials_saved);
  cad = &(player->abilities);
  cpd = &(player->points);  

  if(!*name)
    return -1;

  if((id = get_id_by_name(name)) < 0) {
    return (-1);
  } else {
    sprintf(filename, "%s/%c/%s", PLR_PREFIX, *player_table[id].name,
      player_table[id].name);
    if(!(fl = fopen(filename, "r"))) {
      sprintf(buf, "SYSERR: Couldn't open player file %s", filename);
      if (ch!=NULL)
        mudlog(buf, NRM, LVL_GOD, TRUE);
      return (-2);
    }

    /* initializations necessary to keep some things straight */
    for(i = 0; i < MAX_AFFECT; i++)
      player->affected[i].type = 0;
    for(i = 1; i <= MAX_SKILLS; i++)
      psds->skills[i] = 0;
    player->sex = 0;
    player->class = 0;
    player->level = 0;
    player->height = 0;
    player->weight = 0;
    player->email = NULL;
    player->homepage = NULL;
    player->pagesize = 24;
    player->idleafk = 10;
    player->remort = 0;
    csds->alignment = 0;
    csds->act = 0;
    csds->affected_by = 0;
    csds->affected_by2 = 0;
    csds->real_aff = 0;
    csds->real_aff2 = 0;
    csds->apply_saving_throw[0] = 0;
    csds->apply_saving_throw[1] = 0;
    csds->apply_saving_throw[2] = 0;
    csds->apply_saving_throw[3] = 0;
    csds->apply_saving_throw[4] = 0;
    psds->load_room = 3001;
    psds->invis_level = 0;
    psds->freeze_level = 0;
    psds->wimp_level = 0;
    psds->conditions[0] = 0;
    psds->conditions[1] = 0;
    psds->conditions[2] = 0;
    psds->bad_pws = 0;
    psds->pref = 0;
    psds->pref2 = 0;
    psds->clan = 0;
    psds->race = 0;
    psds->spells_to_learn = 0;
    psds->pkill_points = 0;
    psds->purity = 0;
    psds->quest = 0;
    cpd->gold = 0;
    cpd->bank_gold = 0;
    cpd->exp = 0;
    cpd->hitroll = 0;
    cpd->damroll = 0;

    while(get_line(fl, line)) {
      tag_argument(line, tag);
      num = atoi(line);

      switch (*tag) {
      case 'A':
	if(!strcmp(tag, "Ac  "))
	  cpd->armor = num;
	else if(!strcmp(tag, "Act ")) 
	  csds->act = asciiflag_conv(line);
	else if(!strcmp(tag, "Aff ")) {
	  csds->real_aff = asciiflag_conv(line);
	  csds->affected_by = csds->real_aff;
	} else if(!strcmp(tag, "Aff2")) {
	  csds->real_aff2 = asciiflag_conv(line);
	  csds->affected_by2 = csds->real_aff2;
        }
	else if(!strcmp(tag, "Affs")) {
	  i = 0;
	  do {
	    get_line(fl, line);
	    num6 = 0;
	    sscanf(line, "%d %d %d %d %d %d", &num, &num2, &num3, &num4, &num5,
	      &num6);
	    aff = &(player->affected[i]);
	    aff->type = num;
	    aff->duration = num2;
	    aff->modifier = num3;
	    aff->location = num4;
	    aff->bitvector = num5;
	    aff->bitvector2 = num6;
	    i++;
	  } while (num != 0);
	} else if(!strcmp(tag, "Alin"))
	  csds->alignment = num;
	break;

      case 'B':
	if(!strcmp(tag, "Badp"))
	  psds->bad_pws = num;
	else if(!strcmp(tag, "Bank"))
	  cpd->bank_gold = num;
	else if(!strcmp(tag, "Brth"))
	  player->birth = num;
	break;

      case 'C':
	if(!strcmp(tag, "Cha "))
	  cad->cha = num;
	else if(!strcmp(tag, "Clan"))
	  psds->clan = num;
	else if(!strcmp(tag, "Clas"))
	  player->class = num;
	else if(!strcmp(tag, "Con "))
	  cad->con = num;
	break;

      case 'D':
	if(!strcmp(tag, "Desc")) {
	  buf2 = fread_string(fl, line);
	  if(buf2) {
	    strcpy(player->description, buf2);
	    free(buf2);
	  } else
	    *player->description = '\0';
	} else if(!strcmp(tag, "Dex "))
	  cad->dex = num;
	else if(!strcmp(tag, "Drnk"))
	  psds->conditions[2] = num;
	else if(!strcmp(tag, "Drol"))
	  cpd->damroll = num;
	break;

      case 'E':
	if(!strcmp(tag, "Emal"))
	  player->email = str_dup(line);
	else if(!strcmp(tag, "Exp "))
	  cpd->exp = num;
	break;

      case 'F':
	if(!strcmp(tag, "Frez"))
	  psds->freeze_level = num;
	break;

      case 'G':
	if(!strcmp(tag, "Gold"))
	  cpd->gold = num;
	break;

      case 'H':
	if(!strcmp(tag, "Hit ")) {
	  sscanf(line, "%d/%d", &num, &num2);
	  cpd->hit = num;
	  cpd->max_hit = num2;
	} else if(!strcmp(tag, "Hite"))
	  player->height = num;
	else if(!strcmp(tag, "Http"))
	  player->homepage = str_dup(line);
	else if(!strcmp(tag, "Host")) {
	  strcpy(player->host, line);
	  psds->host = strdup(player->host);
	}
	else if(!strcmp(tag, "Hrol"))
	  cpd->hitroll = num;
	else if(!strcmp(tag, "Hung"))
	 psds->conditions[0] = num;
	break;

      case 'I':
	if(!strcmp(tag, "Id  "))
	  csds->idnum = num;
	else if(!strcmp(tag, "Int "))
	  cad->intel = num;
	else if(!strcmp(tag, "Invs"))
	  psds->invis_level = num;
	else if(!strcmp(tag, "Ign ")) {
	  CREATE(new_ignore, struct ignore_type, 1);
	  new_ignore->idnum = num;
	  new_ignore->next = NULL;
	  new_ignore->prev = NULL;
	  if (!ignore)
	    ignore = new_ignore;
	  else {
	    ignore->prev = new_ignore;
	    new_ignore->next = ignore;
	    ignore = new_ignore;
	  }
	}
	else if(!strcmp(tag, "Idle"))
          player->idleafk = num;
	break;

      case 'L':
	if(!strcmp(tag, "Last"))
	  player->last_logon = num;
	else if(!strcmp(tag, "Lern"))
	  psds->spells_to_learn = num;
	else if(!strcmp(tag, "Levl"))
	  player->level = num;
	break;

      case 'M':
	if(!strcmp(tag, "Mana")) {
	  sscanf(line, "%d/%d", &num, &num2);
	  cpd->mana = num;
	  cpd->max_mana = num2;
	} else if(!strcmp(tag, "Move")) {
	  sscanf(line, "%d/%d", &num, &num2);
	  cpd->move = num;
	  cpd->max_move = num2;
	}
	break;

      case 'N':
	if(!strcmp(tag, "Name")) {
	  strcpy(player->name, line);
	  if(strlen(player->name) == 0)
	    return -1;
	}
	break;
/*
      case 'O':
	if(!strcmp(tag, "Objs") && ch)
	  Crash_load(ch, fl);
	break;
*/
      case 'P':
	if(!strcmp(tag, "Pass"))
	  strcpy(player->pwd, line);
	else if(!strcmp(tag, "Pkpt"))
	  psds->pkill_points = num;
	else if(!strcmp(tag, "Plyd"))
	  player->played = num;
	else if(!strcmp(tag, "Pref"))
	  psds->pref = asciiflag_conv(line);
	else if(!strcmp(tag, "Prf2"))
	  psds->pref2 = asciiflag_conv(line);
	else if(!strcmp(tag, "Pure"))
	  psds->purity = num;
	else if (!strcmp(tag, "Page")) 
	  player->pagesize = (sh_int) num;
	break;

      case 'Q':
	if(!strcmp(tag, "Qust"))
	  psds->quest = num;
	break;

      case 'R':
	if(!strcmp(tag, "Race"))
	  psds->race = num;
	else if(!strcmp(tag, "Room"))
	  psds->load_room = num;
	else if(!strcmp(tag, "Rmrt"))
	  player->remort = num;
	break;

      case 'S':
	if(!strcmp(tag, "Sex "))
	  player->sex = num;
	else if(!strcmp(tag, "Skil")) {
	  do {
	    get_line(fl, line);
	    sscanf(line, "%d %d", &num, &num2);
	      if(num != 0) {
		psds->skills[num] = num2;
	      }
	  } while (num != 0);
	} else if(!strcmp(tag, "Str ")) {
	  sscanf(line, "%d/%d", &num, &num2);
	  cad->str = num;
	  cad->str_add = num2;
	}
	break;

      case 'T':
	if(!strcmp(tag, "Thir"))
	  psds->conditions[1] = num;
	else if(!strcmp(tag, "Thr1"))
	  csds->apply_saving_throw[0] = num;
	else if(!strcmp(tag, "Thr2"))
	  csds->apply_saving_throw[1] = num;
	else if(!strcmp(tag, "Thr3"))
	  csds->apply_saving_throw[2] = num;
	else if(!strcmp(tag, "Thr4"))
	  csds->apply_saving_throw[3] = num;
	else if(!strcmp(tag, "Thr5"))
	  csds->apply_saving_throw[4] = num;
	else if(!strcmp(tag, "Titl")) {
	  if(line)
	    strcpy(player->title, line);
	  else *player->title = '\0';
	  }
	break;

      case 'W':
	if(!strcmp(tag, "Wate"))
	  player->weight = num;
	else if(!strcmp(tag, "Wimp"))
	  psds->wimp_level = num;
	else if(!strcmp(tag, "Wis "))
	  cad->wis = num;
	break;

      default:
	sprintf(buf, "SYSERR: Unknown tag %s in pfile %s", tag, name);
      }
    }
  }
  /* initialization for imms */
  if(player->level >= LVL_IMMORT) {
    for(i = 1; i <= MAX_SKILLS; i++)
      psds->skills[i] = 100;
    psds->conditions[0] = -1;
    psds->conditions[1] = -1;
    psds->conditions[2] = -1;
  }
  csds->idnum = id;

  if (ch)
    ch->ignore = ignore;
  fclose(fl);
  return (id);
}


/*
 * write the vital data of a player to the player file
 *
 * NOTE: load_room should be an *RNUM* now.  It is converted to a vnum here.
 */
void save_char(struct char_data * ch, sh_int load_room)
{
  struct char_file_u player;
  FILE *outfile;
  char outname[40], bits[127];
  int i;
  struct char_special_data_saved *csds;
  struct player_special_data_saved *psds;
  struct ignore_type *ignoring;
  struct char_ability_data *cad;
  struct char_point_data *cpd;
  struct affected_type *aff;

  if (IS_NPC(ch))
    return;

  char_to_store(ch, &player);

  if (ch->desc)
    strncpy(player.host, ch->desc->host, HOST_LENGTH);
  else 
    if (ch->player_specials->saved.host)
      strncpy(player.host, ch->player_specials->saved.host, HOST_LENGTH);
    else
      sprintf(player.host, "<UNKNOWN>");
  player.host[HOST_LENGTH] = '\0';

  if (!PLR_FLAGGED(ch, PLR_LOADROOM)) {
    if (load_room == NOWHERE)
      player.player_specials_saved.load_room = NOWHERE;
    else
      player.player_specials_saved.load_room = world[load_room].number;
  }

  strcpy(player.pwd, GET_PASSWD(ch));

  {
    if(!*player.name)
      return;

    for (i = 0; (*(bits + i) = LOWER(*(player.name + i))); i++);
    sprintf(outname, "%s/%c/%s", PLR_PREFIX, *bits, bits);

    chmod(outname, S_IREAD | S_IWRITE);

    if (!(outfile = fopen(outname, "w"))) {
      sprintf(buf, "error opening output  file: %s", outname);
      perror(buf);
      exit(1);
    }

    /* char_file_u */
    if(player.name)
      fprintf(outfile, "Name: %s\n", player.name);
    if(player.pwd)
      fprintf(outfile, "Pass: %s\n", player.pwd);
    if(player.title) {
      fprintf(outfile, "Titl: %s\n", player.title);
    }
    if(player.description) {
      kill_ems(player.description);
      fprintf(outfile, "Desc:\n%s~\n", player.description);
    }
    if(player.email)
      fprintf(outfile, "Emal: %s\n", player.email);
    if(player.homepage)
      fprintf(outfile, "Http: %s\n", player.homepage);
    if(player.sex)
      fprintf(outfile, "Sex : %d\n", (int)player.sex); 
    if(player.class)
      fprintf(outfile, "Clas: %d\n", (int)player.class); 
    if(player.level)
      fprintf(outfile, "Levl: %d\n", (int)player.level); 
    fprintf(outfile, "Brth: %d\n", (int)player.birth);
    fprintf(outfile, "Plyd: %d\n", (int)player.played);
    fprintf(outfile, "Last: %d\n", (int)player.last_logon);
    fprintf(outfile, "Host: %s\n", player.host);
    fprintf(outfile, "Page: %d\n", (int)player.pagesize);
    fprintf(outfile, "Idle: %d\n", (int)player.idleafk);
    if(player.height)
      fprintf(outfile, "Hite: %d\n", (int)player.height);
    if(player.weight)
      fprintf(outfile, "Wate: %d\n", (int)player.weight);

    /* char_special_data_saved */
    csds = &(player.char_specials_saved);
    if(csds->alignment)
      fprintf(outfile, "Alin: %d\n", csds->alignment);
    fprintf(outfile, "Id  : %d\n", (int)csds->idnum);
    if(csds->act) {
      sprintbits(csds->act, bits);
      fprintf(outfile, "Act : %s\n", bits);
    }
    if(csds->real_aff) {
      sprintbits(csds->real_aff, bits);
      fprintf(outfile, "Aff : %s\n", bits);
    }
    if(csds->real_aff2) {
      sprintbits(csds->real_aff2, bits);
      fprintf(outfile, "Aff2: %s\n", bits);
    }
    if(csds->apply_saving_throw[0])
      fprintf(outfile, "Thr1: %d\n", csds->apply_saving_throw[0]);
    if(csds->apply_saving_throw[1])
      fprintf(outfile, "Thr2: %d\n", csds->apply_saving_throw[1]);
    if(csds->apply_saving_throw[2])
      fprintf(outfile, "Thr3: %d\n", csds->apply_saving_throw[2]);
    if(csds->apply_saving_throw[3])
      fprintf(outfile, "Thr4: %d\n", csds->apply_saving_throw[3]);
    if(csds->apply_saving_throw[4])
      fprintf(outfile, "Thr5: %d\n", csds->apply_saving_throw[4]);

    /* player_special_data_saved */
    psds = &(player.player_specials_saved);
    if(player.level < LVL_IMMORT) {
      fprintf(outfile, "Skil:\n");
      for(i = 1; i <= MAX_SKILLS; i++) {
	if(psds->skills[i])
	  fprintf(outfile, "%d %d\n", i, (int)psds->skills[i]);
      }
      fprintf(outfile, "0 0\n");
    }
    if(psds->wimp_level)
      fprintf(outfile, "Wimp: %d\n", psds->wimp_level);
    if(psds->freeze_level)
      fprintf(outfile, "Frez: %d\n", (int)psds->freeze_level);
    if(psds->invis_level)
      fprintf(outfile, "Invs: %d\n", (int)psds->invis_level);
    if(psds->load_room)
      fprintf(outfile, "Room: %d\n", (int)psds->load_room);
      fprintf(outfile, "Rmrt: %d\n", (int)player.remort);
    if(psds->pref) {
      sprintbits(psds->pref, bits);
      fprintf(outfile, "Pref: %s\n", bits);
    }
    if(psds->pref2) {
      sprintbits(psds->pref2, bits);
      fprintf(outfile, "Prf2: %s\n", bits);
    }
    if(psds->bad_pws)
      fprintf(outfile, "Badp: %d\n", (int)psds->bad_pws);
    if(psds->conditions[0] && player.level < LVL_IMMORT)
      fprintf(outfile, "Hung: %d\n", (int)psds->conditions[0]);
    if(psds->conditions[1] && player.level < LVL_IMMORT)
      fprintf(outfile, "Thir: %d\n", (int)psds->conditions[1]);
    if(psds->conditions[2] && player.level < LVL_IMMORT)
      fprintf(outfile, "Drnk: %d\n", (int)psds->conditions[2]);
    if(psds->clan)
      fprintf(outfile, "Clan: %d\n", (int)psds->clan);
    if(psds->race)
      fprintf(outfile, "Race: %d\n", (int)psds->race);
    if(psds->spells_to_learn)
      fprintf(outfile, "Lern: %d\n", (int)psds->spells_to_learn);
    if(psds->pkill_points)
      fprintf(outfile, "Pkpt: %d\n", (int)psds->pkill_points);
    if(psds->purity)
      fprintf(outfile, "Pure: %d\n", (int)psds->purity);
    if(psds->quest)
      fprintf(outfile, "Qust: %d\n", (int)psds->quest);

    /* char_ability_data */
    cad = &(player.abilities);
    fprintf(outfile, "Str : %d/%d\n", cad->str, cad->str_add);
    fprintf(outfile, "Int : %d\n", cad->intel);
    fprintf(outfile, "Wis : %d\n", cad->wis);
    fprintf(outfile, "Dex : %d\n", cad->dex);
    fprintf(outfile, "Con : %d\n", cad->con);
    fprintf(outfile, "Cha : %d\n", cad->cha);

    /* char_point_data */
    cpd = &(player.points);
    fprintf(outfile, "Hit : %d/%d\n", cpd->hit, cpd->max_hit);
    fprintf(outfile, "Mana: %d/%d\n", cpd->mana, cpd->max_mana);
    fprintf(outfile, "Move: %d/%d\n", cpd->move, cpd->max_move);
    fprintf(outfile, "Ac  : %d\n", cpd->armor);
    if(cpd->gold)
      fprintf(outfile, "Gold: %d\n", cpd->gold);
    if(cpd->bank_gold)
      fprintf(outfile, "Bank: %d\n", cpd->bank_gold);
    if(cpd->exp)
      fprintf(outfile, "Exp : %d\n", cpd->exp);
    if(cpd->hitroll)
      fprintf(outfile, "Hrol: %d\n", cpd->hitroll);
    if(cpd->damroll)
      fprintf(outfile, "Drol: %d\n", cpd->damroll);
    if(ch->ignore)
      for (ignoring = ch->ignore; ignoring; ignoring = ignoring->next)
        fprintf(outfile, "Ign : %ld\n", ignoring->idnum);

    /* affected_type */
    fprintf(outfile, "Affs:\n");
    for(i = 0; i < MAX_AFFECT; i++) {
      aff = &(player.affected[i]);
      if(aff->type)
	fprintf(outfile, "%d %d %d %d %d %d\n", aff->type, aff->duration,
	  aff->modifier, aff->location, (int)aff->bitvector,
	  (int)aff->bitvector2);
    }
    fprintf(outfile, "0 0 0 0 0 0\n");

    fclose(outfile);
    chmod(outname, S_IREAD);
  }
  
  player_table[GET_IDNUM(ch)].id = GET_IDNUM(ch);
  player_table[GET_IDNUM(ch)].level = GET_LEVEL(ch);
  player_table[GET_IDNUM(ch)].race = GET_RACE(ch);
  player_table[GET_IDNUM(ch)].class = GET_CLASS(ch);
  player_table[GET_IDNUM(ch)].clan = GET_CLAN(ch);
  player_table[GET_IDNUM(ch)].last = player.last_logon;
  save_player_index();
}


/* copy data from the file structure to a char struct */
void store_to_char(struct char_file_u * st, struct char_data * ch)
{
  int i;

  /* to save memory, only PC's -- not MOB's -- have player_specials */
  if (ch->player_specials == NULL)
    CREATE(ch->player_specials, struct player_special_data, 1);

  GET_SEX(ch) = st->sex;
  GET_CLASS(ch) = st->class;
  GET_LEVEL(ch) = st->level;
  GET_REMORT(ch) = st->remort;


  ch->player.short_descr = NULL;
  ch->player.long_descr = NULL;
  ch->player.title = str_dup(st->title);
  ch->player.description = str_dup(st->description);

  ch->player.hometown = st->hometown;
  ch->player.time.birth = st->birth;
  ch->player.time.played = st->played;
  ch->player.time.logon = time(0);

  ch->player.weight = st->weight;
  ch->player.height = st->height;

  ch->player.email = st->email;
  ch->player.homepage = st->homepage;
  ch->player.pagesize = st->pagesize;
  ch->player.idleafk = st->idleafk;

  ch->real_abils = st->abilities;
  ch->aff_abils = st->abilities;
  ch->points = st->points;
  ch->char_specials.saved = st->char_specials_saved;
  ch->player_specials->saved = st->player_specials_saved;
  POOFIN(ch) = NULL;
  POOFOUT(ch) = NULL;

  if (ch->points.max_mana < 100)
    ch->points.max_mana = 100;

  ch->player.obj_buffer = NULL;      /*oeditmod*/
  ch->player.mob_buf = NULL;         /*meditmod*/
  ch->player.zone_edit = NOWHERE;    /*reditmod*/
  ch->player.default_zmob = NOBODY;  /*zeditmod*/
  ch->player.zone_flags = 0;         /*zeditmod*/
  ch->player.default_mob = -1;       /* .06 */
  ch->player.default_obj = -1;       /* .06 */
  ch->player.default_room = -1;      /* .06 */
  ch->player.room_edit = NOWHERE;    /*newolc*/

  ch->quitting = FALSE;

/*  ch->char_specials.carry_weight = 0;
  ch->char_specials.carry_items = 0;*/
  if IS_MONK(ch)
    ch->points.armor = (100 - (GET_LEVEL(ch) * 2));
  else
    ch->points.armor = 100;
  ch->points.hitroll = 0;
  ch->points.damroll = 0;

  if (ch->player.name == NULL)
    CREATE(ch->player.name, char, strlen(st->name) + 1);
  strcpy(ch->player.name, st->name);
  strcpy(ch->player.passwd, st->pwd);
  /* dwix: initialization */
  if (GET_RACE(ch) == RACE_VAMPIRE)
    GET_COND(ch, FULL) = -1;

/*  if (GET_LEVEL(ch)==LVL_DEMI)
    GET_MAX_HIT(ch) = 2000; */

  if (IS_MONK(ch)) {
    GET_HITROLL(ch) = (int) (GET_LEVEL(ch) / 2);
    GET_DAMROLL(ch) = (int) (GET_LEVEL(ch) / 2);
    }

  ch->riding = NULL; ch->ridden_by = NULL;

  GET_BINDTELL(ch) = NULL;
  GET_BINDHB(ch)   = NULL;
  GET_BINDWIMP(ch) = NULL;
  GET_BINDGT(ch)   = NULL;

  GET_LAST_MSG(ch) = NULL;
  GET_LAST_MSGER(ch) = NOBODY;

  GET_PROMPT_STRING(ch) = NULL;
  read_saves(ch);
  /* dwix: end */


  /* Add all spell effects */
  for (i = 0; i < MAX_AFFECT; i++) {
    if (st->affected[i].type)
      affect_to_char(ch, &st->affected[i]);
  }

  /*
   * If you're not poisioned and you've been away for more than an hour of
   * real time, we'll set your HMV back to full
   */

  if (!IS_AFFECTED(ch, AFF_POISON) &&
      (((long) (time(0) - st->last_logon)) >= SECS_PER_REAL_HOUR)) {
    GET_HIT(ch) = GET_MAX_HIT(ch);
    GET_MOVE(ch) = GET_MAX_MOVE(ch);
    GET_MANA(ch) = GET_MAX_MANA(ch);
  }
}				/* store_to_char */




/* copy vital data from a players char-structure to the file structure */
void char_to_store(struct char_data * ch, struct char_file_u * st)
{
  int i;
  struct affected_type *af;
  struct obj_data *char_eq[NUM_WEARS];

  /* Unaffect everything a character can be affected by */

  for (i = 0; i < NUM_WEARS; i++) {
    if (GET_EQ(ch, i))
      char_eq[i] = unequip_char(ch, i);
    else
      char_eq[i] = NULL;
  }

  for (af = ch->affected, i = 0; i < MAX_AFFECT; i++) {
    if (af) {
      st->affected[i] = *af;
      st->affected[i].next = 0;
      af = af->next;
    } else {
      st->affected[i].type = 0;	/* Zero signifies not used */
      st->affected[i].duration = 0;
      st->affected[i].modifier = 0;
      st->affected[i].location = 0;
      st->affected[i].bitvector = 0;
      st->affected[i].bitvector2 = 0;
      st->affected[i].next = 0;
    }
  }


  /*
   * remove the affections so that the raw values are stored; otherwise the
   * effects are doubled when the char logs back in.
   */

  while (ch->affected)
    affect_remove(ch, ch->affected);

  if ((i >= MAX_AFFECT) && af && af->next)
    nmlog("SYSERR: WARNING: OUT OF STORE ROOM FOR AFFECTED TYPES!!!");

  ch->aff_abils = ch->real_abils;

  st->birth = ch->player.time.birth;
  st->played = ch->player.time.played;
  st->played += (long) (time(0) - ch->player.time.logon);
  st->last_logon = time(0);

  ch->player.time.played = st->played;
  ch->player.time.logon = time(0);

  st->hometown = ch->player.hometown;
  st->weight = GET_WEIGHT(ch);
  st->height = GET_HEIGHT(ch);
  st->sex = GET_SEX(ch);
  st->class = GET_CLASS(ch);
  st->level = GET_LEVEL(ch);
  st->abilities = ch->real_abils;
  st->points = ch->points;
  st->char_specials_saved = ch->char_specials.saved;
  st->player_specials_saved = ch->player_specials->saved;
  st->email = ch->player.email;
  st->homepage = ch->player.homepage;
  st->pagesize = ch->player.pagesize;
  st->idleafk = ch->player.idleafk;
  st->remort = GET_REMORT(ch);

  if IS_MONK(ch)
    st->points.armor = (100 - (GET_LEVEL(ch) * 2));
  else
    st->points.armor = 100;
  st->points.hitroll = 0;
  st->points.damroll = 0;

  if (GET_TITLE(ch))
    strcpy(st->title, GET_TITLE(ch));
  else
    *st->title = '\0';

  if (ch->player.description)
    strcpy(st->description, ch->player.description);
  else
    *st->description = '\0';

  strcpy(st->name, GET_NAME(ch));

  /* add spell and eq affections back in now */
  for (i = 0; i < MAX_AFFECT; i++) {
    if (st->affected[i].type)
      affect_to_char(ch, &st->affected[i]);
  }

  for (i = 0; i < NUM_WEARS; i++) {
    if (char_eq[i])
      equip_char(ch, char_eq[i], i);
  }
/*   affect_total(ch); unnecessary, I think !?! */
}				/* Char to store */



void save_etext(struct char_data * ch)
{
/* this will be really cool soon */

}


/* create a new entry in the in-memory index table for the player file */
int create_entry(char *name)
{
  int i, j;

  if (top_of_p_table == -1) {
    CREATE(player_table, struct player_index_element, 1);
    top_of_p_table = 0;
  } else {
    for(i = 0; i <= top_of_p_table; i++) {
      if (!player_table[i].name)
        continue;
      if(!str_cmp(name, player_table[i].name)) {
	  return (i);
      }
    }
  }
  for (i = 1; i <= top_of_p_table; i++)
    if (player_table[i].id == 0) {
      CREATE(player_table[i].name, char, strlen(name) + 1);
      for (j = 0; (*(player_table[i].name + j) = LOWER(*(name + j))); j++);
      return(i);
    }
  ++top_of_p_table;
  CREATE(player_table[top_of_p_table].name, char, strlen(name) + 1);

  /* copy lowercase equivalent of name to table field */
  for (i = 0; (*(player_table[top_of_p_table].name + i) = LOWER(*(name + i)));
       i++);

  return (top_of_p_table);
}



/************************************************************************
*  funcs of a (more or less) general utility nature			*
********************************************************************** */


/* read and allocate space for a '~'-terminated string from a given file */
char *fread_string(FILE * fl, char *error)
{
  char buf[MAX_STRING_LENGTH], tmp[512], *rslt;
  register char *point;
  int done = 0, length = 0, templength = 0;

  *buf = '\0';

  do {
    if (!fgets(tmp, 512, fl)) {
      fprintf(stderr, "SYSERR: fread_string: format error at or near %s\n",
	      error);
  system("touch ../.killscript");
    exit(1);
    }
    /* If there is a '~', end the string; else put an "\r\n" over the '\n'. */
    if ((point = strchr(tmp, '~')) != NULL) {
      *point = '\0';
      done = 1;
    } else {
      point = tmp + strlen(tmp) - 1;
      *(point++) = '\r';
      *(point++) = '\n';
      *point = '\0';
    }

    templength = strlen(tmp);

    if (length + templength >= MAX_STRING_LENGTH) {
      nmlog("SYSERR: fread_string: string too large (db.c)");
      nmlog(error);
    system("touch ../.killscript");
    exit(1);
    } else {
      strcat(buf + length, tmp);
      length += templength;
    }
  } while (!done);

  /* allocate space for the new string and copy it */
  if (strlen(buf) > 0) {
    CREATE(rslt, char, length + 1);
    strcpy(rslt, buf);
  } else
    rslt = NULL;

  return rslt;
}


/* release memory allocated for a char struct */
void free_char(struct char_data * ch)
{
  int i;
  struct alias *a;

  void free_alias(struct alias * a);

  while ((a = GET_ALIASES(ch)) != NULL) {
    GET_ALIASES(ch) = (GET_ALIASES(ch))->next;
    free_alias(a);
  }

  if (ch->player_specials != NULL && ch->player_specials != &dummy_mob) {
    if (POOFIN(ch))
      free(POOFIN(ch));
    if (POOFOUT(ch))
      free(POOFOUT(ch));
    if (ch->player.email)
      free(ch->player.email);
    if (ch->player.homepage)
      free(ch->player.homepage);

    /* dwix: cleaning up my stuff - clean up */
    if (GET_LAST_MSG(ch))
      free(GET_LAST_MSG(ch));

    if (GET_BINDHB(ch))
      free(GET_BINDHB(ch));
    if (GET_BINDTELL(ch))
      free(GET_BINDTELL(ch));
    if (GET_BINDWIMP(ch))
      free(GET_BINDWIMP(ch));
    if (GET_BINDGT(ch))
      free(GET_BINDGT(ch));

    if (GET_PROMPT_STRING(ch))
      free(GET_PROMPT_STRING(ch));
    /* dwix: end */
    free(ch->player_specials);
    if (IS_NPC(ch))
      nmlog("SYSERR: Mob had player_specials allocated!");
  }
  if (!IS_NPC(ch) || (IS_NPC(ch) && GET_MOB_RNUM(ch) == -1)) {
    /* if this is a player, or a non-prototyped non-player, free all */
    if (GET_NAME(ch))
      free(GET_NAME(ch));
    if (ch->player.title)
      free(ch->player.title);
    if (ch->player.short_descr)
      free(ch->player.short_descr);
    if (ch->player.long_descr)
      free(ch->player.long_descr);
    if (ch->player.description)
      free(ch->player.description);
  } else if ((i = GET_MOB_RNUM(ch)) > -1) {
    /* otherwise, free strings only if the string is not pointing at proto */
    if (ch->player.name && ch->player.name != mob_proto[i].player.name)
      free(ch->player.name);
    if (ch->player.title && ch->player.title != mob_proto[i].player.title)
      free(ch->player.title);
    if (ch->player.short_descr && ch->player.short_descr != mob_proto[i].player.short_descr)
      free(ch->player.short_descr);
    if (ch->player.long_descr && ch->player.long_descr != mob_proto[i].player.long_descr)
      free(ch->player.long_descr);
    if (ch->player.description && ch->player.description != mob_proto[i].player.description)
      free(ch->player.description);
  }
  while (ch->affected)
    affect_remove(ch, ch->affected);

  free(ch);
}




/* release memory allocated for an obj struct */
void free_obj(struct obj_data * obj)
{
  int nr;
  struct extra_descr_data *this, *next_one;

  if ((nr = GET_OBJ_RNUM(obj)) < 0) {
    if (obj->name)
      free(obj->name);
    if (obj->description)
      free(obj->description);
    if (obj->short_description)
      free(obj->short_description);
    if (obj->action_description)
      free(obj->action_description);
    if (obj->ex_description)
      for (this = obj->ex_description; this; this = next_one) {
	next_one = this->next;
	if (this->keyword)
	  free(this->keyword);
	if (this->description)
	  free(this->description);
	free(this);
      }
  } else {
    if (obj->name && obj->name != obj_proto[nr].name)
      free(obj->name);
    if (obj->description && obj->description != obj_proto[nr].description)
      free(obj->description);
    if (obj->short_description && obj->short_description != obj_proto[nr].short_description)
      free(obj->short_description);
    if (obj->action_description && obj->action_description != obj_proto[nr].action_description)
      free(obj->action_description);
    if (obj->ex_description && obj->ex_description != obj_proto[nr].ex_description)
      for (this = obj->ex_description; this; this = next_one) {
	next_one = this->next;
	if (this->keyword)
	  free(this->keyword);
	if (this->description)
	  free(this->description);
	free(this);
      }
  }

  free(obj);
}



/* read contets of a text file, alloc space, point buf to it */
int file_to_string_alloc(char *name, char **buf)
{
  char temp[MAX_STRING_LENGTH];

  if (*buf)
    free(*buf);

  if (file_to_string(name, temp) < 0) {
    *buf = "";
    return -1;
  } else {
    *buf = str_dup(temp);
    return 0;
  }
}


/* read contents of a text file, and place in buf */
int file_to_string(char *name, char *buf)
{
  FILE *fl;
  char tmp[READ_SIZE+3];
 
  *buf = '\0';

  if (!(fl = fopen(name, "r"))) {
    sprintf(tmp, "Error reading %s", name);
    perror(tmp);
    return (-1);
  }
  do {
    fgets(tmp, READ_SIZE, fl);
    tmp[strlen(tmp) - 1] = '\0'; /* take off the trailing \n */
    strcat(tmp, "\r\n");

    if (!feof(fl)) {
      if (strlen(buf) + strlen(tmp) + 1 > MAX_STRING_LENGTH) {
        sprintf(buf, "SYSERR: %s: string too big (%d max)", name,
		MAX_STRING_LENGTH);
	nmlog(buf);
	*buf = '\0';
	return -1;
      }
      strcat(buf, tmp);
    }
  } while (!feof(fl));

  fclose(fl);

  return (0);
}



/* clear some of the the working variables of a char */
void reset_char(struct char_data * ch)
{
  int i;

  for (i = 0; i < NUM_WEARS; i++)
    GET_EQ(ch, i) = NULL;

  ch->followers = NULL;
  ch->master = NULL;
  ch->in_room = NOWHERE;
  ch->carrying = NULL;
  ch->next = NULL;
  ch->next_fighting = NULL;
  ch->next_in_room = NULL;
  FIGHTING(ch) = NULL;
  ch->char_specials.position = POS_STANDING;
  ch->mob_specials.default_pos = POS_STANDING;
  ch->char_specials.carry_weight = 0;
  ch->char_specials.carry_items = 0;

  if (GET_HIT(ch) <= 0)
    GET_HIT(ch) = 1;
  if (GET_MOVE(ch) <= 0)
    GET_MOVE(ch) = 1;
  if (GET_MANA(ch) <= 0)
    GET_MANA(ch) = 1;

  GET_LAST_TELL(ch) = NOBODY;
}



/* clear ALL the working variables of a char; do NOT free any space alloc'ed */
void clear_char(struct char_data * ch)
{
  memset((char *) ch, 0, sizeof(struct char_data));

  ch->in_room = NOWHERE;
  GET_PFILEPOS(ch) = -1;
  GET_WAS_IN(ch) = NOWHERE;
  GET_POS(ch) = POS_STANDING;
  ch->mob_specials.default_pos = POS_STANDING;

  GET_AC(ch) = 100;		/* Basic Armor */
  if (ch->points.max_mana < 100)
    ch->points.max_mana = 100;
}


void clear_object(struct obj_data * obj)
{
  memset((char *) obj, 0, sizeof(struct obj_data));

  obj->item_number = NOTHING;
  obj->in_room = NOWHERE;
}




/* initialize a new character only if class is set */
void init_char(struct char_data * ch)
{
  int i, newid;

  /* create a player_special structure */
  if (ch->player_specials == NULL)
    CREATE(ch->player_specials, struct player_special_data, 1);

  /* *** if this is our first player --- he be God *** */

  if (top_of_p_table == 0) {
    GET_EXP(ch) = 1;
    GET_LEVEL(ch) = LVL_SIMP;
    SET_BIT(PRF_FLAGS(ch), PRF_NOHIT);
    SET_BIT(PRF_FLAGS(ch), PRF_HOLYLIGHT);

    ch->points.max_hit = 500;
    ch->points.max_mana = 100;
    ch->points.max_move = 82;
  }
  set_title(ch, NULL);

  ch->player.short_descr = NULL;
  ch->player.long_descr = NULL;
  ch->player.description = NULL;
  ch->player.zone_edit = NOWHERE;   /*reditmod*/
  ch->player.room_edit = NOWHERE;
  ch->player.hometown = 1;
  ch->player.pagesize = 24;
  ch->player.idleafk = 10;

  ch->player.time.birth = time(0);
  ch->player.time.played = 0;
  ch->player.time.logon = time(0);
  GET_REMORT(ch) = 0;

  for (i = 0; i < MAX_TONGUE; i++)
    GET_TALK(ch, i) = 0;

  /* make favors for sex */
  if (ch->player.sex == SEX_MALE) {
    switch (GET_RACE(ch)) {
      case RACE_HUMAN: 
       ch->player.weight = number(120, 180);
       ch->player.height = number(160, 200);
       break;
      case RACE_ELF:
       ch->player.weight = number(90, 120);
       ch->player.height = number(55, 65);
       break;
     case RACE_DWARF:
       ch->player.weight = number(135, 170);
       ch->player.height = number(43, 53);
       break;
     case RACE_HOBBIT:
       ch->player.weight = number(57, 72);
       ch->player.height = number(32, 48);
       break;
     }
  } else {
    switch (GET_RACE(ch)) {
      case RACE_HUMAN: 
       ch->player.weight = number(80, 140);
       ch->player.height = number(160, 120);
       break;
      case RACE_ELF:
       ch->player.weight = number(70, 100);
       ch->player.height = number(50, 60);
       break;
     case RACE_DWARF:
       ch->player.weight = number(110, 145);
       ch->player.height = number(41, 51);
       break;
     case RACE_HOBBIT:
       ch->player.weight = number(53, 68);
       ch->player.height = number(30, 46);
       break;
     }
  }

  ch->points.max_mana = 100;
  ch->points.mana = GET_MAX_MANA(ch);
  ch->points.hit = GET_MAX_HIT(ch);
  ch->points.max_move = 82;
  ch->points.move = GET_MAX_MOVE(ch);
  ch->points.armor = 100;


  GET_IDNUM(ch) = 0;
  for (newid = 1; newid < top_of_p_table; newid++) {
    if (player_table[newid].id == 0) {
      GET_PFILEPOS(ch) = player_table[newid].id = GET_IDNUM(ch) = newid;
      break;
    }
  }
    
  if (GET_IDNUM(ch) == 0) {
    top_of_p_table++;
    player_table[top_of_p_table].id = GET_IDNUM(ch) = top_of_p_table;
  }


  for (i = 1; i <= MAX_SKILLS; i++) {
    if (GET_LEVEL(ch) < LVL_IMMORT)
      SET_SKILL(ch, i, 0)
    else
      SET_SKILL(ch, i, 100);
  }

  ch->char_specials.saved.affected_by = 0;

  for (i = 0; i < 5; i++)
    GET_SAVE(ch, i) = 0;

  for (i = 0; i < 3; i++)
    GET_COND(ch, i) = (GET_LEVEL(ch) == LVL_SIMP ? -1 : 24);

  GET_LOADROOM(ch) = NOWHERE;
}



/* returns the real number of the object with given virtual number */
/* reditmod - changed binary search to incremental                 */
/* -JEE- 0.4 - changed to binary & incremental                     */
/* thanks to Johan Eenfeldt for the idea and the code :)  *.06*    */
int real_room(int virtualrm)
{
  int bot, top, mid, nr, found = NOWHERE;

  if(room_sbuf == NOWHERE) room_sbuf = 0;

  /* First binary search. */
  bot = 0;
  top = top_of_world - new_rooms;

  if((world + room_sbuf)->number == virtualrm)
    return room_sbuf;

  for (;;) {
    mid = (bot + top) >> 1;

    if ((world + mid)->number == virtualrm) {
      room_sbuf = mid;
      return mid;
    }
    if (bot >= top)
      break;
    if ((world + mid)->number > virtualrm)
      top = mid - 1;
    else
      bot = mid + 1;
  }

  /* If not found - use linear on the "new" parts. */
  for(nr = top_of_world - new_rooms; nr <= top_of_world; nr++) {
    if(world[nr].number == virtualrm) {
      found = nr;
      break;
    }
  }
  return(found);
}
 
/* meditmod - changed binary search to incremental                 */
/* -JEE- 0.4 - changed to binary & incremental                     */
/* thanks to Johan Eenfeldt for the idea and the code :)  *.06*    */
int real_mobile(int virtualrm)
{
  int bot, top, mid, nr, found = NOWHERE;
 
  /* First binary search. */
  bot = 0;
  top = top_of_mobt - new_mobs;
 
  if((mob_index + mob_sbuf)->virtualrm == virtualrm)
    return mob_sbuf;

  for (;;) {
    mid = (bot + top) >> 1;
 
    if ((mob_index + mid)->virtualrm == virtualrm) {
      mob_sbuf = mid;
      return mid;
    }
    if (bot >= top)
      break;
    if ((mob_index + mid)->virtualrm > virtualrm)
      top = mid - 1;
    else
      bot = mid + 1;
  }
 
  /* If not found - use linear on the "new" parts. */
  for(nr = top_of_mobt - new_mobs; nr <= top_of_mobt; nr++) {
    if(mob_index[nr].virtualrm == virtualrm) {
      found = nr;
      break;
    }
  }
  return(found);
}
 
/* oeditmod - changed binary search to incremental                 */
/* -JEE- 0.4 - changed to binary & incremental                     */
/* thanks to Johan Eenfeldt for the idea and the code :)  *.06*    */
int real_object(int virtualrm)
{
  int bot, top, mid, nr, found = NOWHERE;
 
  /* First binary search. */
  bot = 0;
  top = top_of_objt - new_objects;
 
  if((obj_index + obj_sbuf)->virtualrm == virtualrm)
    return obj_sbuf;

  for (;;) {
    mid = (bot + top) >> 1;
 
    if ((obj_index + mid)->virtualrm == virtualrm) {
      obj_sbuf = mid;
      return mid;
    }
    if (bot >= top)
      break;
    if ((obj_index + mid)->virtualrm > virtualrm)
      top = mid - 1;
    else
      bot = mid + 1;
  }
 
  /* If not found - use linear on the "new" parts. */
  for(nr = top_of_objt - new_objects; nr <= top_of_objt; nr++) {
    if(obj_index[nr].virtualrm == virtualrm) {
      found = nr;
      break;
    }
  }
  return(found);
}

/* the functions */
 
/* This routine transfers between alpha and numeric forms of the
 *  mob_prog bitvector types. This allows the use of the words in the
 *  mob/script files.
 */

 
/*
 * Read a number from a file.
 */
int fread_number(FILE *fp)
{
    int number;
    bool sign;
    char c;

    do {
        c = fgetc(fp);
    } while (isspace(c));

    number = 0;

    sign   = FALSE;
    if (c == '+') {
        c = fgetc(fp);
    } else if (c == '-') {
        sign = TRUE;
        c = fgetc(fp);
    }

    if (!isdigit(c)) {
        nmlog("Fread_number: bad format.");
    system("touch ../.killscript");
        exit(1);
    }

    while (isdigit(c)) {
        number = number * 10 + c - '0';
        c      = fgetc(fp);
    }

    if (sign)
        number = 0 - number;

    if (c == '|')
        number += fread_number(fp);
    else if (c != ' ')
        ungetc(c, fp);

    return number;
}
  
/*
 * Read to end of line (for comments).
 */
void fread_to_eol(FILE *fp)
{
    char c;

    do {
        c = fgetc(fp);
    } while (c != '\n' && c != '\r');

    do {
        c = fgetc(fp);
    } while (c == '\n' || c == '\r');

    ungetc(c, fp);
    return;
}
  
/*
 * Read one word (into static buffer).
 */
char *fread_word(FILE *fp)
{
    static char word[MAX_INPUT_LENGTH];
    char *pword;
    char cEnd;

    do
    {
        cEnd = fgetc(fp);
    }
    while (isspace(cEnd));

    if (cEnd == '\'' || cEnd == '"')
    {
        pword   = word;
    }
    else
    {
        word[0] = cEnd;
        pword   = word+1;
        cEnd    = ' ';
    }

    for (; pword < word + MAX_INPUT_LENGTH; pword++)
    {
        *pword = fgetc(fp);
        if (cEnd == ' ' ? isspace(*pword) || *pword == '~' : *pword == cEnd)
        {
            if (cEnd == ' ' || cEnd == '~')
                ungetc(*pword, fp);
            *pword = '\0';
            return word;
        }
    }

    nmlog("SYSERR: Fread_word: word too long.");
    system("touch ../.killscript");
    exit(1);
    return NULL;
}
  
  
  
struct index_data *get_obj_index (int vnum)
{
  int nr;
  for(nr = 0; nr <= top_of_objt; nr++) {
    if(obj_index[nr].virtualrm == vnum) return &obj_index[nr];
  }
  return NULL;
}
  
struct index_data *get_mob_index (int vnum)
{
  int nr;
  for(nr = 0; nr <= top_of_mobt; nr++) {
    if(mob_index[nr].virtualrm == vnum) return &mob_index[nr];
  }
  return NULL;
}
  


int virtual_room(int real_room)
{
if (real_room < 0) return(-1);
if (real_room > top_of_world) return(-1);

return(world[real_room].number);
}



void tag_argument(char *argument, char *tag)
{
  char *tmp = argument, *ttag = tag, *wrt = argument;
  int i;

  for(i = 0; i < 4; i++)
    *(ttag++) = *(tmp++);
  *ttag = '\0';
  
  while(*tmp == ':' || *tmp == ' ')
    tmp++;

  while(*tmp)
    *(wrt++) = *(tmp++);
  *wrt = '\0';
}


void save_player_index(void)
{
  int i;
  FILE *index_file;

  if(!(index_file = fopen(PLR_INDEX_FILE, "w"))) {
    nmlog("SYSERR:  Could not write player index file");
    return;
  }

  for(i = 1; i <= top_of_p_table; i++)
    if(player_table[i].name) {
      fprintf(index_file, "%s %d %d %d %d %d %d\n", player_table[i].name,
        (int)player_table[i].id, player_table[i].level,
        player_table[i].race, player_table[i].class,
        player_table[i].clan, (int) player_table[i].last);
    }
  fclose(index_file);
}


ACMD(do_thing)
{
  int i, j;
  FILE *fl;
  struct char_file_u player;
  struct char_data *p;
  char arg[256];
  long timeout = 1000;

  one_argument(argument, arg); 

  if(is_abbrev(arg, "cleanup")) {
    send_to_char("ok\r\n", ch);
    for(i = 0; i < top_of_p_table; i ++) {
      if (!player_table[i].name)
        continue;
      for(j = i + 1; j <= top_of_p_table; j ++) {
        if (!player_table[j].name)
          continue;
	if(!strcmp(player_table[i].name, player_table[j].name)) {
	  *player_table[i].name = '\0';
	}
      }
    }
    return;
  }

  if(is_abbrev(arg, "pclean")) {
    send_to_char("ok\r\n", ch);
    for(i = 0; i < top_of_p_table; i++) {
      if (!player_table[i].name)
        continue;
      memset((char *) &player, 0, sizeof(struct char_file_u));
      if((load_char(player_table[i].name, &player, NULL)) < 0)
	continue;
      if(player.level < 1)
	timeout = 0;
      else if(player.level == 1)
	timeout = 4;
      else if(player.level <= 10)
	timeout = 21;
      else if(player.level <= 20)
	timeout = 31;
      else
	timeout = 90;

      timeout *= SECS_PER_REAL_DAY;
      if((((time(0) - player.last_logon) > timeout) || 
	(player.char_specials_saved.act & PLR_DELETED)) &&
	!(player.char_specials_saved.act & PLR_NODELETE)) {
	remove_player(player.char_specials_saved.idnum, TRUE);
	sprintf(buf, "PCLEAN: %s%s Lev:%d Last:%s", 
	  (player.char_specials_saved.act & PLR_DELETED) ? "(PLR_DELE)"
	  : "", player.name, player.level,
	  asctime(localtime(&player.last_logon)));
	nmlog(buf);
      }
    }
    save_player_index();
    return;
  }

  fl = fopen("pfiles/plr_index2", "w");

  CREATE(p, struct char_data, 1);

  for(i = 0; i <= top_of_p_table; i++) {
    memset((char *) &player, 0, sizeof(struct char_file_u));
    if (!player_table[i].name)
      continue;
    load_char(player_table[i].name, &player, NULL);
    clear_char(p);
    store_to_char(&player, p);
    sprintf(buf, "%s %d %d %d %d %d %d\n", player_table[i].name,
	(int) GET_IDNUM(p), GET_LEVEL(p),
	GET_RACE(p), GET_CLASS(p),
	GET_CLAN(p), (int) player.last_logon);
    fprintf(fl, buf);
  }
  fclose(fl);
}


/* dnsmod */
void boot_dns(void)
{
  int i = 0;
  char line[256], name[256];
  FILE *fl;
  struct dns_entry *dns;

  memset((char *) dns_cache, 0, sizeof(struct dns_entry *) * DNS_HASH_NUM);

  if(!(fl = fopen(DNS_FILE, "r"))) {
    nmlog("No DNS cache!");
    return;
  }

  do {
    dns_size++;
    i = 0;
    get_line(fl, line);
    if(*line != '~') {
      CREATE(dns, struct dns_entry, 1);
      dns->name = NULL;
      dns->next = NULL;
      sscanf(line, "%d.%d.%d.%d %s", dns->ip, dns->ip + 1,
	dns->ip + 2, dns->ip + 3, name);
      dns->name = str_dup(name);
      i = (dns->ip[0] * dns->ip[1]) % DNS_HASH_NUM;
      dns->next = dns_cache[i];
      dns_cache[i] = dns;
    }
  } while (!feof(fl) && *line != '~');
  fclose(fl);
}


/* dnsmod */
void save_dns_cache(void)
{
  int i;
  FILE *fl;
  struct dns_entry *dns;

  if(!(fl = fopen(DNS_FILE, "w"))) {
    nmlog("SYSERR: Can't open dns cache file for write!");
    return;
  }

  for(i = 0; i < DNS_HASH_NUM; i++) {
    if(dns_cache[i]) {
      for(dns = dns_cache[i]; dns; dns = dns->next) {
	if(dns->ip[0] != -1) {
	  fprintf(fl, "%d.%d.%d.%d %s\n", dns->ip[0], dns->ip[1],
	    dns->ip[2], dns->ip[3], dns->name);
	}
      }
    }
  }
  fprintf(fl, "~\n");
  fclose(fl);
}


/* dnsmod */
int get_host_from_cache(struct dns_entry *dnsd)
{
  int i;
  struct dns_entry *d;
  char buf[256];

  i = (dnsd->ip[0] * dnsd->ip[1]) % DNS_HASH_NUM;
  if(dns_cache[i]) {
    for(d = dns_cache[i]; d; d = d->next) {
      if(dnsd->ip[0] == d->ip[0] && dnsd->ip[1] == d->ip[1]) {
	if(d->ip[2] == -1) {
	  sprintf(buf, "%d-%d.%s", dnsd->ip[2], dnsd->ip[3], d->name);
	  dnsd->name = str_dup(buf);
	  return TRUE;
	} else if(dnsd->ip[2] == d->ip[2] && d->ip[3] == -1) {
	  sprintf(buf, "%d.%s", dnsd->ip[3], d->name);
	  dnsd->name = str_dup(buf);
	  return TRUE;
	} else if(dnsd->ip[2] == d->ip[2] && dnsd->ip[3] == d->ip[3]) {
	  dnsd->name = str_dup(d->name);
	  return TRUE;
	}
      }
    }
  }
  return FALSE;
}


/* dnsmod */
void add_dns_host(struct dns_entry *dnsd, char *hostname)
{
  int i;
  struct dns_entry *d;

  i = (dnsd->ip[0] + dnsd->ip[1] + dnsd->ip[2]) % DNS_HASH_NUM;
  CREATE(d, struct dns_entry, 1);
  d->ip[0] = dnsd->ip[0];
  d->ip[1] = dnsd->ip[1];
  d->ip[2] = dnsd->ip[2];
  d->ip[3] = dnsd->ip[3];
  d->name = str_dup(hostname);
  d->next = dns_cache[i];
  dns_cache[i] = d;
  save_dns_cache();
}


void remove_player(long id, byte stomp)
{

  if (!id || id >= MAXENTRIES)
    return;

  if(!player_table[(int) id].name) {
    return;
  }

  sprintf(buf, "%s/%c/%s", PLR_PREFIX, *player_table[id].name,
    player_table[id].name);
  chmod(buf, S_IREAD | S_IWRITE);
  unlink(buf);

  sprintf(buf, "%s/%c/%s", RENT_PREFIX, *player_table[id].name,
    player_table[id].name);
  chmod(buf, S_IREAD | S_IWRITE);
  unlink(buf);

  get_filename(player_table[id].name, buf, SAVES_FILE);
  unlink(buf);

  sprintf(buf, "%s/%c/%s", MAIL_PREFIX, *player_table[id].name,
    player_table[id].name);
  chmod(buf, S_IREAD | S_IWRITE);
  unlink(buf);

  if(stomp) {
    *player_table[id].name = '\0';
    player_table[id].id = 0;
  }
}


void clean_pfiles(void)
{
  int i, timeout;
  struct char_file_u plr;

  for(i = 0; i < top_of_p_table; i++) {
      if (!player_table[i].name)
        continue;
      if(player_table[i].level < 1)
	timeout = 0;
      /*else if(player_table[i].level == 1)
	timeout = 4;
      else if(player_table[i].level <= 10)
	timeout = 21;
      else if(player_table[i].level <= 20)
	timeout = 31;*/
      else
	timeout = 90;

      timeout *= SECS_PER_REAL_DAY;
      if((time(0) - player_table[i].last) > timeout) {
        memset((char *) &plr, 0, sizeof(struct char_file_u));
        if((load_char(player_table[i].name, &plr, NULL)) >= 0)
	  /*if(IS_SET(plr.char_specials_saved.act, PLR_NODELETE)) {
	    sprintf(buf, "PCLEAN: (PLR_NODEL) %s skipped",
	      player_table[i].name);
	    nmlog(buf);
	    continue;
	  }*/
	remove_player(plr.char_specials_saved.idnum, TRUE);
	if(player_table[i].id > 0) {
	  sprintf(buf, "PCLEAN: %s Lev:%d Last:%s", 
	    plr.name ? plr.name : "(null)", plr.level,
	    asctime(localtime(&plr.last_logon)));
	  nmlog(buf);
	}
      }
  }
  save_player_index();
}

void strip_ems(char *str)
{
  char *ptr1, *ptr2, *tmp;
  
  tmp = str;
  ptr1 = str;
  ptr2 = str;
  
  while(*ptr1) {
    if((*(ptr2++) = *(ptr1++)) == 13)
      if(*ptr1 == 13)
        ptr1++;
  }
  *ptr2 = '\0';
}
