/* ************************************************************************
*   File: fight.c                                       Part of CircleMUD *
*  Usage: Combat system                                                   *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"

#ifndef __CHECKER__
#include <assert.h>
#else
#include "assert.h"
#endif

#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "handler.h"
#include "interpreter.h"
#include "db.h"
#include "spells.h"
#include "screen.h"
#include "quest.h"

/* Structures */
struct char_data *combat_list = NULL;	/* head of l-list of fighting chars */
struct char_data *next_combat_list = NULL;

/* External structures */
extern struct room_data *world;
extern struct message_list fight_messages[MAX_MESSAGES];
extern struct obj_data *object_list;
extern int pk_allowed;		/* see config.c */
extern int auto_save;		/* see config.c */
extern int max_exp_gain;	/* see config.c */
extern sh_int r_mortal_start_room;
extern struct clan_stuff clans[];
extern struct index_data *mob_index;
extern struct index_data *obj_index;
extern struct room_data *room_combat_damage;
extern struct room_data *room_tick_damage;

/* External procedures */
char *fread_action(FILE * fl, int nr);
char *fread_string(FILE * fl, char *error);
void stop_follower(struct char_data * ch);
ACMD(do_flee);
void hit(struct char_data * ch, struct char_data * victim, int type);
void forget(struct char_data * ch, struct char_data * victim);
void remember(struct char_data * ch, struct char_data * victim);
int ok_damage_shopkeeper(struct char_data * ch, struct char_data * victim);
int mag_savingthrow(struct char_data * ch, int type, int bonus);
void fury_sword(struct char_data *ch);

/* Weapon attack texts */
struct attack_hit_type attack_hit_text[] =
{
  {"hit", "hits"},		/* 0 */
  {"sting", "stings"},
  {"whip", "whips"},
  {"slash", "slashes"},
  {"bite", "bites"},
  {"bludgeon", "bludgeons"},	/* 5 */
  {"crush", "crushes"},
  {"pound", "pounds"},
  {"claw", "claws"},
  {"maul", "mauls"},
  {"thrash", "thrashes"},	/* 10 */
  {"pierce", "pierces"},
  {"blast", "blasts"},
  {"punch", "punches"},
  {"stab", "stabs"}
};

#define IS_WEAPON(type) (((type) >= TYPE_HIT) && ((type) < TYPE_SUFFERING))

/* The Fight related routines */

void appear(struct char_data * ch)
{
  if (affected_by_spell(ch, SPELL_INVISIBLE))
    affect_from_char(ch, SPELL_INVISIBLE);

  REMOVE_BIT(AFF_FLAGS(ch), AFF_INVISIBLE | AFF_HIDE);

  if (GET_LEVEL(ch) < LVL_IMMORT)
    act("$n slowly fades into existence.", FALSE, ch, 0, 0, TO_ROOM);
  else
    act("You feel a strange presence as $n appears, seemingly from nowhere.",
	FALSE, ch, 0, 0, TO_ROOM);
}



void load_messages(void)
{
  FILE *fl;
  int i, type;
  struct message_type *messages;
  char chk[128];

  if (!(fl = fopen(MESS_FILE, "r"))) {
    sprintf(buf2, "Error reading combat message file %s", MESS_FILE);
    perror(buf2);
    exit(1);
  }
  for (i = 0; i < MAX_MESSAGES; i++) {
    fight_messages[i].a_type = 0;
    fight_messages[i].number_of_attacks = 0;
    fight_messages[i].msg = 0;
  }


  fgets(chk, 128, fl);
  while (!feof(fl) && (*chk == '\n' || *chk == '*'))
    fgets(chk, 128, fl);

  while (*chk == 'M') {
    fgets(chk, 128, fl);
    sscanf(chk, " %d\n", &type);
    for (i = 0; (i < MAX_MESSAGES) && (fight_messages[i].a_type != type) &&
	 (fight_messages[i].a_type); i++);
    if (i >= MAX_MESSAGES) {
      fprintf(stderr, "Too many combat messages.  Increase MAX_MESSAGES and recompile.");
      exit(1);
    }
    CREATE(messages, struct message_type, 1);
    fight_messages[i].number_of_attacks++;
    fight_messages[i].a_type = type;
    messages->next = fight_messages[i].msg;
    fight_messages[i].msg = messages;

    messages->die_msg.attacker_msg = fread_action(fl, i);
    messages->die_msg.victim_msg = fread_action(fl, i);
    messages->die_msg.room_msg = fread_action(fl, i);
    messages->miss_msg.attacker_msg = fread_action(fl, i);
    messages->miss_msg.victim_msg = fread_action(fl, i);
    messages->miss_msg.room_msg = fread_action(fl, i);
    messages->hit_msg.attacker_msg = fread_action(fl, i);
    messages->hit_msg.victim_msg = fread_action(fl, i);
    messages->hit_msg.room_msg = fread_action(fl, i);
    messages->god_msg.attacker_msg = fread_action(fl, i);
    messages->god_msg.victim_msg = fread_action(fl, i);
    messages->god_msg.room_msg = fread_action(fl, i);
    fgets(chk, 128, fl);
    while (!feof(fl) && (*chk == '\n' || *chk == '*'))
      fgets(chk, 128, fl);
  }

  fclose(fl);
}


void update_pos(struct char_data * victim)
{
  if(!victim) {
    nmlog("SYSERR: attempting to update_pos null victim");
    return;
  }

  if ((GET_HIT(victim) > 0) && (GET_POS(victim) > POS_STUNNED))
    return;
  else if (GET_HIT(victim) > 0)
    GET_POS(victim) = POS_STANDING;
  else if (GET_HIT(victim) <= -11)
    GET_POS(victim) = POS_DEAD;
  else if (GET_HIT(victim) <= -6)
    GET_POS(victim) = POS_MORTALLYW;
  else if (GET_HIT(victim) <= -3)
    GET_POS(victim) = POS_INCAP;
  else
    GET_POS(victim) = POS_STUNNED;
}


void check_killer(struct char_data * ch, struct char_data * vict)
{
  if ( (!PLR_FLAGGED(vict, PLR_KILLER) && !PLR_FLAGGED(vict, PLR_THIEF)
      && !PLR_FLAGGED(ch, PLR_KILLER) && !IS_NPC(ch) && !IS_NPC(vict) &&
      (ch != vict) && GET_LEVEL(ch) < LVL_IMMORT)
       && !ROOM_FLAGGED(vict->in_room, ROOM_CRIMEOK) ) {
    char buf[256];
    SET_BIT(PLR_FLAGS(ch), PLR_KILLER);
    sprintf(buf, "PC Killer bit set on %s for initiating attack on %s at %s.",
	    GET_NAME(ch), GET_NAME(vict), world[vict->in_room].name);
    mudlog(buf, BRF, LVL_IMMORT, TRUE);
    send_to_char("If you want to be a PLAYER KILLER, so be it...\r\n", ch);
  }
}

/* When ch kills victim */
void change_alignment(struct char_data * ch, struct char_data * victim)
{
  /*
   * new alignment change algorithm: if you kill a monster with alignment A,
   * you move 1/16th of the way to having alignment -A.  Simple and fast.
   */
  GET_ALIGNMENT(ch) += (-GET_ALIGNMENT(victim) - GET_ALIGNMENT(ch)) >> 4;
}

/* start one char fighting another (yes, it is horrible, I know... )  */
void set_fighting(struct char_data * ch, struct char_data * vict)
{
  if (ch == vict)
    return;

    if (!pk_allowed)
    check_killer(ch, vict);

  assert(!FIGHTING(ch));

  ch->next_fighting = combat_list;
  combat_list = ch;
  ch->fight_round = 0;
  if (IS_AFFECTED(ch, AFF_SLEEP))
    affect_from_char(ch, SPELL_SLEEP);

  FIGHTING(ch) = vict;
  GET_POS(ch) = POS_FIGHTING;

}



/* remove a char from the list of fighting chars */
void stop_fighting(struct char_data * ch)
{
  struct char_data *temp;
  struct affected_type *af, *next;
  extern char *spell_wear_off_msg[]; 

  if (ch == next_combat_list)
    next_combat_list = ch->next_fighting;

  REMOVE_FROM_LIST(ch, combat_list, next_fighting);
  ch->next_fighting = NULL;
  FIGHTING(ch) = NULL;
  if (ch->char_specials.blinkon)
    ch->char_specials.blinkon = FALSE;
  if (affected_by_spell(ch, SPELL_BLINK)) {
    for (af = ch->affected; af; af = next) {
      next = af->next;
      if (af->bitvector2 == AFF2_BLINK) {
        affect_remove(ch, af);
        send_to_char(spell_wear_off_msg[SPELL_BLINK], ch);
        send_to_char("\r\n", ch);
      }
    }
  }
  if (ch->riding) GET_POS(ch) = POS_RIDING;
  else if (ch->ridden_by) GET_POS(ch) = POS_RIDDEN;
  else GET_POS(ch) = POS_STANDING;
  update_pos(ch);
  if (MOB_FLAGGED(ch, MOB_CONJURE) && GET_HIT(ch) > 0) {
    act("$s job done, $n fades away.", TRUE, ch, 0, ch->master, TO_ROOM);
    extract_char(ch);
   }
/*  if (MOB_FLAGGED(ch, MOB_HUNTER) && ch->mob_specials.target) 
    hunt_victim(ch->mob_specials.target); */
}



void make_corpse(struct char_data * ch)
{
  struct obj_data *corpse, *o, *blood;
  struct obj_data *money;
  int i;
  extern int max_npc_corpse_time, max_pc_corpse_time;

  struct obj_data *create_money(int amount);

  corpse = create_obj();

  corpse->item_number = 0 - (IS_NPC(ch) ? GET_MOB_VNUM(ch) : GET_IDNUM(ch));
  corpse->in_room = NOWHERE;
  corpse->name = str_dup("corpse");

  sprintf(buf2, "The corpse of %s is lying here.", GET_NAME(ch));
  corpse->description = str_dup(buf2);

  sprintf(buf2, "the corpse of %s", GET_NAME(ch));
  corpse->short_description = str_dup(buf2);

  GET_OBJ_TYPE(corpse) = ITEM_CONTAINER;
  GET_OBJ_WEAR(corpse) = ITEM_WEAR_TAKE;
  GET_OBJ_EXTRA(corpse) = ITEM_NODONATE;
  GET_OBJ_VAL(corpse, 0) = 0;	/* You can't store stuff in a corpse */
  GET_OBJ_VAL(corpse, 3) = 1;	/* corpse identifier */
  GET_OBJ_WEIGHT(corpse) = GET_WEIGHT(ch) + IS_CARRYING_W(ch);
  GET_OBJ_RENT(corpse) = 100000;
  if (IS_NPC(ch))
    GET_OBJ_TIMER(corpse) = max_npc_corpse_time;
  else
    GET_OBJ_TIMER(corpse) = max_pc_corpse_time;

  if(!IS_NPC(ch)) {  /*corpsemod*/
    corpse->obj_flags.owner = GET_IDNUM(ch);
    SET_BIT(GET_OBJ_EXTRA(corpse), ITEM_CORPSE);
  }

  /* transfer character's inventory to the corpse */
  corpse->contains = ch->carrying;
  for (o = corpse->contains; o != NULL; o = o->next_content)
    o->in_obj = corpse;
  object_list_new_owner(corpse, NULL);

  /* transfer character's equipment to the corpse */
  for (i = 0; i < NUM_WEARS; i++)
    if (GET_EQ(ch, i))
      obj_to_obj(unequip_char(ch, i), corpse);

/* BLOOD */
  blood = read_object(real_object(16018), REAL);
  if(blood) {
    if (IS_NPC(ch))
      GET_OBJ_TIMER(blood) = max_npc_corpse_time;
    else
      GET_OBJ_TIMER(blood) = max_pc_corpse_time;
    if (number(1,100) > 80)/*bloodmod*/
     obj_to_obj(blood, corpse);
    else
     extract_obj(blood);
  } else
    mudlog("SYSERR:  Obj #16018 missing.  No blood available.", BRF,
      LVL_IMMORT, TRUE);

  /* transfer gold */
  if (GET_GOLD(ch) > 0) {
    /* following 'if' clause added to fix gold duplication loophole */
    if (IS_NPC(ch) || (!IS_NPC(ch) && ch->desc)) {
      money = create_money(GET_GOLD(ch));
      obj_to_obj(money, corpse);
    }
    GET_GOLD(ch) = 0;
  }
  ch->carrying = NULL;
  IS_CARRYING_N(ch) = 0;
  IS_CARRYING_W(ch) = 0;
  for (o = corpse->contains; o != NULL; o = o->next_content) {
    if(IS_NPC(ch) && MOB_TAG(ch)) /*idmod*/
      OBJ_TAG(o) = MOB_TAG(ch);
  }

  obj_to_room(corpse, ch->in_room);
}





void death_cry(struct char_data * ch)
{
  int door, was_in;

  act("Your blood freezes as you hear $n's death cry.", FALSE, ch, 0, 0, TO_ROOM);
  was_in = ch->in_room;

  for (door = 0; door < NUM_OF_DIRS; door++) {
    if (CAN_GO(ch, door)) {
      ch->in_room = world[was_in].dir_option[door]->to_room;
      act("Your blood freezes as you hear someone's death cry.", FALSE, ch, 0, 0, TO_ROOM);
      ch->in_room = was_in;
    }
  }
}



void raw_kill(struct char_data * ch, struct char_data *killer)
{
  struct char_data *fighter;

  if (FIGHTING(ch)) {
    fighter = FIGHTING(ch);
    stop_fighting(ch);
  }

  while (ch->affected)
    affect_remove(ch, ch->affected);
 /*  death_cry(ch); */
  if (killer)
  GET_LOADROOM(ch) = world[r_mortal_start_room].number;
  if (!MOB_FLAGGED(ch, MOB_CONJURE))
    make_corpse(ch);
  extract_char(ch);
}



void die(struct char_data * ch, struct char_data * killer)
{
  gain_exp(ch, -(GET_EXP(ch) >> 1));
  if (!IS_NPC(ch))
    REMOVE_BIT(PLR_FLAGS(ch), PLR_KILLER | PLR_THIEF);
  raw_kill(ch, killer);
}


void perform_group_gain(struct char_data * ch, int base,
                             struct char_data * victim, int level)
{
  int share, exp;

  /*  base exp divided up by level */

  exp = (GET_LEVEL(ch) * base) / level;
  share = MIN(max_exp_gain, MAX(1, exp));

  if (share > 1) {
    sprintf(buf2, "You receive your share of experience -- %d points.\r\n", share);
    send_to_char(buf2, ch);
  } else
    send_to_char("You receive your share of experience -- one measly little point!\r\n", ch);

  gain_exp(ch, share);
}


void group_gain(struct char_data * ch, struct char_data * victim)
{
  int tot_members, base;
  struct char_data *k;
  struct follow_type *f;
  int level = 0;

  if (!(k = ch->master))
    k = ch;

  if (IS_AFFECTED(k, AFF_GROUP) && (k->in_room == ch->in_room)) {
    tot_members = 1;
    level += GET_LEVEL(k);
    }
  else
    tot_members = 0;

  for (f = k->followers; f; f = f->next)
    if (IS_AFFECTED(f->follower, AFF_GROUP) &&
	f->follower->in_room == ch->in_room && !IS_NPC(f->follower) ) {
      tot_members++;
      level += GET_LEVEL(f->follower);
      }

  /* round up to the next highest tot_members */
  /* total exp for group is doubled--Trebor */

  base = 2 * tot_members * 1000 / 3 / (tot_members + 1);

  if (tot_members < 1)
    base = 0;

  if (IS_AFFECTED(k, AFF_GROUP) && k->in_room == ch->in_room)
    perform_group_gain(k, base, victim, level);

  for (f = k->followers; f; f = f->next)
    if (IS_AFFECTED(f->follower, AFF_GROUP) &&
       f->follower->in_room == ch->in_room && !IS_NPC(f->follower) )
      perform_group_gain(f->follower, base, victim, level);

}



char *replace_string(char *str, char *weapon_singular, char *weapon_plural)
{
  static char buf[256];
  char *cp;

  cp = buf;

  for (; *str; str++) {
    if (*str == '#') {
      switch (*(++str)) {
      case 'W':
	for (; *weapon_plural; *(cp++) = *(weapon_plural++));
	break;
      case 'w':
	for (; *weapon_singular; *(cp++) = *(weapon_singular++));
	break;
      default:
	*(cp++) = '#';
	break;
      }
    } else
      *(cp++) = *str;

    *cp = 0;
  }				/* For */

  return (buf);
}


/* message for doing damage with a weapon */
void dam_message(int dam, struct char_data * ch, struct char_data * victim,
		      int w_type)
{
  char *buf;
  int msgnum;

  static struct dam_weapon_type {
    char *to_room;
    char *to_char;
    char *to_victim;
  } dam_weapons[] = {

    /* use #w for singular (i.e. "slash") and #W for plural (i.e. "slashes") */

    {
      "$n tries to #w $N, but misses.",	/* 0: 0     */
      "You try to #w $N&+y, but miss.",
      "$n&+r tries to #w you, but misses."
    },

    {
      "$n tickles $N as $e #W $M.",	/* 1: 1..2  */
      "You tickle $N&+y as you #w $M.",
      "$n&+r tickles you as $e #W you."
    },

    {
      "$n barely #W $N.",		/* 2: 3..5  */
      "You barely #w $N&+y.",
      "$n&+r barely #W you."
    },

    {
      "$n #W $N.",			/* 3: 6..9  */
      "You #w $N&+y.",
      "$n&+r #W you."
    },

    {
      "$n #W $N hard.",			/* 4: 10..14  */
      "You #w $N&+y hard.",
      "$n&+r #W you hard."
    },

    {
      "$n #W $N very hard.",		/* 5: 15..20  */
      "You #w $N&+y very hard.",
      "$n&+r #W you very hard."
    },

    {
      "$n #W $N extremely hard.",	/* 6: 21..27  */
      "You #w $N&+y extremely hard.",
      "$n&+r #W you extremely hard."
    },

    {
      "$n massacres $N to small fragments with $s #w.", /* 7: 28..35 */
      "You massacre $N&+y to small fragments with your #w.",
      "$n&+r massacres you to small fragments with $s #w."
    },
    {
      "$n obliterates $N with $s deadly #w!!",  /* 8: 36..44   */
      "You obliterate $N&+y with your deadly #w!!",
      "$n&+r obliterates you with $s deadly #w!!"
    },
    {
      "$n matricates $N with $s deadly #w!!",  /* 9: 45..54  */
      "You matricate $N&+y with your deadly #w!!",
      "$n&+r matricates you with $s deadly #w!!"
    },
    {
      "$n mutilates $N with $s skillful #w!!",  /* 10: 55..65  */
      "You mutilate $N&+y with your skillful #w!!",
      "$n&+r mutilates you with $s skillful #w!!"
    },

    {
      "$n annihilates $N with $s masterful #w!!",  /* 11: 66..76  */
      "You annihilate $N&+y with your masterful #w!!",
      "$n&+r annihilates you with $s masterful #w!!"
    },
    {
      "$n atomizes $N with $s killer #w!!",  /* 12: 77..90  */
      "You atomize $N&+y with your killer #w!!",
      "$n&+r atomizes you with $s killer #w!!"
    },
    {
      "$n decimates $N with $s deadly #w!!",  /* 13: 91..104  */
      "You decimate $N&+y with your deadly #w!!",
      "$n&+r decimates you with $s deadly #w!!"
    },

    {
      "$n pulverizes $N with $s powerful #w!!",  /* 14: 105..119  */
      "You pulverize $N&+y with your powerful #w!!",
      "$n&+r pulverizes you with $s powerful #w!!"
    },

    {
      "$n eraticates $N with $s powerful #w!!",  /* 15: 120..135  */
      "You eraticate $N&+y with your powerful #w!!",
      "$n&+r eraticates you with $s powerful #w!!"
    },
    {
      "$n vanquishes $N with $s incredible #w!!",  /* 16: 136..152  */
      "You vanquish $N&+y with your incredible #w!!",
      "$n&+r vanquishes you with $s incredible #w!!"
    },
    {
      "$n Eviscerates $N with $s powerful #w!!",  /* 17: 153..170 */
      "You Eviscerate $N&+y with your powerful #w!!",
      "$n&+r Eviscerates you with $s powerful #w!!"
    },
    {
      "$n BRUTALLY MAULS $N with $s powerful #w!!",  /* 18: 171..189 */
      "You BRUTALLY MAUL $N&+y with your powerful #w!!",
      "$n&+r BRUTALLY MAULS you with $s powerful #w!!"
    },

    {
      "$n DESTROYS $N with $s staggering #w!!",  /* 19: > 190  */
      "You DESTROY $N&+y with your staggering #w!!",
      "$n&+r DESTROYS you with $s staggering #w!!"
    }
  };


  w_type -= TYPE_HIT;		/* Change to base of table with text */

  if (dam == 0)		 msgnum = 0;
  else if (dam <= 2)     msgnum = 1;
  else if (dam <= 5)     msgnum = 2;
  else if (dam <= 9)     msgnum = 3;
  else if (dam <= 14)    msgnum = 4;
  else if (dam <= 20)    msgnum = 5;
  else if (dam <= 27)    msgnum = 6;
  else if (dam <= 35)    msgnum = 7;
  else if (dam <= 44)    msgnum = 8;
  else if (dam <= 54)    msgnum = 9;
  else if (dam <= 65)    msgnum = 10;
  else if (dam <= 77)    msgnum = 11;
  else if (dam <= 90)    msgnum = 12;
  else if (dam <= 104)   msgnum = 13;
  else if (dam <= 119)   msgnum = 14;
  else if (dam <= 135)   msgnum = 15;
  else if (dam <= 152)   msgnum = 16;
  else if (dam <= 170)   msgnum = 17;
  else if (dam <= 189)   msgnum = 18;
  else                   msgnum = 19;

  /* damage message to onlookers */
  buf = replace_string(dam_weapons[msgnum].to_room,
	  attack_hit_text[w_type].singular, attack_hit_text[w_type].plural);
  act(buf, FALSE, ch, NULL, victim, TO_NOTVICT);

  /* damage message to damager */
  send_to_char(CCYEL(ch, C_CMP), ch);
  buf = replace_string(dam_weapons[msgnum].to_char,
	  attack_hit_text[w_type].singular, attack_hit_text[w_type].plural);
  act(buf, FALSE, ch, NULL, victim, TO_CHAR);
  send_to_char(CCNRM(ch, C_CMP), ch);

  /* damage message to damagee */
  send_to_char(CCRED(victim, C_CMP), victim);
  buf = replace_string(dam_weapons[msgnum].to_victim,
	  attack_hit_text[w_type].singular, attack_hit_text[w_type].plural);
  act(buf, FALSE, ch, NULL, victim, TO_VICT | TO_SLEEP);
  send_to_char(CCNRM(victim, C_CMP), victim);
}


/*
 * message for doing damage with a spell or skill
 *  C3.0: Also used for weapon damage on miss and death blows
 */
int skill_message(int dam, struct char_data * ch, struct char_data * vict,
		      int attacktype)
{
  int i, j, nr;
  struct message_type *msg;

  struct obj_data *weap = GET_EQ(ch, WEAR_WIELD);
  if (attacktype == TYPE_FALL) return 1;
  for (i = 0; i < MAX_MESSAGES; i++) {
    if (fight_messages[i].a_type == attacktype) {
      nr = dice(1, fight_messages[i].number_of_attacks);
      for (j = 1, msg = fight_messages[i].msg; (j < nr) && msg; j++)
	msg = msg->next;

      if (!IS_NPC(vict) && PRF_FLAGGED(vict, PRF_NOHIT)) {
	act(msg->god_msg.attacker_msg, FALSE, ch, weap, vict, TO_CHAR);
	act(msg->god_msg.victim_msg, FALSE, ch, weap, vict, TO_VICT);
	act(msg->god_msg.room_msg, FALSE, ch, weap, vict, TO_NOTVICT);
      } else if (dam != 0) {
	if (GET_POS(vict) == POS_DEAD) {
	  send_to_char(CCYEL(ch, C_CMP), ch);
	  act(msg->die_msg.attacker_msg, FALSE, ch, weap, vict, TO_CHAR);
	  send_to_char(CCNRM(ch, C_CMP), ch);

	  send_to_char(CCRED(vict, C_CMP), vict);
	  act(msg->die_msg.victim_msg, FALSE, ch, weap, vict, TO_VICT | TO_SLEEP);
	  send_to_char(CCNRM(vict, C_CMP), vict);

	  act(msg->die_msg.room_msg, FALSE, ch, weap, vict, TO_NOTVICT);
	} else {
	  send_to_char(CCYEL(ch, C_CMP), ch);
	  act(msg->hit_msg.attacker_msg, FALSE, ch, weap, vict, TO_CHAR);
	  send_to_char(CCNRM(ch, C_CMP), ch);

	  send_to_char(CCRED(vict, C_CMP), vict);
	  act(msg->hit_msg.victim_msg, FALSE, ch, weap, vict, TO_VICT | TO_SLEEP);
	  send_to_char(CCNRM(vict, C_CMP), vict);

	  act(msg->hit_msg.room_msg, FALSE, ch, weap, vict, TO_NOTVICT);
	}
      } else if (ch != vict) {	/* Dam == 0 */
	send_to_char(CCYEL(ch, C_CMP), ch);
	act(msg->miss_msg.attacker_msg, FALSE, ch, weap, vict, TO_CHAR);
	send_to_char(CCNRM(ch, C_CMP), ch);

	send_to_char(CCRED(vict, C_CMP), vict);
	act(msg->miss_msg.victim_msg, FALSE, ch, weap, vict, TO_VICT | TO_SLEEP);
	send_to_char(CCNRM(vict, C_CMP), vict);

	act(msg->miss_msg.room_msg, FALSE, ch, weap, vict, TO_NOTVICT);
      }
      return 1;
    }
  }
  return 0;
}
void perform_pkill_death(struct char_data * ch, struct char_data * victim)
{
  int exp;

  if (!PLR_FLAGGED(ch, PLR_COMPETITION) ||
      !PLR_FLAGGED(victim, PLR_COMPETITION)) {
    sprintf(buf, "SYSERR: invalid pkill! %s and %s", GET_NAME(ch),
            GET_NAME(victim));
    mudlog(buf, BRF, LVL_IMMORT, TRUE);
    return;
  }
  exp = ((GET_LEVEL(ch)/10)+1)*100000;
  sprintf(buf, "You receive %d experience.", exp);
  send_to_char(buf, ch);
/*  gain_exp(ch, exp); */
  GET_HIT(victim) = 10;
  stop_fighting(victim); stop_fighting(ch);
  char_from_room(victim); char_to_room(victim, r_mortal_start_room);
/*  sprintf(buf, "And the winner of the Pkill competition between %s and %s is:"
          "%s!!!\r\n", GET_NAME(ch), GET_NAME(victim), GET_NAME(ch));
  send_to_all(buf); */
  sprintf(buf, "clan: %s [%d] %s defeats %s [%d] in clan competition",
          GET_NAME(ch), GET_LEVEL(ch),
          clans[(int) GET_CLAN(ch)].clan_short, GET_NAME(victim),
          GET_LEVEL(victim));
  nmlog(buf);
/*  REMOVE_BIT(PLR_FLAGS(ch), PLR_COMPETITION); */
  REMOVE_BIT(PLR_FLAGS(victim), PLR_COMPETITION);
  ch->player_specials->saved.pkill_points++;
  return;
}


void damage(struct char_data * ch, struct char_data * victim, int dam,
	    int attacktype)
{
  int exp, klvl, vlvl;
  byte  prob, percent;

  if (GET_POS(victim) <= POS_DEAD) {
    nmlog("SYSERR: Attempt to damage a corpse.");
    return;			/* -je, 7/7/92 */
  }

  /* remort_mod - take remorts into account for level */
  klvl = (GET_REMORT(ch) * 100) + GET_LEVEL(ch);
  vlvl = (GET_REMORT(victim) * 100) + GET_LEVEL(victim);

  /* peaceful rooms */
  if (ch != victim && ROOM_FLAGGED(ch->in_room, ROOM_PEACEFUL)) {
    send_to_char("This room just has such a peaceful, easy feeling...\r\n", ch);
    return;
  }

  /* shopkeeper protection */
  if (!ok_damage_shopkeeper(ch, victim))
    return;

  if (MOB_FLAGGED(victim, MOB_NOFIGHT)) {
    act("$N is under the gods' divine protection.  Choose someone else.", 
        FALSE, ch, 0, victim, TO_CHAR);
    return;
  }
  if (victim != ch) {
    if (GET_POS(ch) > POS_STUNNED) {
      if (!(FIGHTING(ch))) {
        if (!IS_NPC(ch))
          change_alignment(ch, victim);
	set_fighting(ch, victim);
      }

      if (IS_NPC(ch) && IS_NPC(victim) && victim->master &&
	  !number(0, 10) && IS_AFFECTED(victim, AFF_CHARM) &&
	  (victim->master->in_room == ch->in_room)) {
	if (FIGHTING(ch))
	  stop_fighting(ch);
	hit(ch, victim->master, TYPE_UNDEFINED);
	return;
      }
    }
     /*idmod*/
    if(GET_LEVEL(ch) >= LVL_IMMORT && IS_NPC(victim))
      MOB_TAG(victim) = GET_IDNUM(ch);

    if (GET_POS(victim) > POS_STUNNED && !FIGHTING(victim)) {
      set_fighting(victim, ch);
      if (MOB_FLAGGED(victim, MOB_MEMORY) && !IS_NPC(ch) &&
	  (GET_LEVEL(ch) < LVL_IMMORT))
	remember(victim, ch);
    }
  }
  if (victim->master == ch)
    stop_follower(victim);

  if (IS_AFFECTED(ch, AFF_INVISIBLE | AFF_HIDE) && attacktype < 399)
    appear(ch);

  if (IS_AFFECTED(victim, AFF_SANCTUARY))
    dam = (dam * .9);		/* 10% damage reduction */

  if (IS_SET(AFF2_FLAGS(victim), AFF2_DIVINE_PROT))
    dam = (dam * .85);          /* 15% damage reduction */

  if (IS_SET(AFF2_FLAGS(victim), AFF2_STONESKIN))
    dam = (dam * .9);		/* 10% damage reduction */

  if (GET_RACE(victim) == RACE_OGRE)
    dam = (dam * .9);		/* 10% damage reduction */

  if (GET_RACE(ch) == RACE_DRAGON)
    dam = (dam * 1.1);		/* 10% damage bonus */

  if ((GET_ALIGNMENT(ch) < -349) && (IS_AFFECTED(victim, AFF_PROTECT_EVIL))) {
    if ( ( (GET_CLASS(victim) == CLASS_PALADIN) &&
      (GET_ALIGNMENT(victim) < -349) ) || ( (GET_CLASS(victim) != CLASS_PALADIN) &&
         (GET_ALIGNMENT(victim) < 350) ) )
      send_to_char("Your evilness neutralizes your protection from evil.\r\n", victim);
    else  {
      act("You are protected from $N's evilness.", FALSE, victim, 0, ch,
           TO_CHAR);
      dam = (dam * .8); 		/* 20% damage reduction */
    }
  }

  if ((GET_ALIGNMENT(ch) > 349) && (IS_AFFECTED(victim, AFF_PROTECT_GOOD))) {
    if ( ( (GET_CLASS(victim) == CLASS_APALADIN) &&
      (GET_ALIGNMENT(victim) > 349) ) || ( (GET_CLASS(victim) != CLASS_APALADIN) &&
        (GET_ALIGNMENT(victim) > -350) ) )
      send_to_char("Your goodness neutralizes your protection from good.\r\n", victim);
    else  {
      act("You are protected from $N's goodness.", FALSE, victim, 0, ch,
           TO_CHAR);
      dam = (dam * .8);			/* 20% damage reduction */
    }
  }

  if (IS_AFFECTED(victim, AFF_BERSERK))
    dam <<= 1;                          /* 2x damage taken */
  if (IS_AFFECTED(ch, AFF_BERSERK))
    dam += (GET_LEVEL(ch) / 2);

  if (!pk_allowed) {
    check_killer(ch, victim);

    if (PLR_FLAGGED(ch, PLR_KILLER))
      dam = 0;
  }
  /* KAKO % dodge skill */
  percent = number(1, 101);
  if(GET_RACE(ch) == RACE_HOBBIT)
    percent -= 10; /* racial modifiers for dodge - Treb 9/96 */
  prob = GET_SKILL(victim, SKILL_DODGE);
  prob >>= 3;
  if (percent < prob && IS_WEAPON(attacktype) && dam != 0 && !IS_NPC(victim) &&
      !IS_WARRIOR(victim) && !IS_CLERIC(victim) && !IS_MAGIC_USER(victim) &&
      !IS_PALADIN(victim) && !IS_APALADIN(victim) && !IS_RANGER(victim) ) {
   act("You skillfully dodge $N's attack!", FALSE, victim, 0, ch, TO_CHAR);
   act("$n gracefully dances around $N's clumsy attack.", FALSE, victim, 0, ch, TO_ROOM);
   dam = 0;
  }
  /* Kako % parry skill */
  percent = number(1, 101);
  prob = GET_SKILL(victim, SKILL_PARRY);
  prob >>= 3;
  if (percent < prob && IS_WEAPON(attacktype) && dam != 0 && !IS_NPC(victim) &&
      !IS_THIEF(victim) && !IS_CLERIC(victim) && !IS_MAGIC_USER(victim) &&
      !IS_BARD(victim) && !IS_APALADIN(victim) && victim->equipment[WEAR_WIELD] ) {
   send_to_char("Wow! Nice parry!\r\n", victim);
   act("$n skillfully parries $N's attack!", FALSE, victim, 0, ch, TO_ROOM);
   dam = 0;
  }
  /* Kako % blur skill */
  if (IS_AFFECTED(victim, AFF_BLUR) && dam != 0 && IS_WEAPON(attacktype))
  if (!mag_savingthrow(ch, SAVING_SPELL, 0)) {

    /* attacker must fail save for blur to work - Trebor*/

    act("$N misses your blurry figure.", FALSE, victim, 0, ch, TO_CHAR);
    act("$N misses $n's blurry outline.", FALSE, victim, 0, ch, TO_ROOM);
    dam = 0;
    }

  /* You can't damage an immortal! */
  if (!IS_NPC(victim) && (PRF_FLAGGED(victim, PRF_NOHIT)))
    dam = 0;

  dam = MAX(MIN(dam, 60000), 0);
  GET_HIT(victim) -= dam;

/* No more gaining experience from doing damage 12-14-2015 */
/*  if (ch != victim && !PLR_FLAGGED(ch, PLR_COMPETITION) &&
      !PLR_FLAGGED(victim, PLR_COMPETITION))
    gain_exp(ch, (MAX(1, (GET_LEVEL(victim) - GET_LEVEL(ch))) * dam));
*/

  update_pos(victim);

  /*
   * skill_message sends a message from the messages file in lib/misc.
   * dam_message just sends a generic "You hit $n extremely hard.".
   * skill_message is preferable to dam_message because it is more
   * descriptive.
   *
   * If we are _not_ attacking with a weapon (i.e. a spell), always use
   * skill_message. If we are attacking with a weapon: If this is a miss or a
   * death blow, send a skill_message if one exists; if not, default to a
   * dam_message. Otherwise, always send a dam_message.
   */
  if (attacktype != TYPE_DEATHTRAP) {
    if (!IS_WEAPON(attacktype))
      skill_message(dam, ch, victim, attacktype);
    else {
      if (GET_POS(victim) == POS_DEAD || dam == 0) {
        if (!skill_message(dam, ch, victim, attacktype))
	  dam_message(dam, ch, victim, attacktype);
      } else
        dam_message(dam, ch, victim, attacktype);
    }
  }

  /* Use send_to_char -- act() doesn't send message if you are DEAD. */
  switch (GET_POS(victim)) {
  case POS_MORTALLYW:
    act("$n is mortally wounded, and will die soon, if not aided.", TRUE, victim, 0, 0, TO_ROOM);
    send_to_char("You are mortally wounded, and will die soon, if not aided.\r\n", victim);
    break;
  case POS_INCAP:
    act("$n is incapacitated and will slowly die, if not aided.", TRUE, victim, 0, 0, TO_ROOM);
    send_to_char("You are incapacitated an will slowly die, if not aided.\r\n", victim);
    break;
  case POS_STUNNED:
    act("$n is stunned, but will probably regain consciousness again.", TRUE, victim, 0, 0, TO_ROOM);
    send_to_char("You're stunned, but will probably regain consciousness again.\r\n", victim);
    break;
  case POS_DEAD:
    act("$n is dead!  R.I.P.", FALSE, victim, 0, 0, TO_ROOM);
    send_to_char("You are dead!  Sorry...\r\n", victim);
    break;

  default:			/* >= POSITION SLEEPING */
    if (dam > (GET_MAX_HIT(victim) >> 2))
      act("That really did HURT!", FALSE, victim, 0, 0, TO_CHAR);

    if (GET_HIT(victim) < (GET_MAX_HIT(victim) >> 2)) {
      sprintf(buf2, "%sYou wish that your wounds would stop BLEEDING so much!%s\r\n",
	      CCRED(victim, C_SPR), CCNRM(victim, C_SPR));
      send_to_char(buf2, victim);
      if (MOB_FLAGGED(victim, MOB_WIMPY) && (ch != victim))
	do_flee(victim, "", 0, 0);
    }
    if (!IS_NPC(victim) && GET_WIMP_LEV(victim) && victim != ch &&
	GET_HIT(victim) < GET_WIMP_LEV(victim) && attacktype < 399 &&
        !PLR_FLAGGED(victim, PLR_COMPETITION)) {
      send_to_char("You wimp out, and attempt to flee!\r\n", victim);
      do_flee(victim, "", 0, 0);
    }
    break;
  }

  if (!IS_NPC(victim) && !(victim->desc) && attacktype < 399) {
    do_flee(victim, "", 0, 0);
    if (!FIGHTING(victim)) {
      act("$n is rescued by divine forces.", FALSE, victim, 0, 0, TO_ROOM);
      GET_WAS_IN(victim) = victim->in_room;
      char_from_room(victim);
      char_to_room(victim, 1);
    }
  }
  if (!AWAKE(victim))
    if (FIGHTING(victim))
      stop_fighting(victim);

  if (GET_POS(victim) == POS_DEAD) {
    if (PLR_FLAGGED(victim, PLR_COMPETITION)) {  /* Pkill code */
      perform_pkill_death(ch, victim);
      return;
    }
    if (IS_NPC(victim) || victim->desc) {
      if (IS_AFFECTED(ch, AFF_GROUP) && (ch->master || ch->followers))
	group_gain(ch, victim);
      else {
	/* Calculate level-difference bonus */
	exp = MIN(max_exp_gain, GET_EXP(victim));

	if (GET_LEVEL(victim) > GET_LEVEL(ch))
/*	  exp += MAX(0, (exp * MIN(4, (GET_LEVEL(victim) - GET_LEVEL(ch)))) >> 3); */
        exp = 1000;
	else
/*	  exp += MAX(0, (exp * MIN(8, (GET_LEVEL(victim) / GET_LEVEL(ch)))) >> 3); */
        exp = 1000 - MIN(1000, ((GET_LEVEL(ch) - (GET_LEVEL(victim) - 1)) * 100));
        exp = MIN(max_exp_gain, exp);

	if (exp > 1) {
	  sprintf(buf2, "You receive %d experience points.\r\n", exp);
	  send_to_char(buf2, ch);
	} else if (exp == 0)
          send_to_char("You don't get SQUAT!\r\n", ch);
        else
	  send_to_char("You receive one lousy experience point.\r\n", ch);
	gain_exp(ch, exp);
      }
    }
    if (!IS_NPC(victim)) {
      sprintf(buf2, "%s killed by %s at %s (#%d)", GET_NAME(victim), GET_NAME(ch),
	      world[victim->in_room].name, world[victim->in_room].number);
      mudlog(buf2, BRF, LVL_IMMORT, TRUE);
      if (MOB_FLAGGED(ch, MOB_MEMORY))
	forget(ch, victim);
    }
    if (attacktype == TYPE_DEATHTRAP) {
      if (!IS_NPC(victim))
        REMOVE_BIT(PLR_FLAGS(victim), PLR_KILLER | PLR_THIEF);
      raw_kill(victim, ch);
    } else
      die(victim, ch);
    if (PRF_FLAGGED(ch, PRF_AUTOLOOT))
     command_interpreter(ch, "get all corpse");
/*    if (PRF_FLAGGED(ch, PRF_AUTOGOLD))
     command_interpreter(ch, "get all.coins corpse"); */
/*    if (PRF_FLAGGED(ch, PRF_AUTOCHECK))
     command_interpreter(ch, "look in corpse"); */
  }

}



void hit(struct char_data * ch, struct char_data * victim, int type)
{
  struct obj_data *wielded = GET_EQ(ch, WEAR_WIELD);
  int w_type, victim_ac, calc_thaco, dam, diceroll;
  int prof_bonus = 0;

  extern int thaco[NUM_CLASSES][LVL_SIMP+1];
  extern struct str_app_type str_app[];
  extern struct dex_app_type dex_app[];

  /* backstab function modified for additional damage by high-level
     thieves */

  int backstab_mult(struct char_data *ch, int level);

  if (ROOM_FLAGGED(ch->in_room, ROOM_NOFIGHT)) {
    send_to_char("It's too tranquil here to fight.\r\n", ch);
    return;
  }

  if (ch->in_room != victim->in_room) {
    if (FIGHTING(ch) && FIGHTING(ch) == victim)
      stop_fighting(ch);
    return;
  }


  if (wielded && GET_OBJ_TYPE(wielded) == ITEM_WEAPON)
    w_type = GET_OBJ_VAL(wielded, 3) + TYPE_HIT;
  else {
    if (IS_NPC(ch) && (ch->mob_specials.attack_type != 0))
      w_type = ch->mob_specials.attack_type + TYPE_HIT;
    else
      w_type = TYPE_HIT;
  }

  /* calculate proficiency bonus, if any */
  if (wielded && IS_OBJ_STAT(wielded, ITEM_TWOHAND) && 
      number(1,101) < GET_SKILL(ch, SKILL_2HAND)) 
     prof_bonus = GET_SKILL(ch, SKILL_2HAND) / 10;
  if (test_mode && prof_bonus) {
    sprintf(buf, "Proficiency bonus: %d\r\n", prof_bonus);
    send_to_char(buf, ch);
  }

  /* Calculate the raw armor including magic armor.  Lower AC is better. */

  if (!IS_NPC(ch))
    calc_thaco = thaco[(int) GET_CLASS(ch)][(int) GET_LEVEL(ch)];
  else {	/* THAC0 for monsters is set in the HitRoll */
    if(GET_LEVEL(ch) >=30)
      calc_thaco = 30 - (GET_LEVEL(ch))/3;
    else
      calc_thaco = 20;
  }

/*  calc_thaco -= str_app[STRENGTH_APPLY_INDEX(ch)].tohit; */
  calc_thaco -= GET_HITROLL(ch);
  calc_thaco -= prof_bonus;
  calc_thaco -= (int) ((GET_INT(ch) - 13) / 1.5);	/* Intelligence helps! */
  calc_thaco -= (int) ((GET_WIS(ch) - 13) / 1.5);	/* So does wisdom */
  if(GET_RACE(ch) == RACE_GARGOYLE && weather_info.sunlight != SUN_DARK)
    calc_thaco += (20 * GET_LEVEL(ch)) / 100;
  diceroll = number(1, 20);

  victim_ac = GET_AC(victim) / 10;
  if (GET_RACE(victim)  == RACE_HOBBIT) 
    victim_ac -= 4;  /*  Hobbit AC mod due to small size Treb 9/96 */ 

  if (AWAKE(victim))
    victim_ac += dex_app[GET_DEX(victim)].defensive;

/*  if(AFF2_FLAGGED(victim, AFF2_STONESKIN) && !affected_by_spell(victim,
    SPELL_STONESKIN))
    victim_ac -= 3; */

  victim_ac = MAX(-30, victim_ac);	/* -30 is lowest-Trebor */

  /* decide whether this is a hit or a miss */
  if ((((diceroll < 20) && AWAKE(victim)) &&
       ((diceroll == 1) || ((calc_thaco - diceroll) > victim_ac)))) {
    if (type == SKILL_BACKSTAB || type == SKILL_CIRCLE)
      damage(ch, victim, 0, SKILL_BACKSTAB);
    else
      damage(ch, victim, 0, w_type);
  } else {
    /* okay, we know the guy has been hit.  now calculate damage. */
    dam = str_app[STRENGTH_APPLY_INDEX(ch)].todam;
    dam += GET_DAMROLL(ch);
    dam += prof_bonus;
    if(GET_RACE(ch) == RACE_GARGOYLE && weather_info.sunlight != SUN_DARK)
      dam -= (20 * GET_LEVEL(ch)) / 100;

    if (wielded) {
      if (GET_OBJ_TYPE(wielded) == ITEM_WEAPON)
        dam += dice(GET_OBJ_VAL(wielded, 1), GET_OBJ_VAL(wielded, 2));
      else
        dam += number(1,10);
    } else {
      if (IS_NPC(ch)) {
	dam += dice(ch->mob_specials.damnodice, ch->mob_specials.damsizedice);
      } else
	dam += number(0, 2);	/* Max. 2 dam with bare hands */
    }

    if (GET_POS(victim) < POS_FIGHTING)
      dam *= 1 + (POS_FIGHTING - GET_POS(victim)) / 3;
    /* Position  sitting  x 1.33 */
    /* Position  resting  x 1.66 */
    /* Position  sleeping x 2.00 */
    /* Position  stunned  x 2.33 */
    /* Position  incap    x 2.66 */
    /* Position  mortally x 3.00 */

    dam = MAX(1, dam);		/* at least 1 hp damage min per hit */

    if (type == SKILL_BACKSTAB) {
      dam *=  backstab_mult(ch, GET_LEVEL(ch)); /* Trebor 8/96 */
      damage(ch, victim, dam, SKILL_BACKSTAB);
    } else if (type == SKILL_DOUBLE_BS) { /*double*/
      dam *=  backstab_mult(ch, GET_LEVEL(ch)) * 1.5; /* Trebor 8/96 */
      damage(ch, victim, dam, SKILL_DOUBLE_BS);
    } else if (type == SKILL_CIRCLE) { /* circle */
      dam *=  backstab_mult(ch, GET_LEVEL(ch)) >> 1;
      damage(ch, victim, dam, SKILL_BACKSTAB);
    } else
      damage(ch, victim, dam, w_type);
  }
}



/* control the fights going on.  Called every 2 seconds from comm.c. */
void perform_violence(void)
{
  struct char_data *ch, *victim;
  extern struct index_data *mob_index;
  int percent;

  for (ch = combat_list; ch; ch = next_combat_list) {
    next_combat_list = ch->next_fighting;
    if (FIGHTING(ch) == NULL || ch->in_room != FIGHTING(ch)->in_room)
      stop_fighting(ch);
    else {
   victim = FIGHTING(ch);
   ch->fight_round++;
   if (!(ch->fight_round % 2) && !IS_AFFECTED(ch, AFF_HASTE))
    continue;
   if (ch->char_specials.blinkon) {
     send_to_char("You blink into existence.\r\n", ch);
     act("$n blinks into existence.", TRUE, ch, 0, 0, TO_ROOM);
     ch->char_specials.blinkon = 0;
   } else if (AFF2_FLAGGED(ch, AFF2_BLINK)) 
     if (number(1, 20) > 15) {
       send_to_char("You blink out of existence.\r\n", ch);
       act("$n blinks out of existence.", TRUE, ch, 0, 0, TO_ROOM);
       ch->char_specials.blinkon = 1;
       continue;
     }
   if (victim->char_specials.blinkon)
      continue;

   if (((ch->fight_round % 4) == 3) && IS_AFFECTED(ch, AFF_SLOW))
    continue; /* Every other odd round skip attack if slowed - Treb 9/96 */
    if (ch->player.roundsout) { /* if ch is out of comission, don't attack! */
      ch->player.roundsout--;   /* GtG 3/8/97 */
      continue;
    }


      if((GET_POS(ch) != POS_SLEEPING) /*&& (GET_POS(ch) != POS_SITTING)
         && (GET_POS(ch) != POS_RESTING)*/) {
        hit(ch, FIGHTING(ch), TYPE_UNDEFINED);
        if (GET_EQ(ch, WEAR_WIELD))
          if (GET_OBJ_VNUM(GET_EQ(ch, WEAR_WIELD))==1471 || GET_OBJ_VNUM(GET_EQ(ch, WEAR_WIELD))==1472)
            fury_sword(ch);
        }

      for (percent = 1; percent < ch->mob_specials.num_attacks &&
                        FIGHTING(ch); percent++) /* multi attack */
        if (GET_POS(ch) != POS_SLEEPING)
          hit(ch, FIGHTING(ch), TYPE_UNDEFINED);

      percent = number(1, 101);
      if ( (!IS_NPC(ch)) && (FIGHTING(ch)) &&
           (percent < GET_SKILL(ch, SKILL_SECOND_ATTACK)) )
        if((GET_POS(ch) != POS_SLEEPING) /*&& (GET_POS(ch) != POS_SITTING)
           && (GET_POS(ch) != POS_RESTING)*/)
          hit(ch, FIGHTING(ch), TYPE_UNDEFINED);

      percent = number(1, 101);
      if ( (!IS_NPC(ch)) && ((!IS_CLERIC(ch) && !IS_MAGIC_USER(ch)) ||
           IS_IMMORTAL(ch)) && (FIGHTING(ch)) &&
           (percent < GET_SKILL(ch, SKILL_THIRD_ATTACK)) )
        if((GET_POS(ch) != POS_SLEEPING) /*&& (GET_POS(ch) != POS_SITTING)
           && (GET_POS(ch) != POS_RESTING)*/)
          hit(ch, FIGHTING(ch), TYPE_UNDEFINED);

      if (IS_MONK(ch) || IS_WARRIOR(ch) || IS_IMMORTAL(ch) )
        {
          percent = number(1, 101);
          if ( (!IS_NPC(ch)) && (FIGHTING(ch)) &&
            (percent < GET_SKILL(ch, SKILL_FOURTH_ATTACK)) )
          if((GET_POS(ch) != POS_SLEEPING) /*&& (GET_POS(ch) != POS_SITTING)
            && (GET_POS(ch) != POS_RESTING)*/)
          hit(ch, FIGHTING(ch), TYPE_UNDEFINED);
        }
       if (IS_MONK(ch) || IS_IMMORTAL(ch))
         {
         percent = number(1, 101);
         if ( (!IS_NPC(ch)) && (FIGHTING(ch)) &&
           (percent < GET_SKILL(ch, SKILL_FIFTH_ATTACK)) )
         if((GET_POS(ch) != POS_SLEEPING) /*&& (GET_POS(ch) != POS_SITTING)
           && (GET_POS(ch) != POS_RESTING)*/)
         hit(ch, FIGHTING(ch), TYPE_UNDEFINED);

         if (IS_IMMORTAL(ch))
         {
         percent = number(1, 101);
         if ( (!IS_NPC(ch)) && (FIGHTING(ch)) &&
           (percent < GET_SKILL(ch, SKILL_SIXTH_ATTACK)) )
         if((GET_POS(ch) != POS_SLEEPING) /*&& (GET_POS(ch) != POS_SITTING)
           && (GET_POS(ch) != POS_RESTING)*/)
         hit(ch, FIGHTING(ch), TYPE_UNDEFINED);
         }
       }

      if((GET_POS(ch) != POS_SLEEPING) /*&& (GET_POS(ch) != POS_SITTING)
         && (GET_POS(ch) != POS_RESTING)*/)
        if (MOB_FLAGGED(ch, MOB_SPEC) && mob_index[GET_MOB_RNUM(ch)].func
           != NULL)
	  (mob_index[GET_MOB_RNUM(ch)].func) (ch, ch, 0, "");
      if(IS_NPC(ch)) {
        if(GET_POS(ch) == POS_SLEEPING && CHECK_WAIT(ch))
          command_interpreter(ch, "wake");
        /*if(((GET_POS(ch) == POS_SITTING) || (GET_POS(ch) == POS_RESTING))
           && CHECK_WAIT(ch))
          GET_POS(ch) = POS_STANDING;*/
      }
    }
  }
}

ACMD(do_peace)
{
  struct char_data *i, *next_i;

  for (i = world[ch->in_room].people; i; i = next_i) {
    act("You are startled by $n's sudden bellow, 'PEACE!'", TRUE, ch, 0, i, TO_VICT);
    next_i = i->next_in_room;
    if (FIGHTING(i))
      stop_fighting(i);
    }
}

void damage_rooms(int interval)
{
  struct char_data *ch, *tmp;
  struct room_data *room;

  if (interval == 1)
    room = room_combat_damage;
  else
    room = room_tick_damage;

  if (!room)
    return;

  for (; room; room = room->next_death) {
    if (room->damage.interval == interval) {
      for (ch = room->people; ch;) {
        tmp = ch->next_in_room;
        if (!IS_IMMORTAL(ch)) {
          if (*room->damage.mesg)
            send_to_char(room->damage.mesg, ch);
          damage(ch, ch, room->damage.dam, TYPE_DEATHTRAP);
        }
        ch = tmp;
      }
    }
  }
}

void fury_sword(struct char_data *ch)
{
  struct char_data *vict;
  int percent, prob, exp;

  if (!(vict = FIGHTING(ch)))
    return;

  percent = ((GET_LEVEL(ch) - GET_LEVEL(vict)) / 5) + 5;
  if (IS_IMMORTAL(ch)) percent = 105;
  prob = number(1, 101);
  if (percent > prob) {
    /* print pretty messages */
    send_to_char("THERE CAN BE ONLY ONE!\r\n", ch);
    act("$n beans you with the bodiless head of $N as $e severs it.", FALSE, ch, 0, vict, TO_NOTVICT);
    act("The last thing you see as life flickers from you is the grinning face of $N as $E stands over your decapitated corpse.", FALSE, vict, 0, ch, TO_CHAR);
    if (IS_NPC(vict) || vict->desc) {
      if (IS_AFFECTED(ch, AFF_GROUP) && (ch->master || ch->followers))
	group_gain(ch, vict);
      else {
	/* Calculate level-difference bonus */
	exp = MIN(max_exp_gain, GET_EXP(vict));

	if (IS_NPC(ch))
	  exp += MAX(0, (exp * MIN(4, (GET_LEVEL(vict) - GET_LEVEL(ch)))) >> 3);
	else
	  exp += MAX(0, (exp * MIN(8, (GET_LEVEL(vict) / GET_LEVEL(ch)))) >> 3);
        exp = MIN(max_exp_gain, exp);

	if (exp > 1) {
	  sprintf(buf2, "You receive %d experience points.\r\n", exp);
	  send_to_char(buf2, ch);
	} else if (exp == 1)
          send_to_char("You don't get SQUAT!\r\n", ch);
        else
	  send_to_char("You receive one lousy experience point.\r\n", ch);
	gain_exp(ch, exp);
      }
    }
    die(vict, ch);
  }

  return;
}
