/* ************************************************************************
*   File: objsave.c                                     Part of CircleMUD *
*  Usage: loading/saving player objects for rent and crash-save           *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"


#include "structs.h"
#include "comm.h"
#include "handler.h"
#include "db.h"
#include "interpreter.h"
#include "utils.h"
#include "spells.h"

/* these factors should be unique integers */
#define RENT_FACTOR 	1
#define CRYO_FACTOR 	4

/* dwix: prototyping internal function to avoid warnings */
int Crash_is_unrentable(struct obj_data * obj);
void encode_mail(char *str);
void decode_mail(char *str);
void dummy(void);

int obj_save_count;	/* ugly hack to limit item saving in rooms */

extern struct str_app_type str_app[];
extern struct room_data *world;
extern struct index_data *mob_index;
extern struct index_data *obj_index;
extern struct descriptor_data *descriptor_list;
extern struct player_index_element *player_table;
extern int top_of_p_table;
extern int min_rent_cost;
extern struct obj_data *obj_proto;

/* Extern functions */
ACMD(do_tell);
SPECIAL(receptionist);
SPECIAL(cryogenicist);
int find_name(char *name);

struct obj_data *Obj_from_store(struct obj_file_elem object)
{
  struct obj_data *obj;
  int j;

  if (real_object(object.item_number) > -1) {
    obj = read_object(object.item_number, VIRTUALRM);
    GET_OBJ_VAL(obj, 0) = object.value[0];
    GET_OBJ_VAL(obj, 1) = object.value[1];
    GET_OBJ_VAL(obj, 2) = object.value[2];
    GET_OBJ_VAL(obj, 3) = object.value[3];
    GET_OBJ_EXTRA(obj) = object.extra_flags;
    GET_OBJ_WEIGHT(obj) = object.weight;
    GET_OBJ_TIMER(obj) = object.timer;
    obj->obj_flags.bitvector = object.bitvector;
    if (!IS_SET(object.bitvector2, 1 << 5))
      obj->obj_flags.bitvector2 = object.bitvector2;
    obj->obj_flags.owner = (object.maker ? object.maker : 0);
    obj->worn_on = object.worn_on;
    obj->restring = object.restring;

    if(IS_SET(object.extra_flags, ITEM_MAIL)) {
      CREATE(obj->mail, struct mail_data, 1);
      obj->mail->from = object.from;
      obj->mail->to = object.to;
      obj->mail->flags = object.mailflags;
      obj->mail->date = object.maildate;
      if(object.subject)
	obj->mail->subject = object.subject;
      else
	obj->mail->subject = NULL;
    }
    if(object.text) {
      if(IS_SET(object.extra_flags, ITEM_MAIL))
	decode_mail(object.text);
      obj->action_description = object.text;
    }
    /* cstmod */
    if(object.short_desc)
      obj->short_description = object.short_desc;
    if(object.long_desc)
      obj->description = object.long_desc;
    if(object.name)
      obj->name = object.name;

    for (j = 0; j < MAX_OBJ_AFFECT; j++)
      obj->affected[j] = object.affected[j];

    return obj;
  } else
    return NULL;
}



int Obj_to_store(struct obj_data * obj, FILE * fl)
{
  int j, i, rnum;
  struct obj_file_elem object;
  struct obj_affected_type *aff;
  char bits[40];

  void sprintbits(long vektor, char *outstring);

  memset((char *) &object, 0, sizeof(struct obj_file_elem));

  object.item_number = GET_OBJ_VNUM(obj);
  rnum = real_object(object.item_number);
  object.value[0] = GET_OBJ_VAL(obj, 0);
  object.value[1] = GET_OBJ_VAL(obj, 1);
  object.value[2] = GET_OBJ_VAL(obj, 2);
  object.value[3] = GET_OBJ_VAL(obj, 3);
  object.extra_flags = GET_OBJ_EXTRA(obj);
  object.weight = GET_OBJ_WEIGHT(obj);
  object.timer = GET_OBJ_TIMER(obj);
  object.bitvector = obj->obj_flags.bitvector;
  object.bitvector2 = obj->obj_flags.bitvector2;
  object.maker = obj->obj_flags.owner;
  object.text = NULL;
  object.restring = obj->restring;
  if(obj->mail) {
    object.to = obj->mail->to;
    object.from = obj->mail->from;
    object.mailflags = obj->mail->flags;
    object.maildate = obj->mail->date;
    object.subject = obj->mail->subject;
  }
  if(obj->action_description) {
    object.text = str_dup(obj->action_description);
    if(IS_SET(GET_OBJ_EXTRA(obj), ITEM_MAIL))
      encode_mail(object.text);
  }
  /* cstmod */
  if (strcmp(obj->short_description, obj_proto[rnum].short_description)) {
    object.short_desc = str_dup(obj->short_description);  }    
  if (strcmp(obj->description, obj_proto[rnum].description)) {
    object.long_desc = str_dup(obj->description);  }    
  if (strcmp(obj->name, obj_proto[rnum].name)) {
    object.name = str_dup(obj->name);  }    

  /* dwix: begin */
  if (obj->contains) object.is_container = 1;
  else object.is_container = 0;
 
  if (obj->next_content)
    object.is_last_content = 0;
  else if (!obj->carried_by)
    object.is_last_content = 1;
  else
    object.is_last_content = 0;
  /* dwix: end */

   if ( (obj->carried_by) || (obj->in_obj) )
    object.worn_on = -1;
  else
    object.worn_on = obj->worn_on;
 
  for (j = 0; j < MAX_OBJ_AFFECT; j++)
    object.affected[j] = obj->affected[j];

/* new code to write to the ascii pfile */

      fprintf(fl, "Vnum: %d\n", object.item_number);
      if(object.value[0] != obj_proto[rnum].obj_flags.value[0])
	fprintf(fl, "Val0: %d\n", object.value[0]);
      if(object.value[1] != obj_proto[rnum].obj_flags.value[1])
	fprintf(fl, "Val1: %d\n", object.value[1]);
      if(object.value[2] != obj_proto[rnum].obj_flags.value[2])
	fprintf(fl, "Val2: %d\n", object.value[2]);
      if(object.value[3] != obj_proto[rnum].obj_flags.value[3])
	fprintf(fl, "Val3: %d\n", object.value[3]);
      if(object.value[4] != obj_proto[rnum].obj_flags.value[4])
        fprintf(fl, "Val4: %d\n", object.value[4]);
      if(object.extra_flags != obj_proto[rnum].obj_flags.extra_flags) {
	sprintbits(object.extra_flags, bits);
	fprintf(fl, "Extr: %s\n", bits);
      }
      if(object.maker)
        fprintf(fl, "Mdby: %ld\n", object.maker);
      if(object.weight != obj_proto[rnum].obj_flags.weight)
	fprintf(fl, "Wate: %d\n", object.weight);
      if(object.timer != obj_proto[rnum].obj_flags.timer)
	fprintf(fl, "Time: %d\n", object.timer);
      if(object.bitvector != obj_proto[rnum].obj_flags.bitvector) {
	sprintbits(object.bitvector, bits);
	fprintf(fl, "Bits: %s\n", bits);
      }
      if(object.bitvector2 != obj_proto[rnum].obj_flags.bitvector2) {
        sprintbits(object.bitvector2, bits);
        fprintf(fl, "Bit2: %s\n", bits);
      }
      if(object.worn_on != -1)
	fprintf(fl, "Worn: %d\n", object.worn_on);
      if(object.is_container != 0)
	fprintf(fl, "Cont: %d\n", object.is_container);
      if(object.is_last_content == 1 && object.worn_on < 0)
	fprintf(fl, "Last: %d\n", object.is_last_content);

      if(object.text && (!(obj_proto[rnum].action_description) ||
	strcmp(object.text, obj_proto[rnum].action_description)))
	fprintf(fl, "Text: \n%s~\n", object.text);

      if(object.from > 0)
	fprintf(fl, "From: %d\n", (int) object.from);
      if(object.to > 0)
	fprintf(fl, "To  : %d\n", (int) object.to);
      fprintf(fl, "Mdte: %d\n", (int) object.maildate);
      if(object.mailflags != 0) {
	sprintbits(object.mailflags, bits);
	fprintf(fl, "Mflg: %s\n", bits);
      }
      if(object.subject)
	fprintf(fl, "Subj: %s\n", object.subject);

      /* cstmod */
      if(object.short_desc)
        fprintf(fl, "Sdsc: %s\n", object.short_desc);
      if(object.long_desc)
        fprintf(fl, "Ldsc: %s\n", object.long_desc);
      if(object.name)
        fprintf(fl, "Name: %s\n", object.name);
        
      fprintf(fl, "Affs:\n");
      for(i = 0; i < MAX_OBJ_AFFECT; i++) {
	aff = &(object.affected[i]);
	if(aff->location)
	  fprintf(fl, "%d %d\n", aff->location, aff->modifier);
      }
      fprintf(fl, "0 0\n~\n");

      if(object.text)
	free(object.text);
	/* cstmod */
      if(object.short_desc)
        free(object.short_desc);
      if(object.long_desc)
        free(object.long_desc);
      if(object.name)
        free(object.name);

  return 1;
}



void Crash_delete_file(char *name)
{
  char filename[40];

  sprintf(filename, "%s/%c/%s", RENT_PREFIX, *name, name);
  remove(filename);
}


/*
int Crash_delete_crashfile(struct char_data * ch)
{
  char fname[MAX_INPUT_LENGTH];
  struct rent_info rent;
  FILE *fl;

  if (!get_filename(GET_NAME(ch), fname, CRASH_FILE))
    return 0;
  if (!(fl = fopen(fname, "rb"))) {
    if (errno != ENOENT) {	* if it fails, NOT because of no file *
      sprintf(buf1, "SYSERR: checking for crash file %s (3)", fname);
      perror(buf1);
    }
    return 0;
  }
  if (!feof(fl))
    fread(&rent, sizeof(struct rent_info), 1, fl);
  fclose(fl);

  if (rent.rentcode == RENT_CRASH)
    Crash_delete_file(GET_NAME(ch));

  return 1;
}
*/

/*
int Crash_clean_file(char *name)
{
  char fname[MAX_STRING_LENGTH], filetype[20];
  struct rent_info rent;
  extern int rent_file_timeout, crash_file_timeout;
  FILE *fl;

  if (!get_filename(name, fname, CRASH_FILE))
    return 0;
  *
   * open for write so that permission problems will be flagged now, at boot
   * time.
   *
  if (!(fl = fopen(fname, "r+b"))) {
    if (errno != ENOENT) {	* if it fails, NOT because of no file *
      sprintf(buf1, "SYSERR: OPENING OBJECT FILE %s (4)", fname);
      perror(buf1);
    }
    return 0;
  }
  if (!feof(fl))
    fread(&rent, sizeof(struct rent_info), 1, fl);
  fclose(fl);

  if ((rent.rentcode == RENT_CRASH) ||
      (rent.rentcode == RENT_FORCED) || (rent.rentcode == RENT_TIMEDOUT)) {
    if (rent.time < time(0) - (crash_file_timeout * SECS_PER_REAL_DAY)) {
      Crash_delete_file(name);
      switch (rent.rentcode) {
      case RENT_CRASH:
	strcpy(filetype, "crash");
	break;
      case RENT_FORCED:
	strcpy(filetype, "forced rent");
	break;
      case RENT_TIMEDOUT:
	strcpy(filetype, "idlesave");
	break;
      default:
	strcpy(filetype, "UNKNOWN!");
	break;
      }
      sprintf(buf, "    Deleting %s's %s file.", name, filetype);
      nmlog(buf);
      return 1;
    }
    * Must retrieve rented items w/in 30 days *
  } else if (rent.rentcode == RENT_RENTED)
    if (rent.time < time(0) - (rent_file_timeout * SECS_PER_REAL_DAY)) {
      Crash_delete_file(name);
      sprintf(buf, "    Deleting %s's rent file.", name);
      nmlog(buf);
      return 1;
    }
  return (0);
}
*/


/*void update_obj_file(void)
{
  int i;

  for (i = 0; i <= top_of_p_table; i++)
    Crash_clean_file((player_table + i)->name);
  return;
}
*/


void Crash_listrent(struct char_data * ch, char *name)
{
  FILE *fl;
  char fname[MAX_INPUT_LENGTH], buf[MAX_STRING_LENGTH];
  char line[MAX_STRING_LENGTH], tag[6];
  int num, player;
  struct obj_file_elem object;
  struct obj_data *obj;
  void tag_argument(char *argument, char *tag);

  if((player = find_name(name)) < 0) {
    send_to_char("No such player.\r\n", ch);
    return;
  }

  sprintf(fname, "%s/%c/%s", RENT_PREFIX,
         *player_table[player].name,
         player_table[player].name);
           
  if (!(fl = fopen(fname, "r"))) {
    sprintf(buf, "%s has no rent file.\r\n", name);
    send_to_char(buf, ch);
    return;
  }
  sprintf(buf, "%s\r\n", fname);

  while (!feof(fl)) {
    get_line(fl, line);
    if (*line!='~') {
      tag_argument(line, tag);
      num = atoi(line);
      if (!strcmp(tag, "Vnum")) {
        object.item_number = num;
        if (real_object(object.item_number) > -1) {
          obj = read_object(object.item_number, VIRTUALRM);
          sprintf(buf, "%s [%5d] (%3d) %-20s\r\n", buf,
                  object.item_number, object.weight, obj->short_description);
          extract_obj(obj);
        }
      }
    }
  }
  page_string(ch->desc, buf, TRUE);
  fclose(fl);
}


/*
int Crash_write_rentcode(struct char_data * ch, FILE * fl, struct rent_info * rent)
{
  if (fwrite(rent, sizeof(struct rent_info), 1, fl) < 1) {
    perror("Writing rent code.");
    return 0;
  }
  return 1;
}
*/


struct obj_data *Crash_load(struct char_data * ch, FILE *fl)
{
  struct obj_file_elem object;
  int containers_deep; /* dwix */
  struct obj_data *containers[50]; /* dwix */
  struct obj_data *objtemp, *top_obj = NULL, *last_obj = NULL;
  char line[MAX_STRING_LENGTH], tag[6];
  int num, num2, i, rnum;
  char buf[MAX_STRING_LENGTH];

  void tag_argument(char *argument, char *tag);
  long asciiflag_conv(char *flag);
/*
  sprintf(buf, "%s entering NoMUD.", GET_NAME(ch));
  mudlog(buf, NRM, MAX(LVL_IMMORT, GET_INVIS_LEV(ch)), TRUE);
*/
  containers_deep = 0; /* dwix: init deepness to 0 */

  while (!feof(fl)) {

    get_line(fl, line);

    if(*line != '~') {

      /*if(!strcmp(line, "Mail")) for future pfile expansion */

      tag_argument(line, tag);
      num = atoi(line);

      switch (*tag) {
      case 'A':
        if(!strcmp(tag, "Affs")) {
	  i = 0;
	  do {
	    get_line(fl, line);
	    sscanf(line, "%d %d", &num, &num2);
	    if(num != 0) {
	      object.affected[i].location = num;
	      object.affected[i].modifier = num2;
	    }
	    i++;
	  } while (num != 0);
        }
        break;

      case 'B':
        if(!strcmp(tag, "Bits"))
  	  object.bitvector = asciiflag_conv(line);
  	if (!strcmp(tag, "Bit2")) {
  	  object.bitvector2 = asciiflag_conv(line);
          if (IS_SET(object.bitvector2, AFF2_RESERVED))
            object.bitvector2 = 0;
        }
        break;

      case 'C':
        if(!strcmp(tag, "Cont"))
	  object.is_container = num;
        break;

      case 'E':
        if(!strcmp(tag, "Extr"))
	  object.extra_flags = asciiflag_conv(line);
        break;

      case 'F':
        if(!strcmp(tag, "From"))
	  object.from = num;
        break;

      case 'L':
        if(!strcmp(tag, "Last"))
	  object.is_last_content = num;
	if(!strcmp(tag, "Ldsc")) {
          object.restring = TRUE;
	  object.long_desc = str_dup(line);
        }
        break;

      case 'M':
        if(!strcmp(tag, "Mflg"))
	  object.mailflags = asciiflag_conv(line);
	if(!strcmp(tag, "Mdte"))
	  object.maildate = num;
	if(!strcmp(tag, "Mdby"))
	   object.maker = num;
        break;

      case 'N':
	if(!strcmp(tag, "Name")) {
          object.restring = TRUE;
	  object.name = str_dup(line);
        }
        break;

      case 'S':
        if(!strcmp(tag, "Subj"))
	  object.subject = str_dup(line);
	if(!strcmp(tag, "Sdsc")) {
          object.restring = TRUE;
	  object.short_desc = str_dup(line);
        }
        break;

      case 'T':
	if(!strcmp(tag, "Text")) {
	  object.text = NULL;
	  object.text = fread_string(fl, buf);
	}
        else if(!strcmp(tag, "Time"))
	  object.timer = num;
        else if(!strcmp(tag, "To  "))
	  object.to = num;
        break;

      case 'V':
        if(!strcmp(tag, "Val0"))
	  object.value[0] = num;
        else if(!strcmp(tag, "Val1"))
	  object.value[1] = num;
        else if(!strcmp(tag, "Val2"))
	  object.value[2] = num;
        else if(!strcmp(tag, "Val3"))
	  object.value[3] = num;
	else if(!strcmp(tag, "Val4"))
	  object.value[4] = num;
        else if(!strcmp(tag, "Vnum")) {
	  object.item_number = num;
	  /* initialize defaults */
	  rnum = real_object(num);
	  object.value[0]		= obj_proto[rnum].obj_flags.value[0];
	  object.value[1]		= obj_proto[rnum].obj_flags.value[1];
	  object.value[2]		= obj_proto[rnum].obj_flags.value[2];
	  object.value[3]		= obj_proto[rnum].obj_flags.value[3];
	  object.extra_flags	= obj_proto[rnum].obj_flags.extra_flags;
	  object.weight		= obj_proto[rnum].obj_flags.weight;
	  object.timer		= obj_proto[rnum].obj_flags.timer;
	  object.bitvector	= obj_proto[rnum].obj_flags.bitvector;
	  object.bitvector2     = obj_proto[rnum].obj_flags.bitvector2;
	  object.worn_on		= -1;
	  object.is_container	= 0;
	  object.is_last_content	= 0;
	  object.from		= NOBODY;
	  object.to		= NOBODY;
	  object.mailflags	= 0;
	  object.maildate       = 0;
	  object.subject	= NULL;
          object.restring = FALSE;
	  if(obj_proto[rnum].action_description)
	    object.text = obj_proto[rnum].action_description;
	  else
	    object.text = NULL;
	  for(i = 0; i < MAX_OBJ_AFFECT; i++) {
	    object.affected[i].location = 0;
	    object.affected[i].modifier = 0;
	  }
	  object.short_desc = NULL;
	  object.long_desc = NULL;
	  object.name = NULL;
        }
        break;

      case 'W':
        if(!strcmp(tag, "Wate"))
	  object.weight = num;
        else if(!strcmp(tag, "Worn"))
	  object.worn_on = num;
        break;
      }
    } else if(object.item_number != NOTHING) {
      if ((objtemp = Obj_from_store(object))) {
	if(ch) {
        /* dwix: begin */
          if (objtemp->worn_on < 0) {
            if (containers_deep)
	      obj_to_obj(objtemp, containers[containers_deep]); /* dwix */
            else
	      obj_to_char(objtemp, ch);
            if (object.is_last_content)
	      containers_deep--; /* dwix */
            if (containers_deep < 0)
	      containers_deep = 0;
            if (object.is_container)
	      containers[++containers_deep] = objtemp; /* dwix */
          } else {
            if (containers_deep)
	      obj_to_obj(objtemp, containers[containers_deep]);
	    else
	      equip_char(ch, objtemp, objtemp->worn_on);
	    if (object.is_last_content)
	      containers_deep--;
            if (containers_deep < 0)
              containers_deep = 0;
            if (object.is_container)
              containers[++containers_deep] = objtemp; /* dwix */
           } /* dwix: end */
	} else if(!top_obj) {
	  top_obj = objtemp;
	  last_obj = top_obj;
	} else {
          if (containers_deep)
	    obj_to_obj(objtemp, containers[containers_deep]); /* dwix */
          else {
	    last_obj->next_content = objtemp;
	    last_obj = objtemp;
	  }
          if (object.is_last_content)
	    containers_deep--; /* dwix */
          if (containers_deep < 0)
	    containers_deep = 0;
          if (object.is_container)
	    containers[++containers_deep] = objtemp; /* dwix */
	}
      }
      object.item_number = NOTHING;
    }
  }

  return(top_obj);
}



int Crash_save(struct char_data *ch, struct obj_data * obj, int
		max_save, FILE * fp)
{
  struct obj_data *tmp;
  int result = 0;
 
  if (obj) {

    for (tmp = obj->contains; tmp; tmp = tmp->next_content)
      GET_OBJ_WEIGHT(obj) -= GET_OBJ_WEIGHT(tmp);
 
    if (Crash_is_unrentable(obj)) {
      Crash_save(ch, obj->next_content, max_save, fp);
      return TRUE;
    }
    if ((max_save == 0 || obj_save_count <= max_save)) {
      result = Obj_to_store(obj, fp);
      obj_save_count++;
    }/* else if(ch->quitting) {
      obj_from_char(obj);
      obj_to_room(obj, real_room(donation_room_1));
    }*/
 
    if(ch) {
      Crash_save(ch, obj->contains, max_save, fp);
      Crash_save(ch, obj->next_content, max_save, fp);
    } 

    if(ch)
      if(ch->quitting)
        extract_obj(obj);

    obj_save_count = 0;

    if (!result)
      return 0;
  }
  return TRUE;
}

void Crash_restore_weight(struct obj_data * obj)
{
  if (obj) {
    Crash_restore_weight(obj->contains);
    Crash_restore_weight(obj->next_content);
    if (obj->in_obj)
      GET_OBJ_WEIGHT(obj->in_obj) += GET_OBJ_WEIGHT(obj);
  }
}



void Crash_extract_objs(struct obj_data * obj)
{
  if (obj) {
    Crash_extract_objs(obj->contains);
    Crash_extract_objs(obj->next_content);
    extract_obj(obj);
  }
}


int Crash_is_unrentable(struct obj_data * obj)
{
  if (!obj)
    return 0;

  if (IS_OBJ_STAT(obj, ITEM_NORENT) || GET_OBJ_RENT(obj) < 0 ||
      GET_OBJ_RNUM(obj) <= NOTHING || GET_OBJ_TYPE(obj) == ITEM_KEY
      || (GET_OBJ_TYPE(obj) == ITEM_POTION && GET_OBJ_MIN(obj) >= 10))
    return 1;
  return 0;
}


void Crash_extract_norents(struct obj_data * obj)
{
  if (obj) {
    Crash_extract_norents(obj->contains);
    Crash_extract_norents(obj->next_content);
    if (Crash_is_unrentable(obj))
      extract_obj(obj);
  }
}


void Crash_extract_expensive(struct obj_data * obj)
{
  struct obj_data *tobj, *max;

  max = obj;
  for (tobj = obj; tobj; tobj = tobj->next_content)
    if (GET_OBJ_RENT(tobj) > GET_OBJ_RENT(max))
      max = tobj;
  extract_obj(max);
}



void Crash_calculate_rent(struct obj_data * obj, int *cost)
{
  if (obj) {
    *cost += MAX(0, GET_OBJ_RENT(obj));
    Crash_calculate_rent(obj->contains, cost);
    Crash_calculate_rent(obj->next_content, cost);
  }
}


void Crash_crashsave(struct char_data * ch, FILE *fp)
{
  int j;

  if (!Crash_save(ch, ch->carrying, 0, fp)) {
    fclose(fp);
    return;
  }
  Crash_restore_weight(ch->carrying);

  for (j = 0; j < NUM_WEARS; j++)
    if (GET_EQ(ch, j)) {
      if (!Crash_save(ch, GET_EQ(ch, j), 0, fp)) {
	return;
      }
      Crash_restore_weight(GET_EQ(ch, j));
    }
  if(!IS_NPC(ch))
    REMOVE_BIT(PLR_FLAGS(ch), PLR_CRASH);
}


/*void Crash_idlesave(struct char_data * ch)
{
  char buf[MAX_INPUT_LENGTH];
  struct rent_info rent;
  int j;
  int cost;
  FILE *fp;

  if (IS_NPC(ch))
    return;

  if (!get_filename(GET_NAME(ch), buf, CRASH_FILE))
    return;
  if (!(fp = fopen(buf, "wb")))
    return;

  for (j = 0; j < NUM_WEARS; j++)
    if (GET_EQ(ch, j))
      obj_to_char(unequip_char(ch, j), ch);

  Crash_extract_norents(ch->carrying);

  cost = 0;
  Crash_calculate_rent(ch->carrying, &cost);
  cost <<= 1;			* forcerent cost is 2x normal rent *
  while ((cost > GET_GOLD(ch) + GET_BANK_GOLD(ch)) && ch->carrying) {
    Crash_extract_expensive(ch->carrying);
    cost = 0;
    Crash_calculate_rent(ch->carrying, &cost);
    cost <<= 1;
  }

  if (!ch->carrying) {
    fclose(fp);
    Crash_delete_file(GET_NAME(ch));
    return;
  }
  rent.net_cost_per_diem = cost;

  rent.rentcode = RENT_TIMEDOUT;
  rent.time = time(0);
  rent.gold = GET_GOLD(ch);
  rent.account = GET_BANK_GOLD(ch);
  if (!Crash_write_rentcode(ch, fp, &rent)) {
    fclose(fp);
    return;
  }
  if (!Crash_save(ch, ch->carrying, fp)) {
    fclose(fp);
    return;
  }
  fclose(fp);

  Crash_extract_objs(ch->carrying);
}
*/


/*void Crash_rentsave(struct char_data * ch, int cost)
{
  char buf[MAX_INPUT_LENGTH];
  struct rent_info rent;
  int j;
  FILE *fp;

  if (IS_NPC(ch))
    return;

  if (!get_filename(GET_NAME(ch), buf, CRASH_FILE))
    return;
  if (!(fp = fopen(buf, "wb")))
    return;

  for (j = 0; j < NUM_WEARS; j++)
    if (GET_EQ(ch, j))
*      obj_to_char(unequip_char(ch, j), ch); *

  Crash_extract_norents(ch->carrying);

  rent.net_cost_per_diem = cost;
  rent.rentcode = RENT_RENTED;
  rent.time = time(0);
  rent.gold = GET_GOLD(ch);
  rent.account = GET_BANK_GOLD(ch);
  if (!Crash_write_rentcode(ch, fp, &rent)) {
    fclose(fp);
    return;
  }
  for (j = 0; j < NUM_WEARS; j++)
    if (ch->equipment[j])
     Crash_save(ch, ch->equipment[j], fp);
 
  if (!Crash_save(ch, ch->carrying, fp)) {
    fclose(fp);
    return;
  }
  fclose(fp);

  for (j = 0; j < NUM_WEARS; j++)
    if (ch->equipment[j])
     Crash_extract_objs(ch->equipment[j]);
  Crash_extract_objs(ch->carrying);
}
*/


/*void Crash_cryosave(struct char_data * ch, int cost)
{
  char buf[MAX_INPUT_LENGTH];
  struct rent_info rent;
  int j;
  FILE *fp;

  if (IS_NPC(ch))
    return;

  if (!get_filename(GET_NAME(ch), buf, CRASH_FILE))
    return;
  if (!(fp = fopen(buf, "wb")))
    return;

  for (j = 0; j < NUM_WEARS; j++)
    if (GET_EQ(ch, j))
      obj_to_char(unequip_char(ch, j), ch);

  Crash_extract_norents(ch->carrying);

  GET_GOLD(ch) = MAX(0, GET_GOLD(ch) - cost);

  rent.rentcode = RENT_CRYO;
  rent.time = time(0);
  rent.gold = GET_GOLD(ch);
  rent.account = GET_BANK_GOLD(ch);
  rent.net_cost_per_diem = 0;
  if (!Crash_write_rentcode(ch, fp, &rent)) {
    fclose(fp);
    return;
  }
  if (!Crash_save(ch, ch->carrying, fp)) {
    fclose(fp);
    return;
  }
  fclose(fp);

  Crash_extract_objs(ch->carrying);
}
*/


/* ************************************************************************
* Routines used for the receptionist					  *
************************************************************************* */

/*void Crash_rent_deadline(struct char_data * ch, struct char_data *
recep,
			      long cost)
{
  long rent_deadline;

  if (!cost)
    return;

  rent_deadline = ((GET_GOLD(ch) + GET_BANK_GOLD(ch)) / cost);
  sprintf(buf,
      "$n tells you, 'You can rent for %ld day%s with the gold you have\r\n"
	  "on hand and in the bank.'\r\n",
	  rent_deadline, (rent_deadline > 1) ? "s" : "");
  act(buf, FALSE, recep, 0, ch, TO_VICT);
}
*/
/*
int Crash_report_unrentables(struct char_data * ch, struct char_data * recep,
			         struct obj_data * obj)
{
  char buf[128];
  int has_norents = 0;

  if (obj) {
    if (Crash_is_unrentable(obj)) {
      has_norents = 1;
      sprintf(buf, "$n tells you, 'You cannot store %s.'", OBJS(obj, ch));
      act(buf, FALSE, recep, 0, ch, TO_VICT);
    }
    has_norents += Crash_report_unrentables(ch, recep, obj->contains);
    has_norents += Crash_report_unrentables(ch, recep, obj->next_content);
  }
  return (has_norents);
}
*/

/*
void Crash_report_rent(struct char_data * ch, struct char_data * recep,
		            struct obj_data * obj, long *cost, long *nitems, int display, int factor)
{
  static char buf[256];

  if (obj) {
    if (!Crash_is_unrentable(obj)) {
      (*nitems)++;
      *cost += MAX(0, (GET_OBJ_RENT(obj) * factor));
      if (display) {
	sprintf(buf, "$n tells you, '%5d coins for %s..'",
		(GET_OBJ_RENT(obj) * factor), OBJS(obj, ch));
	act(buf, FALSE, recep, 0, ch, TO_VICT);
      }
    }
    Crash_report_rent(ch, recep, obj->contains, cost, nitems, display, factor);
    Crash_report_rent(ch, recep, obj->next_content, cost, nitems, display, factor);
  }
}
*/

/*
int Crash_offer_rent(struct char_data * ch, struct char_data * receptionist,
		         int display, int factor)
{
  extern int max_obj_save;	* change in config.c *
  char buf[MAX_INPUT_LENGTH];
  int i;
  long totalcost = 0, numitems = 0, norent = 0;

  norent = Crash_report_unrentables(ch, receptionist, ch->carrying);
  for (i = 0; i < NUM_WEARS; i++)
    norent += Crash_report_unrentables(ch, receptionist, GET_EQ(ch, i));

  if (norent)
    return 0;

  totalcost = min_rent_cost * factor;

  Crash_report_rent(ch, receptionist, ch->carrying, &totalcost, &numitems, display, factor);

  for (i = 0; i < NUM_WEARS; i++)
    Crash_report_rent(ch, receptionist, GET_EQ(ch, i), &totalcost, &numitems, display, factor);

  if (!numitems) {
    act("$n tells you, 'But you are not carrying anything!  Just quit!'",
	FALSE, receptionist, 0, ch, TO_VICT);
    return (0);
  }
  if (numitems > max_obj_save) {
    sprintf(buf, "$n tells you, 'Sorry, but I cannot store more than %d items.'",
	    max_obj_save);
    act(buf, FALSE, receptionist, 0, ch, TO_VICT);
    return (0);
  }
  if (display) {
    sprintf(buf, "$n tells you, 'Plus, my %d coin fee..'",
	    min_rent_cost * factor);
    act(buf, FALSE, receptionist, 0, ch, TO_VICT);
    sprintf(buf, "$n tells you, 'For a total of %ld coins%s.'",
	    totalcost, (factor == RENT_FACTOR ? " per day" : ""));
    act(buf, FALSE, receptionist, 0, ch, TO_VICT);
    if (totalcost > GET_GOLD(ch)) {
      act("$n tells you, '...which I see you can't afford.'",
	  FALSE, receptionist, 0, ch, TO_VICT);
      return (0);
    } else if (factor == RENT_FACTOR)
      Crash_rent_deadline(ch, receptionist, totalcost);
  }
  return (totalcost);
}
*/


int gen_receptionist(struct char_data * ch, struct char_data * recep,
		         int cmd, char *arg, int mode)
{
  /*int cost = 0;
  extern int free_rent;
  sh_int save_room;*/
  char *action_table[] = {"smile", "dance", "sigh", "blush", "burp",
  "cough", "fart", "twiddle", "yawn"};

  ACMD(do_action);

  if (!ch->desc || IS_NPC(ch))
    return FALSE;

  if (!cmd && !number(0, 5)) {
    do_action(recep, "", find_command(action_table[number(0, 8)]), 0);
    return FALSE;
  }
  if (!CMD_IS("offer") && !CMD_IS("rent"))
    return FALSE;
  if (!AWAKE(recep)) {
    send_to_char("She is unable to talk to you...\r\n", ch);
    return TRUE;
  }
  if (!CAN_SEE(recep, ch)) {
    act("$n says, 'I don't deal with people I can't see!'", FALSE, recep, 0, 0, TO_ROOM);
    return TRUE;
  }
  act("$n tells you, 'Rent is free here.  Just quit, and your objects will be saved!'",
	FALSE, recep, 0, ch, TO_VICT);
  return 1;
}


SPECIAL(receptionist)
{
  return (gen_receptionist(ch, me, cmd, argument, RENT_FACTOR));
}


SPECIAL(cryogenicist)
{
  return (gen_receptionist(ch, me, cmd, argument, CRYO_FACTOR));
}


void Crash_save_all(void)
{
  struct descriptor_data *d;
  for (d = descriptor_list; d; d = d->next) {
    if ((d->connected == CON_PLAYING) && !IS_NPC(d->character)) {
      if (PLR_FLAGGED(d->character, PLR_CRASH)) {
        d->character->quitting = FALSE;
	save_rent(d->character);
	save_char(d->character, NOWHERE);
	REMOVE_BIT(PLR_FLAGS(d->character), PLR_CRASH);
      }
    }
  }
}


void encode_mail(char *str)
{
  char *ptr, c;

  for(ptr = str; *ptr; ptr++)
    if(*ptr) {
      c = *ptr;
      if(c >= 'A' && c <= 'Z')
	*ptr = 'A' + 'Z' - c;
      else if(c >= 'a' && c <= 'z')
	*ptr = 'a' + 'z' - c;
      else if(c == '\'')
	*ptr = '`';
      else if(c == '~')
	*ptr = '\'';
      else if(c == '.')
	*ptr = ' ';
      else if(c == ' ')
	*ptr = '.';
    }
}


void decode_mail(char *str)
{
  char *ptr, c;

  for(ptr = str; *ptr; ptr++)
    if(*ptr) {
      c = *ptr;
      if(c >= 'A' && c <= 'Z')
	*ptr = 'A' + 'Z' - c;
      else if(c >= 'a' && c <= 'z')
	*ptr = 'a' + 'z' - c;
      else if(c == '\'')
	*ptr = '~';
      else if(c == '`')
	*ptr = '\'';
      else if(c == '.')
	*ptr = ' ';
      else if(c == ' ')
	*ptr = '.';
    }
}

int load_rent(struct char_data *ch)
{
  FILE *fl;
  char filename[40];
  char *p;

  sprintf(filename, "%s/%c/%s", RENT_PREFIX, *ch->player.name, 
    ch->player.name);
  for(p = filename; *p; p++)
    *p = LOWER(*p); 
  if(!(fl = fopen(filename, "r"))) {
    return (0);
  }

  Crash_load(ch, fl);
  fclose(fl);
  return(1);
}


void save_rent(struct char_data * ch)
{
  FILE *fl;
  char filename[40];
  char *p;

  if((ch->desc && ch->desc->original) || !*ch->player.name)
    return;

  sprintf(filename, "%s/%c/%s", RENT_PREFIX, *ch->player.name,
    ch->player.name);
  for(p = filename; *p; p++)
    *p = LOWER(*p); 

  chmod(filename, S_IREAD | S_IWRITE | S_IRGRP | S_IWGRP);

  if (!(fl = fopen(filename, "w"))) {
    sprintf(buf, "error opening output file: %s", filename);
    perror(buf);
    exit(1);
  }

  Crash_crashsave(ch, fl);
  fclose(fl);
  chmod(filename, S_IREAD | S_IWRITE | S_IRGRP | S_IWGRP);
}
