/* ************************************************************************
*   File: spec_procs.c                                  Part of CircleMUD *
*  Usage: implementation of special procedures for mobiles/objects/rooms  *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"


#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "handler.h"
#include "db.h"
#include "spells.h"


/*   external vars  */
extern struct room_data *world;
extern struct char_data *character_list;
extern struct descriptor_data *descriptor_list;
extern struct index_data *mob_index;
extern struct index_data *obj_index;
extern struct time_info_data time_info;
extern struct command_info cmd_info[];

/* extern functions */
void add_follower(struct char_data * ch, struct char_data * leader);


struct social_type {
  char *cmd;
  int next_line;
};


/* ********************************************************************
*  Special procedures for mobiles                                     *
******************************************************************** */

int spell_sort_info[MAX_SKILLS+1];

extern char *spells[];

void sort_spells(void)
{
  int a, b, tmp;

  /* initialize array */
  for (a = 1; a < MAX_SKILLS; a++)
    spell_sort_info[a] = a;

  /* Sort.  'a' starts at 1, not 0, to remove 'RESERVED' */
  for (a = 1; a < MAX_SKILLS - 1; a++)
    for (b = a + 1; b < MAX_SKILLS; b++)
      if (strcmp(spells[spell_sort_info[a]], spells[spell_sort_info[b]]) > 0) {
	tmp = spell_sort_info[a];
	spell_sort_info[a] = spell_sort_info[b];
	spell_sort_info[b] = tmp;
      }
}


char *how_good(int percent)
{
  static char buf[256];

  if (percent == 0)
    strcpy(buf, " (not learned)");
  else if (percent <= 10)
    strcpy(buf, " (awful)      ");
  else if (percent <= 20)
    strcpy(buf, " (bad)        ");
  else if (percent <= 40)
    strcpy(buf, " (poor)       ");
  else if (percent <= 55)
    strcpy(buf, " (average)    ");
  else if (percent <= 70)
    strcpy(buf, " (fair)       ");
  else if (percent <= 80)
    strcpy(buf, " (good)       ");
  else if (percent <= 85)
    strcpy(buf, " (very good)  ");
  else
    strcpy(buf, " (superb)     ");

  if (test_mode) {
    sprintf(buf, "%d", percent);
  }
  return (buf);
}

char *prac_types[] = {
  "spell",
  "skill"
};

#define LEARNED_LEVEL   0       /* % known which is considered "learned" */
#define MAX_PER_PRAC    1       /* max percent gain in skill per practice */
#define MIN_PER_PRAC    2       /* min percent gain in skill per practice */
#define PRAC_TYPE       3       /* should it say 'spell' or 'skill'?     */

/* actual prac_params are in class.c */
extern int prac_params[4][NUM_CLASSES];

#define LEARNED(ch, skill) ((spell_info[skill].learned[(int)GET_CLASS(ch)]==-1) ? \
   prac_params[LEARNED_LEVEL][(int)GET_CLASS(ch)] : \
   spell_info[skill].learned[(int)GET_CLASS(ch)])
#define MINGAIN(ch) (prac_params[MIN_PER_PRAC][(int)GET_CLASS(ch)])
#define MAXGAIN(ch) (prac_params[MAX_PER_PRAC][(int)GET_CLASS(ch)])
#define SPLSKL(ch) (prac_types[prac_params[PRAC_TYPE][(int)GET_CLASS(ch)]])

void list_skills(struct char_data * ch, struct char_data * vict)
{
  extern char *spells[];
  extern struct spell_info_type spell_info[];
  int i, sortpos, j = 1;

  if(!vict) vict = ch;

  if (IS_NPC(vict)) {
    send_to_char("Mobs are unskilled laborers.\r\n", ch);
    return;
  }
  
  if (!GET_PRACTICES(vict))
    strcpy(buf, "You have no practice sessions remaining.\r\n");
  else
    sprintf(buf, "You have %d practice session%s remaining.\r\n",
	    GET_PRACTICES(vict), (GET_PRACTICES(ch) == 1 ? "" : "s"));

  sprintf(buf, "%sYou know of the following %ss:\r\n", buf, SPLSKL(vict));

  strcpy(buf2, buf);

  for (sortpos = 1; sortpos < MAX_SKILLS; sortpos++) {
    i = spell_sort_info[sortpos];
    if (strlen(buf2) >= MAX_STRING_LENGTH - 32) {
      strcat(buf2, "**OVERFLOW**\r\n");
      break;
    }
    if (GET_LEVEL(vict) >= spell_info[i].min_level[(int) GET_CLASS(vict)] &&
        GET_REMORT(vict) >= spell_info[i].remort[(int) GET_CLASS(vict)]) {
      sprintf(buf, "&+M%-20s &+G%s&+w%s", spells[i], how_good(GET_SKILL(vict, i)),
              (++j % 2) ? "\r\n" : "   ");
      strcat(buf2, buf);
    }
  }

  strcat(buf2, "\r\n");
  page_string(ch->desc, buf2, 1);
}


SPECIAL(guild)
{
  int skill_num, percent;

  extern struct spell_info_type spell_info[];
  extern struct int_app_type int_app[];

  if (IS_NPC(ch) || !CMD_IS("practice"))
    return 0;

  skip_spaces(&argument);

  if (!*argument) {
    list_skills(ch, NULL);
    return 1;
  }
  if (GET_PRACTICES(ch) <= 0) {
    send_to_char("You do not seem to be able to practice now.\r\n", ch);
    return 1;
  }

  skill_num = find_skill_num(argument);

  if (skill_num < 1 ||
      GET_LEVEL(ch) < spell_info[skill_num].min_level[(int) GET_CLASS(ch)] ||
      GET_REMORT(ch) < spell_info[skill_num].remort[(int) GET_CLASS(ch)]) {
    sprintf(buf, "You do not know of that %s.\r\n", SPLSKL(ch));
    send_to_char(buf, ch);
    return 1;
  }
  if (GET_SKILL(ch, skill_num) >= LEARNED(ch, skill_num)) {
    send_to_char("You are already learned in that area.\r\n", ch);
    return 1;
  }
  send_to_char("You practice for a while...\r\n", ch);
  GET_PRACTICES(ch)--;

  percent = GET_SKILL(ch, skill_num);
  percent += MIN(MAXGAIN(ch), MAX(MINGAIN(ch), int_app[GET_INT(ch)].learn));

  SET_SKILL(ch, skill_num, MIN(LEARNED(ch, skill_num), percent));

  if (GET_SKILL(ch, skill_num) >= LEARNED(ch, skill_num))
    send_to_char("You are now learned in that area.\r\n", ch);

  return 1;
}



SPECIAL(dump)
{
  struct obj_data *k;
  int value = 0;

  ACMD(do_drop);
  char *fname(char *namelist);

  for (k = world[ch->in_room].contents; k; k = world[ch->in_room].contents) {
    act("$p vanishes in a puff of smoke!", FALSE, 0, k, 0, TO_ROOM);
    extract_obj(k);
  }

  if (!CMD_IS("drop"))
    return 0;

  do_drop(ch, argument, cmd, 0);

  for (k = world[ch->in_room].contents; k; k = world[ch->in_room].contents) {
    act("$p vanishes in a puff of smoke!", FALSE, 0, k, 0, TO_ROOM);
    value += MAX(1, MIN(50, GET_OBJ_COST(k) / 10));
    extract_obj(k);
  }

  if (value) {
    act("You are awarded for outstanding performance.", FALSE, ch, 0, 0, TO_CHAR);
    act("$n has been awarded for being a good citizen.", TRUE, ch, 0, 0, TO_ROOM);

    if (GET_LEVEL(ch) < 3)
      gain_exp(ch, value);
    else
      GET_GOLD(ch) += value;
  }
  return 1;
}


SPECIAL(mayor)
{
  ACMD(do_gen_door);

  static char open_path[] =
  "W3a3003b33000c111d0d111Oe333333Oe22c222112212111a1S.";

  static char close_path[] =
  "W3a3003b33000c111d0d111CE333333CE22c222112212111a1S.";

  static char *path;
  static int index;
  static bool move = FALSE;

  if (!move) {
    if (time_info.hours == 6) {
      move = TRUE;
      path = open_path;
      index = 0;
    } else if (time_info.hours == 20) {
      move = TRUE;
      path = close_path;
      index = 0;
    }
  }
  if (cmd || !move || (GET_POS(ch) < POS_SLEEPING) ||
      (GET_POS(ch) == POS_FIGHTING))
    return FALSE;

  switch (path[index]) {
  case '0':
  case '1':
  case '2':
  case '3':
    perform_move(ch, path[index] - '0', 1);
    break;

  case 'W':
    GET_POS(ch) = POS_STANDING;
    act("$n awakens and groans loudly.", FALSE, ch, 0, 0, TO_ROOM);
    break;

  case 'S':
    GET_POS(ch) = POS_SLEEPING;
    act("$n lies down and instantly falls asleep.", FALSE, ch, 0, 0, TO_ROOM);
    break;

  case 'a':
    act("$n says 'Hello Honey!'", FALSE, ch, 0, 0, TO_ROOM);
    act("$n smirks.", FALSE, ch, 0, 0, TO_ROOM);
    break;

  case 'b':
    act("$n says 'What a view!  I must get something done about that dump!'",
	FALSE, ch, 0, 0, TO_ROOM);
    break;

  case 'c':
    act("$n says 'Vandals!  Youngsters nowadays have no respect for anything!'",
	FALSE, ch, 0, 0, TO_ROOM);
    break;

  case 'd':
    act("$n says 'Good day, citizens!'", FALSE, ch, 0, 0, TO_ROOM);
    break;

  case 'e':
    act("$n says 'I hereby declare the bazaar open!'", FALSE, ch, 0, 0, TO_ROOM);
    break;

  case 'E':
    act("$n says 'I hereby declare Midgaard closed!'", FALSE, ch, 0, 0, TO_ROOM);
    break;

  case 'O':
    do_gen_door(ch, "gate", 0, SCMD_UNLOCK);
    do_gen_door(ch, "gate", 0, SCMD_OPEN);
    break;

  case 'C':
    do_gen_door(ch, "gate", 0, SCMD_CLOSE);
    do_gen_door(ch, "gate", 0, SCMD_LOCK);
    break;

  case '.':
    move = FALSE;
    break;

  }

  index++;
  return FALSE;
}


/* ********************************************************************
*  General special procedures for mobiles                             *
******************************************************************** */
/* This special procedure makes a mob into a 'rent-a-cleric', who sells spells
   by the sea shore... uuh, maybe not.  Anyway, the mob will also cast certain
   spells on low-level characters in the room for free.  
   By:  Wyatt Bode      Date:  April, 1996
*/
SPECIAL(cleric)
{
  int i;
  char buf[MAX_STRING_LENGTH];
  struct char_data *vict;
  struct price_info {
    short int number;
    char name[25];
    int price;
  } prices[] = {
    /* Spell Num (defined)      Name shown        Price  */
    { SPELL_ARMOR,              "armor          ",   1000 },
    { SPELL_BLESS,              "bless          ",   2000 },
    { SPELL_REMOVE_POISON,      "remove poison  ",   5000 },
    { SPELL_CURE_BLIND,         "cure blindness ",   5000 },
    { SPELL_SHIELD,             "shield         ",   7000 }, 
    { SPELL_CURE_CRITIC,        "critic         ",  15000 },
    { SPELL_SANCTUARY,          "sanctuary      ",  30000 },
    { SPELL_HEAL,               "heal           ",  90000 },

    /* The next line must be last, add new spells above. */ 
    { -1, "\r\n", -1 }
  };

/* NOTE:  In interpreter.c, you must define a command called 'heal' for this
   spec_proc to work.  Just define it as do_not_here, and the mob will take 
   care of the rest.  (If you don't know what this means, look in interpreter.c
   for a clue.)
*/

  if (CMD_IS("heal")) {
    argument = one_argument(argument, buf);

    if (GET_POS(ch) == POS_FIGHTING) return TRUE;

    if (*buf) {
      for (i=0; prices[i].number > SPELL_RESERVED_DBC; i++) {
	if (is_abbrev(buf, prices[i].name))  {
	  if (GET_GOLD(ch) < prices[i].price) {
	    act("$n tells you, 'You don't have enough gold for that spell!'",
		FALSE, (struct char_data *) me, 0, ch, TO_VICT);
	    return TRUE;
	  } else {
	    act("$N gives $n some money.",
		FALSE, (struct char_data *) me, 0, ch, TO_NOTVICT);
	    sprintf(buf, "You give %s %d coins.\r\n", 
		    GET_NAME((struct char_data *) me), prices[i].price);
	    send_to_char(buf, ch);
	    GET_GOLD(ch) -= prices[i].price;
	    /* Uncomment the next line to make the mob get RICH! */
	    /* GET_GOLD((struct char_data *) me) += prices[i].price; */

	    cast_spell((struct char_data *) me, ch, NULL, "", 0, 
                       prices[i].number);
	    return TRUE;
	  
	  }
        }
      }
      act("$n tells you, 'I do not know of that spell!"
	  "  Type 'heal' for a list.'", FALSE, (struct char_data *) me, 
	  0, ch, TO_VICT);
	  
      return TRUE;
    } else {
      act("$n tells you, 'Here is a listing of the prices for my services.'",
	  FALSE, (struct char_data *) me, 0, ch, TO_VICT);
      for (i=0; prices[i].number > SPELL_RESERVED_DBC; i++) {
	sprintf(buf, "%s%d\r\n", prices[i].name, prices[i].price);
	send_to_char(buf, ch);
      }
      return TRUE;
    }
  }

  if (cmd) return FALSE;

  /* pseudo-randomly choose someone in the room */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (!number(0, 3))
      break;
  
  /* change the level at the end of the next line to control free spells */
  if (vict == NULL || IS_NPC(vict) || (GET_LEVEL(vict) > 10))
    return FALSE;

  switch (number(1, GET_LEVEL(vict))) { 
      case 1: cast_spell(ch, vict, NULL, "", 0, SPELL_CURE_LIGHT); break; 
      case 2: cast_spell(ch, vict, NULL, "", 0, SPELL_BLESS); break; 
      case 3: cast_spell(ch, vict, NULL, "", 0, SPELL_ARMOR); break;
      case 4: cast_spell(ch, vict, NULL, "", 0, SPELL_CURE_LIGHT); break; 
      case 5: cast_spell(ch, vict, NULL, "", 0, SPELL_BLESS); break; 
      case 6: cast_spell(ch, vict, NULL, "", 0, SPELL_CURE_CRITIC); break; 
      case 7: cast_spell(ch, vict, NULL, "", 0, SPELL_ARMOR); break;
      case 8: cast_spell(ch, vict, NULL, "", 0, SPELL_CURE_CRITIC); break; 
      case 9: cast_spell(ch, vict, NULL, "", 0, SPELL_ARMOR); break; 
      case 10: 
	/* special wacky thing, your mileage may vary */ 

	act("$n utters the words, 'energizer'.", TRUE, ch, 0, vict, TO_ROOM);
	act("You feel invigorated!", FALSE, ch, 0, vict, TO_VICT);
	GET_MANA(vict) = 
	   MIN(GET_MAX_MANA(vict), MAX((GET_MANA(vict) + 10), number(50, 200)));
	break; 
  }
  return TRUE; 
}               


void npc_steal(struct char_data * ch, struct char_data * victim)
{
  int gold;

  if (IS_NPC(victim))
    return;
  if (GET_LEVEL(victim) >= LVL_IMMORT)
    return;

  if (AWAKE(victim) && (number(0, GET_LEVEL(ch)) == 0)) {
    act("You discover that $n has $s hands in your wallet.", FALSE, ch, 0, victim, TO_VICT);
    act("$n tries to steal gold from $N.", TRUE, ch, 0, victim, TO_NOTVICT);
  } else {
    /* Steal some gold coins */
    gold = (int) ((GET_GOLD(victim) * number(1, 10)) / 100);
    if (gold > 0) {
      GET_GOLD(ch) += gold;
      GET_GOLD(victim) -= gold;
    }
  }
}

SPECIAL(scorpion)
{
  if (cmd)
    return FALSE;

  if (GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  if (FIGHTING(ch) && (FIGHTING(ch)->in_room == ch->in_room) &&
      !number(0, 5)) {
    act("$n stings $N!", 1, ch, 0, FIGHTING(ch), TO_NOTVICT);
    act("$n stings you!", 1, ch, 0, FIGHTING(ch), TO_VICT);
    call_magic(ch, FIGHTING(ch), 0, "", SPELL_POISON, GET_LEVEL(ch), 0, 
	       CAST_SPELL);

    return TRUE;
  }
  return FALSE;
}

SPECIAL(ghoul)
{
  if (cmd)
    return FALSE;
      
  if (GET_POS(ch) != POS_FIGHTING)
    return FALSE;
 
  if (FIGHTING(ch) && (FIGHTING(ch)->in_room == ch->in_room) &&
      !number(0, 5)) {
    act("$n bites $N!", 1, ch, 0, FIGHTING(ch), TO_NOTVICT);
    act("$n bites you!", 1, ch, 0, FIGHTING(ch), TO_VICT);
    call_magic(ch, FIGHTING(ch), 0, "", SPELL_PARALYZE, GET_LEVEL(ch), 0, 
	       CAST_SPELL);
    return TRUE; 
  }
  return FALSE;
}

SPECIAL(snake)
{
  if (cmd)
    return FALSE;

  if (GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  if (FIGHTING(ch) && (FIGHTING(ch)->in_room == ch->in_room) &&
      (number(0, 42 - GET_LEVEL(ch)) == 0)) {
    act("$n bites $N!", 1, ch, 0, FIGHTING(ch), TO_NOTVICT);
    act("$n bites you!", 1, ch, 0, FIGHTING(ch), TO_VICT);
    call_magic(ch, FIGHTING(ch), 0, "", SPELL_POISON, GET_LEVEL(ch), 0,
	       CAST_SPELL);
    return TRUE;
  }
  return FALSE;
}


SPECIAL(thief)
{
  struct char_data *cons;

  if (cmd)
    return FALSE;

  if (GET_POS(ch) != POS_STANDING)
    return FALSE;

  for (cons = world[ch->in_room].people; cons; cons = cons->next_in_room)
    if (!IS_NPC(cons) && (GET_LEVEL(cons) < LVL_IMMORT) && (!number(0, 4))) {
      npc_steal(ch, cons);
      return TRUE;
    }
  return FALSE;
}


SPECIAL(magic_user)
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  if ((GET_LEVEL(ch) > 13) && (number(0, 10) == 0))
    cast_spell(ch, vict, NULL, "", 0, SPELL_SLEEP);

  if ((GET_LEVEL(ch) > 7) && (number(0, 8) == 0))
    cast_spell(ch, vict, NULL, "", 0, SPELL_BLINDNESS);

  if ((GET_LEVEL(ch) > 12) && (number(0, 12) == 0)) {
    if (IS_EVIL(ch))
      cast_spell(ch, vict, NULL, "", 0, SPELL_ENERGY_DRAIN);
    else if (IS_GOOD(ch))
      cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_EVIL);
  }
  if (number(0, 4))
    return TRUE;

  switch (GET_LEVEL(ch)) {
  case 4:
  case 5:
    cast_spell(ch, vict, NULL, "", 0, SPELL_CHILL_TOUCH);
    break;
  case 6:
  case 7:
    cast_spell(ch, vict, NULL, "", 0, SPELL_BURNING_HANDS);
    break;
  case 8:
  case 9:
    cast_spell(ch, vict, NULL, "", 0, SPELL_SHOCKING_GRASP);
    break;
  case 10:
  case 11:
    cast_spell(ch, vict, NULL, "", 0, SPELL_LIGHTNING_BOLT);
    break;
  case 12:
  case 13:
    cast_spell(ch, vict, NULL, "", 0, SPELL_FIREBALL);
    break;
  case 14:
  case 15:
  case 16:
  case 17:
    cast_spell(ch, vict, NULL, "", 0, SPELL_COLOR_SPRAY);
    break;
  default:
    cast_spell(ch, vict, NULL, "", 0, SPELL_MAGIC_MISSILE);
    break;
  }
  return TRUE;

}

SPECIAL(evil_lich)
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  if (IS_AFFECTED(vict, AFF_SANCTUARY)) 
    cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_MAGIC);
  
  else if (IS_MONK(vict) && !IS_AFFECTED(vict, AFF_SLOW))
	 cast_spell(ch, vict, NULL, "", 0, SPELL_SLOW);

       else if (IS_AFFECTED(vict, AFF_HASTE))
	      cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_MAGIC);

  switch (number(1, 7)) {
  case 1:
    if (!IS_AFFECTED(ch, AFF_BLUR))
       cast_spell(ch, ch, NULL, "", 0, SPELL_BLUR);
    else if (!IS_AFFECTED(ch, AFF_HASTE))
       cast_spell(ch, ch, NULL, "", 0, SPELL_HASTE);
	 else if (!number(0, 2))
	    cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
    break;
  case 2:
    if  (!affected_by_spell(vict, SPELL_BLINDNESS))
       cast_spell(ch, ch, NULL, "", 0, SPELL_BLINDNESS);
    else if (!affected_by_spell(ch, SPELL_ARMOR))
       cast_spell(ch, ch, NULL, "", 0, SPELL_ARMOR);
	 else if (!affected_by_spell(ch, SPELL_SHIELD))
	    cast_spell(ch, ch, NULL, "", 0, SPELL_SHIELD);
	     else if  (!affected_by_spell(ch, SPELL_STONESKIN))
	       cast_spell(ch, ch, NULL, "", 0, SPELL_STONESKIN);
		  else cast_spell(ch, ch, NULL, "", 0, SPELL_STRENGTH); 
    break;
  case 3:
    cast_spell(ch, vict, NULL, "", 0, SPELL_FIREBALL);
  case 4:
    if (!IS_AFFECTED(vict, AFF_SLOW))
       cast_spell(ch, vict, NULL, "", 0, SPELL_SLOW);
    else
       cast_spell(ch, vict, NULL, "", 0, SPELL_COLOR_SPRAY);
    break;
  case 5:
     if (IS_AFFECTED(vict, AFF_PROTECT_EVIL)
	  || affected_by_spell(vict, SPELL_ARMOR)
	  || IS_AFFECTED(vict, AFF_BLUR))
	cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_MAGIC);
     else 
	cast_spell(ch, vict, NULL, "", 0, SPELL_LIGHTNING_BOLT);
     break;
  default:
    cast_spell(ch, vict, NULL, "", 0, SPELL_MAGIC_MISSILE);
    break;
  }
  return TRUE;
}

SPECIAL(guardian)
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  if ((GET_LEVEL(ch) > 0) && (number(0, 0) == 0))
    cast_spell(ch, vict, NULL, "", 0, SPELL_SLEEP);

  if ((GET_LEVEL(ch) > 0) && (number(0, 0) == 0))
    cast_spell(ch, vict, NULL, "", 0, SPELL_BLINDNESS);

  if ((GET_LEVEL(ch) > 0) && (number(0, 0) == 0)) {
    if (IS_EVIL(ch))
      cast_spell(ch, NULL, NULL, "", 0, SPELL_HEAL);
    else if (IS_GOOD(ch))
      cast_spell(ch, NULL, NULL, "", 0, SPELL_HEAL);
  }
  if (number(0, 4))
    return TRUE;

  switch (GET_LEVEL(ch)) {
  case 4:
  case 5:
    cast_spell(ch, vict, NULL, "", 0, SPELL_MAGIC_MISSILE);
    break;
  case 6:
  case 7:
    cast_spell(ch, vict, NULL, "", 0, SPELL_CHILL_TOUCH);
    break;
  case 8:
  case 9:
    cast_spell(ch, vict, NULL, "", 0, SPELL_BURNING_HANDS);
    break;
  case 10:
  case 11:
    cast_spell(ch, vict, NULL, "", 0, SPELL_SHOCKING_GRASP);
    break;
  case 12:
  case 13:
    cast_spell(ch, vict, NULL, "", 0, SPELL_LIGHTNING_BOLT);
    break;
  case 14:
  case 15:
  case 16:
  case 17:
    cast_spell(ch, vict, NULL, "", 0, SPELL_COLOR_SPRAY);
    break;
  default:
    cast_spell(ch, vict, NULL, "", 0, SPELL_FIREBALL);
    break;
  }
  return TRUE;

}

SPECIAL(familiar)
{
 ACMD(do_tell);
 int cmd_tell = find_command("tell");
 switch (number(0,100)) {
  case 0: 
   do_tell(ch, "So, what are we doing again?", cmd_tell, 0);
   return TRUE;
  case 1:
   do_tell(ch, "Whoa!  Did you see that??? Oh, never mind... it was nothing.",
	   cmd_tell, 0);
   return TRUE;
  default:
   return FALSE;
 }
}


/* ********************************************************************
*  Special procedures for mobiles                                      *
******************************************************************** */

SPECIAL(guild_guard)
{
  int i;
  extern int guild_info[][3];
  struct char_data *guard = (struct char_data *) me;
  char *buf = "The guard humiliates you, and blocks your way.\r\n";
  char *buf2 = "The guard humiliates $n, and blocks $s way.";

  if (!IS_MOVE(cmd) || IS_AFFECTED(guard, AFF_BLIND))
    return FALSE;

  if (GET_LEVEL(ch) >= LVL_IMMORT)
    return FALSE;

  for (i = 0; guild_info[i][0] != -1; i++) {
    if ((IS_NPC(ch) || GET_CLASS(ch) != guild_info[i][0]) &&
	world[ch->in_room].number == guild_info[i][1] &&
	cmd == guild_info[i][2]) {
      send_to_char(buf, ch);
      act(buf2, FALSE, ch, 0, 0, TO_ROOM);
      return TRUE;
    }
  }

  return FALSE;
}


SPECIAL(kalpet)
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0,60))  {
  case 0:
    do_say(ch, "I just love my master!", 0, 0);
    return (1);
  case 1:
    do_say(ch, "Kalanor is great!  Don't you think?", 0, 0);
    return (1);
  case 2:
    do_say(ch, "I will always follow Kalanor.", 0, 0);
    return (1);
  case 3:
    do_say(ch, "I worship Kalanor.", 0, 0);
    return (1);
  default:
    return (0);
  }
}

SPECIAL (TKCers)
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0,50))  {
  case 0:
    do_say(ch, "You are an idiot!", 0, 0);
    return (1);
  case 1:
    do_say(ch, "Mess with that elf babe, and she'll kick your butt", 0, 0);
    return (1);
  case 2:
    do_say(ch, "Taylor's the coolest god on NoMud! Don't you think?", 0, 0);
    return (1);
  case 3:
    do_say(ch, "That kind of talk will get you squashed like a bug!", 
0, 0);
    return (1);
  case 4:
    do_say(ch, "We need more men like Mayor Abe, men who take bribes", 0, 0);
    return (1);
  case 5:
    do_say(ch, "I think the mayor's policies stink", 0, 0);
    return (1);
  case 6:
    do_say(ch, "Welmar is a prissy wus and his Captain is even worse!", 
0, 0);
    return (1);
  case 7:
    do_say(ch, "Who the hell are you anyway!", 0, 0);
    return (1);
  case 8:
    do_say(ch, "If I had a dog HALF as ugly as you I'd shave its ass\n\rand teach it to walk backwards!", 0, 0);
    return (1);
  case 9:
    do_say(ch, "Just what we need another stinkin suckup!", 0, 0);
    return (1);
  case 10:
    do_say(ch, "First thing we do, let's kill all the peacekeepers!", 0, 0);
    return (1);
  case 11:
    do_say(ch, "Kalanor is great!  Don't you think?", 0, 0);
    return (1);
  case 12:
    do_say(ch, "Where the hell's my wallet!", 0, 0);
    return (1);
  case 13:
    do_say(ch, "We should finish off the monks once and for all", 0, 0);
    return (1);
  case 14:
    do_say(ch, "You lookin at me, boy!  You besta not be looking at me!\n\ror I'll cut you!", 0, 0);
    return (1);
  default:
  return (0);
 }
}

SPECIAL (CDAers)
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0,50))  {
  case 0:
    do_say(ch, "Hey buddy, can you spare a dime?", 0, 0);
    return (1);
  case 1:
    do_say(ch, "A horse! A horse! My kingdom for a horse!", 
0, 0);
    return (1);
  case 2:
    do_say(ch, "Garon would never sing his own praises, but I will.", 0, 0);
    return (1);
  case 3:
    do_say(ch, "Pygar is a kewl immort. We should all try to be like him.",
0, 0);
    return (1);
  case 4:
    do_say(ch, "We need more immorts like Desmond... just to keep up with the ladies.", 0, 0);
    return (1);
  case 5:
    do_say(ch, "But soft, what light through yonder window breaks?", 0, 0);
    return (1);
  case 6:
    do_say(ch, "If I can sell one painting, my ship will come in!",
0, 0);
    return (1);
  case 7:
    do_say(ch, "The CDA doesn't reject anyone. It's open to all races and classes", 0, 0);
    return (1);
  case 8:
    do_say(ch, "The CDA Clan Hall is north at the Poor Alley.", 0, 0);
    return (1);
  case 9:
    do_say(ch, "Ask Garon or any CDA immort to set you to clan CDA.", 0, 0);
    return (1);
  case 10:
    do_say(ch, "Ahh, I remember the good ole days of the CDA!", 0, 0);
    return (1);
  case 11:
    do_say(ch, "Did you know that the CDA was one of the first clans on NoMud?", 0, 0);
    return (1);
  case 12:
    do_say(ch, "All the world's a stage!", 0, 0);
    return (1);
  case 13:
    do_say(ch, "Don't look now, but your shoe is untied.", 0, 0);
    return (1);
  case 14:
    do_say(ch, "Carpe Diem, my friend!", 0, 0);
    return (1);
  default:
  return (0);
 }
}


SPECIAL(leon_lai)
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0,10))  {
  case 0:
    do_say(ch, "I'm Leon Lai", 0, 0);
    return (1);
  case 1:
    act("$n sings 'I, I, I, was born in beijing.'", FALSE, ch, 0, 0, TO_ROOM);
 /*   do_say(ch, "I, I, I, was born in beijing.", 0, 0); */
    return (1);
  case 2:
/*   do_say(ch, "Oh! my love, You're ichi-ban....", 0, 0); */
    act("$n sings 'Oh! my love, You're ichi-ban....'", FALSE, ch, 0, 0, TO_ROOM);

    return (1);
  case 3:
/*    do_say(ch, "SAYONARA-OH! SAYONARA-OH!", 0, 0); */
     act("$n sings 'SAYONARA-OH! SAYONARA-OH!'", FALSE, ch, 0, 0, TO_ROOM);

    return (1);
  default:
    return (0);
  }
}

SPECIAL(puff)
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0, 60)) {
  case 0:
    do_say(ch, "My god!  It's full of stars!", 0, 0);
    return (1);
  case 1:
    do_say(ch, "How'd all those fish get up here?", 0, 0);
    return (1);
  case 2:
    do_say(ch, "I'm a very female dragon.", 0, 0);
    return (1);
  case 3:
    do_say(ch, "I've got a peaceful, easy feeling.", 0, 0);
    return (1);
  default:
    return (0);
  }
}

SPECIAL(minstrel)
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0, 60)) {
  case 0:
    do_say(ch, "Parsley, Sage, Rosemary and Thyme.", 0, 0);
    return (1);
  case 1:
    do_say(ch, "Any tune you'd like to hear?  Hum a few bars and I'll fake it.", 0, 0);
    return (1);
  case 2:
    do_say(ch, "Would you like to hear the Tale of Sir Robin?", 0, 0);
    return (1);
  case 3:
    do_say(ch, "My dream is to one day sing at the Abbey.", 0, 0);
    return (1);
  default:
    return (0);
  }
}


SPECIAL(Happy_Noodle_Boy)
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0,100))  {
  case 0:
    do_say(ch, "BOW DOWN!!  OR I WILL UNLEASH MY ZOINKY ARMY OF SURLY CRACK BABIES!!", 0, 0);
    return (1);
  case 1:
    do_say(ch, "DO YOU NOT FEAR THE AWESOME POWER OF TILE GROUT?!", 0, 0);
    return (1);
  case 2:
    do_say(ch, "JOIN MY LEGION OF DARKNESS, MY FROWNY FACE EMPIRE!!!!", 0, 0);
    return (1);
  case 3:
    do_say(ch, "GO FORTH AND BECOME A HAPPY CABBAGE!!!", 0, 0);
    return (1);
  case 4:
    do_say(ch, "THESE HANDS!!  I CAN'T GET THEM OFF MY WRISTS!!", 0, 0);
    return (1);
  case 5:
    do_say(ch, "OTAN JIYOU BE O ME DETOU GOZAIMASU!!", 0, 0);
    return (1);
  case 6:
    do_say(ch, "YOU REMEMBER THIS LESSON I TEACH YOU!!  USE IT FOR GOOD!!!  USE IT ON NACHOS!!!", 0, 0);
    return (1);
  case 7:
    do_say(ch, "REVEALED BY YOUR OWN BUTT-THRUSTING STUPIDITY!!  I SNATCH THEE!!", 0, 0);
    return (1);
  case 8:
    do_say(ch, "BE FREE, LITTLE POP-TART!", 0, 0);
    return (1);
  case 9:
    do_say(ch, "YOU DENY ME FRESHMAKER?!?!?!?!?!?", 0, 0);
    return (1);
  case 10:
    do_say(ch, "IF MY DEMANDS ARE NOT MET BY SOMETIME, I WILL BEGIN BUTTERING YOUR LOVED ONES!!  SOME WEENIES HAVE CHEESE ON THE INSIDE!", 0, 0);
    return (1);
  case 11:
    do_say(ch, "I HAVE SUPER DUPER RASH!!!", 0, 0);
    return (1);
  case 12:
    do_say(ch, "HOW CAN THIS BE!!?  YOU... YOU'RE A MARSHMALLOW PEEP AREN'T YOU!!?", 0, 0);
    return (1);
  case 13:
    do_say(ch, "I AM SUPERDEFORMED!!  GRR!!", 0, 0);
    return (1);
  case 14:
    act("$n climbs on your head.", FALSE, ch, 0, 0, TO_ROOM);
    default:
    return (0);
  }
}


SPECIAL(wilson)
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0, 60)) {
  case 0:
    do_say(ch, "My father is a blacksmith", 0, 0);
    return (1);
  case 1:
    do_say(ch, "My mother is a famous sorceress", 0, 0);
    return (1);
  case 2:
    do_say(ch, "I am lost. Could you help me find my parents?", 0, 0);
    return (1);
  case 3:
    do_say(ch, "I am protected by powerful magic. Fighting me would be foolish.", 0, 0);
    return (1);
  default:
    return (0);
  }
}

SPECIAL(demon)
{
	ACMD(do_say);
	
	if (cmd)
		return (0);
		
	switch (number(0,10)) {
	case 0:
		do_say(ch, "You UGLY humans... hardly worth the FLESH you're printed on...", 0, 0);
		return (1);
	case 1:
		do_say(ch, "Bow down before Krom or I shall dismember you!!", 0, 0);
		return (1);
	case 2:
		do_say(ch, "Your pain shall be LEGENDARY... even in HELL!", 0, 0);
		return (1);
	case 3:
		do_say(ch, "I shall kill you ALL and let GOD (Krom) sort you out!", 0, 0);
		return(1);
	default:
		return (0);
	}
}

SPECIAL(barny)
{
	ACMD(do_say);
	
	if (cmd)
		return (0);
		
	switch (number(0,60)) {
	case 0:
		do_say(ch, "Iiiiii LOVE you!", 0, 0);
		return (1);
	case 1:
		do_say(ch, "Iiiiii loove you... yooou loove mee...", 0, 0);
		return (1);
	case 2:
		do_say(ch, "Will you be my friend?? I LOVE you...", 0, 0);
		return (1);
	case 3: 
		act("$n produces a lute from nowhere and starts strumming a painfully familliar song...", FALSE, ch, 0, 0, TO_ROOM);
		return (1);
	case 4:
		act("$n wraps his big purple arms around you and squeeezes you with love!", FALSE, ch, 0, 0, TO_ROOM);
	default:
		return (0);
	}
}
SPECIAL(baby)
{
	ACMD(do_say);
	
	if (cmd)
		return (0);
		
	switch (number(0,20)) {
	case 1:
		act("$n gurgles contentedly.", TRUE, ch, 0, 0, TO_ROOM);
		return (1);
	case 2:
		do_say(ch, "dada", 0, 0);
		return (1);
	case 3: 
		act("$n reaches out and grabs your nose with her chubby fingers.",
		TRUE, ch, 0, 0, TO_ROOM);
		return (1);
	default:
		return (0);
	}
}

SPECIAL(fido)
{

  struct obj_data *i, *temp, *next_obj;

  if (cmd || !AWAKE(ch))
    return (FALSE);

  for (i = world[ch->in_room].contents; i; i = i->next_content) {
/*    if (GET_OBJ_TYPE(i) == ITEM_DRINKCON && GET_OBJ_VAL(i, 2) == LIQ_BLOOD) {
      act("$n laps up some blood.", FALSE, ch, 0, 0, TO_ROOM);
      extract_obj(i);
    } */
    if (GET_OBJ_TYPE(i) == ITEM_CONTAINER && GET_OBJ_VAL(i, 3)) {
      act("$n savagely devours a corpse.", FALSE, ch, 0, 0, TO_ROOM);
      for (temp = i->contains; temp; temp = next_obj) {
	next_obj = temp->next_content;
	obj_from_obj(temp);
	obj_to_room(temp, ch->in_room);
      }
      extract_obj(i);
      return (TRUE);
    }
  }
  switch(number(0, 20)) {
  case 0:
    act("$n barks wildly.", FALSE, ch, 0, 0, TO_ROOM);
    return (TRUE);
  case 1:
    act("$n growls fiercely at you.", FALSE, ch, 0, 0, TO_ROOM);
    return (TRUE);
  case 2:
    act("$n bares $s fangs and snarls.", FALSE, ch, 0, 0, TO_ROOM);
    return (TRUE);
  case 3:
    act("$n pees on your foot.", FALSE, ch, 0, 0, TO_ROOM);
    return (TRUE);
  case 4:
    act("You are knocked off your feet by $n.", FALSE, ch, 0, 0, TO_ROOM);
    return (TRUE);
  case 5:
    act("$n wimpers pitifully for some food.", FALSE, ch, 0, 0, TO_ROOM);
    return (TRUE);
  default:
  return (FALSE);
  }
}

SPECIAL(blob)
{
  struct obj_data *i;

  if (cmd || !AWAKE(ch))
    return (FALSE);

  for (i = world[ch->in_room].contents; i; i = i->next_content) {
    if (!CAN_WEAR(i, ITEM_WEAR_TAKE))
      continue;
    if (GET_OBJ_TYPE(i) != ITEM_DRINKCON && GET_OBJ_COST(i) >= 15)
      continue;
    act("$n sucks up some trash.", FALSE, ch, 0, 0, TO_ROOM);
    obj_from_room(i);
    obj_to_char(i, ch);
    return TRUE;
  }
  return (FALSE);
}


SPECIAL(janitor)
{
  struct obj_data *i;

  if (cmd || !AWAKE(ch))
    return (FALSE);

  for (i = world[ch->in_room].contents; i; i = i->next_content) {
    if (!CAN_WEAR(i, ITEM_WEAR_TAKE))
      continue;
    if (GET_OBJ_TYPE(i) != ITEM_DRINKCON && GET_OBJ_COST(i) >= 15)
      continue;
    act("$n picks up some trash.", FALSE, ch, 0, 0, TO_ROOM);
    obj_from_room(i);
    obj_to_char(i, ch);
    return TRUE;
  }

  switch (number(0, 20)) {
  case 0:
    act("$n says, 'Look at the mess you made!'", FALSE, ch, 0, 0, TO_ROOM);
    act("$n glares at you", FALSE, ch, 0, 0, TO_ROOM);
    return (TRUE);
  case 1:
    act("$n whistles, 'I've been working on the railroad!'", FALSE, ch, 0, 0, TO_ROOM);
    return (TRUE);
  case 2:
    act("$n swipes at a fido with his broom.", FALSE, ch, 0, 0, TO_ROOM);
    return (TRUE);
  default:
  return FALSE;
  }
}


SPECIAL(cityguard)
{
  struct char_data *tch, *evil;
  int max_evil;

  if (cmd || !AWAKE(ch) || FIGHTING(ch))
    return FALSE;

  max_evil = 1000;
  evil = 0;

  for (tch = world[ch->in_room].people; tch; tch = tch->next_in_room) {
    if (!IS_NPC(tch) && CAN_SEE(ch, tch) && IS_SET(PLR_FLAGS(tch), PLR_KILLER)
	&& !PRF_FLAGGED(tch, PRF_NOHASSLE)) {
      act("$n screams 'HEY!!!  You're one of those PLAYER KILLERS!!!!!!'", FALSE, ch, 0, 0, TO_ROOM);
      hit(ch, tch, TYPE_UNDEFINED);
      return (TRUE);
    }
  }

  for (tch = world[ch->in_room].people; tch; tch = tch->next_in_room) {
    if (!IS_NPC(tch) && CAN_SEE(ch, tch) && IS_SET(PLR_FLAGS(tch), PLR_THIEF)
	&& !PRF_FLAGGED(tch, PRF_NOHASSLE)){
      act("$n screams 'HEY!!!  You're one of those PLAYER THIEVES!!!!!!'", FALSE, ch, 0, 0, TO_ROOM);
      hit(ch, tch, TYPE_UNDEFINED);
      return (TRUE);
    }
  }

  for (tch = world[ch->in_room].people; tch; tch = tch->next_in_room) {
    if (CAN_SEE(ch, tch) && FIGHTING(tch)) {
      if ((GET_ALIGNMENT(tch) < max_evil) &&
	  (IS_NPC(tch) || IS_NPC(FIGHTING(tch)))) {
	max_evil = GET_ALIGNMENT(tch);
	evil = tch;
      }
    }
  }

  if (evil && (GET_ALIGNMENT(FIGHTING(evil)) >= 0)) {
    act("$n screams 'PROTECT THE INNOCENT!  BANZAI!  CHARGE!  ARARARAGGGHH!'", FALSE, ch, 0, 0, TO_ROOM);
    hit(ch, evil, TYPE_UNDEFINED);
    return (TRUE);
  }
  return (FALSE);
}


SPECIAL(dragon) /*jabber*/
{
  struct char_data *vict;
  if (cmd)
    return FALSE;
  if  (GET_POS(ch) != POS_FIGHTING)
    return FALSE;
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 6)) 
     break;
  if (vict == NULL)
    vict = FIGHTING(ch);
  if ((GET_LEVEL(ch) > 15))
   act("&+r$n lets out a roar as a burst of flame engulfs $N!", 1, ch, 0,
       FIGHTING(ch),  TO_NOTVICT);
   act("&+r$n lets out a roar as a burst of flame engulfs you!", 1, ch, 0,
       FIGHTING(ch), TO_VICT); 
   call_magic(ch, FIGHTING(ch), 0, NULL, SPELL_FIREBALL, GET_LEVEL(ch), 0,
              CAST_SPELL);
 return TRUE;
}

SPECIAL(dinos) /*jabber*/
{
  extern int cmd_emote;

  ACMD(do_say);
  ACMD(do_echo);
    if (cmd || FIGHTING(ch))
      return FALSE;
   
    if (number(0, 4))
      return FALSE;

    switch (mob_index[ch->nr].virtualrm) {

    case 2701:
      do_echo(ch, "looks for his family.", cmd_emote, SCMD_EMOTE);
        return TRUE;

    case 2702:
      do_say(ch, "Yep Yep Yep!", 0, 0);
        return TRUE;

    case 2704:
      do_echo(ch, "roars at you, preparing to attack.", cmd_emote, SCMD_EMOTE); 
        return TRUE;

    default:
      return FALSE;
    }
}
SPECIAL(mudzones) /*jabber*/
{
  extern int cmd_emote;

  ACMD(do_say);
  ACMD(do_echo);

    if (cmd || FIGHTING(ch))
      return FALSE;

    if (number(0, 4))
      return FALSE;

    switch (mob_index[ch->nr].virtualrm) {

      case 907: /*Minos*/
        do_say(ch, "Nothing gets your day started like a Mort for breakfast!", 0, 0);
          return TRUE;

      case 21023: /*Sparta Bishop*/
        do_echo(ch, "gets tangled up in his robes.", cmd_emote, SCMD_EMOTE);
          return TRUE;
 
      case 21038: /*Sparta Child*/
        do_say(ch, "Have you seen my mother?", 0, 0);
          return TRUE;

      case 12023: /*Master Scheduler*/    
        do_echo(ch, "writes your name in his book.", cmd_emote, SCMD_EMOTE);
          return TRUE;
      
      case 5002: /*Coral Snake*/
        do_echo(ch, "hisses at you.", cmd_emote, SCMD_EMOTE);
          return TRUE;

      case 7001: /*small bat in sewers*/
        do_echo(ch, "flaps his wings in your face.", cmd_emote, SCMD_EMOTE); 
          return TRUE;

      case 7002: /*sewer rat*/
        do_say(ch, "You look like a tasty morsel", 0, 0);
          return TRUE;

      case 24300: /*Inigo Montoya*/
        do_say(ch, "My name is Inigo Montoya, you killed my father, Prepare to DIE!", 0, 0);
          return TRUE;
       
      case 2131: /*Chaos Robot*/
        do_say(ch, "Screech...Kill!!....Kill!!", 0, 0);
        return TRUE;
      
      case 2133: /*Jogurt*/
        do_say(ch, "Hello there, I've been waiting for you.", 0, 0);
        return TRUE;
        
      default:
        return FALSE;
  }
}

SPECIAL(elves) /*jabber*/
{
  extern int cmd_emote;

  ACMD(do_say);
  ACMD(do_echo);

    if (cmd || FIGHTING(ch))
      return FALSE;

    if (number(0, 4))
      return FALSE;

    switch (mob_index[ch->nr].virtualrm) {
      
      case 1305:
        do_say(ch, "Stop and state your business here at the palace.",
               0, 0);
        return TRUE;
      
      case 1306:
        do_echo(ch, "writes of the world in his book.", cmd_emote,
                    SCMD_EMOTE);
        return TRUE;

      case 1307:
        do_say(ch, "Stop!  Noone can see the Speaker unannounced.", 0,
               0);
        return TRUE;

      case 1308:
        do_say(ch, "Hello, what can I do for you my child.", 0, 0);
        return TRUE;

      case 1309:
        do_say(ch, "Get out of here or I will turn you into a kender!",
               0, 0);
        return TRUE;

      case 1312: /*use for mobs 6212 and 6213*/
      case 1313: /*use for mobs 6212 and 6213*/
        do_say(ch, "Hello, may your day be as sweet as Elven Blossom Wine.", 0, 0);
        return TRUE;

      case 1314:
        do_say(ch, "Mommy, I need to go to the bathroom.", 0, 0);
        return TRUE;

      case 1315:
        do_echo(ch, "sweeps the sidewalk with her broom.", cmd_emote,
                    SCMD_EMOTE); 
        return TRUE;

      case 1316:
        do_say(ch, "I have some fresh Quith Pa in the oven.", 0, 0);
        return TRUE;
    
      case 1317:
        do_say(ch, "I can fix your weapon like new.", 0, 0);
        return TRUE;
   
      case 1318:
        do_say(ch, "I know I have that map in one of my pouches!", 0, 0);
        return TRUE;
       
      case 1319:
        do_say(ch, "One moment, and I'll get that dress I altered for you", 0, 0);
        return TRUE;

      case 1324:
        do_echo(ch, "chants jumprope rhymes while turning the rope.",
                    cmd_emote, SCMD_EMOTE);
        return TRUE;

      default:
        return FALSE;
    }
}


SPECIAL(pet_shops)
{
  char buf[MAX_STRING_LENGTH], pet_name[256];
  int pet_room;
  struct char_data *pet;

  pet_room = ch->in_room + 1;

  if (CMD_IS("list")) {
    send_to_char("Available pets are:\r\n", ch);
    for (pet = world[pet_room].people; pet; pet = pet->next_in_room) {
      sprintf(buf, "%8d - %s\r\n", 3 * GET_EXP(pet), GET_NAME(pet));
      send_to_char(buf, ch);
    }
    return (TRUE);
  } else if (CMD_IS("buy")) {

    argument = one_argument(argument, buf);
    argument = one_argument(argument, pet_name);

    if (!(pet = get_char_room(buf, pet_room))) {
      send_to_char("There is no such pet!\r\n", ch);
      return (TRUE);
    }
    if (GET_GOLD(ch) < (GET_EXP(pet) * 3)) {
      send_to_char("You don't have enough gold!\r\n", ch);
      return (TRUE);
    }
    GET_GOLD(ch) -= GET_EXP(pet) * 3;

    pet = read_mobile(GET_MOB_RNUM(pet), REAL);
    GET_EXP(pet) = 0;
    SET_BIT(AFF_FLAGS(pet), AFF_CHARM);
    SET_BIT(MOB_FLAGS(pet), MOB_SENTINEL);

    if (*pet_name) {
      sprintf(buf, "%s %s", pet->player.name, pet_name);
      /* free(pet->player.name); don't free the prototype! */
      pet->player.name = str_dup(buf);

      sprintf(buf, "%sA small sign on a chain around the neck says 'My name is %s'\r\n",
	      pet->player.description, CAP(pet_name));
      /* free(pet->player.description); don't free the prototype! */
      pet->player.description = str_dup(buf);
    }
    char_to_room(pet, ch->in_room);
    add_follower(pet, ch);

    /* Be certain that pets can't get/carry/use/wield/wear items */
    IS_CARRYING_W(pet) = 1000;
    IS_CARRYING_N(pet) = 100;

    send_to_char("May you enjoy your pet.\r\n", ch);
    act("$n buys $N as a pet.", FALSE, ch, 0, pet, TO_ROOM);

    return 1;
  }
  /* All commands except list and buy */
  return 0;
}


/* phonebooth teleporter special for Rose Mall */
#define NUM_DEST 7 /* number of destinations available */
SPECIAL(phone_booth) {
  int i, number= 0;
  int charges[NUM_DEST][3] = {
    /* ph_num *//* cost */      /* room */
    {132027,    200000,         11028}, /* over LBT */
    {132830,    40000,          834},   /* scarborough fair */
    {132210,    50000,          21001}, /* port of new sparta */
    {132120,    30000,          12051}, /* east gate of rome */
    {132020,    300000,         2001},  /* entrance to floating castle */
    {132008,    100000,         800},   /* the foyer (brady house) */
    {132049,    100000,         4900}   /* battle of the 5 armies */
  };

  if(CMD_IS("list")) {
    sprintf(buf, "Bell Kiwi Payphone.  To reach your destination, 'dial' the appropriate\r\n");
    sprintf(buf, "%sphone number.\r\n\r\n", buf);
    sprintf(buf, "%sNumber    Cost     Destination\r\n", buf);
    sprintf(buf, "%s------------------------------------------------------------------------\r\n", buf);
    for(i = 0; i < NUM_DEST; i++)
      sprintf(buf, "%s%5d   %6d   %s\r\n", buf, charges[i][0], charges[i][1],
	world[real_room(charges[i][2])].name);
    send_to_char(buf, ch);
    return(TRUE);
  } else if(CMD_IS("dial")) {
    number = atoi(argument);
    for(i = 0; i < NUM_DEST; i++) {
      if(number == charges[i][0]) {
	if(charges[i][1] > GET_GOLD(ch)) {
	  send_to_char("You don't have enough gold!\r\n", ch);
	  return(TRUE);
	} else {
	  GET_GOLD(ch) -= charges[i][1];
	  sprintf(buf, "You insert %d gold coins, dial %d, and are sucked into the phone!\r\n",
	    charges[i][1], charges[i][0]);
	  send_to_char(buf, ch);
	  sprintf(buf, "$n inserts a pocket full of gold in the phone, pushes some numbers, and\r\n");
	  sprintf(buf, "%sis sucked into the phone!\r\n", buf);
	  act(buf, TRUE, ch, 0, FALSE, TO_ROOM);
	  char_from_room(ch);
	  char_to_room(ch, real_room(charges[i][2]));
	  look_at_room(ch, 0);
	  return(TRUE);
	}
      }
    }
  }
  return(FALSE);
}


SPECIAL(lynx){
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(1,30)) {
   case 1:      
	act("$n winds $mself around your legs and purrs.", 
	    TRUE, ch, 0, 0, TO_ROOM);
	return 1;
   case 2: 
	do_say(ch, "meow", 0, 0);
	return 1;
   case 3: 
	act("$n rubs his head in your face affectionately. 'ewww cat fur!'",
	    TRUE, ch, 0, 0, TO_ROOM);
	return 1;
   default:
	return 0;
  } 
}


/* ********************************************************************
*  Special procedures for objects                                     *
******************************************************************** */


SPECIAL(bank)
{
  int amount;

  if ( (CMD_IS("balance") || CMD_IS("deposit") || CMD_IS("withdraw")) &&
       (IS_MONK(ch)) ) {
    sprintf(buf, "You should stay away from such worldly things.\r\n");
    send_to_char(buf, ch);
    sprintf(buf, "$n curses the worldliness of $s thoughts.\r\n");
    act(buf, TRUE, ch, 0, FALSE, TO_ROOM);
    return 1;
    }

  if (CMD_IS("balance")) {
    if (GET_BANK_GOLD(ch) > 0)
      sprintf(buf, "Your current balance is %d coins.\r\n",
	      GET_BANK_GOLD(ch));
    else
      sprintf(buf, "You currently have no money deposited.\r\n");
    send_to_char(buf, ch);
    return 1;
  } else if (CMD_IS("deposit")) {
    if ((amount = atoi(argument)) <= 0) {
      send_to_char("How much do you want to deposit?\r\n", ch);
      return 1;
    }
    if (GET_GOLD(ch) < amount) {
      send_to_char("You don't have that many coins!\r\n", ch);
      return 1;
    }
    GET_GOLD(ch) -= amount;
    GET_BANK_GOLD(ch) += amount;
    sprintf(buf, "You deposit %d coins.\r\n", amount);
    send_to_char(buf, ch);
    act("$n makes a bank transaction.", TRUE, ch, 0, FALSE, TO_ROOM);
    return 1;
  } else if (CMD_IS("withdraw")) {
    if ((amount = atoi(argument)) <= 0) {
      send_to_char("How much do you want to withdraw?\r\n", ch);
      return 1;
    }
    if (GET_BANK_GOLD(ch) < amount) {
      send_to_char("You don't have that many coins deposited!\r\n", ch);
      return 1;
    }
    GET_GOLD(ch) += amount;
    GET_BANK_GOLD(ch) -= amount;
    sprintf(buf, "You withdraw %d coins.\r\n", amount);
    send_to_char(buf, ch);
    act("$n makes a bank transaction.", TRUE, ch, 0, FALSE, TO_ROOM);
    return 1;
  } else
    return 0;
}

SPECIAL(gym)
{
  int stat, success = FALSE, try;
  char statname[15];

  if (!CMD_IS("train"))
    return 0;

  argument = one_argument(argument, buf);
  
  if (!*buf) {
    send_to_char("Train a stat!  Read the sign for information.\r\n", ch);
    return 1;
  }
  if (GET_POS(ch) == POS_FIGHTING) return 1;
  
  /* which stat do we want to train? */
  if (is_abbrev(buf, "str")) {
    stat = 1; 
    sprintf(statname, "strength");
  } else if (is_abbrev(buf, "int")) {
    stat = 2; 
    sprintf(statname, "intelligence");
  } else if (is_abbrev(buf, "cha")) {
    stat = 3;
    sprintf(statname,"charisma");
  } else if (is_abbrev(buf, "wis")) {
    stat = 4;
    sprintf(statname, "wisdom");
  } else if (is_abbrev(buf, "dex")) {
    stat = 5;
    sprintf(statname, "dexterity");
  } else if (is_abbrev(buf, "con")) {
    stat = 6;
    sprintf(statname, "constitution");
  } else { 
    send_to_char("That is not a trainable stat.\r\n", ch); 
    return 1; 
  }

  if (GET_GOLD(ch) < 1000000) {
    send_to_char("You cannot afford to train!\r\n", ch);
    return 1;
  }
  if (GET_PRACTICES(ch) < 3) {
    send_to_char("You don't have enough practices to train.\r\n", ch);
    return 1;
  }
  send_to_char("One million gold received, Thank you.\r\n", ch);
  GET_GOLD(ch) -= 1000000;
  send_to_char("-----------------------------------------------\r\n", ch);
  for (try = 1; try <= 3; try++) {
    sprintf(buf, "Attempt %d at training %s - ", try, statname);
    if (number(1,101) < 33) {
      strcat(buf, "&+GSuccessful!&+w\r\n");
      success = TRUE;
      send_to_char(buf, ch);
      break;
    } else {
      strcat(buf, "&+rFailed&+w\r\n");
      send_to_char(buf, ch);
    }
  }
  send_to_char("-----------------------------------------------\r\n", ch);
  if (!success) {
    send_to_char("You failed to train.\r\n", ch);
    return 1; 
  }
  GET_PRACTICES(ch) -= 3;
  switch (stat) {
    case 1: if (ch->real_abils.str < 18) { ch->real_abils.str++;
                                    GET_STR(ch)++; }
            else if (ch->real_abils.str_add < 100) { ch->real_abils.str_add += 10;
                                          GET_ADD(ch) += 10; }
            sprintf(buf, "Your strength is now %d/%d", GET_STR(ch), 
                    GET_ADD(ch));
            break;
    case 2: if (ch->real_abils.intel < 18) { ch->real_abils.intel++;
                                    GET_INT(ch)++; }
            sprintf(buf, "Your intelligence is now %d", GET_INT(ch)); 
            break;             
    case 3: if (ch->real_abils.cha < 18) { ch->real_abils.cha++;
                                    GET_CHA(ch)++; }
            sprintf(buf, "Your charisma is now %d", GET_CHA(ch)); 
            break;
    case 4: if (ch->real_abils.wis < 18) { ch->real_abils.wis++;
                                    GET_WIS(ch)++; }
            sprintf(buf, "Your wisdom is now %d", GET_WIS(ch)); 
            break;
    case 5: if (ch->real_abils.dex < 18) { ch->real_abils.dex++;
                                    GET_DEX(ch)++; }
            sprintf(buf, "Your dexterity is now %d", GET_DEX(ch)); 
            break;
    case 6: if (ch->real_abils.con < 18) { ch->real_abils.con++;
                                    GET_CON(ch)++; }
            sprintf(buf, "Your constitution is now %d", GET_CON(ch)); 
            break;            
  }
  send_to_char(buf, ch);
  send_to_char("\r\nThree practice sessions being deducted now.\r\n", ch);
  return 1;
}

SPECIAL(slots)
{
   int i, roll[3], winnings, win[3];
   char tmp1[100];
   struct obj_data *obj;
   
   if (!CMD_IS("use"))
     return 0;
   one_argument(argument, arg);
   obj = (struct obj_data *) me;
   if (!is_abbrev(arg, obj->name))
     return 0;
   
   if (GET_OBJ_VAL(obj, 0) > GET_GOLD(ch)) {
     sprintf(buf, "This machine charges %d gold.  You don't have enough.",
             GET_OBJ_VAL(obj, 0));
    send_to_char(buf, ch);
    return 1;
  }
  GET_GOLD(ch) -= GET_OBJ_VAL(obj, 0);
  GET_OBJ_VAL(obj, 1) += GET_OBJ_VAL(obj, 0);
   
  *tmp1 = '\0';
  for (i = 0; i < 3; i++) {
    switch(number(1, 15)) {
      case 1: case 6: case 10: case 13: case 15:
        roll[i] = 1;
        win[i] = 5;
        sprintf(tmp1, "%s  &+m[BAR]  ", tmp1);
      break; 
      case 2: case 7: case 11: case 14:
        roll[i] = 2;
        win[i] = 25;
        sprintf(tmp1, "%s  &+y[BELL]  ", tmp1);
      break;
      case 3: case 8: case 12:
        roll[i] = 3;
        win[i] = 50;
        sprintf(tmp1, "%s  &+Y[LEMON]  ", tmp1);
      break;
      case 4: case 9:
        roll[i] = 4;
        win[i] = 100;
        sprintf(tmp1, "%s  &+g[DOLLAR]  ", tmp1);
      break;
      case 5:
        roll[i] = 5;
        win[i] = 15;
        sprintf(tmp1, "%s  &+r[CHERRY]  ", tmp1);
      break;
    }  
  }   
  strcat(tmp1, "&+w");
  sprintf(buf, "You insert %d coins and pull the handle.\r\n%s\r\n", 
          GET_OBJ_VAL(obj, 0), tmp1);
  act("$n inserts some coins into the slot machine and pulls the handle.",
     FALSE, ch, 0, 0, TO_ROOM);
  winnings = (win[0]) * GET_OBJ_VAL(obj, 0);
  if ((roll[1] == roll[2] && roll[2] == roll[0])) {
    if (roll[1] == 5) {
      winnings = GET_OBJ_VAL(obj, 1);
      sprintf(buf, "%sYou win the Jackpot!!!! %d coins!!!\r\n", buf, winnings);
      sprintf(buf1, "%s spews from the machine as $n wins the Jackpot!!",
              CAP(money_desc(winnings)));
    } else {
      sprintf(buf, "%sYou win %d coins!\r\n", buf, winnings);
      sprintf(buf1, "$n wins %s!", money_desc(winnings));
    }
    act(buf1, FALSE, ch, 0, 0, TO_ROOM);
    GET_GOLD(ch) += winnings;
  } else {
    sprintf(buf, "%sYou lose!\r\n", buf);
  }
  send_to_char(buf, ch);
  WAIT_STATE(ch, PULSE_VIOLENCE * 2);
  return 1;
}

SPECIAL(elminster) /*vnum 1409*/
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  if (IS_AFFECTED(vict, AFF_SANCTUARY)
      || affected_by_spell(vict, SPELL_REVERSAL)
      || IS_AFFECTED(vict, AFF_HASTE)
      || affected_by_spell(vict, SPELL_ARMOR)
      || affected_by_spell(vict, SPELL_SHIELD)
      || affected_by_spell(vict, SPELL_STONESKIN))    
    cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_MAGIC);

  if (!IS_AFFECTED(vict, AFF_SLOW))
    cast_spell(ch, vict, NULL, "", 0, SPELL_SLOW);

  if (!IS_AFFECTED(ch, AFF_BLUR))
    cast_spell(ch, ch, NULL, "", 0, SPELL_BLUR);
  
  if (!IS_AFFECTED(ch, AFF_REVERSAL))
    cast_spell(ch, ch, NULL, "", 0, SPELL_REVERSAL);

  if (!IS_AFFECTED(ch, AFF_HASTE))
    cast_spell(ch, ch, NULL, "", 0, SPELL_HASTE);    

    cast_spell(ch, vict, NULL, "", 0, SPELL_VAMPIRIC_TOUCH);

  switch (number(1, 7)) {
  case 1:
    cast_spell(ch, vict, NULL, "", 0, SPELL_VAMPIRIC_TOUCH);
    cast_spell(ch, vict, NULL, "", 0, SPELL_FIREBALL);
    break;

  case 2:
    cast_spell(ch, vict, NULL, "", 0, SPELL_VAMPIRIC_TOUCH);
    cast_spell(ch, vict, NULL, "", 0, SPELL_ENERGY_BLAST);
    break;

  case 3:
    cast_spell(ch, vict, NULL, "", 0, SPELL_VAMPIRIC_TOUCH);
    cast_spell(ch, vict, NULL, "", 0, SPELL_VAMPIRIC_TOUCH);
    break;

  case 4:
    cast_spell(ch, vict, NULL, "", 0, SPELL_MAGIC_MISSILE);
    cast_spell(ch, vict, NULL, "", 0, SPELL_COLOR_SPRAY);
    break;

  case 5:
    cast_spell(ch, vict, NULL, "", 0, SPELL_CALL_LIGHTNING);
    break;
 
  default:
    cast_spell(ch, vict, NULL, "", 0, SPELL_MAGIC_MISSILE);
    cast_spell(ch, vict, NULL, "", 0, SPELL_VAMPIRIC_TOUCH);
    break;
  }
  return TRUE;
}

SPECIAL(hoareb) /*vnum 1424 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
  switch (number(1, 4)) {

  case 1:
    cast_spell(ch, vict, NULL, "", 0, SPELL_CALL_LIGHTNING);
    break;

  case 2:
    cast_spell(ch, vict, NULL, "", 0, SPELL_HARM);
    break;

  case 3:
    cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
    break;

  default:
    break;
  }
return TRUE;
}

SPECIAL(rewel)  /* vnum 1425 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
  
  return TRUE;
}

SPECIAL(munro)  /* vnum 1426 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
    
switch (number(1, 8)) {

  case 1:
    cast_spell(ch, vict, NULL, "", 0, SPELL_CALL_LIGHTNING);
    break;

  case 2:
    cast_spell(ch, vict, NULL, "", 0, SPELL_HARM);
    break;

  case 3:
    cast_spell(ch, vict, NULL, "", 0, SPELL_EARTHQUAKE);
    break;
  
  case 4:
    cast_spell(ch, vict, NULL, "", 0, SPELL_PWR_WRD_STUN);
    break;

  case 5:
    if (IS_EVIL(vict))
      cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_EVIL);
    else
      cast_spell(ch, vict, NULL, "", 0, SPELL_BLINDNESS);
    break;

  case 6:
    cast_spell(ch, vict, NULL, "", 0, SPELL_FEAR);
    break;
  
  default:
    cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
    break;
  }
  return TRUE;
} 
SPECIAL(midnight) /* vnum 1406 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);


  if (IS_AFFECTED(vict, AFF_SANCTUARY))
    cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_MAGIC);

  if (!IS_AFFECTED(vict, AFF_SLOW))
	 cast_spell(ch, vict, NULL, "", 0, SPELL_SLOW);

  switch (number(1, 8)) {
  case 1:
    if (!IS_AFFECTED(ch, AFF_BLUR))
       cast_spell(ch, ch, NULL, "", 0, SPELL_BLUR);
    if (!IS_AFFECTED(ch, AFF_HASTE))
       cast_spell(ch, ch, NULL, "", 0, SPELL_HASTE);
	 else if  (!affected_by_spell(vict, SPELL_BLINDNESS))
            cast_spell(ch, vict, NULL, "", 0, SPELL_BLINDNESS);
    break;

  case 2:
    cast_spell(ch, vict, NULL, "", 0, SPELL_FIREBALL);
    break;
  
  case 3:
    cast_spell(ch, vict, NULL, "", 0, SPELL_LIGHTNING_BOLT);
    break;

  case 4:
    cast_spell(ch, vict, NULL, "", 0, SPELL_VAMPIRIC_TOUCH);
    break;

  case 5:
    if (!IS_AFFECTED(vict, AFF_BLIND))
       cast_spell(ch, vict, NULL, "", 0, SPELL_BLINDNESS);
    if (!IS_AFFECTED(vict, AFF_POISON))
       cast_spell(ch, vict, NULL, "", 0, SPELL_POISON);
    break;

  case 6:
    cast_spell(ch, vict, NULL, "", 0, SPELL_ENERGY_BLAST);
    break;

  default:
    cast_spell(ch, ch, NULL, "", 0, SPELL_REVERSAL);
    break;
  }
return TRUE;
}


SPECIAL(lhaeo) /* vnum 1449 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  switch (number(1,6)) {
  case 1:
    cast_spell(ch, vict, NULL, "", 0, SPELL_SLEEP);
    break;
  case 2:
    cast_spell(ch, vict, NULL, "", 0, SPELL_PWR_WRD_STUN);
    break;
  case 3: /* Will 'Word of Recall' work? if not just delete it */
    cast_spell(ch, vict, NULL, "", 0, SPELL_WORD_OF_RECALL);
    break;
  default:
    break;
 }
return TRUE;
}



SPECIAL(adon) /* vnum 1404 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  switch (number(1,4)) {

  case 1:
    cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
    break;
  
  case 2:
    cast_spell(ch, vict, NULL, "", 0, SPELL_BLINDNESS);
    break; 
  
  default:
    break;
  }
return TRUE;
}

SPECIAL(citizen) /* vnum 1401 */
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0,20))  {
  case 0:
    do_say(ch, "Welcome to Shadowdale!", 0, 0);
    return (1);
  case 1:
    do_say(ch, "I hear Elminster turned his last uninvited guest into a newt!",0, 0);
    return (1);
  case 2:
    do_say(ch, "Rumor has it some Zhentish dogs have been spotted east of town.", 0, 0);
    return (1);
  case 3:
    do_say(ch, "I hear little Timmy, Latha's boy, fell into the crevasse!", 0, 0);
    return (1);
  case 4:
    do_say(ch, "Rumor has it one of the tower guards disappeared while on duty in the cellar!", 0, 0);
    return (1);
  case 5: 
    do_say(ch, "A company of those wacky SPOOOOONERS! came through town yesterday. We're always glad to see them!", 0, 0);
    return (1);
  case 6:
    do_say(ch, "The SPOONERS! are even more popular than the Knights of Myth Drannor!", 0, 0);
    return (1);
  
  case 7:
    do_say(ch, "Elminster is so wonderous the almighty Fury even visits him from time to time!", 0, 0);
    return (1);
  default:
    return (0);
  }
}



SPECIAL(rocker)  /* vnum 1416 */
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0,30))  {
  case 0:
    do_say(ch, "Hangman, Hangman, wait a little while.", 0, 0);
    return (1);
  case 1:
    do_say(ch, "I think I see my brother coming, riding many miles.",0,0);
    return (1);
  case 2:
    do_say(ch, "I think I see my sister coming, riding many miles.",0,0);
    return (1);
  case 3:
    do_say(ch, "Brother ya get me some silver, did ya get a little gold?",0,0);
    return (1);
  case 4:
    do_say(ch, "What did you bring me my brother, to keep me from the gallows pole?", 0, 0);
    return (1);
  case 5:
    do_say(ch, "Sister ya get some silver, did ya get a little gold?",0,0);
    return (1);
  case 6:
    do_say(ch, "What did you bring me my sister, to keep me from the gallows pole?", 0, 0);
    return (1);
  default:
    return (0);
  }
}
SPECIAL(brother) /* vnum 1417 */
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0,7))  {
  case 0: /* here I actually used a printf switch...is it right? */
    do_say(ch, "Brother I brought you some silver, I brought a little gold, I brought a little of everything to keep you from the gallows pole!", 0,0);
    return (1);
  default:
    return (0);
 }
}
SPECIAL(friend) /* vnum 1459 */
{
  ACMD(do_say);

  if (cmd)
    return (0);

  switch (number(0,7))  {
  case 0:                     /* the same switch is used here..correct? */
    do_say(ch, "I couldn't get no silver, I couldn't get no gold, you know that we're too damn poor, keep you from the gallows pole!", 0, 0);
    return (1);
  default:
    return (0);
  }
}



SPECIAL(saurial) /* vnum 1415 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  switch (number(1,4)) {

  case 1:
    cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
    break;

  case 2:
    if (IS_EVIL(vict))
       cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_EVIL);
    else
       cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
    break;

  default:
    break;
  }
return TRUE;
}
 
SPECIAL(storm) /* vnum 1402 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);


  switch (number(1,5)) {

  case 1:
     cast_spell(ch, vict, NULL, "", 0, SPELL_FIREBALL);
     break;

  case 2:
     cast_spell(ch, vict, NULL, "", 0, SPELL_BURNING_HANDS);
     break;

  case 3:
     cast_spell(ch, vict, NULL, "", 0, SPELL_CALL_LIGHTNING);
     break;

  default:
     break;
  }
return TRUE;
}
 
   


SPECIAL(torm) /* vnum 1435 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  switch (number(1,4)) {

  case 1:
     cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
     break;

  case 2:
     if (IS_EVIL(vict))
        cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_EVIL);
     else 
        cast_spell(ch, ch, NULL, "", 0, SPELL_HEAL);
     break;
  
  default:
     break;
  }
return TRUE;
}

SPECIAL(bane) /* vnum 1437 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  switch (number(1,4)) {

  case 1:
     if (IS_GOOD(vict))
        cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_GOOD);
     else
        cast_spell(ch, vict, NULL, "", 0, SPELL_FIREBALL);
     break;

  case 2:
     cast_spell(ch, vict, NULL, "", 0, SPELL_ENERGY_DRAIN);
     break;

  default:
     break;

  }
return TRUE;
}
 

SPECIAL(myrkul) /* vnum 1442 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  switch (number(1,4)) {

  case 1:
     if (IS_GOOD(vict))
       cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_GOOD);

       else if (IS_EVIL(vict))
         cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_EVIL);

         else
           cast_spell(ch, vict, NULL, "", 0, SPELL_FIREBALL);
     break;

  case 2:
     cast_spell(ch, vict, NULL, "", 0, SPELL_BLINDNESS);
     cast_spell(ch, vict, NULL, "", 0, SPELL_POISON);
  break;

  case 3:
     cast_spell(ch, vict, NULL, "", 0, SPELL_LIGHTNING_BOLT);
  break;

  case 4:
     cast_spell(ch, vict, NULL, "", 0, SPELL_VAMPIRIC_TOUCH);
  break;
  }
return TRUE;
}

SPECIAL(mystra) /* vnum 1448 */
{
  struct char_data *vict;

  if (cmd || GET_POS(ch) != POS_FIGHTING)
    return FALSE;

  /* pseudo-randomly choose someone in the room who is fighting me */
  for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == ch && !number(0, 4))
      break;

  /* if I didn't pick any of those, then just slam the guy I'm fighting */
  if (vict == NULL)
    vict = FIGHTING(ch);

  if (IS_AFFECTED(vict, AFF_SANCTUARY) || 
      IS_AFFECTED(vict, AFF_REVERSAL))

    cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_MAGIC);

  if (!IS_AFFECTED(vict, AFF_SLOW))
    cast_spell(ch, vict, NULL, "", 0, SPELL_SLOW);

  if (!IS_AFFECTED(ch, AFF_BLUR))
    cast_spell(ch, ch, NULL, "", 0, SPELL_BLUR);
  
  if (!IS_AFFECTED(ch, AFF_REVERSAL))
    cast_spell(ch, ch, NULL, "", 0, SPELL_REVERSAL);

  if (!IS_AFFECTED(ch, AFF_HASTE))
    cast_spell(ch, ch, NULL, "", 0, SPELL_HASTE);

  switch (number(1,5)) {

  case 1:
     cast_spell(ch, vict, NULL, "", 0, SPELL_ENERGY_BLAST);
  break;

  case 2:
     cast_spell(ch, vict, NULL, "", 0, SPELL_CALL_LIGHTNING);
  break;

  case 3:
     cast_spell(ch, vict, NULL, "", 0, SPELL_FIREBALL);
  break;

  default:
     cast_spell(ch, vict, NULL, "", 0, SPELL_MAGIC_MISSILE);
  break;
  }
return TRUE;
}

SPECIAL(hanilzone)
{
  extern int cmd_emote;
  struct char_data *vict;
  ACMD (do_say);
  ACMD (do_echo);

  if (cmd)
    return FALSE; /* you must return either TRUE or FALSE */
  
 
  switch (mob_index[ch->nr].virtualrm) {
  case 6702:
    if (FIGHTING(ch))
      return FALSE;
    if (number (0, 4))    /* use 0, not o */
      return FALSE;
    do_echo(ch, "watches the ceremony attentively.",cmd_emote, SCMD_EMOTE);
    return TRUE;
  case 6704:
    if (FIGHTING(ch))
      return FALSE;
    if (number (0, 4))
      return FALSE;
    do_echo(ch, "plunges a knife into his sacrificial victim.", cmd_emote, 
	    SCMD_EMOTE);
    return TRUE;
  case 6706:
    switch (number(0, 3)) {
    case 0: 
      do_echo(ch,"stands at attention.",cmd_emote, SCMD_EMOTE);
      break;
    case 1: 
      do_say(ch, "Welcome, you are expected.", 0, 0); /* you had a typo here */
      break;
    default:
      return FALSE;
    }
  case 6711:
    if (!FIGHTING(ch))
      return FALSE;
    vict = FIGHTING(ch);
    cast_spell(ch,vict,NULL,"",0, SPELL_ENERGY_DRAIN);
    return TRUE;
  case 6712:
    if (!FIGHTING(ch))
      return FALSE;
    vict = FIGHTING(ch);
    cast_spell(ch,vict,NULL,"",0, SPELL_ENERGY_DRAIN);
    return TRUE;
  case 6713:
    if (!FIGHTING(ch))
      return FALSE;
    vict = FIGHTING(ch);
    cast_spell(ch,vict,NULL,"",0, SPELL_FIREBALL);
    return TRUE;
  case 6714:
    if (FIGHTING(ch))
      return FALSE;
    switch (number(0, 10)) {   /* switch, not if */
    case 0:
      do_say(ch, "off with his head!",0,0);
      break;
    case 1:
      do_say(ch,"stretch his neck!",0,0);
      break;
    case 2:
      do_echo(ch,"bangs his gavel to resume order in the court.", cmd_emote,
	      SCMD_EMOTE);
      break;
    default:
      return FALSE;
    }
  case 6720:
    if (FIGHTING(ch))
      return FALSE;
    do_echo(ch, "shambles about.", cmd_emote, SCMD_EMOTE);
    return TRUE;
  case 6721:
    if (FIGHTING(ch))
      return FALSE;
    if (number(0,5)){
    case 0:
      do_echo(ch, "begins to sharpen the blade of his axe.", cmd_emote, 
	      SCMD_EMOTE);
      break;
    case 1:
      do_echo(ch, "arranges some nooses on to the Gallows.", cmd_emote, 
	      SCMD_EMOTE);
      break;
    default:
      return FALSE;
    }
  case 6723:
    if (FIGHTING(ch))
      return FALSE;
    switch (number(0,6)){
    case 0: 
      do_say(ch, "You must stop the Army of the Dead at all costs!",0,0);
      break;
    case 1:
      do_say(ch, "Will you help me rid the world of evil?",0,0);
      break;
    case 2:
      do_say(ch, "Find the holy warrior in hell, he is your key to victory!",0,
	     0);
      break;
    }
  }
return FALSE;
}  /* you forgot a closing brace */

/* I think that all of them */

SPECIAL (finalfantasy)
{
  ACMD (do_say);
  ACMD (do_echo);
  extern int cmd_emote;

  switch (mob_index[ch->nr].virtualrm) {
  case 7700:
    switch (number (0, 4)) {
    case 0:
      if (FIGHTING (ch))
	return FALSE;
      do_echo (ch, "stops a traveller and questions them.", cmd_emote,
	       SCMD_EMOTE);
      return TRUE;
    case 1:
      if (FIGHTING (ch))
	return FALSE;
      do_say (ch, "You're not allowed in there!", cmd_emote, SCMD_EMOTE);
      return TRUE;
    default:		/* if the number wasn't one of the ones above */
      return FALSE;

    }

  case 7701:
    switch (number (0, 5))
      {

      case 0:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, FIGHTING(ch), NULL, "", 0, SPELL_LIGHTNING_BOLT);
	return TRUE;
      case 1:
	if (FIGHTING (ch))
	  return FALSE;
	do_say (ch, "My kingdom will NEVER fall!", cmd_emote, SCMD_EMOTE);
	return TRUE;
      case 2:
	if (FIGHTING (ch))
	  return FALSE;
	do_say (ch, "God speed valiant warrior.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      case 3:
	if (FIGHTING (ch))
	  return FALSE;
	do_say (ch, "Find my daughters token and her treasure is yours.",
		cmd_emote, SCMD_EMOTE);
	return TRUE;
      case 4:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "sighs deeply and mummbles something about Amanda's favorite book.", cmd_emote, SCMD_EMOTE);
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7703:
    switch (number (0, 2))
      {
      case 0:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "yawns lazily displaying her razor sharp fangs.",
		 cmd_emote, SCMD_EMOTE);
	return TRUE;
      case 1:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "purrs loudly.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7704:
    switch (number (0, 3))
      {
      case 0:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, FIGHTING(ch), NULL, "", 0, SPELL_VAMPIRIC_TOUCH);
	return TRUE;
      case 1:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "calls forth his dark magics and creates another undead monster.", 0, 0);
	return TRUE;
      case 2:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "cackles maniacly.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7705:
    switch (number (0, 2))
      {
      case 0:
	do_echo (ch, "guzzles ale from a golden keg on the counter.",
		 cmd_emote, SCMD_EMOTE);
	return TRUE;
      case 1:
	do_say (ch, "Your old blade was dull....let us toast to your new weapon!", cmd_emote, SCMD_EMOTE);
	return TRUE;
      case 2:
	do_echo (ch, "lets you try some of his potent brew from the golden keg.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7706:
    switch (number (0, 2))
      {
      case 0:
	do_say (ch, "You break it, you buy it.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      case 1:
	do_say (ch, "My magic craftsmanship is the best in the land.",
		cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7707:
    switch (number (0, 2))
      {
      case 0:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, ch, NULL, "", 0, SPELL_HEAL);
	return TRUE;
      case 1:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "tosses her head back and calls a water elemetal to aid her.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7708:
    switch (number (0, 2))
      {
      case 0:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, FIGHTING(ch), NULL, "", 0, SPELL_FIREBALL);
	return TRUE;
      case 1:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "strains his body and a beam of flame from his hand melts the rocks in front of him.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7709:
    switch (number (0, 2))
      {
      case 0:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, FIGHTING(ch), NULL, "", 0, SPELL_EARTHQUAKE);
	return TRUE;
      case 1:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "casualy tosses a house sized boulder at an apprentice drenched in sweat.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7710:
    switch (number (0, 2))
      {
      case 0:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, FIGHTING(ch), NULL, "", 0, SPELL_CALL_LIGHTNING);
	return TRUE;
      case 1:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "flys circles around a dozen yound pupils while stinging them with lightning.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7714:

    if (cmd)
      return FALSE;

    if (GET_POS (ch) != POS_FIGHTING)
      return FALSE;

    if (FIGHTING (ch) && (FIGHTING (ch)->in_room == ch->in_room) &&
	!number (0, 5))
      {
	act ("$n bites $N!", 1, ch, 0, FIGHTING (ch), TO_NOTVICT);
	act ("$n bites you!", 1, ch, 0, FIGHTING (ch), TO_VICT);
	call_magic (ch, FIGHTING (ch), 0, "", SPELL_POISON, GET_LEVEL(ch), 0,
		    CAST_SPELL);

	return TRUE;
      }


  case 7724:

    if (cmd)
      return FALSE;

    if (GET_POS (ch) != POS_FIGHTING)
      return FALSE;

    if (FIGHTING (ch) && (FIGHTING (ch)->in_room == ch->in_room) &&
	!number (0, 5))
      {
	act ("$n stings $N!", 1, ch, 0, FIGHTING (ch), TO_NOTVICT);
	act ("$n stings you!", 1, ch, 0, FIGHTING (ch), TO_VICT);
	call_magic (ch, FIGHTING (ch), 0, "", SPELL_POISON, GET_LEVEL(ch), 0,
		    CAST_SPELL);

	return TRUE;
      }


  case 7719:

    if (cmd)
      return FALSE;

    if (GET_POS (ch) != POS_FIGHTING)
      return FALSE;

    if (FIGHTING (ch) && (FIGHTING (ch)->in_room == ch->in_room) &&
	!number (0, 5))
      {
	act ("$n claws $N!", 1, ch, 0, FIGHTING (ch), TO_NOTVICT);
	act ("$n claws you!", 1, ch, 0, FIGHTING (ch), TO_VICT);
	call_magic (ch, FIGHTING(ch), 0, "", SPELL_PARALYZE, GET_LEVEL(ch), 0,
		    CAST_SPELL);
	return TRUE;
      }


  case 7725:
    switch (number (0, 2))
      {
      case 0:
	do_say (ch, "My food is cooked fresh!", cmd_emote, SCMD_EMOTE);
	return TRUE;
      case 1:
	do_say (ch, "My wife makes all the bags by hand.", cmd_emote,
		SCMD_EMOTE);
	return TRUE;
      case 2:
	do_say (ch, "My brews are all well aged.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }

  case 7736:
    switch (number (0, 3))
      {
      case 0:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, FIGHTING(ch), NULL, "", 0, SPELL_WEB);
	return TRUE;
      case 1:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, ch, NULL, "", 0, SPELL_FAERIE_FIRE);
	return TRUE;
      case 2:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, ch, NULL, "", 0, SPELL_CURE_SERIOUS);
	return TRUE;
      case 3:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "sighs heavily while mummbling about her crystal being the key....", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }


  case 7739:
    switch (number (0, 3))
      {
      case 0:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, FIGHTING(ch), NULL, "", 0, SPELL_FIREBALL);
	return TRUE;
      case 1:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, ch, NULL, "", 0, SPELL_FAERIE_FIRE);
	return TRUE;
      case 2:
	if (!FIGHTING (ch))
	  return FALSE;
	cast_spell (ch, ch, NULL, "", 0, SPELL_CURE_SERIOUS);
	return TRUE;
      case 3:
	if (FIGHTING (ch))
	  return FALSE;
	do_echo (ch, "sobs quietly to herself.", cmd_emote, SCMD_EMOTE);
	return TRUE;
      default:		/* if the number wasn't one of the ones above */
	return FALSE;
      }
  }
  return FALSE;
}

SPECIAL(portal)
{
  struct obj_data *obj, *obj2;
  int room;

  if (!CMD_IS("enter"))
    return 0;
  one_argument(argument, arg);
  obj = (struct obj_data *) me;
  
  if (!(obj2 = get_obj_in_list_vis(ch, arg, world[ch->in_room].contents))) {
    return 0;
  } else {
    if (obj != obj2)
      return 0;
  }

  room = real_room(GET_OBJ_VAL(obj, 0));
  if (room <= 0) {
    act("$n tries to enter $p, but $e is repulsed.", TRUE, ch, obj, 0, 
        TO_ROOM);
    act("You try to enter $p, but you are repulsed.", FALSE, ch, obj, 0,
        TO_CHAR);
    return 1;
  }
  act("$n steps up to $p and disappears.", TRUE, ch, obj, 0, TO_ROOM);
  act("You step up to $p and inside.", FALSE, ch, obj, 0, TO_CHAR);
  char_from_room(ch);
  char_to_room(ch, room);
  act("$n steps out of mid-air.", TRUE, ch, 0, 0, TO_ROOM);
  look_at_room(ch, 1);
  return 1;
}

SPECIAL(nomudcon)
{
  if (!CMD_IS("say")) 
    return 0;
  skip_spaces(&argument);
  if (!*argument)
    return 0;
  if (!strcmp(argument, "jabber Jabber JABBER")) {
    act("$n disappears in a mushroom cloud.", FALSE, ch, 0, 0, TO_ROOM);
    char_from_room(ch);
    char_to_room(ch, real_room(2762));
    act("$n arrives from a puff of smoke.", FALSE, ch, 0, 0, TO_ROOM);
    look_at_room(ch, 0);
    return 1;
  }
  return 0;
}
  
SPECIAL(furytwin) /* 1458 */
{
  struct char_data *tch, *vict, *next_tch;
  struct obj_data *obj;
  int found, tmp1, i;
  void perform_tell(struct char_data *ch, struct char_data *vict, char *arg);
  void perform_wear(struct char_data * ch, struct obj_data * obj, int where);
  
  ACMD(do_say);
  ACMD(do_tell); 
  ACMD(do_disarm);
  ACMD(do_dazzle);
  ACMD(do_charge);
  
  if (cmd) 
    return FALSE;
  
  if (!FIGHTING(ch) && GET_HIT(ch) < GET_MAX_HIT(ch)) {
    GET_HIT(ch) = GET_MAX_HIT(ch);
    act("&+r$n &+yrestores himself!", FALSE, ch, 0, 0, TO_ROOM);
    do_say(ch, "It's good to be a &+rGOD!&+w", 0, 0);
  }
  
  GET_ALIGNMENT(ch) = 0;
  if (!FIGHTING(ch)) {
    for (tch = world[ch->in_room].people; tch; tch = tch->next_in_room) {
      if (IS_NPC(tch) || IS_IMMORTAL(tch) || !CAN_SEE(ch, tch))
        continue;
      if (ch->char_specials.daytimer >= 8) {
        ch->char_specials.daytimer = 0;
        act("&+w$n screams '&+rYou have been warned!! Prepare to meet your destiny!!!!&+w'\r\n",
            FALSE, ch, 0, 0, TO_ROOM);
        hit(ch, tch, TYPE_UNDEFINED);
        return TRUE;
      }    
      perform_tell(ch, tch, 
          "&+r$N&+y, I suggest you leave before I am forced to take your head!&+w\r\n");
      ch->char_specials.daytimer++;
    }
    return FALSE;
  } else {
  /* ******************************************************************* */
  /* here are the fighting parts                                         */
  /* ******************************************************************* */
  
    ch->char_specials.daytimer = 0;
    for (vict = world[ch->in_room].people; vict; vict = vict->next_in_room) {
      if (IS_NPC(vict) || FIGHTING(vict) != ch)
        continue;
      if (isname(GET_NAME(vict), "crunk akuma minion venom girlyman"))
        break;
      else if (IS_MONK(vict))
        break;
      else if (!number(0, 4))
        break;
    }
    if (vict == NULL)
      vict = FIGHTING(ch);
  
    if (GET_POS(ch) != POS_FIGHTING) {
      GET_POS(ch) = POS_FIGHTING;
      act("$n stands up.", 1, ch, 0, 0, TO_ROOM);
    }
  
    if (!GET_EQ(ch, WEAR_WIELD)) {
      found = FALSE;
      for (obj = ch->carrying; obj; obj = obj->next)
        if (GET_OBJ_VNUM(obj) == 1471 || GET_OBJ_VNUM(obj) == 1472) {
          found = TRUE;
          break;
        }
      if (found) {
        perform_wear(ch, obj, WEAR_WIELD);
      }
    }
  
/*
    if (ch->player.roundsout) {
      ch->player.roundsout = 0;
      GET_HIT(ch) += (GET_MAX_HIT(ch) - GET_HIT(ch)) / 2; 
      act("&+r$n&+y concentrates for a moment...$e seems healthier.&+w", 
          TRUE, ch, 0, 0, TO_ROOM);
      do_say(ch, "&+yYou chose...poorly", 0, 0);
   
      tch = read_mobile(1466, VIRTUALRM);
      char_to_room(tch, ch->in_room);
      act("&+gOne of &+rFury's&+g helpers appears.&+w", FALSE, ch, 0, 0,
          TO_ROOM);
    }    
*/
  
    /* he regenerates some every round of combat */
    found = ( GET_HIT(ch) * 100 ) / GET_MAX_HIT(ch);
    if (found < 30)
      GET_HIT(ch)+=500;
    else if (found < 50)
      GET_HIT(ch)+=400;
    else if (found < 70) 
      GET_HIT(ch)+=300;
    

    /* cast dispel magic if they have a certain flag(s) */
    if (IS_AFFECTED(vict, AFF_SANCTUARY) || IS_AFFECTED(vict, SPELL_REVERSAL)
        || IS_AFFECTED(vict, SPELL_HASTE) || IS_AFFECTED(vict, SPELL_ARMOR)
        || IS_AFFECTED(vict, SPELL_SHIELD) || IS_AFFECTED(vict, SPELL_STONESKIN))
      cast_spell(ch, vict, NULL, "", 0, SPELL_DISPEL_MAGIC);

    if (!IS_AFFECTED(vict, AFF_SLOW))
      cast_spell(ch, vict, NULL, "", 0, SPELL_SLOW);
 
    if (!number(0, 100)) {
      GET_HIT(ch) = GET_MAX_HIT(ch);
      do_say(ch, "&+wIt's good to be a &+rGOD!&+w", 0, 0);
    }
    
    found = number(1, 104);
    if (found < 20) {
      do_disarm(ch, GET_NAME(vict), 0, 0);
    } else if (found < 40) {
      if ((GET_MAX_HIT(ch) - GET_HIT(ch)) > 1500) {
        GET_HIT(ch) += 1500;
        act("&+r$n&+y looks healthier...bet you wish you could do that!&+w",
            TRUE, ch, 0, 0, TO_ROOM);
      }
    } else if (found < 50 && !vict->player.roundsout) {
      do_dazzle(ch, GET_NAME(vict), 0, 0);
    } else if (found < 60) {
      tch = read_mobile(1466, VIRTUALRM);
      char_to_room(tch, ch->in_room);
      act("&+gOne of &+rFury's&+g helpers appears.&+w", FALSE, ch, 0, 0, 
          TO_ROOM);
    } else if (found < 80) {
      do_charge(ch, "", 0, 0);
    } else if (found < 82) {
      do_say(ch, "Go AWAY!&+w", 0, 0);
    } else if (found < 84) {
      do_say(ch, "&+rI'm INVINCIBLE!!!!", 0, 0);
    } else if (found < 86) {
      do_say(ch, "&+y'Tis but a scratch!", 0, 0);
    } else if (found < 88) {
      do_say(ch, "&+BCome on you pansy!", 0, 0);
    } else if (found < 90) {
        do_say(ch, "I'll do you for that!", 0, 0);
    } else if (found < 92) {
      do_say(ch, "Leave me alone, I'm trying to build here.", 0, 0);
    } else if (found < 94) {
      act("&+r$n&+y lets a real rip roarer of a fart. Eww Gross!'", 1, ch,
          0, 0, TO_ROOM); 
      for (tch = world[ch->in_room].people; tch; tch = next_tch) {
        next_tch = tch->next_in_room;
        if (!IS_NPC(tch)) 
          call_magic(ch, tch, 0, "", SPELL_POISON, 1, 0, CAST_SPELL);
      }
    } else if (found < 97) {
      if (PLR_FLAGGED(vict, PRF_NOHIT))
        return TRUE; 
      tmp1 = GET_HIT(vict) / 2;
      GET_HIT(vict) -= tmp1;
      GET_HIT(ch) += tmp1;
      if (GET_HIT(ch) < 0)
        GET_HIT(ch) = GET_MAX_HIT(ch);
      GET_HIT(ch) = MIN(GET_HIT(ch), GET_MAX_HIT(ch));
      act("&+r$n&+y points a finger at &+g$N&+y and chuckles as &+g$N&+y weakens!&+w", 
          TRUE, ch, 0, vict,  TO_NOTVICT);
      act("&+r$n&+y points a finger at you and you feel considerably weaker!&+w",
          TRUE, ch, 0, vict, TO_VICT);
    } else if (found < 102) {   
      act("&+yBeams of &+ggreen&+y light shoots from &+r$n's&+y eyes and strike &+m$N!&+w",
          TRUE, ch, 0, vict,  TO_NOTVICT);
      act("&+yBeams of &+ggreen&+y light shoots from &+r$n's&+y eyes and strike YOU!&+w", 
          TRUE, ch, 0, vict, TO_VICT);
      for (i = 0; i < NUM_WEARS; i++) {
        if (GET_EQ(vict, i)) {
          obj_to_room(unequip_char(vict, i), ch->in_room);
        }
      }
      for (i = 1; i < 4; i++) { 
        tch = read_mobile(3068, VIRTUALRM);
      /*  free(GET_NAME(tch));
        GET_NAME(tch)=str_dup("masssive flubber");
        free(tch->player.long_descr);
        tch->player.long_descr = str_dup("A very large Flubber");
        free(tch->player.description);
        tch->player.description = str_dup("This Flubber is one of the largest you have ever seen");
        free(tch->player.short_descr);
        tch->player.short_descr = str_dup("massive Flubber");
*/
        GET_MAX_HIT(tch) = 2000;
        GET_HIT(tch) = 2000;    
        char_to_room(tch, ch->in_room);
      }      
    } else if (found < 104) {
      act("&+r$n&+y points at &+g$N&+y and says, 'Go to Hell. Do not pass GO.  Do not collect 200 gold.&+w'", 
          TRUE, ch, 0, vict, TO_NOTVICT);
      act("&+r$n&+y points at &+gyou&+y and says, 'Go to Hell.&+w'", TRUE, ch, 
          0, vict,  TO_VICT);
      char_from_room(vict);
      char_to_room(vict, real_room(6700));
      send_to_char("Welcome to Hell\r\n", vict);
    }
    return TRUE;
  }
  return FALSE;
}
  

SPECIAL(furyhelper) /* vnum 1466 */
{
  struct char_data *vict, *tch;

  if (cmd || !AWAKE(ch) || FIGHTING(ch) || !ch->in_room)
    return FALSE;

  for (vict = world[ch->in_room].people; vict ; vict = tch) {
    tch = vict->next_in_room;
    if (IS_NPC(vict) && GET_MOB_VNUM(vict) == 1458 && FIGHTING(vict)) {
      act("$n &+yscreams &+rSTOP ATTACKING MY FRIEND!!!&+y and jumps to the aid of $N!",
    FALSE, ch, 0, vict, TO_ROOM);
      hit(ch, FIGHTING(vict), TYPE_UNDEFINED);
      FIGHTING(FIGHTING(vict)) = ch;
      return TRUE;
    }
  }
  return FALSE;
}

SPECIAL (twinsroom) /* room vnum 1549 */
{
  struct char_data *tch;

  for (tch = world[ch->in_room].people; tch; tch = tch->next_in_room) {
    if ((IS_NPC(tch)) && (GET_MOB_VNUM(tch) == 1458))
      if (tch->player.roundsout) {
        tch->player.roundsout = 0;
        act("&+bFury's &+rEVIL&+b twin&+w shakes himself back to attention.",
            FALSE, tch, 0, 0, TO_ROOM);
        return TRUE;
      }
  }
  return FALSE;
}

