/* ************************************************************************
*   File: spells.c                                      Part of CircleMUD *
*  Usage: Implementation of "manual spells".  Circle 2.2 spell compat.    *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */


#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "spells.h"
#include "handler.h"
#include "db.h"
#include "interpreter.h" /* dwix */

extern struct room_data *world;
extern struct obj_data *object_list;
extern struct char_data *character_list;
//extern struct cha_app_type cha_app[];
extern struct int_app_type int_app[];
extern struct index_data *obj_index;

extern struct weather_data weather_info;
extern struct descriptor_data *descriptor_list;
extern struct zone_data *zone_table;
extern char *dirs[];
extern char *dir_abbrev[];

extern int mini_mud;
extern int pk_allowed;

extern struct default_mobile_stats *mob_defaults;
extern char weapon_verbs[];
extern int *max_ac_applys;
extern struct apply_mod_defaults *apmd;
int control_weather;

void clearMemory(struct char_data * ch);
void act(char *str, int i, struct char_data * c, struct obj_data * o,
	      void *vict_obj, int j);

void damage(struct char_data * ch, struct char_data * victim,
	         int damage, int weapontype);

void weather_change(void);
void weight_change_object(struct obj_data * obj, int weight);
void add_follower(struct char_data * ch, struct char_data * leader);
int mag_savingthrow(struct char_data * ch, int type, int bonus);
int House_can_enter(struct char_data * ch, sh_int house);

ACMD(do_flee); /* dwix: prototype to aviod warnings */

#define NOEFFECT "Your spell has no apparent effect.\r\n"
/*
 * Special spells appear below.
 */

ASPELL(spell_create_water)
{
  int water;

  void name_to_drinkcon(struct obj_data * obj, int type);
  void name_from_drinkcon(struct obj_data * obj);

  if (ch == NULL || obj == NULL)
    return;
  level = MAX(MIN(level, LVL_SIMP), 1);

  if (GET_OBJ_TYPE(obj) == ITEM_DRINKCON) {
    if ((GET_OBJ_VAL(obj, 2) != LIQ_WATER) && (GET_OBJ_VAL(obj, 1) != 0)) {
      name_from_drinkcon(obj);
      GET_OBJ_VAL(obj, 2) = LIQ_SLIME;
      name_to_drinkcon(obj, LIQ_SLIME);
    } else {
      water = MAX(GET_OBJ_VAL(obj, 0) - GET_OBJ_VAL(obj, 1), 0);
      if (water > 0) {
	GET_OBJ_VAL(obj, 2) = LIQ_WATER;
	GET_OBJ_VAL(obj, 1) += water;
	weight_change_object(obj, water);
	name_from_drinkcon(obj);
	name_to_drinkcon(obj, LIQ_WATER);
	act("$p is filled.", FALSE, ch, obj, 0, TO_CHAR);
      }
    }
  }
}


ASPELL(spell_recall)
{
  extern sh_int r_mortal_start_room;

  if (victim == NULL || IS_NPC(victim))
    return;

  act("$n disappears.", TRUE, victim, 0, 0, TO_ROOM);
  char_from_room(victim);
  char_to_room(victim, r_mortal_start_room);
  act("$n appears in the middle of the room.", TRUE, victim, 0, 0, TO_ROOM);
  look_at_room(victim, 0);
}


ASPELL(spell_teleport)
{
  int to_room;
  extern int top_of_world;

  if (victim != NULL)
    return;

  do {
    to_room = number(0, top_of_world);
  } while (ROOM_FLAGGED(to_room, ROOM_PRIVATE | ROOM_DEATH));

  if ( (GET_LEVEL(victim) >= LVL_IMMORT) && (ch != victim) ) {
    send_to_char("You can't get rid of a god that easily...\r\n", ch);
    act("Haha!  $N tried to recall you!", TRUE, victim, 0, ch, TO_CHAR);
    return;
  }

  if(ROOM_FLAGGED(victim->in_room, ROOM_NORECALL)) {
    send_to_char("You feel momentary vertigo as your spirit bounces back in on itself.\r\n", ch);
    act("$n flickers and fades, then returns to normal.\r\n", TRUE, victim,
     0, 0, TO_ROOM);
    return;
  }

  act("$n slowly fades out of existence and is gone.",
      FALSE, victim, 0, 0, TO_ROOM);
  if (affected_by_spell(victim, SPELL_WEB))
      affect_from_char(victim, SPELL_WEB);
  char_from_room(victim);
  char_to_room(victim, to_room);
  act("$n slowly fades into existence.", FALSE, victim, 0, 0, TO_ROOM);
  look_at_room(victim, 0);
}

#define SUMMON_FAIL "You failed.\r\n"

ASPELL(spell_summon)
{
  int room;
  if (ch == NULL || victim == NULL)
    return;

  if (GET_LEVEL(victim) > MIN(LVL_IMMORT - 1, level + 3)) {
    send_to_char(SUMMON_FAIL, ch);
    return;
  }

  if (MOB_FLAGGED(victim, MOB_NOSUMMON)) {
    act("$E just doesn't want to come to you.  Sorry, you can't do it to $M.", 
        FALSE, ch, 0, victim, TO_CHAR);
    return;
  }
  if (ROOM_FLAGGED(ch->in_room, ROOM_CRIMEOK) || 
      ROOM_FLAGGED(victim->in_room, ROOM_NOSUMMON)) {
    act("Your spell has been nullified by a greater (divine?) force.",
        FALSE, ch, 0, victim, TO_CHAR);
    return;
  }
  if (PLR_FLAGGED(victim, PLR_AUTOBOT)) {
    sprintf(buf, "SUMMON attempt: %s tried to summon the BOT %s to %s (%d)",
            GET_NAME(ch), GET_NAME(victim), world[ch->in_room].name, 
            world[ch->in_room].number);
    mudlog(buf, BRF, LVL_IMMORT, TRUE);
    send_to_char("You are not allowed to summon bots.  Now go away!\r\n", ch);
    return;
  }
  if (!pk_allowed) {
/*    if (MOB_FLAGGED(victim, MOB_AGGRESSIVE)) {
      act("As the words escape your lips and $N travels\r\n"
	  "through time and space towards you, you realize that $E is\r\n"
	  "aggressive and might harm you, so you wisely send $M back.",
	  FALSE, ch, 0, victim, TO_CHAR);
      return;
    } */
    if (!IS_NPC(victim) && !PRF_FLAGGED(victim, PRF_SUMMONABLE) &&
	!PLR_FLAGGED(victim, PLR_KILLER)) {
      sprintf(buf, "%s just tried to summon you to: %s.\r\n"
	      "%s failed because you have summon protection on.\r\n"
	      "Type NOSUMMON to allow other players to summon you.\r\n",
	      GET_NAME(ch), world[ch->in_room].name,
	      (ch->player.sex == SEX_MALE) ? "He" : "She");
      send_to_char(buf, victim);

      sprintf(buf, "You failed because %s has summon protection on.\r\n",
	      GET_NAME(victim));
      send_to_char(buf, ch);

      sprintf(buf, "%s failed summoning %s to %s.",
	      GET_NAME(ch), GET_NAME(victim), world[ch->in_room].name);
      mudlog(buf, BRF, LVL_IMMORT, TRUE);
      return;
    }
  }

  if (MOB_FLAGGED(victim, MOB_NOCHARM) ||
      (IS_NPC(victim) && mag_savingthrow(victim, SAVING_SPELL, 0))) {
    send_to_char(SUMMON_FAIL, ch);
    return;
  }

  act("$n disappears suddenly.", TRUE, victim, 0, 0, TO_ROOM);

  room = ch->in_room;
  char_from_room(victim);
  char_to_room(victim, room);
  if (affected_by_spell(victim, SPELL_WEB))
    affect_from_char(victim, SPELL_WEB);
  act("$n arrives suddenly.", TRUE, victim, 0, 0, TO_ROOM);
  act("$n has summoned you!", FALSE, ch, 0, victim, TO_VICT);
  look_at_room(victim, 0);
}



ASPELL(spell_locate_object)
{
  struct obj_data *i;
  char name[MAX_INPUT_LENGTH];
  int j;

  if (tstring && *tstring)
    strcpy(name, tstring);
  j = level >> 1;

  for (i = object_list; i && (j > 0); i = i->next) {
    if (!isname(name, i->name))
      continue;

    if (i->carried_by)
      sprintf(buf, "%s is being carried by %s.\n\r",
	      i->short_description, PERS(i->carried_by, ch));
    else if (i->in_room != NOWHERE)
      sprintf(buf, "%s is in %s.\n\r", i->short_description,
	      world[i->in_room].name);
    else if (i->in_obj)
      sprintf(buf, "%s is in %s.\n\r", i->short_description,
	      i->in_obj->short_description);
    else if (i->worn_by)
      sprintf(buf, "%s is being worn by %s.\n\r",
	      i->short_description, PERS(i->worn_by, ch));
    else
      sprintf(buf, "%s's location is uncertain.\n\r",
	      i->short_description);

    CAP(buf);
    send_to_char(buf, ch);
    j--;
  }

  if (j == level >> 1)
    send_to_char("You sense nothing.\n\r", ch);
}



ASPELL(spell_charm)
{
  struct affected_type af;

  if (victim == NULL || ch == NULL)
    return;

  if (victim == ch)
    send_to_char("You like yourself even better!\r\n", ch);
  else if (!IS_NPC(victim) && !PRF_FLAGGED(victim, PRF_SUMMONABLE))
    send_to_char("You fail because SUMMON protection is on!\r\n", ch);
  else if (IS_AFFECTED(victim, AFF_SANCTUARY))
    send_to_char("Your victim is protected by sanctuary!\r\n", ch);
  else if (MOB_FLAGGED(victim, MOB_NOCHARM))
    send_to_char("Your victim resists!\r\n", ch);
  else if (IS_AFFECTED(ch, AFF_CHARM))
    send_to_char("You can't have any followers of your own!\r\n", ch);
  else if (IS_AFFECTED(victim, AFF_CHARM) || level < GET_LEVEL(victim))
    send_to_char("You fail.\r\n", ch);
  /* player charming another player - no legal reason for this */
  else if (!pk_allowed && !IS_NPC(victim))
    send_to_char("You fail - shouldn't be doing it anyway.\r\n", ch);
  else if (MOB_FLAGGED(victim, MOB_NOCHARM))
    send_to_char("Go find another, this one knows better than to follow you.\r\n", ch);
  else if (circle_follow(victim, ch))
    send_to_char("Sorry, following in circles can not be allowed.\r\n", ch);
  else if (MOB_FLAGGED(victim, MOB_MOUNTABLE))
    send_to_char("You can't charm mounts, only love them.\r\n", ch);
  else if (mag_savingthrow(victim, SAVING_PARA, 0))
    send_to_char("Your victim resists!\r\n", ch);
  else {
    if (victim->master)
      stop_follower(victim);

    add_follower(victim, ch);

    af.type = SPELL_CHARM;

    if (GET_INT(victim))
      af.duration = 24 * 18 / GET_INT(victim);
    else
      af.duration = 24 * 18;

    af.modifier = 0;
    af.location = 0;
    af.bitvector = AFF_CHARM;
    affect_to_char(victim, &af);

    act("Isn't $n just such a nice fellow?", FALSE, ch, 0, victim, TO_VICT);
    if (IS_NPC(victim)) {
      REMOVE_BIT(MOB_FLAGS(victim), MOB_AGGRESSIVE);
      REMOVE_BIT(MOB_FLAGS(victim), MOB_SPEC);
    }
  }
}



ASPELL(spell_identify)
{
  int i;
  int found;

  struct time_info_data age(struct char_data * ch);

  extern char *spells[];

  extern char *item_types[];
  extern char *extra_bits[];
  extern char *apply_types[];
  extern char *affected_bits[];

  if (obj) {
    send_to_char("You feel informed:\r\n", ch);
    sprintf(buf, "Object '%s', Item type: ", obj->short_description);
    sprinttype(GET_OBJ_TYPE(obj), item_types, buf2);
    strcat(buf, buf2);
    strcat(buf, "\r\n");
    send_to_char(buf, ch);

    if (obj->obj_flags.bitvector) {
      send_to_char("Item will give you following abilities:  ", ch);
      sprintbit(obj->obj_flags.bitvector, affected_bits, buf);
      strcat(buf, "\r\n");
      send_to_char(buf, ch);
    }
    send_to_char("Item is: ", ch);
    sprintbit(GET_OBJ_EXTRA(obj), extra_bits, buf);
    strcat(buf, "\r\n");
    send_to_char(buf, ch);

    sprintf(buf, "Weight: %d, Value: %d, Rent: %d, Min Level: %d:%d,\r\n",
	    GET_OBJ_WEIGHT(obj), GET_OBJ_COST(obj), GET_OBJ_RENT(obj), 
            obj->obj_flags.min_remort, GET_OBJ_MIN(obj));
    send_to_char(buf, ch);

    switch (GET_OBJ_TYPE(obj)) {
    case ITEM_SCROLL:
    case ITEM_POTION:
    case ITEM_PILL:
      sprintf(buf, "This %s casts: ", item_types[(int) GET_OBJ_TYPE(obj)]);

      if (GET_OBJ_VAL(obj, 1) >= 1)
	sprintf(buf, "%s %s", buf, spells[GET_OBJ_VAL(obj, 1)]);
      if (GET_OBJ_VAL(obj, 2) >= 1)
	sprintf(buf, "%s %s", buf, spells[GET_OBJ_VAL(obj, 2)]);
      if (GET_OBJ_VAL(obj, 3) >= 1)
	sprintf(buf, "%s %s", buf, spells[GET_OBJ_VAL(obj, 3)]);
      sprintf(buf, "%s\r\n", buf);
      send_to_char(buf, ch);
      break;
    case ITEM_WAND:
    case ITEM_STAFF:
      sprintf(buf, "This %s casts: ", item_types[(int) GET_OBJ_TYPE(obj)]);
      sprintf(buf, "%s %s\r\n", buf, spells[GET_OBJ_VAL(obj, 3)]);
      sprintf(buf, "%sIt has %d maximum charge%s and %d remaining.\r\n", buf,
	      GET_OBJ_VAL(obj, 1), GET_OBJ_VAL(obj, 1) == 1 ? "" : "s",
	      GET_OBJ_VAL(obj, 2));
      send_to_char(buf, ch);
      break;
    case ITEM_WEAPON:
      sprintf(buf, "Damage Dice is '%dD%d'", GET_OBJ_VAL(obj, 1),
	      GET_OBJ_VAL(obj, 2));
      sprintf(buf, "%s for an average per-round damage of %.1f.\r\n", buf,
	      (((GET_OBJ_VAL(obj, 2) + 1) / 2.0) * GET_OBJ_VAL(obj, 1)));
      send_to_char(buf, ch);
      break;
    case ITEM_ARMOR:
      sprintf(buf, "AC-apply is %d\r\n", GET_OBJ_VAL(obj, 0));
      send_to_char(buf, ch);
      break;
    case ITEM_LIV_ARMOR:
      sprintf(buf, "AC Value is wearer's level * %d/%d + %d, Max AC is %d\r\n",
	GET_OBJ_VAL(obj, 1), GET_OBJ_VAL(obj, 2), GET_OBJ_VAL(obj, 0),
	GET_OBJ_VAL(obj, 3));
      send_to_char(buf, ch);
      break;
    case ITEM_ECONOMIZER:
      sprintf(buf, "Mana savings is %d%%\r\n", GET_OBJ_VAL(obj, 0));
      send_to_char(buf, ch);
      break;
    case ITEM_LOCKPICK:
      sprintf(buf, "Quality is %d\r\n", GET_OBJ_VAL(obj, 0));
      if (GET_OBJ_VAL(obj, 1))
       sprintf(buf, "%sThese picks are broken.", buf);
      send_to_char(buf, ch);
      break;
    }
    found = FALSE;
    for (i = 0; i < MAX_OBJ_AFFECT; i++) {
      if ((obj->affected[i].location != APPLY_NONE) &&
	  (obj->affected[i].modifier != 0)) {
	if (!found) {
	  send_to_char("Can affect you as :\r\n", ch);
	  found = TRUE;
	}
	sprinttype(obj->affected[i].location, apply_types, buf2);
	sprintf(buf, "   Affects: %s By %d\r\n", buf2, obj->affected[i].modifier);
	send_to_char(buf, ch);
      }
    }
  } else if (victim) {		/* victim */
    sprintf(buf, "Name: %s\r\n", GET_NAME(victim));
    send_to_char(buf, ch);
    if (!IS_NPC(victim)) {
      sprintf(buf, "%s is %d years, %d months, %d days and %d hours old.\r\n",
	      GET_NAME(victim), age(victim).year, age(victim).month,
	      age(victim).day, age(victim).hours);
      send_to_char(buf, ch);
    }
    sprintf(buf, "Height %d cm, Weight %d pounds\r\n",
	    GET_HEIGHT(victim), GET_WEIGHT(victim));
    sprintf(buf, "%sLevel: %d, Hits: %d, Mana: %d\r\n", buf,
	    GET_LEVEL(victim), GET_HIT(victim), GET_MANA(victim));
    sprintf(buf, "%sAC: %d, Hitroll: %d, Damroll: %d\r\n", buf,
	    GET_AC(victim), GET_HITROLL(victim), GET_DAMROLL(victim));
    sprintf(buf, "%sStr: %d/%d, Int: %d, Wis: %d, Dex: %d, Con: %d, Cha: %d\r\n",
	    buf, GET_STR(victim), GET_ADD(victim), GET_INT(victim),
	GET_WIS(victim), GET_DEX(victim), GET_CON(victim), GET_CHA(victim));
    send_to_char(buf, ch);

  }
}

ASPELL(spell_scry)
{
  int i;
  int found;

  struct time_info_data age(struct char_data * ch);

  extern char *spells[];

  extern char *item_types[];
  extern char *extra_bits[];
  extern char *apply_types[];
  extern char *affected_bits[];

  if (obj) {
    if((GET_LEVEL(ch) + 5) < GET_OBJ_MIN(obj)) {
      send_to_char("You get only a cloudy outline of the item.  It's complexity overwhelms\r\nyou.\r\n", ch);
      return;
    }
    send_to_char("You feel informed:\r\n", ch);
    sprintf(buf, "Object '%s', Item type: ", obj->short_description);
    sprinttype(GET_OBJ_TYPE(obj), item_types, buf2);
    strcat(buf, buf2);
    strcat(buf, "\r\n");
    send_to_char(buf, ch);

    if (obj->obj_flags.bitvector) {
      send_to_char("Item will give you following abilities:  ", ch);
      sprintbit(obj->obj_flags.bitvector, affected_bits, buf);
      strcat(buf, "\r\n");
      send_to_char(buf, ch);
    }
    send_to_char("Item is: ", ch);
    sprintbit(GET_OBJ_EXTRA(obj), extra_bits, buf);
    strcat(buf, "\r\n");
    send_to_char(buf, ch);

    sprintf(buf, "Weight: %d, Value: %d, Rent: %d, Min Level: %d:%d,\r\n",
	    GET_OBJ_WEIGHT(obj), GET_OBJ_COST(obj), GET_OBJ_RENT(obj), 
            obj->obj_flags.min_remort, GET_OBJ_MIN(obj));
    send_to_char(buf, ch);

    switch (GET_OBJ_TYPE(obj)) {
    case ITEM_SCROLL:
    case ITEM_POTION:
    case ITEM_PILL:
      sprintf(buf, "This %s casts: ", item_types[(int) GET_OBJ_TYPE(obj)]);

      if (GET_OBJ_VAL(obj, 1) >= 1)
	sprintf(buf, "%s %s", buf, spells[GET_OBJ_VAL(obj, 1)]);
      if (GET_OBJ_VAL(obj, 2) >= 1)
	sprintf(buf, "%s %s", buf, spells[GET_OBJ_VAL(obj, 2)]);
      if (GET_OBJ_VAL(obj, 3) >= 1)
	sprintf(buf, "%s %s", buf, spells[GET_OBJ_VAL(obj, 3)]);
      sprintf(buf, "%s\r\n", buf);
      send_to_char(buf, ch);
      break;
    case ITEM_WAND:
    case ITEM_STAFF:
      sprintf(buf, "This %s casts: ", item_types[(int) GET_OBJ_TYPE(obj)]);
      sprintf(buf, "%s %s\r\n", buf, spells[GET_OBJ_VAL(obj, 3)]);
      sprintf(buf, "%sIt has %d maximum charge%s and %d remaining.\r\n", buf,
	      GET_OBJ_VAL(obj, 1), GET_OBJ_VAL(obj, 1) == 1 ? "" : "s",
	      GET_OBJ_VAL(obj, 2));
      send_to_char(buf, ch);
      break;
    case ITEM_WEAPON:
      sprintf(buf, "Damage Dice is '%dD%d'", GET_OBJ_VAL(obj, 1),
	      GET_OBJ_VAL(obj, 2));
      sprintf(buf, "%s for an average per-round damage of %.1f.\r\n", buf,
	      (((GET_OBJ_VAL(obj, 2) + 1) / 2.0) * GET_OBJ_VAL(obj, 1)));
      send_to_char(buf, ch);
      break;
    case ITEM_ARMOR:
      sprintf(buf, "AC-apply is %d\r\n", GET_OBJ_VAL(obj, 0));
      send_to_char(buf, ch);
      break;
    case ITEM_LIV_ARMOR:
      sprintf(buf, "AC Value is wearer's level * %d/%d + %d, Max AC is %d\r\n",
	GET_OBJ_VAL(obj, 1), GET_OBJ_VAL(obj, 2), GET_OBJ_VAL(obj, 0),
	GET_OBJ_VAL(obj, 3));
      send_to_char(buf, ch);
      break;
    case ITEM_ECONOMIZER:
      sprintf(buf, "Mana savings is %d%%\r\n", GET_OBJ_VAL(obj, 0));
      send_to_char(buf, ch);
      break;
    case ITEM_LOCKPICK:
      sprintf(buf, "Quality is %d\r\n", GET_OBJ_VAL(obj, 0));
      if (GET_OBJ_VAL(obj, 1))
       sprintf(buf, "%sThese picks are broken.", buf);
      send_to_char(buf, ch);
      break;
    }
    found = FALSE;
    for (i = 0; i < MAX_OBJ_AFFECT; i++) {
      if ((obj->affected[i].location != APPLY_NONE) &&
	  (obj->affected[i].modifier != 0)) {
	if (!found) {
	  send_to_char("Can affect you as :\r\n", ch);
	  found = TRUE;
	}
	sprinttype(obj->affected[i].location, apply_types, buf2);
	sprintf(buf, "   Affects: %s By %d\r\n", buf2, obj->affected[i].modifier);
	send_to_char(buf, ch);
      }
    }
  } else if (victim) {		/* victim */
    if((GET_LEVEL(ch) + 5) < GET_LEVEL(victim)) {
      send_to_char("You see only a cloudy outline.\r\n", ch);
      return;
    }
    sprintf(buf, "Name: %s\r\n", GET_NAME(victim));
    send_to_char(buf, ch);
    if (!IS_NPC(victim)) {
      sprintf(buf, "%s is %d years, %d months, %d days and %d hours old.\r\n",
	      GET_NAME(victim), age(victim).year, age(victim).month,
	      age(victim).day, age(victim).hours);
      send_to_char(buf, ch);
    }
    sprintf(buf, "Height %d cm, Weight %d pounds\r\n",
	    GET_HEIGHT(victim), GET_WEIGHT(victim));
    sprintf(buf, "%sLevel: %d, Hits: %d, Mana: %d\r\n", buf,
	    GET_LEVEL(victim), GET_HIT(victim), GET_MANA(victim));
    sprintf(buf, "%sAC: %d, Hitroll: %d, Damroll: %d\r\n", buf,
	    GET_AC(victim), GET_HITROLL(victim), GET_DAMROLL(victim));
    sprintf(buf, "%sStr: %d/%d, Int: %d, Wis: %d, Dex: %d, Con: %d, Cha: %d\r\n",
	    buf, GET_STR(victim), GET_ADD(victim), GET_INT(victim),
	GET_WIS(victim), GET_DEX(victim), GET_CON(victim), GET_CHA(victim));
    send_to_char(buf, ch);

  }
}


ASPELL(spell_enchant_weapon)
{
  int i;

  if (ch == NULL || obj == NULL)
    return;

  if ((GET_OBJ_TYPE(obj) == ITEM_WEAPON) &&
      !IS_SET(GET_OBJ_EXTRA(obj), ITEM_MAGIC)) {

    for (i = 0; i < MAX_OBJ_AFFECT; i++)
      if (obj->affected[i].location != APPLY_NONE)
	return;

    SET_BIT(GET_OBJ_EXTRA(obj), ITEM_MAGIC);

    obj->affected[0].location = APPLY_HITROLL;
    obj->affected[0].modifier = 1 + (level >= 18);

    obj->affected[1].location = APPLY_DAMROLL;
    obj->affected[1].modifier = 1 + (level >= 20);

    if (IS_GOOD(ch)) {
      SET_BIT(GET_OBJ_EXTRA(obj), ITEM_ANTI_EVIL);
      act("$p glows blue.", FALSE, ch, obj, 0, TO_CHAR);
    } else if (IS_EVIL(ch)) {
      SET_BIT(GET_OBJ_EXTRA(obj), ITEM_ANTI_GOOD);
      act("$p glows red.", FALSE, ch, obj, 0, TO_CHAR);
    } else {
      act("$p glows yellow.", FALSE, ch, obj, 0, TO_CHAR);
    }
  }
}


ASPELL(spell_detect_poison)
{
  if (victim) {
    if (victim == ch) {
      if (IS_AFFECTED(victim, AFF_POISON))
        send_to_char("You can sense poison in your blood.\r\n", ch);
      else
        send_to_char("You feel healthy.\r\n", ch);
    } else {
      if (IS_AFFECTED(victim, AFF_POISON))
        act("You sense that $E is poisoned.", FALSE, ch, 0, victim, TO_CHAR);
      else
        act("You sense that $E is healthy.", FALSE, ch, 0, victim, TO_CHAR);
    }
  }

  if (obj) {
    switch (GET_OBJ_TYPE(obj)) {
    case ITEM_DRINKCON:
    case ITEM_FOUNTAIN:
    case ITEM_FOOD:
      if (GET_OBJ_VAL(obj, 3))
	act("You sense that $p has been contaminated.",FALSE,ch,obj,0,TO_CHAR);
      else
	act("You sense that $p is safe for consumption.", FALSE, ch, obj, 0,
	    TO_CHAR);
      break;
    default:
      send_to_char("You sense that it should not be consumed.\r\n", ch);
    }
  }
}
/*               KAKO                */
ASPELL (spell_astral_walk)
 {

  int room;
  if (victim == NULL || ch == NULL)
    return;

  if ( (GET_LEVEL(victim) > level ) && IS_NPC(victim) )
  {
   act ("You cannot go to $M.", FALSE, ch, 0, victim, TO_CHAR);
   return;
  }
  if (MOB_FLAGGED(victim, MOB_NOASTRAL)) {
   act("You try to open a portal to $N, but $E closed it.\r\n",
       FALSE, ch, 0, victim, TO_CHAR);
   return;
  }
  if ( (GET_LEVEL(victim) >= LVL_IMMORT ) && !IS_NPC(victim) )
  {
   act("$N decided to ignore your feeble attempt at reaching $M.",
       FALSE, ch, 0, victim, TO_CHAR);
   act("$n attempted to open a portal to you.", FALSE, ch, 0, victim, 
        TO_VICT);
   return;
  }
  room = victim->in_room;
  if ( ROOM_FLAGGED(room, ROOM_IMPL) ||
       ROOM_FLAGGED(room, ROOM_GODROOM) ||
       ROOM_FLAGGED(room, ROOM_NOASTRAL) ||
       ROOM_FLAGGED(room, ROOM_CLANONLY) ) {
    act("Your spell has been nullified by a greater (divine?) force.",
        FALSE, ch, 0, victim, TO_CHAR);
    return;
  }
  if ( ROOM_FLAGGED(room, ROOM_HOUSE) &&
      !House_can_enter(ch, world[room].number) ) {
    act("That's private property -- no trespassing!", FALSE, ch, 0, 
        victim, TO_CHAR);
    return;
  }
  if ( (ROOM_FLAGGED(room, ROOM_PRIVATE)) &&
        world[room].people && 
        world[room].people->next_in_room ) {
     act("There's a private conversation going on in that room.", FALSE, 
          ch, 0, victim, TO_CHAR);
     return;
  }

act("A shimmering blue disc appears before you, and you step through to $N.", 
     FALSE, ch, 0, victim, TO_CHAR);
 act("$n opens a dimensional gateway and steps through.", TRUE, ch, 0, 0, 
      TO_ROOM);
 char_from_room(ch);
 char_to_room(ch, room);
 act("$n steps out of mid-air.", TRUE, ch, 0, 0, TO_ROOM);
 look_at_room(ch,0);
}

ASPELL(spell_energy_drain)
{
  int loss;

  GET_ALIGNMENT(ch) -= 25;
  GET_ALIGNMENT(ch) = MAX(GET_ALIGNMENT(ch), -1000);
  if (mag_savingthrow(victim, SAVING_SPELL, 0))
   return;

  loss = MAX(40000, dice(7, GET_LEVEL(ch)));
  GET_EXP(victim) -= loss;
  GET_EXP(victim) = MAX(0, GET_EXP(victim));
  if (GET_LEVEL(ch) < LVL_IMMORT)
   GET_EXP(ch) += (loss >> 2);
  
}

ASPELL(spell_create_familiar)  /*  -=FAMILIAR=- */
{
  struct char_data *pet;
  struct follow_type *k;
  int mob_num;

  if (GET_LEVEL(ch) < LVL_IMMORT) {
    send_to_char("Sorry, this spell isn't ready...\r\n", ch);
    GET_MANA(ch) += cost;
    return;
  }
  for (k = ch->followers; k; k = k->next)
   if (MOB_FLAGGED(k->follower, MOB_FAMILIAR)) {
    send_to_char("You already have a familiar.\r\n", ch);
    return;
   }
  if (!(mob_num = ch->player_specials->saved.familiar) || 
      !real_mobile(mob_num))
    mob_num = 3094;
  if (tstring && atoi(tstring) > 0) 
    mob_num = atoi(tstring);
  
  pet = read_mobile(real_mobile(mob_num), REAL);
  GET_EXP(pet) = 0;
  sprintf(buf, "%s's familiar, %s, is standing here watching you.\r\n", 
          GET_NAME(ch), pet->player.short_descr);
  pet->player.long_descr = str_dup(buf);
  SET_BIT(MOB_FLAGS(pet), MOB_FAMILIAR); 
  SET_BIT(MOB_FLAGS(pet), MOB_SENTINEL);
  if (MOB_FLAGGED(pet, MOB_SPEC))
    REMOVE_BIT(MOB_FLAGS(pet), MOB_SPEC);
  if (MOB_FLAGGED(pet, MOB_AGGRESSIVE))
    REMOVE_BIT(MOB_FLAGS(pet), MOB_AGGRESSIVE);
  GET_DEFAULT_POS(pet) = POS_STANDING;
  GET_POS(pet) = POS_STANDING;
  GET_MAX_HIT(pet) = GET_MAX_HIT(ch);
  GET_HIT(pet) = GET_MAX_HIT(pet);
  ch->riding = NULL;
  ch->ridden_by = NULL;
  

  act("$n creates a $N as $s familiar.", TRUE, ch, 0, pet, TO_ROOM);
  char_to_room(pet, ch->in_room);
  add_follower(pet, ch);

  send_to_char("You create your familiar...\r\n", ch);
}

ASPELL(spell_clone)
{
  send_to_char("Oops, not ready yet..\r\n", ch);
}

ASPELL(spell_flame_blade)
{
  struct obj_data *j;
  int o_num, time;

  if (ch->equipment[WEAR_WIELD]) {
    send_to_char("You're already wielding something!\r\n", ch);
    GET_MANA(ch) += cost;
    return;
  }

  o_num = real_object(55);
  j = read_object(o_num, REAL);
  if(!j) {
    send_to_char("An bug from the ether plane steals your blade!\r\n", ch);
    GET_MANA(ch) += cost;
    mudlog("SYSERR: Obj #46 doesn't exist.  No flame blade!", BRF, 
      LVL_IMMORT, TRUE);
    return;
  }
  time = (GET_LEVEL(ch) >> 2);
  SET_BIT(GET_OBJ_EXTRA(j), ITEM_CREATED);
  GET_OBJ_TIMER(j) = time;
  send_to_char("You feel your hand brighten, and a blade suddenly appears.\r\n",
               ch);
  act("A flaming scimitar suddenly sprouts from $n's hand.", TRUE, ch, 0, 
      0, TO_ROOM);
  equip_char(ch, j, WEAR_WIELD);
}

/* NEW SPELLS -- Myst */

ASPELL(spell_fear)
{
  if (victim == NULL)
    return;

  if (!pk_allowed && !IS_NPC(ch) && !IS_NPC(victim))
    return;

  if (!(FIGHTING(ch)))
    set_fighting(ch, victim);

  if (mag_savingthrow(victim, SAVING_SPELL, 0))
    send_to_char(NOEFFECT, ch);
  else {
    act("$n cracks under the pressure!", TRUE, victim, 0, 0, TO_ROOM);
    act("Aiee! You're frightened by $N!", FALSE, victim, 0, ch, TO_CHAR);
    do_flee(victim, "", 0, 0);
  }

  WAIT_STATE(ch, PULSE_VIOLENCE);

  return;
}


ASPELL(spell_paralyze)
{

  if (victim == NULL)
    return;

  if (!(FIGHTING(ch)))
    set_fighting(ch, victim);
  if (MOB_FLAGGED(victim, MOB_NOBASH) || ((GET_LEVEL(victim) > GET_LEVEL(ch)) &&
      (mag_savingthrow(victim, SAVING_PARA, 0)))) 
    send_to_char(NOEFFECT, ch);
  else {
    WAIT_STATE(victim, PULSE_VIOLENCE * 2);
    act("You are paralyzed momentarily by $N!", FALSE, victim, 0, ch, TO_CHAR);
    act("$n is momentarily paralyzed!", TRUE, victim, 0, 0, TO_ROOM);
  }
  WAIT_STATE(ch, PULSE_VIOLENCE * 3);
}


ASPELL(spell_power_word_stun)
{

  if (victim == NULL)
    return;

  if (!(FIGHTING(ch)))
    set_fighting(ch, victim);

  if (MOB_FLAGGED(victim, MOB_NOBASH) || ((GET_HIT(victim) > 120) &&
      (mag_savingthrow(victim, SAVING_SPELL, 0)))) 
    send_to_char(NOEFFECT, ch);
  else {
    GET_POS(victim) = POS_SITTING;
    WAIT_STATE(victim, PULSE_VIOLENCE * 2);
    act("You are stunned by the words $N uttered!", 
        FALSE, victim, 0, ch, TO_CHAR);
    act("The words appear to stun $n!", TRUE, victim, 0, 0, TO_ROOM);
  }
  WAIT_STATE(ch, PULSE_VIOLENCE);
}


ASPELL(spell_gate)
{
  struct affected_type af;
  struct char_data *mob;
  struct follow_type *k;
  bool exists = FALSE;
  int r_num, number;

  for (k = ch->followers; k; k = k->next) 
    if (IS_AFFECTED(k->follower, AFF_GATED)) {
      exists = TRUE;
      break;
    }
  if (exists) {
    send_to_char("Wisely, you decide that another gated being would be beyond your control.\r\n", ch);
    return;
  }

  number = 16101;				/* devils */
  if (GET_LEVEL(ch) > 28) number = 16102;
  if (GET_LEVEL(ch) > 31) number = 16103;
  if (GET_LEVEL(ch) > 34) number = 16104;
  if (GET_LEVEL(ch) > 38) number = 16105;
  if (GET_LEVEL(ch) > 43) number = 16106;
  if (GET_LEVEL(ch) > 48) number = 16107;
  if (GET_LEVEL(ch) > 53) number = 16108;
  if (GET_LEVEL(ch) >= 60) number = 16109;
/*  if (GET_ALIGNMENT(ch) > -350) number += 9;*/	/* devas */
/*  if (GET_ALIGNMENT(ch) > 349) number += 9;*/	/* angels */
  if ((r_num = real_mobile(number)) < 0) {
    send_to_char("Nothing to gate - I guess the planes are empty.\r\n", ch);
    return;
  }
  mob = read_mobile(r_num, REAL);
  act("You've gated $N!", FALSE, ch, 0, mob, TO_CHAR);
  act("As $n finishes chanting, $N appears from a cloud of smoke!",
      TRUE, ch, 0, mob, TO_ROOM);
  char_to_room(mob, ch->in_room);
  add_follower(mob, ch);

  af.type = SPELL_CHARM;
  af.duration = 23;
  af.modifier = 0;
  af.location = 0;
  af.bitvector = AFF_CHARM;
  affect_to_char(mob, &af);

  af.type = SPELL_GATE;
  af.duration = 23;
  af.modifier = 0;
  af.location = 0;
  af.bitvector = AFF_GATED;
  affect_to_char(mob, &af); 

  REMOVE_BIT(MOB_FLAGS(mob), MOB_AGGRESSIVE);
  return;

}


ASPELL(spell_dispel_magic)
{
  static struct affected_type *af, *next;
  extern char *spell_wear_off_msg[];

  char *to_room = NULL;
  int prob, percent;
  int banish = FALSE;
  int i;

  if (victim == NULL && obj == NULL)
    return;

  if (victim) {
    if (IS_AFFECTED(victim, AFF_SANCTUARY) && IS_NPC(victim))
     REMOVE_BIT(AFF_FLAGS(victim), AFF_SANCTUARY);
    for (af = victim->affected; af; af = next) {
      next = af->next;
      if ((af->type > 0) && (af->type <= MAX_SPELLS)) {
        prob = 50 + (GET_LEVEL(ch) - GET_LEVEL(victim)) * 5;
        prob = MAX(1, MIN(prob, 100));
	percent = number (1, 101);
        if (percent <= prob) {
          if (af->type == SPELL_GATE)
            banish = TRUE;
          if (*spell_wear_off_msg[af->type]) {
            send_to_char(spell_wear_off_msg[af->type], victim);
            send_to_char("\r\n", victim);
          }
          affect_remove(victim, victim->affected);
        }
      }
    }
    if (banish) {
      extract_char(victim);
      act("You've completely dispelled $N!", TRUE, victim, 0, ch, TO_CHAR);
      act("$n dispels $N completely!", FALSE, ch, 0, victim, TO_NOTVICT);
    }
  }

  if (obj) {
    if(IS_SET(GET_OBJ_EXTRA(obj), ITEM_MAGIC))
     REMOVE_BIT(GET_OBJ_EXTRA(obj), ITEM_MAGIC);
    for(i = 0; i < MAX_OBJ_AFFECT; i++) {
     obj->affected[i].location = APPLY_NONE;
     obj->affected[i].modifier = 0;
    }
    to_room = "$N unenchants the $p.";
  }
  return;
}


ASPELL(spell_pacify)
{
  struct char_data *vict;

  WAIT_STATE(ch, PULSE_VIOLENCE);

  if (victim == NULL || !FIGHTING(victim))
    return;

  if (mag_savingthrow(victim, SAVING_SPELL, 0)) {
    send_to_char("Your soothing strains seem to have no effect.\r\n", ch);
    return;
  }

  for (vict = world[victim->in_room].people; vict; vict = vict->next_in_room)
    if (FIGHTING(vict) == victim) 
      stop_fighting(vict);

  stop_fighting(victim);
  send_to_char("You don't feel much like fighting.\r\n", victim);
  act("$N has pacified $n.", FALSE, victim, 0, ch, TO_ROOM);

  return;
}

ASPELL(spell_enrage)
{

  return;
}

ASPELL(spell_create_golem)
{
  struct char_data *mob;
  int r_num, number;

  number = 21;
  if ((r_num = real_mobile(number)) < 0) {
    send_to_char("Your golem crumbles to dust.\r\n", ch);
    return;
  }
  mob = read_mobile(r_num, REAL);
  GET_MAX_HIT(mob) = GET_MAX_HIT(ch) + dice(GET_CON(ch), 2);
  GET_HIT(mob) = GET_MAX_HIT(mob);
  SET_BIT(AFF_FLAGS(mob), AFF_CHARM);
  act("You've created $N!", FALSE, ch, 0, mob, TO_CHAR);
  act("As $n finishes casting, $N comes to life!", TRUE, ch, 0, mob, TO_ROOM);
  char_to_room(mob, ch->in_room);
  add_follower(mob, ch);

  return;

}

ASPELL(spell_control_weather)
{
 if (!tstring) {
   GET_MANA(ch) += cost;
   return; 
 }
 if (str_cmp(tstring, "better") && str_cmp(tstring, "worse")) {
   GET_MANA(ch) += cost;
   return; 
 }
 if (!str_cmp(tstring, "better"))
   control_weather = -1; 
 if (!str_cmp(tstring, "worse"))
   control_weather = 1; 
 sprintf(buf, "The weather starts to look %s.\r\n", tstring); 
 send_to_char(buf, ch); 
 weather_change(); 
 control_weather = 0; 
 return; 
}

ASPELL(spell_decompose)
{
  struct obj_data *i;
  int virtualrm; 
  bool found; 
  found = FALSE; 
  for (i = object_list; i; i = i->next_content) {
    if ((virtualrm = GET_OBJ_VNUM(i)) == -1)
      if (!(IS_SET(GET_OBJ_EXTRA(i), ITEM_CORPSE) && (i->obj_flags.owner !=
        GET_IDNUM(ch)) && i->obj_flags.owner && !IS_NPC(ch)
        && GET_LEVEL(ch) < 100))
      GET_OBJ_TIMER(i) = GET_OBJ_TIMER(i) - 1; 
    found = TRUE;
  }
  return;
}

ASPELL(spell_rowans_pizza)
{
 int throw_dir;
 int percent, prob;
 struct char_data *vict;
 
 if (!FIGHTING(ch)) { /* offensive only! */
   send_to_char("This is an offensive spell only.\r\n", ch);
   GET_MANA(ch) += cost;
   return;
 }
 vict = FIGHTING(ch);
if (tstring)
  throw_dir = search_block(tstring, dirs, FALSE);
 else 
  throw_dir = number(0, NUM_OF_DIRS-1);
 prob = GET_LEVEL(ch);
 percent = number(1,101);
if ((throw_dir < 0) || (percent > prob)) {
   throw_dir = number(0, NUM_OF_DIRS-1);
 } 
 sprintf(buf, "Trying '%s'\n\r", dirs[throw_dir]);
 send_to_char(buf, ch);
 
 sprintf(buf, "You watch in awe as a large pepperoni and pineapple pizza zooms by and leaves %s.\n\r",
         dirs[throw_dir]);
 send_to_room(buf, ch->in_room);

 if (!EXIT(ch, throw_dir) || EXIT(ch, throw_dir)->to_room == NOWHERE) {
   sprintf(buf, "You watch the pizza go &+R*SPLAT*&+w against the wall.\r\n");
   send_to_room(buf, ch->in_room);
   return;
 }
 
 if (IS_SET(EXIT(ch, throw_dir)->exit_info, EX_CLOSED)) {
   sprintf(buf, "You watch the pizza go &+R*SPLAT*&+w against the wall.\r\n");
   send_to_room(buf, ch->in_room);
   return;
 }
 
 if (ROOM_FLAGGED(EXIT(ch, throw_dir)->to_room, ROOM_NOMOB | ROOM_DEATH | 
     ROOM_HOUSE | ROOM_IMPL | ROOM_GODROOM | ROOM_SIMP | ROOM_CLANONLY))
   return;

 if (!mag_savingthrow(vict, SAVING_SPELL, 0)) {
   stop_fighting(vict);
   send_to_char("You see the pizza and must have it!  You start to chase the wonderful pie!\r\n", vict);
   act("$n sees the pizza and wanders off after it.", FALSE, vict, NULL, NULL, TO_ROOM);
   command_interpreter(vict, dir_abbrev[throw_dir]);
 } 
}

ASPELL(spell_paladin_fury)
{
  int hr, dr, factor, save_spell;
  struct affected_type af[3];
  
  if (ch->char_specials.daytimer && !IS_IMMORTAL(ch)) {
    send_to_char("You cannot call upon your god again so soon.\r\n", ch);
    GET_MANA(ch) += cost;
    return;
  }
  if (GET_MANA(ch) < 999) {
    send_to_char("You haven't the energy to cast that spell!\r\n", ch);
    return;
  }
  if (GET_MOVE(ch) < 40) {
    send_to_char("You haven't the stamina for that!\r\n", ch);
    return;
  }
  
/*  if (IS_AFFECTED(ch, AFF_HOLY_RAGE) && GET_ALIGN(ch) >=900)
    factor = 2;
  else */
    factor = 1;
  
  hr = 127 - GET_HITROLL(ch);
  dr = 127 - GET_DAMROLL(ch);
  save_spell = -127 - GET_SAVE(ch, SAVING_SPELL);
  
  
  af[0].location = APPLY_HITROLL;
  af[0].type = SPELL_PALADIN_FURY;
  af[0].bitvector = 0;
  af[0].modifier = hr;
  af[0].duration = 1;
  
  af[1].location = APPLY_DAMROLL;
  af[1].type = SPELL_PALADIN_FURY;
  af[1].bitvector = 0;
  af[1].modifier = dr;
  af[1].duration = 1;
  
  af[2].location = APPLY_SAVING_SPELL;
  af[2].type = SPELL_PALADIN_FURY;
  af[2].bitvector = 0;
  af[2].modifier = save_spell;
  af[2].duration = 1;
  
  affect_join(ch, af, 0, FALSE, 0, FALSE);
  affect_join(ch, af+1, 0, FALSE, 0, FALSE);
  affect_join(ch, af+2, 0, FALSE, 0, FALSE);

  send_to_char("You call upon your deity who blesses you with &+WPaladin's Fury&+w.\r\n", ch);
  act("$n calls upon $s deity who blesses $m with &+WPaladin's Fury&+w.",
      TRUE, ch, 0, 0, TO_ROOM);
  GET_MOVE(ch) -= 40;
  GET_MANA(ch) -= 999;
  ch->char_specials.daytimer = 10;
}

ASPELL(spell_blade_legend)
{
  struct obj_data *j;
  int manacost;
  
  if (ch->equipment[WEAR_WIELD]) {
    send_to_char("You're already wielding something!\r\n", ch);
    GET_MANA(ch) += cost;
    return;
  }
  
  j = read_object(1207, VIRTUALRM);
  if (!j) {
    send_to_char("A bug from the ethereal plane steals your blade!\r\n", ch);
    GET_MANA(ch) += cost;
    mudlog("SYSERR: Obj #1207 doesn't exist.  No Blade of Legend!", BRF, 
           LVL_IMMORT, TRUE);
    return;
  }
  GET_OBJ_VAL(j, 2) = 2;
  j->affected[0].location = APPLY_AC;
  SET_BIT(GET_OBJ_EXTRA(j), ITEM_CREATED);
  if (GET_LEVEL(ch) >= 90) {
    GET_OBJ_VAL(j, 1) = 100;
    j->affected[0].modifier = -50;
    manacost = 1999;
  } else if (GET_LEVEL(ch) >= 75) {
    GET_OBJ_VAL(j, 1) = 85;
    j->affected[0].modifier = -35;
    manacost = 1499;
  } else {
    GET_OBJ_VAL(j, 1) = 70;
    j->affected[0].modifier = -20;
    manacost = 999;
  }
  
/*  if (IS_AFFECTED(ch, AFF_HOLY_RAGE) && GET_ALIGN(ch) >=900)
    GET_OBJ_TIMER(j) = 6;
  else */
    GET_OBJ_TIMER(j) = 3;
    
  if (GET_MANA(ch) < manacost) {
    send_to_char("You haven't the energy to cast that spell!\r\n", ch);
    extract_obj(j);
    return;
  }
  
  GET_MANA(ch) -= manacost;  
  send_to_char("Your deity sends to you the &+WBlade of Legend&+w.\r\n", ch);
  act("$n is surrounded by a holy glow and receives the &+WBlade of Legend&+w.",
      FALSE, ch, 0, 0, TO_ROOM);
  GET_MANA(ch) -= manacost;  
  equip_char(ch, j, WEAR_WIELD);
}

ASPELL(spell_stratos_racing)
{
  struct char_data *vict, *cupra, *milk, *beach, *helmet;
  struct follow_type *k;
  int num, loaded = 0;
  ACMD(do_assist);
  
  if (!(vict = FIGHTING(ch))) {
    send_to_char("This spell can only be cast during battle.\r\n", ch);
    GET_MANA(ch) += cost;
    return;
  }
 
  if (ch->followers)
    for (k = ch->followers; k; k = k->next)
      if (MOB_FLAGGED(k->follower, MOB_CONJURE)) {
        send_to_char("You can only conjure once per battle.\r\n", ch);
        return;
      }
  num = number(1,4);
  
  if (number(1,100) > 25) { /* cupra */
    cupra = read_mobile(1460, VIRTUALRM);
    GET_MAX_HIT(cupra) = GET_MAX_HIT(ch);
    GET_LEVEL(cupra) = GET_LEVEL(ch);
    GET_HIT(cupra) = GET_MAX_HIT(cupra);
    SET_BIT(MOB_FLAGS(cupra), MOB_CONJURE);
    char_to_room(cupra, ch->in_room);
    add_follower(cupra, ch);
    do_assist(cupra, GET_NAME(ch), 0, 0);
    loaded++;
  }
  if ((number(1,100) > 25) && loaded < num) { /* milk truck */
    milk = read_mobile(1462, VIRTUALRM);
    GET_MAX_HIT(milk) = GET_MAX_HIT(ch);
    GET_LEVEL(milk) = GET_LEVEL(ch);
    GET_HIT(milk) = GET_MAX_HIT(milk);
    SET_BIT(MOB_FLAGS(milk), MOB_CONJURE);
    char_to_room(milk, ch->in_room);
    add_follower(milk, ch);
    do_assist(milk, GET_NAME(ch), 0, 0);
    loaded++;
  }
  if ((number(1,100) > 25) && loaded < num) { /* helmet */
    helmet = read_mobile(1463, VIRTUALRM);
    GET_MAX_HIT(helmet) = GET_MAX_HIT(ch);
    GET_LEVEL(helmet) = GET_LEVEL(ch);
    GET_HIT(helmet) = GET_MAX_HIT(helmet);
    SET_BIT(MOB_FLAGS(helmet), MOB_CONJURE);
    char_to_room(helmet, ch->in_room);
    add_follower(helmet, ch);
    do_assist(helmet, GET_NAME(ch), 0, 0);
    loaded++;
  }
  if ((number(1,100) > 25) && loaded < num) { /* beach ball */
    beach = read_mobile(1461, VIRTUALRM);
    GET_MAX_HIT(beach) = GET_MAX_HIT(ch);
    GET_LEVEL(beach) = GET_LEVEL(ch);
    GET_HIT(beach) = GET_MAX_HIT(beach);
    SET_BIT(MOB_FLAGS(beach), MOB_CONJURE);
    char_to_room(beach, ch->in_room);
    add_follower(beach, ch);
    do_assist(beach, GET_NAME(ch), 0, 0);
    loaded++;
  }
  sprintf(buf, "You grin as %d vehicles come to your aid.", loaded);
  act(buf, FALSE, ch, 0, 0, TO_CHAR);
  sprintf(buf, "From out of nowhere, %d strange vehicles come to $n's aid.",
          loaded);
  act(buf, FALSE, ch, 0, 0, TO_ROOM);
}

ASPELL(spell_portal)
 {

  int room;
  struct obj_data *portal;

  if (victim == NULL || ch == NULL)
    return;

  if ( (GET_LEVEL(victim) > level ) && IS_NPC(victim) )
  {
   act ("You cannot go to $M.", FALSE, ch, 0, victim, TO_CHAR);
   return;
  }
  if (MOB_FLAGGED(victim, MOB_NOASTRAL)) {
   act("You try to open a portal to $N, but $E closed it.\r\n",
       FALSE, ch, 0, victim, TO_CHAR);
   return;
  }
  if ( (GET_LEVEL(victim) >= LVL_IMMORT ) && !IS_NPC(victim) )
  {
   act("$N decided to ignore your feeble attempt at reaching $M.",
       FALSE, ch, 0, victim, TO_CHAR);
   act("$n attempted to open a portal to you.", FALSE, ch, 0, victim,
        TO_VICT);
   return;
  }
  room = victim->in_room;
  if ( ROOM_FLAGGED(room, ROOM_IMPL) ||
       ROOM_FLAGGED(room, ROOM_GODROOM) ||
       ROOM_FLAGGED(room, ROOM_NOASTRAL) ||
       ROOM_FLAGGED(room, ROOM_CLANONLY) ) {
    act("Your spell has been nullified by a greater (divine?) force.",
        FALSE, ch, 0, victim, TO_CHAR);
    return;
  }
  if ( ROOM_FLAGGED(room, ROOM_HOUSE) &&
      !House_can_enter(ch, world[room].number) ) {
    act("That's private property -- no trespassing!", FALSE, ch, 0,
        victim, TO_CHAR);
    return;
  }
  if ( (ROOM_FLAGGED(room, ROOM_PRIVATE)) &&
        world[room].people &&
        world[room].people->next_in_room ) {
     act("There's a private conversation going on in that room.", FALSE,
          ch, 0, victim, TO_CHAR);
     return;
  }
  portal = read_object(31, VIRTUALRM);
  if (!portal) {
    send_to_char("Your portal fails to open.\r\n", ch);
    GET_MANA(ch) += cost;
    mudlog("SYSERR: Obj #31 doesn't exist.  No portal!", BRF, LVL_IMMORT,
      TRUE);
    return;
  }
  GET_OBJ_TIMER(portal) = 2;
  SET_BIT(GET_OBJ_EXTRA(portal), ITEM_CREATED);
  GET_OBJ_VAL(portal, 0) = world[room].number;
  obj_to_room(portal, ch->in_room);
  act("A shimmering blue disc appears before you.", FALSE, ch, 0, victim, 
      TO_CHAR);
  act("$n opens a dimensional gateway.", TRUE, ch, 0, 0, TO_ROOM);
  
}
