/* ************************************************************************
*   File: structs.h                                     Part of CircleMUD *
*  Usage: header file for central structures and contstants               *
*                                                                         *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  Copyright (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */


/* preamble *************************************************************/

#define NOWHERE    -1    /* nil reference for room-database	*/
#define NOTHING	   -1    /* nil reference for objects		*/
#define NOBODY	   -1    /* nil reference for mobiles		*/

#define SPECIAL(name) \
   int (name)(struct char_data *ch, void *me, int cmd, char *argument)

/* function macro for special input parsers.  If player is playing
   (CON_PLAYING) and ch->player.parser isn't NULL, the special parser is
   called.  If it returns 0, player input follows it's normal course.  A
   nonzero value means the parser found something it was looking for and
   acted on it.  Your special parser, if coded in this way, will
   peacefully coexist with command_interpreter, and players can continue
   to use mud commands and communicate while they're considering an input
   to the parser.  The "data" is optional and allows you to use one
   parser to have several contexts (maybe you want it to ask two different
   yes/no answers).
   NOTE:  This was designed with the housing authority mobs in mind, but
   intended to be generic enough for use in other code.  If you're using
   a parser with a "me", you'll have to include some code in the parser
   itself to make sure "me" exists and is in the right place.
   BIG WARNING!!:  Make sure you don't mix up rooms/mobs, and objects in
   the ch->player.parse_me pointer.  As long as you don't try to use the
   same parser for more than one of these, you'll be ok.
*/  /*1.0*/ /*menumod*/
#define PARSER(name) \
   int (name)(struct char_data *ch, void *me, char *inp, void *data)

#define FORMAT_INDENT		(1 << 0)
/* room-related defines *************************************************/


/* The cardinal directions: used as index to room_data.dir_option[] */
#define NORTH          0
#define EAST           1
#define SOUTH          2
#define WEST           3
#define UP             4
#define DOWN           5
#define NORTHEAST      6
#define SOUTHEAST      7
#define SOUTHWEST      8
#define NORTHWEST      9

#define NUM_OF_DIRS	10	/* number of directions in a room (nsewud) */


/* Room flags: used in room_data.room_flags */
/* WARNING: In the world files, NEVER set the bits marked "R" ("Reserved") */
#define ROOM_DARK		(1 << 0)   /* Dark			*/
#define ROOM_DEATH		(1 << 1)   /* Death trap		*/
#define ROOM_NOMOB		(1 << 2)   /* MOBs not allowed		*/
#define ROOM_INDOORS		(1 << 3)   /* Indoors			*/
#define ROOM_PEACEFUL		(1 << 4)   /* Violence not allowed	*/
#define ROOM_SOUNDPROOF		(1 << 5)   /* Shouts, gossip blocked	*/
#define ROOM_NOTRACK		(1 << 6)   /* Track won't go through	*/
#define ROOM_NOMAGIC		(1 << 7)   /* Magic not allowed		*/
#define ROOM_TUNNEL		(1 << 8)   /* room for only 1 pers	*/
#define ROOM_PRIVATE		(1 << 9)   /* Can't teleport in		*/
#define ROOM_GODROOM		(1 << 10)  /* LVL_GOD+ only allowed	*/
#define ROOM_HOUSE		(1 << 11)  /* (R) Room is a house	*/
#define ROOM_HOUSE_CRASH	(1 << 12)  /* (R) House needs saving	*/
#define ROOM_ATRIUM		(1 << 13)  /* (R) The door to a house	*/
#define ROOM_OLC		(1 << 14)  /* (R) Modifyable/!compress	*/
#define ROOM_BFS_MARK		(1 << 15)  /* (R) breath-first srch mrk	*/
#define ROOM_IMPL               (1 << 16)  /* KAKO     Imp Only room    */
#define ROOM_NOASTRAL           (1 << 17)  /* KAKO   Can't astral there */
#define ROOM_CRIMEOK            (1 << 18)  /* KAKO You can pkill, etc.  */
#define ROOM_NOFIGHT            (1 << 19)  /* KAKO You cannot fight here*/
#define ROOM_SIMP               (1 << 20)  /* KAKO 105s only!           */
#define ROOM_NORECALL		(1 << 21)  /* KAKO can't use "recall"   */
#define ROOM_CLANONLY		(1 << 22)  /* Sam  *clanroom*           */
#define ROOM_ENH_HEAL		(1 << 23)  /* KAKO heal 20% faster here */
#define ROOM_DONATION		(1 << 24)  /* donated stuff goes here   */
#define ROOM_NOCAST             (1 << 25)  /* GtG potions ok but no cast */
#define ROOM_NOPOTION           (1 << 26)  /* GtG cast ok but no potions */
#define ROOM_DELETED            (1 << 27)  /* GTG to delete a room      */
#define ROOM_NOSUMMON           (1 << 28)  /* GtG can't be summoned from here */


/* Exit info: used in room_data.dir_option.exit_info */
#define EX_ISDOOR		(1 << 0)   /* Exit is a door		*/
#define EX_CLOSED		(1 << 1)   /* The door is closed	*/
#define EX_LOCKED		(1 << 2)   /* The door is locked	*/
#define EX_PICKPROOF		(1 << 3)   /* Lock can't be picked	*/
#define EX_SECRET               (1 << 4)   /* Exit is secret            */


/* Sector types: used in room_data.sector_type */
#define SECT_INSIDE          0		   /* Indoors			*/
#define SECT_CITY            1		   /* In a city			*/
#define SECT_FIELD           2		   /* In a field		*/
#define SECT_FOREST          3		   /* In a forest		*/
#define SECT_HILLS           4		   /* In the hills		*/
#define SECT_MOUNTAIN        5		   /* On a mountain		*/
#define SECT_WATER_SWIM      6		   /* Swimmable water		*/
#define SECT_WATER_NOSWIM    7		   /* Water - need a boat	*/
#define SECT_UNDERWATER	     8		   /* Underwater		*/
#define SECT_FLYING	     9		   /* Wheee!			*/
#define SECT_DESERT         10             /* In a desert -- KAKO       */


/* char and mob-related defines *****************************************/


/* PC classes */
#define CLASS_UNDEFINED		-1
#define CLASS_NEWBIE            0
#define CLASS_CITIZEN           0
#define CLASS_MAGIC_USER	1
#define CLASS_CLERIC		2
#define CLASS_THIEF		3
#define CLASS_WARRIOR		4
#define CLASS_PALADIN		5
#define CLASS_BARD		6
#define CLASS_APALADIN		7
#define CLASS_RANGER		8
#define CLASS_MONK		9
#define CLASS_SAMURAI		10
#define CLASS_COLORMAGE		11

#define NUM_CLASSES	  12  /* This must be the number of classes!! */

/* PC races */
#define RACE_UNDEFINED    -1
#define RACE_HUMAN         0
#define RACE_ELF           1
#define RACE_DWARF         2
#define RACE_HOBBIT        3
#define RACE_OGRE	   4
#define RACE_VAMPIRE       5
#define RACE_TROLL	   6
#define RACE_FAIRY	   7
#define RACE_DRAGON	   8
#define RACE_TIEFLING      9
#define RACE_THRIKREEN     10
#define RACE_GARGOYLE	   11

#define NUM_RACES          12

/* PC clans */
#define CLAN_UNDEFINED    -1
#define CLAN_NONE          0
#define CLAN_DKC           1
#define CLAN_RWC           2
#define CLAN_SDC           3
#define CLAN_KKC           4
#define CLAN_CDA           5
#define CLAN_ACOMA         6
#define CLAN_WRC	   7
#define CLAN_COS           8
#define CLAN_ETC           9
#define CLAN_LOTR          10
#define CLAN_JEDI          11
#define CLAN_WMS	   12
#define CLAN_TKC	   13
#define CLAN_WAR   	   14
#define CLAN_AG   	   15
#define CLAN_RDS   	   16
#define CLAN_CSR	   17
#define CLAN_CNW           18
#define CLAN_COTN          19 
#define CLAN_DOD           20
#define CLAN_TSO           21
#define CLAN_DAGGER        22
#define CLAN_DMA	   23

#define NUM_CLANS          24

/* NPC classes (currently unused - feel free to implement!) */
#define CLASS_DEFAULT           30
#define CLASS_AMORPHOUS		31
#define CLASS_AMPHIBIAN		32
#define CLASS_BIRD		33
#define CLASS_CREATED		34
#define CLASS_DEMONIC		35
#define CLASS_DINOSAUR		36
#define CLASS_DRACOIC		37
#define CLASS_ELEMENTAL		38
#define CLASS_FISH		39
#define CLASS_FAE		40
#define CLASS_GIANT		41
#define CLASS_GOBLINOID		42
#define CLASS_HUMANOID		43
#define CLASS_INSECT		44
#define CLASS_MAGICAL		45
#define CLASS_MAMMAL		46
#define CLASS_OUTER_PLANER	47
#define CLASS_PLANT		48
#define CLASS_REPTILE		49
#define CLASS_UNDEAD		50
/*zeditmod zone view flags */
#define ZFL_DOORS               (1 << 0)        /* show door reset */
#define ZFL_MOBS                (1 << 1)        /* show mob loads  */
#define ZFL_OLOADS              (1 << 2)        /* show obj loads  */
#define ZFL_OGIVES              (1 << 3)        /* show obj gives  */
#define ZFL_OEQUIPS             (1 << 4)        /* show obj equips */
#define ZFL_OPUTS               (1 << 5)        /* show obj in obj's*/
#define ZFL_REMOVES             (1 << 6)        /* show obj removes*/
#define ZFL_DEFAULT             (1 << 7)        /* show default mob*/
/* end zeditmod*/
 

/* Sex */
#define SEX_NEUTRAL   0
#define SEX_MALE      1
#define SEX_FEMALE    2


/* Positions */
#define POS_DEAD       0	/* dead			*/
#define POS_MORTALLYW  1	/* mortally wounded	*/
#define POS_INCAP      2	/* incapacitated	*/
#define POS_STUNNED    3	/* stunned		*/
#define POS_SLEEPING   4	/* sleeping		*/
#define POS_RESTING    5	/* resting		*/
#define POS_SITTING    6	/* sitting		*/
#define POS_FIGHTING   7	/* fighting		*/
#define POS_STANDING   8	/* standing		*/
#define POS_RIDING     9        /* riding               */
#define POS_RIDDEN    10        /* ridden (whee!)       */
 

/* Player flags: used by char_data.char_specials.act */
#define PLR_KILLER	(1 << 0)   /* Player is a player-killer		*/
#define PLR_THIEF	(1 << 1)   /* Player is a player-thief		*/
#define PLR_FROZEN	(1 << 2)   /* Player is frozen			*/
#define PLR_DONTSET     (1 << 3)   /* Don't EVER set (ISNPC bit)	*/
#define PLR_WRITING	(1 << 4)   /* Player writing (board/mail/olc)	*/
#define PLR_MAILING	(1 << 5)   /* Player is writing mail		*/
#define PLR_CRASH	(1 << 6)   /* Player needs to be crash-saved	*/
#define PLR_SITEOK	(1 << 7)   /* Player has been site-cleared	*/
#define PLR_NOSHOUT	(1 << 8)   /* Player not allowed to shout/goss	*/
#define PLR_NOTITLE	(1 << 9)   /* Player not allowed to set title	*/
#define PLR_DELETED	(1 << 10)  /* Player deleted - space reusable	*/
#define PLR_LOADROOM	(1 << 11)  /* Player uses nonstandard loadroom	*/
#define PLR_NOWIZLIST	(1 << 12)  /* Player shouldn't be on wizlist	*/
#define PLR_NODELETE	(1 << 13)  /* Player shouldn't be deleted	*/
#define PLR_INVSTART	(1 << 14)  /* Player should enter game wizinvis	*/
#define PLR_PLAYING	(1 << 15)  /* Player is currently playing	*/
/*     KAKO      */
#define PLR_AFK         (1 << 16)  /* Player is AFK                     */
#define PLR_BANNED      (1 << 17)  /* Player is banned			*/
/* meditmod */
#define PLR_OLEDIT      (1 << 18)  /* Player is OLC'ing                 */
#define PLR_PROBATION   (1 << 19)  /* Player is on probation            */
#define PLR_COMPETITION (1 << 20)  /* Player is in pkill competition    */
#define PLR_ELDER       (1 << 21)  /* Player is an inactive original god */
#define PLR_TAGGED      (1 << 22)  /* Player is IT                      */
#define PLR_AUTOBOT     (1 << 23)  /* Player is a bot                   */
#define PLR_NOFINGER    (1 << 24)  /* Player can't be fingered		*/


/* Mobile flags: used by char_data.char_specials.act */
#define MOB_SPEC         (1 << 0)  /* Mob has a callable spec-proc	*/
#define MOB_SENTINEL     (1 << 1)  /* Mob should not move		*/
#define MOB_SCAVENGER    (1 << 2)  /* Mob picks up stuff on the ground	*/
#define MOB_ISNPC        (1 << 3)  /* (R) Automatically set on all Mobs	*/
#define MOB_AWARE	 (1 << 4)  /* Mob can't be backstabbed		*/
#define MOB_AGGRESSIVE   (1 << 5)  /* Mob hits players in the room	*/
#define MOB_STAY_ZONE    (1 << 6)  /* Mob shouldn't wander out of zone	*/
#define MOB_WIMPY        (1 << 7)  /* Mob flees if severely injured	*/
#define MOB_AGGR_EVIL	 (1 << 8)  /* auto attack evil PC's		*/
#define MOB_AGGR_GOOD	 (1 << 9)  /* auto attack good PC's		*/
#define MOB_AGGR_NEUTRAL (1 << 10) /* auto attack neutral PC's		*/
#define MOB_MEMORY	 (1 << 11) /* remember attackers if attacked	*/
#define MOB_HELPER	 (1 << 12) /* attack PCs fighting other NPCs	*/
#define MOB_NOCHARM	 (1 << 13) /* Mob can't be charmed		*/
#define MOB_NOSUMMON	 (1 << 14) /* Mob can't be summoned		*/
#define MOB_NOSLEEP	 (1 << 15) /* Mob can't be slept		*/
#define MOB_NOBASH	 (1 << 16) /* Mob can't be bashed (e.g. trees)	*/
#define MOB_NOBLIND	 (1 << 17) /* Mob can't be blinded		*/
#define MOB_NOASTRAL     (1 << 18) /* Mob can't be astraled to          */
#define MOB_FAMILIAR     (1 << 19) /* Mob is a familiar -=FAMILIAR=-    */
#define MOB_CONJURE      (1 << 20) /* Mob is conjured... -=CONJURE=-    */
#define MOB_MOUNTABLE    (1 << 21) /* Mob can be mounted.. -=MOUNT=-    */
#define MOB_HUNTER       (1 << 22) /* Mob will hunt people              */
#define MOB_NOFIGHT      (1 << 23) /* Mob can't be attacked             */
#define MOB_NONSTRIKE    (1 << 24) /* Mob has better nstrike defense    */


/* Preference flags: used by char_data.player_specials.pref */
#define PRF_BRIEF       (1 << 0)  /* Room descs won't normally be shown	*/
#define PRF_COMPACT     (1 << 1)  /* No extra CRLF pair before prompts	*/
#define PRF_DEAF	(1 << 2)  /* Can't hear shouts			*/
#define PRF_NOTELL	(1 << 3)  /* Can't receive tells		*/
#define PRF_DISPHP	(1 << 4)  /* Display hit points in prompt	*/
#define PRF_DISPMANA	(1 << 5)  /* Display mana points in prompt	*/
#define PRF_DISPMOVE	(1 << 6)  /* Display move points in prompt	*/
#define PRF_AUTOEXIT	(1 << 7)  /* Display exits in a room		*/
#define PRF_NOHASSLE	(1 << 8)  /* Aggr mobs won't attack		*/
#define PRF_QUEST	(1 << 9)  /* On quest				*/
#define PRF_SUMMONABLE	(1 << 10) /* Can be summoned			*/
#define PRF_NOREPEAT	(1 << 11) /* No repetition of comm commands	*/
#define PRF_HOLYLIGHT	(1 << 12) /* Can see in dark			*/
#define PRF_COLOR_1	(1 << 13) /* Color (low bit)			*/
#define PRF_COLOR_2	(1 << 14) /* Color (high bit)			*/
#define PRF_NOWIZ	(1 << 15) /* Can't hear wizline			*/
#define PRF_LOG1	(1 << 16) /* On-line System Log (low bit)	*/
#define PRF_LOG2	(1 << 17) /* On-line System Log (high bit)	*/
#define PRF_NOAUCT	(1 << 18) /* Can't hear auction channel		*/
#define PRF_NOGOSS	(1 << 19) /* Can't hear gossip channel		*/
#define PRF_NOGRATZ	(1 << 20) /* Can't hear grats channel		*/
#define PRF_ROOMFLAGS	(1 << 21) /* Can see room flags (ROOM_x)	*/
#define PRF_NOHIT       (1 << 22) /* Char cannot be hit KAKO            */
#define PRF_DISPALIGN	(1 << 23) /* Disply alignment in prompt KAKO	*/
#define PRF_DISPTOGO	(1 << 24) /* Display exp to go KAKO		*/
#define PRF_AUTOLOOT    (1 << 25) /* Auto-looting KAKO                  */
#define PRF_NOCLAN      (1 << 26) /* Can't hear clan channel            */
#define PRF_ALLCLAN     (1 << 27) /* Get other clan messages?           */
#define PRF_AUTOTICK    (1 << 28) /* Auto-ticking (how'd you guess?)    */
#define PRF_AUTOSPLIT   (1 << 29) /* Auto-Split Kyle....Kalam           */
#define PRF_NOPAGE      (1 << 30) /* Can't be paged                     */
#define PRF_DISPMOBHP   (1 << 31) /* Display mobs HP durring fight Kalam*/
/* #define PRF_CONCENTRATE (1 << 32)  Player is concentrating            */

/* More pref flags: used in char_data.player_specials.pref2; */
#define PRF2_BD_CHK      (1 << 0)  /* Player gets board update notification */
#define PRF2_DISPROOM    (1 << 1)  /* Immort has room # in prompt */
#define PRF2_NOSPDBLD    (1 << 2)  /* Player doesn't want to speedbuild */
#define PRF2_DISPTIME    (1 << 3)  /* Player has time in prompt */
#define PRF2_NOMOUNT     (1 << 4)  /* Player doesn't want to be mounted */

/* Affect bits: used in char_data.char_specials.saved.affected_by */
/* WARNING: In the world files, NEVER set the bits marked "R" ("Reserved") */
#define AFF_BLIND             (1 << 0)	   /* (R) Char is blind		*/
#define AFF_INVISIBLE         (1 << 1)	   /* Char is invisible		*/
#define AFF_DETECT_ALIGN      (1 << 2)	   /* Char is sensitive to align*/
#define AFF_DETECT_INVIS      (1 << 3)	   /* Char can see invis chars  */
#define AFF_DETECT_MAGIC      (1 << 4)	   /* Char is sensitive to magic*/
#define AFF_SENSE_LIFE        (1 << 5)	   /* Char can sense hidden life*/
#define AFF_WATERWALK	      (1 << 6)	   /* Char can walk on water	*/
#define AFF_SANCTUARY         (1 << 7)	   /* Char protected by sanct.	*/
#define AFF_GROUP             (1 << 8)	   /* (R) Char is grouped	*/
#define AFF_CURSE             (1 << 9)	   /* Char is cursed		*/
#define AFF_INFRAVISION       (1 << 10)	   /* Char can see in dark	*/
#define AFF_POISON            (1 << 11)	   /* (R) Char is poisoned	*/
#define AFF_PROTECT_EVIL      (1 << 12)	   /* Char protected from evil  */
#define AFF_PROTECT_GOOD      (1 << 13)	   /* Char protected from good  */
#define AFF_SLEEP             (1 << 14)	   /* (R) Char magically asleep	*/
#define AFF_NOTRACK	      (1 << 15)	   /* Char can't be tracked	*/
#define AFF_BLUR	      (1 << 16)	   /* Char is blurry    	*/
#define AFF_WATERBREATH	      (1 << 17)	   /* Char can breath underH2O	*/
#define AFF_SNEAK             (1 << 18)	   /* Char can move quietly	*/
#define AFF_HIDE              (1 << 19)	   /* Char is hidden		*/
#define AFF_FLYING	      (1 << 20)	   /* Char is flying		*/
#define AFF_CHARM             (1 << 21)	   /* Char is charmed		*/
#define AFF_LIT		      (1 << 22)    /* Char is lit up            */
#define AFF_DET_DOOR          (1 << 23)    /* Char can sense secret doors */
#define AFF_GATED	      (1 << 24)    /* Char is gated? *shrug*    */
#define AFF_HASTE	      (1 << 25)    /* Char is hasty (haste spell) */
#define AFF_WEB               (1 << 26)    /* Char is webbed            */
#define AFF_SLOW              (1 << 27)    /* Char is slowed            */
#define AFF_BERSERK           (1 << 28)    /* Char is berserk           */
#define AFF_REVERSAL	      (1 << 29)    /* Char is reversed          */
#define AFF_NOWEB             (1 << 30)    /* dwix: Char cannot be webbed */
#define AFF_NOPOISON          (1 << 31)    /* dwix: Char cannot be poisoned */
/* note: Hmmm apparently (1 << 32) doesn't work.... so I moved 
  AFF_NOWEB to (1 << 30) */
/* note: since integers are 32 bits on this machine - these are all the 
   AFF_ flags possible.  noweb is not really useful - and should be easy to
   remove all references to it (the only case is in the web spell code)
   -- dwix */

/* a clean set of AFF flags - Sam */
#define AFF2_STONESKIN		(1 <<  0)	/* char has stoneskin */
#define AFF2_HOLY_AURA          (1 <<  1)       /* holy aura */
#define AFF2_DIVINE_PROT        (1 <<  2)       /* divine protection */
#define AFF2_BLINK              (1 <<  3)       /* blink spell */
#define AFF2_RESERVED           (1 << 31)       /* RESERVED for error check */

/* Modes of connectedness: used by descriptor_data.state */
#define CON_PLAYING	 0		/* Playing - Nominal state	*/
#define CON_CLOSE	 1		/* Disconnecting		*/
#define CON_GET_NAME	 2		/* By what name ..?		*/
#define CON_NAME_CNFRM	 3		/* Did I get that right, x?	*/
#define CON_PASSWORD	 4		/* Password:			*/
#define CON_NEWPASSWD	 5		/* Give me a password for x	*/
#define CON_CNFPASSWD	 6		/* Please retype password:	*/
#define CON_QSEX	 7		/* Sex?				*/
#define CON_QCLASS	 8		/* Class?			*/
#define CON_RMOTD	 9		/* PRESS RETURN after MOTD	*/
#define CON_MENU	 10		/* Your choice: (main menu)	*/
#define CON_EXDESC	 11		/* Enter a new description:	*/
#define CON_CHPWD_GETOLD 12		/* Changing passwd: get old	*/
#define CON_CHPWD_GETNEW 13		/* Changing passwd: get new	*/
#define CON_CHPWD_VRFY   14		/* Verify new password		*/
#define CON_DELCNF1	 15		/* Delete confirmation 1	*/
#define CON_DELCNF2	 16		/* Delete confirmation 2	*/
#define CON_QROLLS	 17		/* Rolling stats		*/
#define CON_QRACE	 18		/* Race?			*/
#define CON_MAIL_SUBJ	 19		/* Get mail subject		*/
#define CON_QANSI	 20		/* Ask for ANSI			*/


/* Character equipment positions: used as index for char_data.equipment[] */
/* NOTE: Don't confuse these constants with the ITEM_ bitvectors
   which control the valid places you can wear a piece of equipment */
#define WEAR_LIGHT      0
#define WEAR_FINGER_R   1
#define WEAR_FINGER_L   2
#define WEAR_NECK_1     3
#define WEAR_NECK_2     4
#define WEAR_BODY       5
#define WEAR_HEAD       6
#define WEAR_LEGS       7
#define WEAR_FEET       8
#define WEAR_HANDS      9
#define WEAR_ARMS      10
#define WEAR_SHIELD    11
#define WEAR_ABOUT     12
#define WEAR_WAIST     13
#define WEAR_WRIST_R   14
#define WEAR_WRIST_L   15
#define WEAR_WIELD     16
#define WEAR_HOLD      17

#define NUM_WEARS      18	/* This must be the # of eq positions!! */


/* object-related defines ********************************************/


/* Item types: used by obj_data.obj_flags.type_flag */
#define ITEM_LIGHT      1		/* Item is a light source	*/
#define ITEM_SCROLL     2		/* Item is a scroll		*/
#define ITEM_WAND       3		/* Item is a wand		*/
#define ITEM_STAFF      4		/* Item is a staff		*/
#define ITEM_WEAPON     5		/* Item is a weapon		*/
#define ITEM_FIREWEAPON 6		/* Unimplemented		*/
#define ITEM_MISSILE    7		/* Unimplemented		*/
#define ITEM_TREASURE   8		/* Item is a treasure, not gold	*/
#define ITEM_ARMOR      9		/* Item is armor		*/
#define ITEM_POTION    10 		/* Item is a potion		*/
#define ITEM_WORN      11		/* Unimplemented		*/
#define ITEM_OTHER     12		/* Misc object			*/
#define ITEM_TRASH     13		/* Trash - shopkeeps won't buy	*/
#define ITEM_TRAP      14		/* Unimplemented		*/
#define ITEM_CONTAINER 15		/* Item is a container		*/
#define ITEM_NOTE      16		/* Item is note 		*/
#define ITEM_DRINKCON  17		/* Item is a drink container	*/
#define ITEM_KEY       18		/* Item is a key		*/
#define ITEM_FOOD      19		/* Item is food			*/
#define ITEM_MONEY     20		/* Item is money (gold)		*/
#define ITEM_PEN       21		/* Item is a pen		*/
#define ITEM_BOAT      22		/* Item is a boat		*/
#define ITEM_FOUNTAIN  23		/* Item is a fountain		*/
#define ITEM_FLIGHT    24               /* KAKO item is for flying      */
#define ITEM_SCUBA     25               /* KAKO Item is SCUBA gear      */
#define ITEM_LOCKPICK  26               /* KAKO Item is lockpick        */
#define ITEM_GIFT      27		/* KAKO Item is a gift box!     */
#define ITEM_SEAT      28               /* CHARON Item is a seat        */
#define ITEM_LIV_ARMOR 29		/* Item is living armor		*/
#define ITEM_ECONOMIZER 30		/* Item is spell economizer	*/
#define ITEM_AMMO      31               /* rangemod Item is ammo        */
#define ITEM_BOW       32               /* rangemod Item is a bow       */
#define ITEM_GUN       33               /* rangemod Item is a gun       */
#define ITEM_QUIVER    34               /* rangemod Item is a quiver    */
#define ITEM_PILL      35               /* pillmod Item is a pill       */

#define AMMO_ARROW     1                /* Ammo is an arrow (bow)       */
#define AMMO_BULLET    2                /* Ammo is a bullet (gun)       */

#define GUN_NORMAL     1                /* Gun is normal double-action */
#define GUN_SHOTGUN    2                /* Gun is shotgun (spray)      */
#define GUN_SEMI       3                /* Gun is semi-automatic       */


/* mail flags: used by obj_data.mail.flags */
#define MAIL_NEW		(1 << 0)  /* mail is unread		*/

/* Take/Wear flags: used by obj_data.obj_flags.wear_flags */
#define ITEM_WEAR_TAKE		(1 << 0)  /* Item can be takes		*/
#define ITEM_WEAR_FINGER	(1 << 1)  /* Can be worn on finger	*/
#define ITEM_WEAR_NECK		(1 << 2)  /* Can be worn around neck 	*/
#define ITEM_WEAR_BODY		(1 << 3)  /* Can be worn on body 	*/
#define ITEM_WEAR_HEAD		(1 << 4)  /* Can be worn on head 	*/
#define ITEM_WEAR_LEGS		(1 << 5)  /* Can be worn on legs	*/
#define ITEM_WEAR_FEET		(1 << 6)  /* Can be worn on feet	*/
#define ITEM_WEAR_HANDS		(1 << 7)  /* Can be worn on hands	*/
#define ITEM_WEAR_ARMS		(1 << 8)  /* Can be worn on arms	*/
#define ITEM_WEAR_SHIELD	(1 << 9)  /* Can be used as a shield	*/
#define ITEM_WEAR_ABOUT		(1 << 10) /* Can be worn about body 	*/
#define ITEM_WEAR_WAIST 	(1 << 11) /* Can be worn around waist 	*/
#define ITEM_WEAR_WRIST		(1 << 12) /* Can be worn on wrist 	*/
#define ITEM_WEAR_WIELD		(1 << 13) /* Can be wielded		*/
#define ITEM_WEAR_HOLD		(1 << 14) /* Can be held		*/


/* Extra object flags: used by obj_data.obj_flags.extra_flags */
#define ITEM_GLOW          (1 << 0)	/* Item is glowing		*/
#define ITEM_HUM           (1 << 1)	/* Item is humming		*/
#define ITEM_NORENT        (1 << 2)	/* Item cannot be rented	*/
#define ITEM_NODONATE      (1 << 3)	/* Item cannot be donated	*/
#define ITEM_NOINVIS	   (1 << 4)	/* Item cannot be made invis	*/
#define ITEM_INVISIBLE     (1 << 5)	/* Item is invisible		*/
#define ITEM_MAGIC         (1 << 6)	/* Item is magical		*/
#define ITEM_NODROP        (1 << 7)	/* Item is cursed: can't drop	*/
#define ITEM_BLESS         (1 << 8)	/* Item is blessed		*/
#define ITEM_ANTI_GOOD     (1 << 9)	/* Not usable by good people	*/
#define ITEM_ANTI_EVIL     (1 << 10)	/* Not usable by evil people	*/
#define ITEM_ANTI_NEUTRAL  (1 << 11)	/* Not usable by neutral people	*/
#define ITEM_ANTI_MAGIC_USER (1 << 12)	/* Not usable by mages		*/
#define ITEM_ANTI_CLERIC   (1 << 13)	/* Not usable by clerics	*/
#define ITEM_ANTI_THIEF	   (1 << 14)	/* Not usable by thieves	*/
#define ITEM_ANTI_WARRIOR  (1 << 15)	/* Not usable by warriors	*/
#define ITEM_NOSELL	   (1 << 16)	/* Shopkeepers won't touch it	*/
#define ITEM_ANTI_MORT     (1 << 17)    /* Unusable by mortals KAKO     */
#define ITEM_ANTI_SMORT    (1 << 18)    /* Unusable by S_mortals KAKO   */
#define ITEM_ANTI_PALADIN  (1 << 19)    /* Not usable by paladins       */
#define ITEM_ANTI_BARD	   (1 << 20)	/* Not usable by bards		*/
#define ITEM_ANTI_APALADIN (1 << 21)    /* Not usable by anti paladins  */
#define ITEM_ANTI_RANGER   (1 << 22)    /* Not usable by rangers        */
#define ITEM_CREATED       (1 << 23)    /* Do not use!!!!!!!!!!!        */
#define ITEM_CORPSE        (1 << 24)    /* Do not use!! corpsemod       */
#define ITEM_ANTI_MONK     (1 << 25)    /* dwix: Not usable by monks    */
#define ITEM_MAIL	   (1 << 26)	/* item is mail.  don't touch   */
#define ITEM_TWOHAND       (1 << 27)    /* item requires 2 hands        */


/* Modifier constants used with obj affects ('A' fields) */
#define APPLY_NONE              0	/* No effect			*/
#define APPLY_STR               1	/* Apply to strength		*/
#define APPLY_DEX               2	/* Apply to dexterity		*/
#define APPLY_INT               3	/* Apply to constitution	*/
#define APPLY_WIS               4	/* Apply to wisdom		*/
#define APPLY_CON               5	/* Apply to constitution	*/
#define APPLY_CHA		6	/* Apply to charisma		*/
#define APPLY_CLASS             7	/* Reserved			*/
#define APPLY_LEVEL             8	/* Reserved			*/
#define APPLY_AGE               9	/* Apply to age			*/
#define APPLY_CHAR_WEIGHT      10	/* Apply to weight		*/
#define APPLY_CHAR_HEIGHT      11	/* Apply to height		*/
#define APPLY_MANA             12	/* Apply to max mana		*/
#define APPLY_HIT              13	/* Apply to max hit points	*/
#define APPLY_MOVE             14	/* Apply to max move points	*/
#define APPLY_GOLD             15	/* Reserved			*/
#define APPLY_EXP              16	/* Reserved			*/
#define APPLY_AC               17	/* Apply to Armor Class		*/
#define APPLY_HITROLL          18	/* Apply to hitroll		*/
#define APPLY_DAMROLL          19	/* Apply to damage roll		*/
#define APPLY_SAVING_PARA      20	/* Apply to save throw: paralz	*/
#define APPLY_SAVING_ROD       21	/* Apply to save throw: rods	*/
#define APPLY_SAVING_PETRI     22	/* Apply to save throw: petrif	*/
#define APPLY_SAVING_BREATH    23	/* Apply to save throw: breath	*/
#define APPLY_SAVING_SPELL     24	/* Apply to save throw: spells	*/


/* Container flags - value[1] */
#define CONT_CLOSEABLE      (1 << 0)	/* Container can be closed	*/
#define CONT_PICKPROOF      (1 << 1)	/* Container is pickproof	*/
#define CONT_CLOSED         (1 << 2)	/* Container is closed		*/
#define CONT_LOCKED         (1 << 3)	/* Container is locked		*/


/* Some different kind of liquids for use in values of drink containers */
#define LIQ_WATER      0
#define LIQ_BEER       1
#define LIQ_WINE       2
#define LIQ_ALE        3
#define LIQ_DARKALE    4
#define LIQ_WHISKY     5
#define LIQ_LEMONADE   6
#define LIQ_FIREBRT    7
#define LIQ_LOCALSPC   8
#define LIQ_SLIME      9
#define LIQ_MILK       10
#define LIQ_TEA        11
#define LIQ_COFFE      12
#define LIQ_BLOOD      13
#define LIQ_SALTWATER  14
#define LIQ_CLEARWATER 15
#define LIQ_COKE       16
#define LIQ_KOOLAID    17
#define LIQ_EGGNOG     18
#define LIQ_CIDER      19


/* other miscellaneous defines *******************************************/


/* Player conditions */
#define DRUNK        0
#define FULL         1
#define THIRST       2


/* Sun state for weather_data */
#define SUN_DARK	0
#define SUN_RISE	1
#define SUN_LIGHT	2
#define SUN_SET		3


/* Sky conditions for weather_data */
#define SKY_SIZZLING	0
#define SKY_SUNNY	1
#define SKY_CLEAR	2
#define SKY_PARTCLOUD	3
#define SKY_CLOUDY	4
#define SKY_RAINING	5
#define SKY_POURING	6
#define SKY_THUNDERSTORM	7
#define SKY_SNOWING	8
#define SKY_SNOWSTORM	9
#define SKY_BLIZZARD	10


/* Rent codes */
#define RENT_UNDEF      0
#define RENT_CRASH      1
#define RENT_RENTED     2
#define RENT_CRYO       3
#define RENT_FORCED     4
#define RENT_TIMEDOUT   5

/* File system consts */
#define FS_FILE_CHAR '#'
#define FS_DATA_CHAR '@'

#define FS_DIR_HOME 0
#define FS_DIR_PUB  1


/* other #defined constants **********************************************/

/*
 * **DO**NOT** blindly change the number of levels in your MUD merely by
 * changing these numbers and without changing the rest of the code to match.
 * Other changes throughout the code are required.  See coding.doc for
 * details.
 */
#define GRP_LIMIT	6
#define LVL_RECALL	5

#define LVL_SIMP        105    /*   Super Imp   */
#define LVL_IMPL	104    /*     KAKO      */
#define LVL_GRGOD	103    /* New God Lvls  */
#define LVL_GOD		102
#define LVL_IMMORT	101
#define LVL_DEMI	100

#define LVL_FREEZE	LVL_GRGOD

#define OPT_USEC	100000	/* 10 passes per second */
#define PASSES_PER_SEC	(1000000 / OPT_USEC)
#define RL_SEC		* PASSES_PER_SEC

#define PULSE_ZONE      (10 RL_SEC)
#define PULSE_MOBILE    (10 RL_SEC)
#define PULSE_VIOLENCE  (2 RL_SEC)

#define SMALL_BUFSIZE		1024
#define LARGE_BUFSIZE		(12 * 1024)
#define GARBAGE_SPACE		32

#define MAX_STRING_LENGTH	32768
#define MAX_INPUT_LENGTH	256	/* Max length per *line* of input */
#define MAX_RAW_INPUT_LENGTH	512	/* Max size of *raw* input */
#define MAX_MESSAGES		60
#define MAX_NAME_LENGTH		20  /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_PWD_LENGTH		10  /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_TITLE_LENGTH	80  /* Used in char_file_u *DO*NOT*CHANGE* */
#define HOST_LENGTH		30  /* Used in char_file_u *DO*NOT*CHANGE* */
#define EXDSCR_LENGTH		1024 /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_TONGUE		3   /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_SKILLS		200 /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_AFFECT		32  /* Used in char_file_u *DO*NOT*CHANGE* */
#define MAX_OBJ_AFFECT		6 /* Used in obj_file_elem *DO*NOT*CHANGE* */


/**********************************************************************
* Structures                                                          *
**********************************************************************/


typedef signed char		sbyte;
typedef unsigned char		ubyte;
typedef signed short int	sh_int;
typedef unsigned short int	ush_int;
typedef char			bool;
typedef char			byte;

typedef sh_int	room_num;
typedef sh_int	obj_num;


/* Extra description: used in objects, mobiles, and rooms */
struct extra_descr_data {
   char	*keyword;                 /* Keyword in look/examine          */
   char	*description;             /* What to see                      */
   struct extra_descr_data *next; /* Next in list                     */
};


/* object-related structures ******************************************/


/* object flags; used in obj_data */
struct obj_flag_data {
   int	value[5];	/* Values of the item (see list)    */
   byte type_flag;	/* Type of item			    */
   int	wear_flags;	/* Where you can wear it	    */
   int	extra_flags;	/* If it hums, glows, etc.	    */
   int	weight;		/* Weigt what else                  */
   int	cost;		/* Value when sold (gp.)            */
   int	cost_per_day;	/* Cost to keep pr. real day        */
   int	timer;		/* Timer for object                 */
   long	bitvector;	/* To set chars bits                */
   long bitvector2;     /* To set AFF2_ bits                */
   int  min_level;      /* KAKO minimum level to wear/wield */
   long owner;          /* corpsemod - corpse owner         */
   int min_remort;      /* minimum remorts to wer           */
};


/* Used in obj_file_elem *DO*NOT*CHANGE* */
struct obj_affected_type {
   byte location;      /* Which ability to change (APPLY_XXX) */
   sbyte modifier;     /* How much it changes by              */
};


/* ================== Memory Structure for Objects ================== */
struct obj_data {
   obj_num item_number;		/* Where in data-base			*/
   room_num in_room;		/* In what room -1 when conta/carr	*/

   struct obj_flag_data obj_flags;/* Object information               */
   struct obj_affected_type affected[MAX_OBJ_AFFECT];  /* affects */

   char	*name;                    /* Title of object :get etc.        */
   char	*description;		  /* When in room                     */
   char	*short_description;       /* when worn/carry/in cont.         */
   char	*action_description;      /* What to write when used          */
   struct extra_descr_data *ex_description; /* extra descriptions     */
   struct char_data *carried_by;  /* Carried by :NULL in room/conta   */
   struct char_data *sitter;    /* seatmod who is sitting on me?    */
   struct char_data *worn_by;	  /* Worn by?			      */
   sh_int worn_on;		  /* Worn where?		      */
   byte restring;                 /* object is restrung?              */

   struct obj_data *in_obj;       /* In what object NULL when none    */
   struct obj_data *contains;     /* Contains objects                 */

   struct obj_data *next_content; /* For 'contains' lists             */
   struct obj_data *next;         /* For the object list              */
   struct mail_data *mail;	  /* for letters/parcels	      */
};
/* ======================================================================= */

struct mail_data {
   long from;			  /* sender id				*/
   long to;			  /* recipient id			*/
   long flags;			  /* mail flags bitvector		*/
   char *subject;		  /* mail subject		 	*/
   time_t date;                   /* mail date sent                     */
};


/* ====================== File Element for Objects ======================= */
/*                 BEWARE: Changing it will ruin rent files		   */
struct obj_file_elem {
   obj_num item_number;

   int	value[4];
   int	extra_flags;
   int	weight;
   int	timer;
   long	bitvector;
   long bitvector2;
   long maker;
   byte restring;
   sh_int worn_on;
   sh_int is_container;
   sh_int is_last_content;
   char *text;					/* for notes/mail 	*/
   long from;					/* for mail	  	*/
   long to;					/* for mail		*/
   long mailflags;				/* for mail		*/
   time_t maildate;                             /* for mail             */
   char *subject;				/* for mail subject	*/
   char *short_desc;                            /* cstmod save short desc */
   char *long_desc;                              /* cstmod save long desc  */
   char *name;                                  /* cstmod save alias list */
   struct obj_affected_type affected[MAX_OBJ_AFFECT];
};


/* header block for rent files.  BEWARE: Changing it will ruin rent files  */
struct rent_info {
   int	time;
   int	rentcode;
   int	net_cost_per_diem;
   unsigned int	gold;
   int	account;
   int	nitems;
   int	spare0;
   int	spare1;
   int	spare2;
   int	spare3;
   int	spare4;
   int	spare5;
   int	spare6;
   int	spare7;
};
/* ======================================================================= */


/* room-related structures ************************************************/


struct room_direction_data {
   char	*general_description;       /* When look DIR.			*/

   char	*keyword;		/* for open/close			*/

   sh_int exit_info;		/* Exit info				*/
   obj_num key;			/* Key's number (-1 for no key)		*/
   room_num to_room;		/* Where direction leads (NOWHERE)	*/
};

struct damage_room_data {
  char *mesg;                   /* message for damage                   */
  int dam;                      /* Amount of damage (defaults to 30000) */
  long neutralizer;             /* neutralizer AFF_ flags               */
  long neutralizer2;            /* neutralizer AFF2_ flags              */
  int interval;                /* how often? (0 single, 1 CR, 2 tick)   */
};

struct prog_data {
  char *keyword;                /* keywords to trigger prog */
  struct prog_cmd_data *cmd;    /* commands in prog */
  struct prog_data *next;       /* next prog in list */
};

struct prog_cmd_data {
  char *command;                /* command to execute */
  struct prog_cmd_data *next;   /* next command */
};

/* ================== Memory Structure for room ======================= */
struct room_data {
   room_num number;		/* Rooms number	(vnum)		      */
   sh_int zone;                 /* Room zone (for resetting)          */
   int	sector_type;            /* sector type (move/hide)            */
   char	*name;                  /* Rooms name 'You are ...'           */
   char	*description;           /* Shown when entered                 */
   struct extra_descr_data *ex_description; /* for examine/look       */
   struct room_direction_data *dir_option[NUM_OF_DIRS]; /* Directions */
   struct damage_room_data damage; /* GtG damage room code */
   int room_flags;		/* DEATH,DARK ... etc                 */

   byte light;                  /* Number of lightsources in room     */
   SPECIAL(*func);

   struct obj_data *contents;   /* List of items in room              */
   struct char_data *people;    /* List of NPC / PC in room           */
   struct room_data *next_death; /* GPN link for room_damage_list       */

   struct prog_data *prog;       /* GPN linked list of triggers for progs */
};
/* ====================================================================== */


/* char-related structures ************************************************/


/* memory structure for characters */
struct memory_rec_struct {
   long	id;
   struct memory_rec_struct *next;
};

typedef struct memory_rec_struct memory_rec;


/* This structure is purely intended to be an easy way to transfer */
/* and return information about time (real or mudwise).            */
struct time_info_data {
   byte hours, day, month;
   sh_int year;
};


/* These data contain information about a players time data */
struct time_data {
   time_t birth;    /* This represents the characters age                */
   time_t logon;    /* Time of the last logon (used to calculate played) */
   int	played;     /* This is the total accumulated time played in secs */
};


/* general player-related info, usually PC's and NPC's */
struct char_player_data {
   char	passwd[MAX_PWD_LENGTH+1]; /* character's password      */
   char	*name;	       /* PC / NPC s name (kill ...  )         */
   char	*short_descr;  /* for NPC 'actions'                    */
   char	*long_descr;   /* for 'look'			       */
   char	*description;  /* Extra descriptions                   */
   char	*title;        /* PC / NPC's title                     */
   byte sex;           /* PC / NPC's sex                       */
   byte class;         /* PC / NPC's class		       */
   byte level;         /* PC / NPC's level                     */
   int remort;         /* PC / NPC's remort                    */
   int	hometown;      /* PC s Hometown (zone)                 */
   struct time_data time;  /* PC's AGE in days                 */
   ubyte weight;       /* PC / NPC's weight                    */
   ubyte height;       /* PC / NPC's height                    */
   int zone_edit;      /* PC Zone being edited *reditmod*      */
   int room_edit;      /* PC Zone being edited * .06 *         */
   int zone_flags;     /* PC Zone flags        *zeditmod*      */
   int default_zmob;   /* PC Zone mob default  *zeditmod*      */
   struct obj_data *obj_buffer;/*PC object edit buffer*oeditmod*/
   struct char_data *mob_buf;/*PC mob edit buffer     *meditmod*/
   sh_int default_mob; /* PC OLC default mob proto    *.06*    */
   sh_int default_obj; /* PC OLC default obj proto    *.06*    */
   sh_int default_room;/* PC OLC default room proto   *.06*    */
   struct cm_spell_data *spell_edit;/* PC CM OLC Spell	       */
   long idtag;         /* mob id tag *idmod*                   */
   long trust;         /* PC trusted by player *corpsemod*     */
   byte palming;       /* PC is attmpting a palm               */
   char *email;	       /* Player's email address	       */
   char *homepage;     /* Player's web address		       */
   int roundsout;      /* Number of rounds a char cannot attack */
   sh_int pagesize;    /* number of lines before pausing       */
   sh_int idleafk;     /* minutes until AFK flag is auto set   */
   /*menumod*/
   PARSER(*parser);    /* PC special parser function  *1.0*    */
   void *parse_me;     /* parser room/mob/obj context *1.0*    */
   void *parse_data;   /* parser extra data/context   *1.0*    */
   /*menumod end*/
   
};


/* Char's abilities.  Used in char_file_u *DO*NOT*CHANGE* */
struct char_ability_data {
   sbyte str;
   sbyte str_add;      /* 000 - 100 if strength 18             */
   sbyte intel;
   sbyte wis;
   sbyte dex;
   sbyte con;
   sbyte cha;
};


/* Char's points.  Used in char_file_u *DO*NOT*CHANGE* */
struct char_point_data {
   sh_int mana;
   sh_int max_mana;     /* Max move for PC/NPC			   */
   sh_int hit;
   sh_int max_hit;      /* Max hit for PC/NPC                      */
   sh_int move;
   sh_int max_move;     /* Max move for PC/NPC                     */

   sh_int armor;        /* Internal -100..100, external -10..10 AC */
   unsigned int	gold;           /* Money carried                           */
   unsigned int	bank_gold;	/* Gold the char has in a bank account	   */
   int	exp;            /* The experience of the player            */

   sbyte hitroll;       /* Any bonus or penalty to the hit roll    */
   sbyte damroll;       /* Any bonus or penalty to the damage roll */
};


/* 
 * char_special_data_saved: specials which both a PC and an NPC have in
 * common, but which must be saved to the playerfile for PC's.
 *
 * WARNING:  Do not change this structure.  Doing so will ruin the
 * playerfile.  If you want to add to the playerfile, use the spares
 * in player_special_data.
 */
struct char_special_data_saved {
   int	alignment;		/* +-1000 for alignments                */
   long	idnum;			/* player's idnum; -1 for mobiles	*/
   long	act;			/* act flag for NPC's; player flag for PC's */

   long	affected_by;		/* Bitvector for spells/skills affected by */
   long real_aff;               /* Bitvector for perm affects (affset)     */
   long	affected_by2;		/* Bitvector for spells/skills affected by */
   long real_aff2;              /* Bitvector for perm affects (affset)     */
   sh_int apply_saving_throw[5]; /* Saving throw (Bonuses)		*/
};


/* Special playing constants shared by PCs and NPCs which aren't in pfile */
struct char_special_data {
   struct char_data *fighting;	/* Opponent				*/
   struct char_data *hunting;	/* Char hunted by this char		*/
   byte blinkon;                /* char is blinked or not?              */

   byte position;		/* Standing, fighting, sleeping, etc.	*/

   int	carry_weight;		/* Carried weight			*/
   byte carry_items;		/* Number of items carried		*/
   int	timer;			/* Timer for update			*/
   int  daytimer;               /* used for once-a-day things           */

   struct char_special_data_saved saved; /* constants saved in plrfile	*/
};


/*
 *  If you want to add new values to the playerfile, do it here.  DO NOT
 * ADD, DELETE OR MOVE ANY OF THE VARIABLES - doing so will change the
 * size of the structure and ruin the playerfile.  However, you can change
 * the names of the spares to something more meaningful, and then use them
 * in your new code.  They will automatically be transferred from the
 * playerfile into memory when players log in.
 */
struct player_special_data_saved {
   byte skills[MAX_SKILLS+1];	/* array of skills plus skill 0		*/
   byte PADDING0;		/* used to be spells_to_learn		*/
   bool talks[MAX_TONGUE];	/* PC s Tongues 0 for NPC		*/
   int	wimp_level;		/* Below this # of hit points, flee!	*/
   byte freeze_level;		/* Level of god who froze char, if any	*/
   sh_int invis_level;		/* level of invisibility		*/
   room_num load_room;		/* Which room to place char in		*/
   long	pref;			/* preference flags for PC's.		*/
   long pref2;                  /* 2nd set of preference flags          */
   ubyte bad_pws;		/* number of bad password attemps	*/
   sbyte conditions[3];         /* Drunk, full, thirsty			*/
   char *host;

   /* spares below for future expansion.  You can change the names from
      'sparen' to something meaningful, but don't change the order.  */

   byte clan;
   byte race;
   ubyte spare2;
   ubyte spare3;
   ubyte spare4;
   ubyte spare5;
   int spells_to_learn;		/* How many can you learn yet this level*/
   int familiar;
   int pkill_points;
   int purity;		/* dwix: purity for monks */
   int quest;
   int spare11;
   int spare12;
   int spare13;
   int spare14;
   int spare15;
   int spare16;
   long	spare17;
   long	spare18;
   long	spare19;
   long	spare20;
   long	spare21;
};
/* IGNORE */
struct ignore_type {
   long idnum;
   struct ignore_type *next;
   struct ignore_type *prev;
};

/*
 * Specials needed only by PCs, not NPCs.  Space for this structure is
 * not allocated in memory for NPCs, but it is for PCs and the portion
 * of it labelled 'saved' is saved in the playerfile.  This structure can
 * be changed freely; beware, though, that changing the contents of
 * player_special_data_saved will corrupt the playerfile.
 */
struct player_special_data {
   int advance;
   int taggedby;

   struct player_special_data_saved saved;

   char	*poofin;		/* Description on arrival of a god.     */
   char	*poofout;		/* Description upon a god's exit.       */
   struct alias *aliases;	/* Character's aliases			*/
   long last_tell;		/* idnum of last tell from		*/
   void *last_olc_targ;		/* olc control				*/
   int last_olc_mode;		/* olc control				*/
   long pchan;                   /* Pchannels                            */
   struct obj_data *targobj;   /* object to be modified */

   /* dwix: begin */
   char *bindhb;                /* String to send to on heartbeat       */
   char *bindwimp;              /* string to send on wimp               */
   char *bindtell;              /* string to send on tell               */
   char *bindgt;                /* string to send on gt                 */

   char *last_msg;              /* String sent on the last message      */
   long last_msger;             /* idnum of the last messager           */

   char *prompt_string;         /* This fellows prompt                  */

   int pubdir;                  /* is the user in the public dir?       */
   /* dwix: end */
};


/* Specials used by NPCs, not PCs */
struct mob_special_data {
   byte last_direction;     /* The last direction the monster went     */
   int	attack_type;        /* The Attack Type Bitvector for NPC's     */
   byte default_pos;        /* Default position for NPC                */
   memory_rec *memory;	    /* List of attackers to remember	       */
   byte damnodice;          /* The number of damage dice's	       */
   byte damsizedice;        /* The size of the damage dice's           */
   int wait_state;	    /* Wait state for bashed mobs	       */
   struct char_data *target;/* Target for HUNTER mobs                  */ 
   int num_attacks;         /* Number of NPC attacks per round         */
};


/* An affect structure.  Used in char_file_u *DO*NOT*CHANGE* */
struct affected_type {
   sh_int type;          /* The type of spell that caused this      */
   sh_int duration;      /* For how long its effects will last      */
   sbyte modifier;       /* This is added to apropriate ability     */
   byte location;        /* Tells which ability to change(APPLY_XXX)*/
   long	bitvector;       /* Tells which bits to set (AFF_XXX)       */
   long	bitvector2;      /* Tells which bits to set (AFF2_XXX)      */

   struct affected_type *next;
};


/* Structure used for chars following other chars */
struct follow_type {
   struct char_data *follower;
   struct follow_type *next;
};


/* ================== Structure for player/non-player ===================== */
struct char_data {
   int pfilepos;			 /* playerfile pos		  */
   sh_int nr;                            /* Mob's rnum			  */
   room_num in_room;                     /* Location (real room number)	  */
   room_num was_in_room;		 /* location for linkdead people  */

   struct char_player_data player;       /* Normal data                   */
   struct char_ability_data real_abils;	 /* Abilities without modifiers   */
   struct char_ability_data aff_abils;	 /* Abils with spells/stones/etc  */
   struct char_point_data points;        /* Points                        */
   struct char_special_data char_specials;	/* PC/NPC specials	  */
   struct player_special_data *player_specials; /* PC specials		  */
   struct mob_special_data mob_specials;	/* NPC specials		  */

   struct affected_type *affected;       /* affected by what spells       */
   struct obj_data *equipment[NUM_WEARS];/* Equipment array               */

   struct obj_data *carrying;            /* Head of list                  */
   struct descriptor_data *desc;         /* NULL for mobiles              */
   struct ignore_type *ignore;

   struct obj_data *mail;		/* mail objects (in post office)  */

   struct char_data *next_in_room;     /* For room->people - list         */
   struct char_data *next;             /* For either monster or ppl-list  */
   struct char_data *next_fighting;    /* For fighting list               */

   struct follow_type *followers;        /* List of chars followers       */
   struct char_data *master;             /* Who is char following?        */
   struct char_data *riding;             /* MOUNT you are riding someone  */
   struct char_data *ridden_by;          /* MOUNT someone is riding you?  */
   struct obj_data *sitting_on;   /* seatmod What am i sitting on? */
   int mpactnum;
   int fight_round;                      /* For use with haste/slow       */
   byte quitting;			/* eeew it's a hack!		  */
};
/* ====================================================================== */


/* ==================== File Structure for Player ======================= */
/*             BEWARE: Changing it will ruin the playerfile		  */
struct char_file_u {
   /* char_player_data */
   char	name[MAX_NAME_LENGTH+1];
   char	description[EXDSCR_LENGTH];
   char	title[MAX_TITLE_LENGTH+1];
   byte sex;
   byte class;
   byte level;
   sh_int hometown;
   time_t birth;   /* Time of birth of character     */
   int	played;    /* Number of secs played in total */
   ubyte weight;
   ubyte height;
   int remort;

   char	pwd[MAX_PWD_LENGTH+1];    /* character's password */

   struct char_special_data_saved char_specials_saved;
   struct player_special_data_saved player_specials_saved;
   struct char_ability_data abilities;
   struct char_point_data points;
   struct affected_type affected[MAX_AFFECT];

   time_t last_logon;		/* Time (in secs) of last logon */
   char host[HOST_LENGTH+1];	/* host of last logon */
   char *email;			/* player email address */
   char *homepage;		/* player web address */
   sh_int pagesize;
   sh_int idleafk;
};
/* ====================================================================== */


/* descriptor-related structures ******************************************/


struct txt_block {
   char	*text;
   int aliased;
   struct txt_block *next;
};


struct txt_q {
   struct txt_block *head;
   struct txt_block *tail;
};


struct descriptor_data {
   int	descriptor;		/* file descriptor for socket		*/
   char	host[HOST_LENGTH+1];	/* hostname				*/
   char ip[16];                 /* ip address for verification          */
   byte	bad_pws;		/* number of bad pw attemps this login	*/
   byte idle_tics;		/* tics idle at password prompt		*/
   int	connected;		/* mode of 'connectedness'		*/
   int	wait;			/* wait for how many loops		*/
   int	desc_num;		/* unique num assigned to desc		*/
   time_t login_time;		/* when the person connected		*/
   char *showstr_head;		/* for keeping track of an internal str	*/
   char **showstr_vector;	/* for paging through texts		*/
   int  showstr_count;		/* number of pages to page through	*/
   int  showstr_page;		/* which page are we currently showing?	*/
   char	**str;			/* for the modify-str system		*/
   char *backstr;               /* added for handling abort buffers     */
   int	max_str;		/*		-			*/
   long	mail_to;		/* name for mail system			*/
   char *subj;			/* subject for mail system		*/
   int	prompt_mode;		/* control of prompt-printing		*/
   char	inbuf[MAX_RAW_INPUT_LENGTH];  /* buffer for raw input		*/
   char	last_input[MAX_INPUT_LENGTH]; /* the last input			*/
   char small_outbuf[SMALL_BUFSIZE];  /* standard output buffer		*/
   char *output;		/* ptr to the current output buffer	*/
   int  bufptr;			/* ptr to end of current output		*/
   int	bufspace;		/* space left in the output buffer	*/
   struct txt_block *large_outbuf; /* ptr to large buffer, if we need it */
   struct txt_q input;		/* q of unprocessed input		*/
   struct char_data *character;	/* linked to char			*/
   struct char_data *original;	/* original char if switched		*/
   struct descriptor_data *snooping; /* Who is this char snooping	*/
   struct descriptor_data *snoop_by; /* And who is snooping this char	*/
   struct descriptor_data *next; /* link to next descriptor		*/
   char dns_is_cached;		 /* dns cache lookup worked    * dnsmod */
   int enhanced_display;         /* enhanced display?                   */
};


/* other miscellaneous structures ***************************************/


struct msg_type {
   char	*attacker_msg;  /* message to attacker */
   char	*victim_msg;    /* message to victim   */
   char	*room_msg;      /* message to room     */
};


struct message_type {
   struct msg_type die_msg;	/* messages when death			*/
   struct msg_type miss_msg;	/* messages when miss			*/
   struct msg_type hit_msg;	/* messages when hit			*/
   struct msg_type god_msg;	/* messages when hit on god		*/
   struct message_type *next;	/* to next messages of this kind.	*/
};


struct message_list {
   int	a_type;			/* Attack type				*/
   int	number_of_attacks;	/* How many attack messages to chose from. */
   struct message_type *msg;	/* List of messages.			*/
};


struct dex_skill_type {
   sh_int p_pocket;
   sh_int p_locks;
   sh_int traps;
   sh_int sneak;
   sh_int hide;
};


struct dex_app_type {
   sh_int reaction;
   sh_int miss_att;
   sh_int defensive;
};


struct str_app_type {
   sh_int tohit;    /* To Hit (THAC0) Bonus/Penalty        */
   sh_int todam;    /* Damage Bonus/Penalty                */
   sh_int carry_w;  /* Maximum weight that can be carrried */
   sh_int wield_w;  /* Maximum weight that can be wielded  */
};


struct wis_app_type {
   byte bonus;       /* how many practices player gains per lev */
};


struct int_app_type {
   byte learn;       /* how many % a player learns a spell/skill */
};


struct con_app_type {
   sh_int hitp;
   sh_int shock;
};


struct weather_data {
   int	humidity;	/* How humid is it in % */
   int	change;	/* How fast and what way does it change. */
   int	sky;	/* How is the sky. */
   int	sunlight;	/* And how much sun. */
};


struct title_type {
   char	*title_m;
   char	*title_f;
   int	exp;
};

struct clan_stuff {
   char *abbrev;
   char *clan_short;
   char *clan_long; 
   int donate;
};

/* element in monster and object index-tables   */
struct index_data {
   int	virtualrm;    /* virtual number of this mob/obj           */
   int	number;     /* number of existing units of this mob/obj	*/
   SPECIAL(*func);
   byte deleted;    /* deleteme flag */
};

struct wholist_entry {
  long idnum;
  struct wholist_entry *next;
};
